/*
 * Decompiled with CFR 0.152.
 */
package fi.csc.chipster.rest.websocket;

import fi.csc.chipster.rest.CredentialsProvider;
import fi.csc.chipster.rest.websocket.RetryHandler;
import fi.csc.chipster.rest.websocket.WebSocketClientEndpoint;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeoutException;
import javax.websocket.ClientEndpointConfig;
import javax.websocket.CloseReason;
import javax.websocket.DeploymentException;
import javax.websocket.Endpoint;
import javax.websocket.EndpointConfig;
import javax.websocket.MessageHandler;
import javax.websocket.Session;
import javax.ws.rs.core.UriBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.glassfish.tyrus.client.ClientManager;

public class WebSocketClient
implements WebSocketClientEndpoint.EndpointListener {
    public static final Logger logger = LogManager.getLogger();
    private static final long PING_INTERVAL = 60000L;
    private String name;
    private ClientManager client;
    private WebSocketClientEndpoint endpoint;
    private RetryHandler retryHandler;
    private Timer pingTimer = new Timer("ping timer", true);
    private String uri;
    private MessageHandler.Whole<String> messageHandler;
    private CredentialsProvider credentials;

    public WebSocketClient(String uri, MessageHandler.Whole<String> messageHandler, boolean retry, String name, CredentialsProvider credentials) throws InterruptedException, WebSocketErrorException, WebSocketClosedException {
        this.name = name;
        this.uri = uri;
        this.messageHandler = messageHandler;
        this.credentials = credentials;
        if (retry) {
            this.retryHandler = new RetryHandler(name);
        }
        logger.info("websocket client " + name + " connecting to " + uri);
        this.connect();
    }

    private void connect() throws WebSocketErrorException, InterruptedException, WebSocketClosedException {
        ClientEndpointConfig cec = ClientEndpointConfig.Builder.create().build();
        this.client = ClientManager.createClient();
        try {
            UriBuilder uriBuilder = UriBuilder.fromUri((String)this.uri);
            if (this.credentials != null) {
                uriBuilder = uriBuilder.queryParam("token", new Object[]{this.credentials.getPassword().toString()});
            }
            this.endpoint = new WebSocketClientEndpoint(this.messageHandler, this);
            this.client.connectToServer((Endpoint)this.endpoint, cec, new URI(uriBuilder.toString()));
        }
        catch (IOException | URISyntaxException | DeploymentException e) {
            throw new WebSocketErrorException(e);
        }
        this.endpoint.waitForConnection();
        this.pingTimer.schedule(new TimerTask(){

            @Override
            public void run() {
                try {
                    WebSocketClient.this.ping();
                }
                catch (IOException | InterruptedException | TimeoutException e) {
                    logger.error("failed to send a ping", (Throwable)e);
                }
            }
        }, 60000L, 60000L);
    }

    public void waitForConnection() throws InterruptedException, WebSocketClosedException, WebSocketErrorException {
        if (this.endpoint == null) {
            throw new IllegalStateException("not connected");
        }
        this.endpoint.waitForConnection();
    }

    public void sendText(String text) throws InterruptedException, IOException {
        this.endpoint.sendText(text);
    }

    public void shutdown() throws IOException {
        logger.debug("shutdown websocket client " + this.name);
        if (this.retryHandler != null) {
            this.retryHandler.close();
        }
        this.pingTimer.cancel();
        this.endpoint.close();
        try {
            if (!this.endpoint.waitForDisconnect(1L)) {
                logger.warn("failed to close the websocket client " + this.name);
            }
        }
        catch (InterruptedException e) {
            logger.warn("failed to close the websocket client " + this.name, (Throwable)e);
        }
        this.client.shutdown();
    }

    public void ping() throws IOException, TimeoutException, InterruptedException {
        this.endpoint.ping();
    }

    @Override
    public void onOpen(Session session, EndpointConfig config) {
        logger.info("websocket client " + this.name + " connected succesfully: " + this.uri);
        if (this.retryHandler != null) {
            this.retryHandler.reset();
        }
    }

    @Override
    public void onClose(Session session, CloseReason reason) {
        logger.info("websocket client " + this.name + " closed: " + reason.getReasonPhrase());
        if (this.retryHandler != null) {
            while (this.retryHandler.onDisconnect(reason)) {
                try {
                    Thread.sleep(this.retryHandler.getDelay() * 1000L);
                    this.connect();
                    break;
                }
                catch (WebSocketClosedException | WebSocketErrorException | InterruptedException e) {
                    logger.error("error in reconnection", (Throwable)e);
                }
            }
        }
    }

    @Override
    public void onError(Session session, Throwable thr) {
        logger.warn("websocket client " + this.name + " error: " + thr.getMessage(), thr);
        if (this.retryHandler != null) {
            while (this.retryHandler.onConnectFailure((Exception)thr)) {
                try {
                    Thread.sleep(this.retryHandler.getDelay() * 1000L);
                    this.connect();
                    break;
                }
                catch (WebSocketClosedException | WebSocketErrorException | InterruptedException e) {
                    logger.error("error in reconnection", (Throwable)e);
                }
            }
        }
    }

    public static class WebSocketErrorException
    extends Exception {
        public WebSocketErrorException(Throwable throwable) {
            super(throwable);
        }
    }

    public static class WebSocketClosedException
    extends Exception {
        public WebSocketClosedException(CloseReason closeReason) {
            super(closeReason.getCloseCode() + closeReason.getReasonPhrase());
        }
    }
}

