/*
 * Decompiled with CFR 0.152.
 */
package fi.csc.chipster.auth.resource;

import fi.csc.chipster.auth.model.Token;
import fi.csc.chipster.auth.resource.AuthPrincipal;
import fi.csc.chipster.rest.RestUtils;
import fi.csc.chipster.rest.exception.NotAuthorizedException;
import fi.csc.chipster.rest.hibernate.HibernateUtil;
import fi.csc.chipster.rest.hibernate.Transaction;
import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import javax.annotation.security.RolesAllowed;
import javax.ws.rs.DELETE;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.GET;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.NotFoundException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Path(value="tokens")
public class TokenResource {
    public static final String TOKENS = "tokens";
    private static final String TOKEN_HEADER = "chipster-token";
    private static final Duration CLIENT_TOKEN_LIFETIME = Duration.of(3L, ChronoUnit.DAYS);
    private static final Duration CLIENT_TOKEN_MAX_LIFETIME = Duration.of(10L, ChronoUnit.DAYS);
    private static final Duration SERVER_TOKEN_LIFETIME = Duration.of(6L, ChronoUnit.HOURS);
    private static final Duration SERVER_TOKEN_MAX_LIFETIME = Duration.of(24L, ChronoUnit.HOURS);
    private Timer cleanUpTimer;
    private static Duration CLEAN_UP_INTERVAL = Duration.of(30L, ChronoUnit.MINUTES);
    private static Logger logger = LogManager.getLogger();
    private HibernateUtil hibernate;

    public TokenResource(HibernateUtil hibernate) {
        this.hibernate = hibernate;
        this.cleanUpTimer = new Timer("token db cleanup", true);
        this.cleanUpTimer.schedule(new TimerTask(){

            @Override
            public void run() {
                try {
                    TokenResource.this.cleanUp();
                }
                catch (Exception e) {
                    logger.warn("token clean up failed", (Throwable)e);
                }
            }
        }, 0L, CLEAN_UP_INTERVAL.toMillis());
    }

    @POST
    @RolesAllowed(value={"password"})
    @Produces(value={"application/json"})
    @Transaction
    public Response createToken(@Context SecurityContext sc) {
        AuthPrincipal principal = (AuthPrincipal)sc.getUserPrincipal();
        String username = sc.getUserPrincipal().getName();
        if (username == null) {
            throw new NotAuthorizedException("username is null");
        }
        Token token = this.createToken(username, principal.getRoles());
        this.getHibernate().session().save((Object)token);
        return Response.ok((Object)token).build();
    }

    public Token createToken(String username, HashSet<String> roles) {
        UUID tokenKey = RestUtils.createUUID();
        String rolesJson = RestUtils.asJson(roles);
        LocalDateTime now = LocalDateTime.now();
        Token token = new Token(username, tokenKey, now, now, rolesJson);
        token.setValid(this.getTokenNextExpiration(token));
        return token;
    }

    private void cleanUp() {
        Instant begin = Instant.now();
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime clientOldestValidCreationTime = now.minus(CLIENT_TOKEN_MAX_LIFETIME);
        LocalDateTime serverOldestValidCreationTime = now.minus(SERVER_TOKEN_MAX_LIFETIME);
        logger.info("cleaning up expired tokens");
        logger.info("client token max lifetime is " + CLIENT_TOKEN_MAX_LIFETIME + ", deleting client tokens created before " + clientOldestValidCreationTime);
        logger.info("server token max lifetime is " + SERVER_TOKEN_MAX_LIFETIME + " deleting server tokens created before " + serverOldestValidCreationTime);
        this.getHibernate().beginTransaction();
        List tokens = this.getHibernate().session().createQuery("from Token", Token.class).list();
        logger.info("before clean up db contains " + tokens.size() + " tokens");
        int deleteCount = 0;
        for (Token t : tokens) {
            if (t.getValid().isBefore(now)) {
                logger.info("deleting expired token " + t.getTokenKey() + " " + t.getUsername() + ", was valid until " + t.getValid());
                this.getHibernate().session().delete((Object)t);
                ++deleteCount;
                continue;
            }
            boolean delete = false;
            if (t.getRoles().contains("server")) {
                if (t.getCreationTime().isBefore(serverOldestValidCreationTime)) {
                    delete = true;
                }
            } else if (t.getCreationTime().isBefore(clientOldestValidCreationTime)) {
                delete = true;
            }
            if (!delete) continue;
            logger.info("deleting token " + t.getTokenKey() + " " + t.getUsername() + ", max life time reached, was created " + t.getCreationTime());
            this.getHibernate().session().delete((Object)t);
            ++deleteCount;
        }
        this.getHibernate().commit();
        logger.info("deleted " + deleteCount + " expired token(s) in " + Duration.between(begin, Instant.now()).toMillis() + " ms");
    }

    @GET
    @RolesAllowed(value={"server"})
    @Produces(value={"application/json"})
    @Transaction
    public Response checkToken(@HeaderParam(value="chipster-token") String requestToken, @Context SecurityContext sc) {
        Token dbToken = this.getToken(requestToken);
        if (dbToken.getValid().isAfter(LocalDateTime.now())) {
            return Response.ok((Object)dbToken).build();
        }
        throw new NotFoundException("token expired");
    }

    @POST
    @Path(value="{refresh}")
    @RolesAllowed(value={"client", "server"})
    @Produces(value={"application/json"})
    @Transaction
    public Response refreshToken(@Context SecurityContext sc) {
        String token = ((AuthPrincipal)sc.getUserPrincipal()).getTokenKey();
        Token dbToken = this.getToken(token);
        this.failIfTokenExpired(dbToken);
        if (dbToken.getUsername().equals("comp")) {
            logger.info("REFRESH " + dbToken.getValid() + " " + this.getTokenNextExpiration(dbToken));
        }
        dbToken.setValid(this.getTokenNextExpiration(dbToken));
        return Response.ok((Object)dbToken).build();
    }

    @DELETE
    @RolesAllowed(value={"client"})
    @Transaction
    public Response delete(@Context SecurityContext sc) {
        AuthPrincipal principal = (AuthPrincipal)sc.getUserPrincipal();
        UUID uuid = this.parseUUID(principal.getTokenKey());
        Token dbToken = (Token)this.getHibernate().session().get(Token.class, (Serializable)uuid);
        if (dbToken == null) {
            throw new NotFoundException();
        }
        this.getHibernate().session().delete((Object)dbToken);
        return Response.noContent().build();
    }

    private void failIfTokenExpired(Token token) {
        if (!token.getValid().isAfter(LocalDateTime.now())) {
            throw new ForbiddenException("token expired");
        }
        if (this.getTokenFinalExpiration(token).isBefore(LocalDateTime.now())) {
            throw new ForbiddenException("token expired, max lifetime reached");
        }
    }

    private LocalDateTime getTokenNextExpiration(Token token) {
        LocalDateTime finalExpiration;
        LocalDateTime nextCandidateExpiration = this.getTokenNextCandidateExpiration(token);
        return nextCandidateExpiration.isBefore(finalExpiration = this.getTokenFinalExpiration(token)) ? nextCandidateExpiration : finalExpiration;
    }

    private LocalDateTime getTokenNextCandidateExpiration(Token token) {
        return token.getRoles().contains("server") ? LocalDateTime.now().plus(SERVER_TOKEN_LIFETIME) : LocalDateTime.now().plus(CLIENT_TOKEN_LIFETIME);
    }

    private LocalDateTime getTokenFinalExpiration(Token token) {
        return token.getRoles().contains("server") ? token.getCreationTime().plus(SERVER_TOKEN_MAX_LIFETIME) : token.getCreationTime().plus(CLIENT_TOKEN_MAX_LIFETIME);
    }

    private UUID parseUUID(String token) {
        try {
            return UUID.fromString(token);
        }
        catch (IllegalArgumentException e) {
            throw new NotAuthorizedException("token is not a valid UUID");
        }
    }

    private Token getToken(String tokenString) {
        if (tokenString == null) {
            throw new NotFoundException("chipster-token header is null");
        }
        UUID uuid = this.parseUUID(tokenString);
        Token dbToken = (Token)this.getHibernate().session().get(Token.class, (Serializable)uuid);
        if (dbToken == null) {
            throw new NotFoundException("token not found");
        }
        return dbToken;
    }

    private HibernateUtil getHibernate() {
        return this.hibernate;
    }
}

