# TOOL ordination-pca.R: PCA (Principal component analysis. The number of principal component to save is controlled through the explained variablity. All principal components are saved until the explained variability is exceeded, but at least 3 components are always saved.)
# INPUT normalized.tsv: normalized.tsv TYPE GENE_EXPRS 
# OUTPUT pca.tsv: pca.tsv 
# OUTPUT variance.tsv: variance.tsv 
# OUTPUT loadings.tsv: loadings.tsv
# PARAMETER OPTIONAL expvar: "Amount of variation to explain" TYPE PERCENT DEFAULT 80 (Amount of experimental variation to explain)
# PARAMETER OPTIONAL scaling: Scaling TYPE [yes: yes, no: no] DEFAULT no (Scale the data to have a unit variance)
# PARAMETER OPTIONAL centering: Centering TYPE [yes: yes, no: no] DEFAULT yes (Scale the data to have the same mean)

# JTT, 27.6.2006
# MG, 29.3.2010 to enable coloring by cluster feature 
# IS, 12.10.2012 to cope with tables with gene descriptions (that typically contain 's)
# MK, 16.09.2013 modified to produce variance explanation table
# EK, 02.07.2014 modified to produce loadings table

# Parameter settings for testing purposes
# explained.variation<-c(80)
# scaling<-c("no")
# centering<-c("yes")

# Loads the normalized data
file <- 'normalized.tsv'
dat <- read.table(file, header=TRUE, sep='\t', quote='', row.names=1, check.names=FALSE)

# Separates expression values and other columns
dat2<-dat[,grep("chip", names(dat))]

# Transposes the data
dat2 <- t(dat2)

# PCA calculations
if(scaling=="yes" & centering=="yes") {
	pc<-prcomp(dat2, scale=T, center=T)
}
if(scaling=="yes" & centering=="no") {
	pc<-prcomp(dat2, scale=T, center=F)
}
if(scaling=="no" & centering=="yes") {
	pc<-prcomp(dat2, scale=F, center=T)
}
if(scaling=="no" & centering=="no") {
	pc<-prcomp(dat2, scale=F, center=F)
}

# How many PCs to save? If too low, three will be printed at minimum
no<-as.vector(head(which(summary(pc)$importance[3,]>=(expvar/100)), n=1)[1])
if(no<3) {
	no<-c(3)
}

# Converting PCs from matrix format into data frame
pcs<-as.data.frame(pc$x[,1:no])

# Giving the PC headers new names
names(pcs)<-paste("chip.", names(pcs), sep="")
pcs<-round(pcs,digits=2)

# Saving the PCs with data
write.table(data.frame(pcs), "pca.tsv", sep="\t", row.names=T, col.names=T, quote=F)

# Variance explained
write.table(round(data.frame(summary(pc)$importance[,1:no]),3), "variance.tsv", sep="\t", row.names=T, col.names=T, quote=F)

# Output component loadings. Add ".chip" to column names so that min and max can be easily found with the tool "Calculate descriptiver statistics".
loadings<-round(data.frame(pc$rotation[,1:no]),6)
colnames(loadings)<-paste("chip.", colnames(loadings), sep="")
write.table(loadings, "loadings.tsv", sep="\t", row.names=T, col.names=T, quote=F)

# EOF
