/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.chart.demo;


import java.io.*; 

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;

import org.jfree.chart.BioChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.event.ClusteringTreeChangeEvent;
import org.jfree.chart.event.PlotChangeEvent;
import org.jfree.chart.event.SelectionChangeEvent;
import org.jfree.chart.event.PlotChangeListener;
import org.jfree.chart.plot.HCPlot;
import org.jfree.chart.plot.HCTreeNodeInfo;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.data.hc.*;
import org.jfree.data.som.*;
import org.jfree.ui.ApplicationFrame;
import org.jfree.ui.RefineryUtilities;

import org.jfree.data.hc.*;

/**
 * A small demo for hierarchical clustering and heatmap.
 * @author  viski project
 */
public class HCDemo1 extends ApplicationFrame implements PlotChangeListener {

    /**
     * Creates a new demo instance.
     *
     * @param title  the frame title.
     */
    public HCDemo1(String title,String filename) {

        super(title);
        HCDataset dataset = createDataset(filename);
        JFreeChart chart = createChart(dataset);
	int width = dataset.getHeatMap().getColumnsCount()*40; // reasonable
	int height = dataset.getHeatMap().getRowCount()*40; // reasonable

	/* if we are working with large datasets, make them a bit smaller
	 * so we don't run out of memory. ~10MB is a reasonable
	 * upper limit.
	 */
	while (width * height > 10000000) {
	    width /= 2;
	    height /= 2;
	}
        ChartPanel chartPanel = new ChartPanel(
	    chart,
	    width,
	    height,
	    width/2,
	    height/2,
	    width*2,
	    height*2,
	    true,
	    true,
	    true,
	    true,
	    false,
	    true
	);
	JScrollPane scroll = new JScrollPane(chartPanel);
	HCPlot plot = (HCPlot)(chart.getPlot());
	plot.addChangeListener(this);
        chartPanel.addChartMouseListener((HCPlot)chart.getPlot());
	JPopupMenu popup = chartPanel.getPopupMenu();
	//popup 
	JMenuItem xItem = new JMenuItem( "Expand all" );
	//localizationResources.getString("Expand all")
	//xItem.setActionCommand(SAVE_COMMAND);
	//xItem.addActionListener(this);
	popup.add(xItem);

	chartPanel.setPopupMenu(popup);
        setContentPane(scroll);
	printHCDataset(dataset);

    }

    /**
     * Implements PlotChangeListener
     */
    public void plotChanged (PlotChangeEvent event) {

	if (event instanceof SelectionChangeEvent) {

	    HCPlot plot = (HCPlot)(event.getPlot());
	    System.out.println(plot.getSelection());

	} else if (event instanceof ClusteringTreeChangeEvent) {

	    HCTreeNodeInfo info =
		((ClusteringTreeChangeEvent)event).getTreeNodeInfo();
	    System.out.println("selection changed at subtree "+
		    info);
	    
	}

    }

    /**
     * Returns a sample dataset.
     * 
     * @return The dataset.
     */
    private static HCDataset createDataset(String filename) {
	BufferedReader input;
	HCDataset dataset;
	HeatMap heatMap;
	HCTreeNode leftTree;
	HCTreeNode topTree;
	
	try {
	    input =  
		new BufferedReader(
		    //new InputStreamReader(System.in)
		    new InputStreamReader(
			new FileInputStream(filename)
		    )
		);
	} catch (Exception e) {
	    System.out.println("1: "+e);
	    e.printStackTrace();
	    return null;
	}

	try {
	    leftTree = readTree(input,0);
	    topTree = readTree(input,0);
	} catch (Exception e) {
	    System.out.println("2: "+e);
	    e.printStackTrace();
	    leftTree = null;
	    topTree = null;
	}
	try {
	    heatMap = readHeatMap(input);
	} catch (Exception e) {
	    System.out.println("3: "+e);
	    e.printStackTrace();
	    heatMap = null;
	}
	try {
	    dataset = new HCDataset (heatMap,leftTree,topTree);
	} catch (Exception e) {
	    System.out.println("4: "+e);
	    e.printStackTrace();
	    dataset=null;
	}
	return dataset;
    }

    private static HeatMap readHeatMap(BufferedReader input)
	throws Exception {

        String line;
	HeatMap heatMap;
	int x,y, width, height;

	line = input.readLine();
	width = Integer.parseInt(line);
	line = input.readLine();
	height = Integer.parseInt(line);

	heatMap = new HeatMap ("X",height,width);

	for (x = 0; x < width; x++) {
	    for (y = 0; y < height; y++) {
		line = input.readLine();
		heatMap.update(y,x,Double.parseDouble(line));
	    }
	}
	for (x = 0; x < width; x++) {
	    heatMap.setColumnName(x,"t"+x);
	}
	for (y = 0; y < height; y++) {
	    int i=y;
	    String s="";
	    int j;
	    for (j = 0; j < 3; j++) {
		s = (char)((int)('A') + (i % 5)) + s;
		i/=5; // truncates decimals
	    }
	    heatMap.setRowName(y,"G_"+s);
	}

	return heatMap;
    }

    private static HCTreeNode readTree(BufferedReader input,int index)
	throws Exception {

        String line;
	HCTreeNode node;
	HCTreeNode leftChild;
	HCTreeNode rightChild;

	line = input.readLine();
	if (line.equals("x")) return null;

	if (line.equals("l")) return new HCTreeNode(0,index);

	node = new HCTreeNode(Double.parseDouble(line));
	leftChild = readTree(input, index);
	node.setLeftChild(leftChild);
	rightChild = readTree(input, node.getDataRange().getRightBound()+1);
	node.setRightChild(rightChild);

	return node;
    }

    private static void printTree(HCTreeNode node, int level) {
	int i;
	System.out.print("---+");
	if ((node.getLeftChild() == null) && (node.getRightChild() == null))
	    System.out.println(node.getDataRange());
	if (node.getLeftChild() != null) printTree(node.getLeftChild(),level+1);
	if (node.getRightChild() != null) {
	    for (i=0; i<level; i++) { System.out.print("   |"); }
	    System.out.print("   \\");
	    printTree(node.getRightChild(),level+1);
	}

    }

    private static void printHCDataset(HCDataset dataset) {

	/*
	int x,y,width,height;
	HeatMap heatMap;
	
	System.out.println("left tree:");
	printTree(dataset.getLeftClusteringTree(),0);
	System.out.println("top tree:");
	printTree(dataset.getTopClusteringTree(),0);

	heatMap = dataset.getHeatMap();
	width = heatMap.getColumnsCount();
	height = heatMap.getRowCount();
	System.out.println("heatmap(" + width + ", " + height + ")");
	for (x = 0; x < width; x ++) {
	    for (y = 0; y < height; y ++) {
		System.out.print(heatMap.get(y,x));
		System.out.print(", ");
	    }
	    System.out.println("");
	}
	*/

    }

    /**
     * Creates a sample chart.
     * 
     * @param dataset  the dataset.
     * 
     * @return The chart.
     */
    private static JFreeChart createChart(HCDataset dataset) {
        
        // create the chart...
        JFreeChart chart = BioChartFactory.createHCChart(
            "Hierarchical Clustering Demo",// chart title
            dataset,                  // data
            true,                     // tooltips?
            false                     // URLs?
        );
        
        return chart;
        
    }
    
    /**
     * Starting point for the demonstration application.
     *
     * @param args  command line arguments.
     */
    public static void main(String[] args) {

	HCDemo1 demo;
	if (args.length < 1) {
	    System.out.println("Please give datafile name as a commandline argument.");
	    System.exit(1);
	} else if (args.length > 1) {
	    System.out.println("No more than one command line argument is allowed.");
	    System.exit(1);
	}
	demo = new HCDemo1("HC Chart Demo",args[0]);
	demo.pack();
	RefineryUtilities.centerFrameOnScreen(demo);
	demo.setVisible(true);


    }

}
