/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.data.hc;

/**
 * A class that specifies an integer range.
 * @author  viski project
 */
public class DataRange implements Cloneable{

    private int leftBound;
    private int rightBound;

    /**
     * Creates a new real number range.
     *
     * @param leftBound  the minimum value.
     * @param rightBound  the maximum value.
     */
    public DataRange(int leftBound, int rightBound) {

	if (rightBound < leftBound) {
	    this.leftBound = Integer.MAX_VALUE;
	    this.rightBound = Integer.MIN_VALUE;
	} else {
	    this.leftBound = leftBound;
	    this.rightBound = rightBound;
	}

    }

    /**
     * Checks whether the range is empty. This is true, when
     * the maximum variable is less than the minimum.
     *
     * @return  A boolean.
     */
    public boolean isEmpty() {

	return (this.rightBound<this.leftBound);

    }

    /**
     * Returns the minimum value.
     *
     * @return  The minimum value.
     * @throws RuntimeException  if the minimum value is null ie. has not been set.
     */
    public int getLeftBound() {

	if (this.isEmpty()) throw new RuntimeException("getLeftBound() was called on an empty DataRange.");
	return this.leftBound;

    }
    
    /**
     * Returns the maximum value.
     *
     * @return  The maximum value.
     * @throws RuntimeException  if the maximum value is null ie. has not been set.
     */
    public int getRightBound() {

	if (this.isEmpty()) throw new RuntimeException("getRightBound() was called on an empty DataRange.");
	return this.rightBound;

    }
    
    /**
     * Returns a string with the current bounds.
     *
     * @return  The bounds.
     */
    public String toString() {
	if (this.isEmpty()) return "[]";
	return "[" + this.leftBound + "," + this.rightBound + "]";
    }

    /**
     * Returns the length of the interval.
     *
     * @return  The length of the interval or zero, if the range is empty.
     */
    public int getWidth() {

        if (this.isEmpty()) return 0;
        else return this.rightBound-this.leftBound+1;

    }

    /**
     * Checks whether a datarange contains another datarange.
     *
     * @param that  the datarange.
     * @return  A boolean.
     */
    public boolean contains(DataRange that) {

	// if that is empty, it is contained by any range.
	// This implies that empty ranges contain other empty
	// ranges. Probably what we want.
	if (that.isEmpty()) return true;

	// Empty ranges won't contain other empty ranges.
	if (this.isEmpty()) return false;

	// does it contain index?
	if (that.getLeftBound() < this.leftBound) return false;
	if (that.getRightBound() > this.rightBound) return false;

	// yes.
	return true;
    }
    
    /**
     * Checks whether an index is in range.
     *
     * @param index  the index.
     * @return  A boolean.
     */
    public boolean contains(int index) {

	// is the range empty?
	if (this.isEmpty()) return false;

	// does it contain index?
	if (index < this.leftBound) return false;
	if (index > this.rightBound) return false;

	// yes.
	return true;
    }
    
    /**
     * Imports the bounds of a DataRange-object to this object.
     * 
     * @param range  the range to be imported.
     */
    public void add(DataRange range) throws DataRangeMismatchException {

	int newLeft;
	int newRight;

	try {

	    newLeft = range.getLeftBound();
	    newRight = range.getRightBound();
	    
	} catch (Exception e) {

	    // the range to add is empty, don't do a thing
	    return;

	}

	// if this is empty, we simply copy the range to be added here.
	if (this.isEmpty()) {
	
	    this.leftBound = newLeft;
	    this.rightBound = newRight;

	} else {

	    if (newLeft < this.leftBound) {
		if (newRight < this.leftBound-1)
		    throw new DataRangeMismatchException();
		this.leftBound = newLeft;
	    }
	    if (newRight > this.rightBound) {
		if (newLeft > this.rightBound+1)
		    throw new DataRangeMismatchException();
		this.rightBound = newRight;
	    }

	}

    }

    /**
     * Returns a new DataRange object with the same bounds.
     *
     * @return  A new DataRange.
     */
    public Object clone() {

	return new DataRange(this.leftBound,this.rightBound);

    }

    /**
     * Checks whether two ranges are equal.
     *
     * @param range  the range to be tested with this one.
     * @return  A boolean.
     */
    public boolean equals(DataRange range) {
	if (this.isEmpty() && range.isEmpty()) return true;
	if (this.isEmpty() != range.isEmpty()) return false;
	if (range.getLeftBound() != this.getLeftBound()) return false;
	if (range.getRightBound() != this.getRightBound()) return false;
	return true;
    }

}
