/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart;

import java.math.BigDecimal;
import java.util.*;
import java.awt.Transparency;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;

import org.jfree.chart.encoders.EncoderUtil;
import org.jfree.chart.encoders.ImageFormat;

/**
 * A collection of utility methods for JFreeChart.  Includes methods for 
 * converting multiple charts to image formats (PNG and JPEG).
 */
public class MultiChartUtilities {
    
    /**
     * Saves multiple charts at the same time to a file in PNG-format.
     *
     * @param file  the file name (<code>null</code> not permitted).
     * @param charts  the list of charts (<code>null</code> not permitted).
     * @param width  the image width.
     * @param height  the image height.
     *
     * @throws IOException  if there are any I/O errors.
     * @throws IllegalArgumentException  if file or charts is null.
     */
    public static void saveChartsAsPNG(File file,
                           List charts,
                           int width,
                           int height) throws IOException {

        if (file == null) {
            throw new IllegalArgumentException("Null 'file' argument.");
        }
        if (charts == null) {
            throw new IllegalArgumentException("Null 'charts' argument.");
        }
        OutputStream out = new BufferedOutputStream(new FileOutputStream(file));
        writeChartsAsPNG(out, charts, width, height);
        out.close();
    }        

    /**
     * Writes the charts to an OutputStream.
     *
     * @param out  the output stream (<code>null</code> not permitted).
     * @param charts  the list of charts (<code>null</code> not permitted).
     * @param width  the image width.
     * @param height  the image height.
     *
     * @throws IOException if there are any I/O errors.
     * @throws IllegalArgumentException  if out or charts is null.
     */
    public static void writeChartsAsPNG(OutputStream out,
                           List charts,
                           int width,
                           int height) throws IOException {

        if (out == null) {
            throw new IllegalArgumentException("Null 'out' argument.");
        }
        if (charts == null) {
            throw new IllegalArgumentException("Null 'charts' argument.");
        }

        Iterator i = charts.iterator();

        //number of columns in "saving grid" (a changing method could be added)
        int saveColumns = 2;

        //number of rows in "saving grid"
        int saveRows = new BigDecimal((double)charts.size()).divide(
                                      new BigDecimal((double)saveColumns), BigDecimal.ROUND_UP).intValue();

        BufferedImage picture = new BufferedImage(width*saveColumns,height*saveRows,BufferedImage.TYPE_INT_ARGB);

        Graphics2D g2 = (Graphics2D)picture.createGraphics();

        int x = 0;
        int y = 0;

	while(i.hasNext()) {
            for(int j = 0; j < saveColumns; ++j) {
                if (i.hasNext()) {
                    BufferedImage chartImage = 
                    ((JFreeChart)i.next()).createBufferedImage(
                    width, height
                    );
                    g2.drawImage(chartImage, null, x, y);
                    x = x + width;
                }
            }
            x = 0;
            y = y + height;
        }
        g2.dispose();
        EncoderUtil.writeBufferedImage(
            picture, ImageFormat.PNG, out
        ); 
    }     

    /**
     * Saves multiple charts at the same time to a file in JPEG-format.
     *
     * @param file  the file name (<code>null</code> not permitted).
     * @param charts  the list of charts (<code>null</code> not permitted).
     * @param width  the image width.
     * @param height  the image height.
     *
     * @throws IOException if there are any I/O errors.
     * @throws IllegalArgumentException  if file or charts is null.
     */
    public static void saveChartsAsJPEG(File file,
                           List charts,
                           int width,
                           int height) throws IOException {

        if (file == null) {
            throw new IllegalArgumentException("Null 'file' argument.");
        }
        if (charts == null) {
            throw new IllegalArgumentException("Null 'charts' argument.");
        }
        OutputStream out = new BufferedOutputStream(new FileOutputStream(file));
        writeChartsAsJPEG(out, charts, width, height);
        out.close();
    }        

    /**
     * Writes the charts to an OutputStream.
     *
     * @param out  the output stream (<code>null</code> not permitted).
     * @param charts  the list of charts (<code>null</code> not permitted).
     * @param width  the image width.
     * @param height  the image height.
     *
     * @throws IOException if there are any I/O errors.
     * @throws IllegalArgumentException  if out or charts is null.
     */
    public static void writeChartsAsJPEG(OutputStream out,
                           List charts,
                           int width,
                           int height) throws IOException {

        if (out == null) {
            throw new IllegalArgumentException("Null 'out' argument.");
        }
        if (charts == null) {
            throw new IllegalArgumentException("Null 'charts' argument.");
        }

        Iterator i = charts.iterator();

        //number of columns in "saving grid" (a changing method could be added)
        int saveColumns = 2;

        //number of rows in "saving grid"
        int saveRows = new BigDecimal((double)charts.size()).divide(
                                      new BigDecimal((double)saveColumns), BigDecimal.ROUND_UP).intValue();


        BufferedImage picture = new BufferedImage(width*saveColumns,height*saveRows,BufferedImage.TYPE_INT_ARGB);

        Graphics2D g2 = (Graphics2D)picture.createGraphics();

        int x = 0;
        int y = 0;

	while(i.hasNext()) {
            for(int j = 0; j < saveColumns; ++j) {
                if (i.hasNext()) {
                    BufferedImage chartImage = 
                    ((JFreeChart)i.next()).createBufferedImage(
                    width, height
                    );
                    g2.drawImage(chartImage, null, x, y);
                    x = x + width;
                }
            }
            x = 0;
            y = y + height;
        }
	g2.dispose();

        EncoderUtil.writeBufferedImage(
            picture, ImageFormat.JPEG, out
        );
   }     
}

