/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.chart.demo;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;

import org.jfree.chart.BioChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.SOMPlot;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.data.som.*;
import org.jfree.ui.ApplicationFrame;
import org.jfree.ui.RefineryUtilities;


/**
 * Demo program for SOM charts
 */
public class SOMChartDemo1 extends ApplicationFrame {

    public static final int DEFAULT_WIDTH = 10;
    public static final int DEFAULT_HEIGHT = 10;
    
    /**
     * Creates a new demo instance.
     *
     * @param title  the frame title.
     */
    public SOMChartDemo1(String title, int columns, int rows) {

        super(title);
        SOMDataset dataset = createDataset(columns, rows);
        JFreeChart chart = createChart(dataset);
        ChartPanel chartPanel = new ChartPanel(chart, false);
        chartPanel.addChartMouseListener((SOMPlot)chart.getPlot());
        chartPanel.setFocusable(true);
        chartPanel.setPreferredSize(new Dimension(400, 400));
        setContentPane(chartPanel);

    }

    /**
     * Returns a random sample dataset.
     * 
     * @return The dataset.
     */
    public static SOMDataset createDataset() {
        return SOMChartDemo1.createDataset(DEFAULT_WIDTH, DEFAULT_HEIGHT);
    }
    
    /**
     * Returns a random sample dataset.
     * 
     * @param columns number of columns.
     * @param rows  number of rows.
     * @return The dataset.
     */
    public static SOMDataset createDataset(int columns, int rows) {
        SOMDataset dataset = null;
        try {
            dataset = new SOMDataset(columns, rows);
            
            for (int i=0; i < rows; ++i) {
                for (int j=0; j < columns; ++j) {
                    String[] s = new String[] {"BLUE", "private static SOMDataset createDataset()"};
                    double[] w = {Math.random(), Math.random(), Math.random()};
                    float r = (float)Math.random();
                    float g = (float)Math.random();
                    float b = (float)Math.random();
                    Color clr = new Color(r,g,b);
                    SOMDataItem sdi = new SOMDataItem(clr, s, w);
                    dataset.addValue(j, i, sdi);
                }
            }
            
        } catch (Exception e) {
            System.err.println("SOM data initialization failed.");
            System.exit(1);
        }
        return dataset;
        
    }
    
    /**
     * Creates a sample chart.
     * 
     * @param dataset  the dataset.
     * 
     * @return The chart.
     */
    private static JFreeChart createChart(SOMDataset dataset) {
        
        // create the chart...
        JFreeChart chart = BioChartFactory.createSOMChart(
            "SOM Chart Demo",         // chart title
            dataset,                  // data
            true,                     // tooltips?
            false                     // URLs?
        );
        
        return chart;
        
    }
    
    /**
     * Starting point for the demonstration application.
     *
     * @param args  ignored.
     */
    public static void main(String[] args) {

        int columns = DEFAULT_WIDTH;
        int rows = DEFAULT_HEIGHT;
        if (args.length > 0) {
            try {
            columns = Integer.parseInt(args[0], 10);
            } catch (NumberFormatException e) {
                ;
            }
        }
        if (args.length > 1) {
            try {
            rows = Integer.parseInt(args[1], 10);
            } catch (NumberFormatException e) {
                ;
            }
        }
        
        SOMChartDemo1 demo = new SOMChartDemo1("SOM Chart Demo", columns, rows);
        demo.pack();
        RefineryUtilities.centerFrameOnScreen(demo);
        demo.setVisible(true);

    }

}
