/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.analysis;

import java.io.File;
import java.util.List;
import net.sf.picard.analysis.AlignmentSummaryMetrics;
import net.sf.picard.cmdline.CommandLineProgram;
import net.sf.picard.cmdline.Option;
import net.sf.picard.cmdline.Usage;
import net.sf.picard.io.IoUtil;
import net.sf.picard.metrics.AggregateMetricCollector;
import net.sf.picard.metrics.MetricCollector;
import net.sf.picard.metrics.MetricsFile;
import net.sf.picard.reference.ReferenceSequenceFileWalker;
import net.sf.picard.util.CollectionUtil;
import net.sf.picard.util.Histogram;
import net.sf.picard.util.IlluminaUtil;
import net.sf.picard.util.Log;
import net.sf.samtools.AlignmentBlock;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceDictionary;
import net.sf.samtools.util.CoordMath;
import net.sf.samtools.util.SequenceUtil;
import net.sf.samtools.util.StringUtil;

public class CollectAlignmentSummaryMetrics
extends CommandLineProgram {
    private static final int MAPPING_QUALITY_THRESHOLD = 20;
    private static final int BASE_QUALITY_THRESHOLD = 20;
    private static final int ADAPTER_MATCH_LENGTH = 16;
    private static final int MAX_ADAPTER_ERRORS = 1;
    private byte[][] ADAPTER_SEQUENCES;
    private static final Log log = Log.getInstance(CollectAlignmentSummaryMetrics.class);
    @Usage
    public String USAGE = "Reads a SAM or BAM file and writes a file containing summary alignment metrics.\n";
    @Option(shortName="I", doc="SAM or BAM file")
    public File INPUT;
    @Option(shortName="O", doc="File to write insert size metrics to")
    public File OUTPUT;
    @Option(shortName="R", doc="Reference sequence file")
    public File REFERENCE_SEQUENCE;
    @Option(doc="If true (default), \"unsorted\" SAM/BAM files will be considerd coordinate sorted", shortName="AS")
    public Boolean ASSUME_SORTED = Boolean.TRUE;
    @Option(doc="Paired end reads above this insert size will be considered chimeric along with inter-chromosomal pairs.")
    public int MAX_INSERT_SIZE = 100000;
    @Option
    public List<String> ADAPTER_SEQUENCE = CollectionUtil.makeList(IlluminaUtil.AdapterPair.SINGLE_END.get5PrimeAdapter(), IlluminaUtil.AdapterPair.SINGLE_END.get3PrimeAdapter(), IlluminaUtil.AdapterPair.PAIRED_END.get5PrimeAdapter(), IlluminaUtil.AdapterPair.PAIRED_END.get3PrimeAdapter(), IlluminaUtil.AdapterPair.INDEXED.get5PrimeAdapter(), IlluminaUtil.AdapterPair.INDEXED.get3PrimeAdapter());
    @Option(shortName="BS", doc="Whether the SAM or BAM file consists of bisulfite sequenced reads.  ")
    public boolean IS_BISULFITE_SEQUENCED = false;
    private ReferenceSequenceFileWalker referenceSequenceWalker;
    private SAMFileHeader samFileHeader;

    public static void main(String[] argv) {
        System.exit(new CollectAlignmentSummaryMetrics().instanceMain(argv));
    }

    @Override
    protected int doWork() {
        this.prepareAdapterSequences();
        IoUtil.assertFileIsReadable(this.INPUT);
        IoUtil.assertFileIsReadable(this.REFERENCE_SEQUENCE);
        IoUtil.assertFileIsWritable(this.OUTPUT);
        SAMFileReader in = new SAMFileReader(this.INPUT);
        this.assertCoordinateSortOrder(in);
        this.referenceSequenceWalker = new ReferenceSequenceFileWalker(this.REFERENCE_SEQUENCE);
        this.samFileHeader = in.getFileHeader();
        if (!this.samFileHeader.getSequenceDictionary().isEmpty()) {
            SequenceUtil.assertSequenceDictionariesEqual((SAMSequenceDictionary)this.samFileHeader.getSequenceDictionary(), (SAMSequenceDictionary)this.referenceSequenceWalker.getSequenceDictionary());
        } else {
            log.warn(this.INPUT.getAbsoluteFile() + " has no sequence dictionary.  If any reads " + "in the file are aligned then alignment summary metrics collection will fail.");
        }
        MetricCollector<AlignmentSummaryMetrics, SAMRecord> unpairedCollector = this.constructCollector(AlignmentSummaryMetrics.Category.UNPAIRED);
        MetricCollector<AlignmentSummaryMetrics, SAMRecord> firstOfPairCollector = this.constructCollector(AlignmentSummaryMetrics.Category.FIRST_OF_PAIR);
        MetricCollector<AlignmentSummaryMetrics, SAMRecord> secondOfPairCollector = this.constructCollector(AlignmentSummaryMetrics.Category.SECOND_OF_PAIR);
        MetricCollector<AlignmentSummaryMetrics, SAMRecord> pairCollector = this.constructCollector(AlignmentSummaryMetrics.Category.PAIR);
        for (SAMRecord record : in) {
            if (record.getReadPairedFlag()) {
                if (record.getFirstOfPairFlag()) {
                    firstOfPairCollector.addRecord(record);
                } else {
                    secondOfPairCollector.addRecord(record);
                }
                pairCollector.addRecord(record);
                continue;
            }
            unpairedCollector.addRecord(record);
        }
        in.close();
        firstOfPairCollector.onComplete();
        secondOfPairCollector.onComplete();
        pairCollector.onComplete();
        unpairedCollector.onComplete();
        MetricsFile file = this.getMetricsFile();
        if (firstOfPairCollector.getMetrics().TOTAL_READS > 0L) {
            pairCollector.getMetrics().BAD_CYCLES = firstOfPairCollector.getMetrics().BAD_CYCLES + secondOfPairCollector.getMetrics().BAD_CYCLES;
            file.addMetric(firstOfPairCollector.getMetrics());
            file.addMetric(secondOfPairCollector.getMetrics());
            file.addMetric(pairCollector.getMetrics());
        }
        if (unpairedCollector.getMetrics().TOTAL_READS > 0L) {
            file.addMetric(unpairedCollector.getMetrics());
        }
        file.write(this.OUTPUT);
        return 0;
    }

    private void assertCoordinateSortOrder(SAMFileReader in) {
        switch (in.getFileHeader().getSortOrder()) {
            case coordinate: {
                break;
            }
            case unsorted: {
                if (this.ASSUME_SORTED.booleanValue()) break;
            }
            default: {
                log.warn("May not be able collect summary statistics in file " + this.INPUT.getAbsoluteFile() + " because it is not sorted in coordinate order.  If any of the reads are aligned this will blow up.");
            }
        }
    }

    protected void prepareAdapterSequences() {
        int count = this.ADAPTER_SEQUENCE.size();
        this.ADAPTER_SEQUENCES = new byte[count * 2][];
        for (int i = 0; i < count; ++i) {
            String adapter = this.ADAPTER_SEQUENCE.get(i).toUpperCase();
            this.ADAPTER_SEQUENCES[i] = StringUtil.stringToBytes((String)adapter);
            this.ADAPTER_SEQUENCES[i + count] = StringUtil.stringToBytes((String)SequenceUtil.reverseComplement((String)adapter));
        }
    }

    protected boolean isAdapterSequence(byte[] read) {
        StringUtil.toUpperCase((byte[])read);
        for (byte[] adapter : this.ADAPTER_SEQUENCES) {
            int lastKmerStart = adapter.length - 16;
            for (int adapterStart = 0; adapterStart < lastKmerStart; ++adapterStart) {
                int errors = 0;
                for (int i = 0; i < 16 && errors <= 1; ++i) {
                    if (read[i] == adapter[i + adapterStart]) continue;
                    ++errors;
                }
                if (errors > true) continue;
                return true;
            }
        }
        return false;
    }

    private MetricCollector<AlignmentSummaryMetrics, SAMRecord> constructCollector(AlignmentSummaryMetrics.Category category) {
        AggregateMetricCollector<AlignmentSummaryMetrics, SAMRecord> collector = new AggregateMetricCollector<AlignmentSummaryMetrics, SAMRecord>(new ReadCounter(), new QualityMappingCounter());
        collector.setMetrics(new AlignmentSummaryMetrics());
        ((AlignmentSummaryMetrics)collector.getMetrics()).CATEGORY = category;
        return collector;
    }

    private class QualityMappingCounter
    implements MetricCollector<AlignmentSummaryMetrics, SAMRecord> {
        private final Histogram<Long> mismatchHistogram = new Histogram();
        private final Histogram<Integer> badCycleHistogram = new Histogram();
        private AlignmentSummaryMetrics metrics;

        private QualityMappingCounter() {
        }

        @Override
        public void addRecord(SAMRecord record) {
            if (record.getNotPrimaryAlignmentFlag()) {
                return;
            }
            if (record.getReadUnmappedFlag()) {
                byte[] readBases = record.getReadBases();
                for (int i = 0; i < readBases.length; ++i) {
                    if (!SequenceUtil.isNoCall((byte)readBases[i])) continue;
                    this.badCycleHistogram.increment(CoordMath.getCycle((boolean)record.getReadNegativeStrandFlag(), (int)readBases.length, (int)i));
                }
            } else {
                boolean highQualityMapping = this.isHighQualityMapping(record);
                if (highQualityMapping) {
                    ++this.metrics.PF_HQ_ALIGNED_READS;
                }
                byte[] readBases = record.getReadBases();
                byte[] refBases = CollectAlignmentSummaryMetrics.this.referenceSequenceWalker.get(record.getReferenceIndex()).getBases();
                byte[] qualities = record.getBaseQualities();
                int refLength = refBases.length;
                long mismatchCount = 0L;
                for (AlignmentBlock alignmentBlock : record.getAlignmentBlocks()) {
                    int readIndex = alignmentBlock.getReadStart() - 1;
                    int refIndex = alignmentBlock.getReferenceStart() - 1;
                    int length = alignmentBlock.getLength();
                    for (int i = 0; i < length && refIndex + i < refLength; ++i) {
                        int readBaseIndex = readIndex + i;
                        boolean mismatch = !SequenceUtil.basesEqual((byte)readBases[readBaseIndex], (byte)refBases[refIndex + i]);
                        boolean bisulfiteBase = false;
                        if (mismatch && CollectAlignmentSummaryMetrics.this.IS_BISULFITE_SEQUENCED && (record.getReadNegativeStrandFlag() && (refBases[refIndex + i] == 71 || refBases[refIndex + i] == 103) && (readBases[readBaseIndex] == 65 || readBases[readBaseIndex] == 97) || !record.getReadNegativeStrandFlag() && (refBases[refIndex + i] == 67 || refBases[refIndex + i] == 99) && readBases[readBaseIndex] == 84 || readBases[readBaseIndex] == 116)) {
                            bisulfiteBase = true;
                            mismatch = false;
                        }
                        if (highQualityMapping) {
                            ++this.metrics.PF_HQ_ALIGNED_BASES;
                            if (!bisulfiteBase) {
                                this.metrics.incrementErrorRateDenominator();
                            }
                            if (qualities[readBaseIndex] >= 20) {
                                ++this.metrics.PF_HQ_ALIGNED_Q20_BASES;
                            }
                            if (mismatch) {
                                ++mismatchCount;
                            }
                        }
                        if (!mismatch && !SequenceUtil.isNoCall((byte)readBases[readBaseIndex])) continue;
                        this.badCycleHistogram.increment(CoordMath.getCycle((boolean)record.getReadNegativeStrandFlag(), (int)readBases.length, (int)i));
                    }
                }
                this.mismatchHistogram.increment(mismatchCount);
            }
        }

        private boolean isHighQualityMapping(SAMRecord record) {
            return !record.getReadFailsVendorQualityCheckFlag() && record.getMappingQuality() >= 20;
        }

        @Override
        public void onComplete() {
            this.metrics.PF_HQ_MEDIAN_MISMATCHES = this.mismatchHistogram.getMedian();
            this.metrics.PF_HQ_ERROR_RATE = this.mismatchHistogram.getSum() / (double)this.metrics.getErrorRateDenominator();
            this.metrics.BAD_CYCLES = 0L;
            for (Histogram.Bin cycleBin : this.badCycleHistogram.values()) {
                double badCyclePercentage = cycleBin.getValue() / (double)this.metrics.TOTAL_READS;
                if (!(badCyclePercentage >= 0.8)) continue;
                ++this.metrics.BAD_CYCLES;
            }
        }

        @Override
        public void setMetrics(AlignmentSummaryMetrics metrics) {
            this.metrics = metrics;
        }

        @Override
        public AlignmentSummaryMetrics getMetrics() {
            return this.metrics;
        }
    }

    private class ReadCounter
    implements MetricCollector<AlignmentSummaryMetrics, SAMRecord> {
        private long numPositiveStrand = 0L;
        private final Histogram<Integer> readLengthHistogram = new Histogram();
        private AlignmentSummaryMetrics metrics;
        private long chimeras;
        private long adapterReads;

        private ReadCounter() {
        }

        @Override
        public void addRecord(SAMRecord record) {
            if (record.getNotPrimaryAlignmentFlag()) {
                return;
            }
            ++this.metrics.TOTAL_READS;
            this.readLengthHistogram.increment(record.getReadBases().length);
            if (!record.getReadFailsVendorQualityCheckFlag()) {
                ++this.metrics.PF_READS;
                if (this.isNoiseRead(record)) {
                    ++this.metrics.PF_NOISE_READS;
                } else if (record.getReadUnmappedFlag()) {
                    if (CollectAlignmentSummaryMetrics.this.isAdapterSequence(record.getReadBases())) {
                        ++this.adapterReads;
                    }
                } else {
                    ++this.metrics.PF_READS_ALIGNED;
                    if (!record.getReadNegativeStrandFlag()) {
                        ++this.numPositiveStrand;
                    }
                    if (record.getReadPairedFlag() && !record.getMateUnmappedFlag()) {
                        Integer mateMq;
                        ++this.metrics.READS_ALIGNED_IN_PAIRS;
                        if (!(Math.abs(record.getInferredInsertSize()) <= CollectAlignmentSummaryMetrics.this.MAX_INSERT_SIZE && record.getReferenceIndex().equals(record.getMateReferenceIndex()) || (mateMq = record.getIntegerAttribute("MQ")) != null && (mateMq < 20 || record.getMappingQuality() < 20))) {
                            ++this.chimeras;
                        }
                    }
                }
            }
        }

        @Override
        public void onComplete() {
            this.metrics.PCT_PF_READS = (double)this.metrics.PF_READS / (double)this.metrics.TOTAL_READS;
            this.metrics.PCT_PF_READS_ALIGNED = (double)this.metrics.PF_READS_ALIGNED / (double)this.metrics.PF_READS;
            this.metrics.PCT_READS_ALIGNED_IN_PAIRS = (double)this.metrics.READS_ALIGNED_IN_PAIRS / (double)this.metrics.PF_READS_ALIGNED;
            this.metrics.MEAN_READ_LENGTH = this.readLengthHistogram.getMean();
            this.metrics.STRAND_BALANCE = (double)this.numPositiveStrand / (double)this.metrics.PF_READS_ALIGNED;
            this.metrics.PCT_ADAPTER = (double)this.adapterReads / (double)this.metrics.PF_READS;
            this.metrics.PCT_CHIMERAS = (double)this.chimeras / (double)this.metrics.PF_HQ_ALIGNED_READS;
        }

        private boolean isNoiseRead(SAMRecord record) {
            Object noiseAttribute = record.getAttribute("XN");
            return noiseAttribute != null && noiseAttribute.equals(1);
        }

        @Override
        public void setMetrics(AlignmentSummaryMetrics metrics) {
            this.metrics = metrics;
        }

        @Override
        public AlignmentSummaryMetrics getMetrics() {
            return this.metrics;
        }
    }
}

