/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */


package org.jfree.data.som;

import java.util.*;
import java.awt.Color;
import java.io.Serializable;

import org.jfree.data.general.AbstractDataset;


/**
 * A dataset for dealing with SOM-map-cells.
 *
 * @author  viski-project, Department of Computer Science, Univ. Helsinki
 */
public class SOMDataset extends AbstractDataset implements Cloneable, Serializable {

    /** Datastorage. */
    protected SOMDataItem[][] data;

    /**
     * Creates a new SOMDataset with given columns and rows.
     * 
     * @param columns  the number of columns.
     * @param rows  the number of rows.
     * @throws IllegalArgumentException If columns <= 0 or rows <= 0
     */
    public SOMDataset(int columns, int rows) throws IllegalArgumentException {
        if (columns <= 0)
            throw new IllegalArgumentException("Non-positive column count given to SOMDataset");
        if (rows <= 0)
            throw new IllegalArgumentException("Non-positive row count given to SOMDataset");
        data = new SOMDataItem[rows][columns];
    }

    /**
     * Returns the value at a specific (x,y) point in the datamatrix.
     *
     * @return  A SOMDataItem.
     * @throws IndexOutOfBoundsException
     */
    public SOMDataItem getValue(int x, int y) throws IndexOutOfBoundsException {
        return data[y][x];
    }

    /**
     * Returns the number of columns in a SOMDataset.
     *
     * @return  The number of columns.
     */
    public int getColumnCount() {
        return data[0].length;
    }

    /**
     * Returns the number of rows in a SOMDataset.
     *
     * @return  The number of rows.
     */
    public int getRowCount() {
        return data.length;
    }

    /**
     * Creates a new SOMDataItem from the params and adds it to the dataset.
     *
     * @param x  the x-coordinate to add to in the matrix.
     * @param y  the y-coordinate to add to in the matrix.
     * @param color  the color of the dataitem.
     * @param description  the textual description of the dataitem.
     * @param neuronWeights  the numerical data of the dataitem.
     *
     * @throws IndexOutOfBoundsException if x&lt;=0 or y&lt;=0
     * @throws NullPointerException If any parameter is null.
     * @throws IllegalArgumentException If descriptions or neuronWeights do not contain any items.
     */
    public void addValue(int x, int y, Color color, String[] description, double[] neuronWeights) throws IndexOutOfBoundsException {
        data[y][x] = new SOMDataItem(color, description, neuronWeights);
    }

    /**
     * Adds a SOMDataItem to the dataset.
     *
     * @param x  the x-coordinate to add to in the matrix.
     * @param y  the y-coordinate to add to in the matrix.
     * @param item  the SOMDataItem to add.
     *
     * @throws IndexOutOfBoundsException if x&lt;=0 or y&lt;=0
     * @throws NullPointerException  if item is null.
     */
    public void addValue(int x, int y, SOMDataItem item) throws IndexOutOfBoundsException {
        if (item == null)
            throw new NullPointerException("item given to addValue was null.");
        data[y][x] = item;
    }

    /**
     * Returns the neighbours of a dataitem in a {@link List}-object. 
     * The distances are Euclidean distances
     * between the RGB-values of this data item and all other data items.
     *
     * @param x  the x-coordinate in the matrix to start the search from.
     * @param y  the y-coordinate in the matrix to start the search from.
     * @param maxDistance  the distance between a neighbour (ie. similar color) and a non-neighbour
     * @param includeCenter  Include the item at (x,y) in the list
     *
     * @return The list of neighbours.
     * @throws IndexOutOfBoundsException  if (x,y) is not inside the dataset.
     * @throws IllegalArgumentException  If maxDistance < 0
     */
    public List getNeighbors(int x, int y, int maxDistance, boolean includeCenter)
    throws IndexOutOfBoundsException, IllegalArgumentException {
        SOMDataItem center = getValue(x, y);
        
        return getNeighbors(center, maxDistance, includeCenter);
    }

    /**
     * Returns the neighbours of a dataitem in a {@link List}-object. 
     * The distances are Euclidean distances
     * between the RGB-values of this data item and all other data items.
     *
     * @param center the cell to which all other cells are compared
     * @param maxDistance  the distance between a neighbour (ie. similar color) and a non-neighbour
     * @param includeCenter  Include center in the list
     *
     * @return The list of neighbours.
     * @throws NullPointerException If center is null.
     * @throws IllegalArgumentException If maxDistance < 0.
     */
    public List getNeighbors(SOMDataItem center, int maxDistance, boolean includeCenter) 
    throws IllegalArgumentException, NullPointerException {
        if (center == null)
            throw new NullPointerException("center given to getNeighbors() was null");
        if (maxDistance < 0)
            throw new IllegalArgumentException("maxDistance given to getNeighbors() was negative.");
        Color centerColor = center.getColor();
        LinkedList list = new LinkedList();

        Iterator i = new SOMDatasetIterator(this);
        while (i.hasNext()) {
            SOMDataItem item = (SOMDataItem) i.next();
            if (includeCenter || item != center) {
                double diff = colorDistance(item.getColor(), centerColor);
                if (diff <= maxDistance)
                    list.add(item);
            }
        }

        return list;
    }

    /**
     * This method calculates the distance between two sets of integer color-values.
     *
     * @return  The difference  in color-values.
     * @throws  NullPointerException  if color1 or color2 is null.
     */
    private double colorDistance(Color color1, Color color2) {
        int r1 = color1.getRed();
        int g1 = color1.getGreen();
        int b1 = color1.getBlue();

        int r2 = color2.getRed();
        int g2 = color2.getGreen();
        int b2 = color2.getBlue();

        return Math.sqrt((r1-r2)*(r1-r2)+(g1-g2)*(g1-g2)+(b1-b2)*(b1-b2));
    }
    
    /**
     * Returns a {@link List} of {@link SOMDataItem} objects the are contained
     * inside the rectangle defined by two {@link SOMDataItem} objects.
     *
     * @param item1 corner of the rectangle
     * @param item2 corner of the rectangle
     *
     * @return The list of {@link SOMDataItem}s inside the area.
     * @throws NullPointerException If item1 == null or item2 == null
     * @throws IllegalArgumentException If item1 or item2 do not belong to this dataset.
     */
    public List getArea(SOMDataItem item1, SOMDataItem item2) 
    throws IllegalArgumentException, NullPointerException {
        if (item1 == null || item2 == null)
            throw new NullPointerException("item1 or item2 given to getArea() was null");
        int[] xy1 = getCoordinates(item1);
        int[] xy2 = getCoordinates(item2);
        
        if (xy1 == null || xy2 == null)
            throw new IllegalArgumentException();
        
        int width = Math.abs(xy1[0] - xy2[0]) + 1;
        int height = Math.abs(xy1[1] - xy2[1]) + 1;
        int startX = Math.min(xy1[0], xy2[0]);
        int startY = Math.min(xy1[1], xy2[1]);
        LinkedList list = new LinkedList();
        
        for (int y=0; y < height; ++y) {
            for (int x=0; x < width; ++x) {
                list.add(this.data[startY + y][startX + x]);
            }
        }
        
        return list;
    }
    
    /**
     * Returns the coordinates of a SOMDataItem.
     *
     * @param item  the item whose coordinates we want.
     * 
     * @return  The coordinates as int[]. int[0] is the X-coordinate
     * and int[1] is the Y-coordinate.
     */
    private int[] getCoordinates(SOMDataItem item) {
        int[] xy = new int[2];
        
        for (int y=0; y < this.data.length; ++y) {
            for (int x=0; x < this.data[0].length; ++x) {
                if (item == this.data[y][x]) {
                    xy[0] = x;
                    xy[1] = y;
                    return xy;
                }
            }
        }
        
        return null;
    }
    
    /**
     * This method changes the color hue of all dataitems in a dataset for
     * given amount, negative or positive. The  color-values of each dataitem
     * are retrieved, changed and new rgb-values are calculated and set.
     *
     * @param angle  the amount to increase the hue angle.
     * @throws IllegalArgumentException If Math.abs(angle) >= 360.
     */
    public void changeHueValues(int angle) 
    throws IllegalArgumentException {
        if (Math.abs(angle) > 360)
            throw new IllegalArgumentException("angle value given to changeHueValues() was not in [-360,360].");
        Iterator i = new SOMDatasetIterator(this);
        while (i.hasNext()) {
            SOMDataItem item = (SOMDataItem)i.next();
            
            if (item != null) {
                Color c = item.getColor();
                float[] hsb = Color.RGBtoHSB(
                        c.getRed(),
                        c.getGreen(),
                        c.getBlue(),
                        null);
                
                hsb[0] += angle/360.0 + 1.0;
                hsb[0] %= 1.0;
                item.setColor(Color.getHSBColor(hsb[0], hsb[1], hsb[2]));
            }
        }
    }

    /**
     * This method returns an iterator that iterates all it's dataitems
     *
     * @return iterator
     */
    public Iterator iterator() {
        return new SOMDatasetIterator(this);
    }
    
    /**
     * Deselects all the dataitems in this SOMDataset.
     */
    public void deselectAll() {
        Iterator i = iterator();
        while (i.hasNext()) {
            SOMDataItem item = (SOMDataItem)i.next();
            item.setSelected(false);
        }
    }
    
    /**
     * An inner class to deal with SOMDataItem-objects in the SOMDataset. This
     * class implements the {@link Iterator} interface
     */
    private class SOMDatasetIterator implements Iterator {
        private SOMDataset dataset;
        private int x;
        private int y;

        /**
         * Creates a new SOMDatasetIterator for a dataset.
         *
         * @param dataset  the SOMDataset to be iterated.
         */
        public SOMDatasetIterator(SOMDataset dataset) {
            this.dataset = dataset;
            this.x = 0;
            this.y = 0;
        }
            
        /**
         * Returns false, if the pointer already is at the last item.
         *
         * @return  A boolean.
	 *
	 * @throws NullPointerException  if this.dataset is null.
         */
        public boolean hasNext() {
            return !((this.y == this.dataset.getRowCount()) && (this.x == 0));
        }

        /**
         * Moves the pointer to the next item.
         *
         * @return  The next SOMDataItem.
	 * @throws NullPointerException  if this.dataset is null.
         */
        public Object next() {
            SOMDataItem item;
            item = this.dataset.getValue(this.x++, this.y);

            this.x %= this.dataset.getColumnCount();
            if (this.x == 0)
                this.y++;
            
            return item;
        }

        /**
         * Should remove an item from the iteration list.
         * This method does nothing.
	 * @throws UnsupportedOperationException  always.
         */
        public void remove() throws UnsupportedOperationException {
            throw new UnsupportedOperationException("SOMDataset iterators do not implement remove().");
        }

    }
    
    /**
     * Tests if this object is equal to another.
     *
     * @param obj  the other object.
     *
     * @return true, if obj equals to this, false otherwise.
     */
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }

        if (!(obj instanceof SOMDataset)) {
            return false;
        }
        SOMDataset that = (SOMDataset) obj;
        int cols = this.getColumnCount();
        int rows = this.getRowCount();
        if (that.getColumnCount() != cols ||
                that.getRowCount() != rows) {
            return false;
        }

        Iterator i1 = this.iterator();
        Iterator i2 = that.iterator();
        while (i1.hasNext()) {
            SOMDataItem item1 = (SOMDataItem)i1.next();
            SOMDataItem item2 = (SOMDataItem)i2.next();
            if (item1 == null && item2 == null) {
                ;
            }
            else if (item1 == null || item2 == null || 
                     item1.equals(item2) == false)
                return false;
        }
        
        return true;
    }
    
    /**
     * Returns a clone of the dataset.
     * 
     * @return A clone.
     * 
     * @throws CloneNotSupportedException This class will not throw this 
     *         exception, but subclasses (if any) might.
     */
    public Object clone() throws CloneNotSupportedException {
        SOMDataset clone = (SOMDataset) super.clone();
        clone.data = (SOMDataItem[][]) this.data.clone();
        return clone;    
    }
    
}

