

package org.jfree.data.hc.junit;

import org.jfree.data.hc.HCTreeNode;
import org.jfree.data.hc.DataRange;
import org.jfree.data.hc.DataRangeMismatchException;
import org.jfree.data.hc.NotAChildException;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * Some tests for the <code>org.jfree.data</code> package that can be run 
 * using JUnit.  You can find more information about JUnit at 
 * <a href="http://www.junit.org">http://www.junit.org</a>.
 */
public class HCTreeNodeTests extends TestCase {

    /**
     * Constructs a new set of tests.
     *
     * @param name  the name of the tests.
     */
    public HCTreeNodeTests(String name) {

        super(name);

    }

    /**
     * Creates an unfinalized test tree.
     */
    private HCTreeNode createTree() 
	throws DataRangeMismatchException {

	HCTreeNode tree;
	HCTreeNode n;
	tree = new HCTreeNode(3);
	n = new HCTreeNode(2);
	tree.setLeftChild(n);
	n = new HCTreeNode(0,0);
	tree.getLeftChild().setLeftChild(n);
	n = new HCTreeNode(0,1);
	tree.getLeftChild().setRightChild(n);
	n = new HCTreeNode(0,2);
	tree.setRightChild(n);
	return tree;

    }

    /**
    /**
     * Returns a test suite to the JUnit test runner.
     *
     * @return The test suite.
     */
    public static Test suite() {
        return new TestSuite(HCTreeNodeTests.class);
    }

    /**
     * Runs the test suite using JUnit's text-based runner.
     * 
     * @param args  ignored.
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }

    /**
     * Tests setting and getting the parent-child relationships of nodes.
     */
    public static void testGetSetParentAndLeftChild()
	throws DataRangeMismatchException {

	HCTreeNode n1;
	HCTreeNode n2;
	HCTreeNode n3;
	HCTreeNode n4;

	n1 = new HCTreeNode(1);
	n2 = new HCTreeNode(2);
	n3 = new HCTreeNode(3);
	n4 = new HCTreeNode(4);

	// initial relations
	assertTrue(n1.getParent() == null);
	assertTrue(n1.getLeftChild() == null);
	assertTrue(n1.getRightChild() == null);

	// first, let's try setParent the wrong way
	// and check we got exception and nothing changed.
	try {
	    n2.setParent(n1);
	    assertTrue(false);
	} catch (NotAChildException e) {
	    ;
	}
	assertTrue(n2.getParent() == null);
	assertTrue(n1.getLeftChild() == null);
	assertTrue(n1.getRightChild() == null);

	// now setLeftChild().
	n1.setLeftChild(n2);
	assertTrue(n1.getLeftChild() == n2);
	assertTrue(n2.getParent() == n1);

	// replace it.
	n1.setLeftChild(n3);
	assertTrue(n1.getLeftChild() == n3);
	assertTrue(n3.getParent() == n1);
	assertTrue(n2.getParent() == null);

	// setRightChild() the wrong way.
	try {
	    n1.setRightChild(n3);
	    assertTrue(false);
	} catch (RuntimeException e) {
	    ;
	}
	assertTrue(n1.getRightChild() == null);
	assertTrue(n1.getLeftChild() == n3);
	assertTrue(n3.getParent() == n1);

	// now setRightChild().
	n1.setRightChild(n2);
	assertTrue(n1.getRightChild() == n2);
	assertTrue(n2.getParent() == n1);

	// setLeftChild() the wrong way.
	try {
	    n1.setLeftChild(n2);
	    assertTrue(false);
	} catch (RuntimeException e) {
	    ;
	}
	assertTrue(n1.getRightChild() == n2);
	assertTrue(n1.getLeftChild() == n3);
	assertTrue(n2.getParent() == n1);

	// and the wierd cases.
	n1.setRightChild(n2);
	n1.setLeftChild(n3);
	assertTrue(n1.getRightChild() == n2);
	assertTrue(n1.getLeftChild() == n3);
	assertTrue(n2.getParent() == n1);
	assertTrue(n3.getParent() == n1);

	// and setting a child null.
	n1.setLeftChild(null);
	assertTrue(n1.getLeftChild() == null);
	assertTrue(n3.getParent() == null);

	// set a new parent for a node that already has one.

	// in the begining n4 is the right child of n1.
	assertTrue(n1.getRightChild() == n2);
	n4.setLeftChild(n2);
	assertTrue(n2.getParent() == n4);
	// check that n4 is not a child of n1 any more.
	assertFalse(n1.getRightChild() == n2);
	assertTrue(n4.getLeftChild() == n2);
	// and back so we will test the setRightChild version as well.
	n1.setRightChild(n2);
	assertTrue(n4.getLeftChild() == null);
	assertTrue(n1.getRightChild() == n2);

    }

    /**
     * Tests setting and getting the parent-child relationships of nodes.
     */
    public void testGetDataRange() 
        throws DataRangeMismatchException {

	HCTreeNode n;

	n = new HCTreeNode(2);
	assertTrue (n.getDataRange().equals(new DataRange(0,-1)));

	n = new HCTreeNode(2,5);
	assertTrue (n.getDataRange().equals(new DataRange(5,5)));

	n = createTree();

	assertTrue(n.getDataRange().equals(new DataRange(0,2)));

    }

    /**
     * Tests getHeight method.
     */
    public void testGetHeight() {

	HCTreeNode n;

	n = new HCTreeNode(2);
	assertTrue (n.getHeight() == 2);

    }

    /**
     * Tests getRoot method.
     */
    public void testGetRoot()
        throws DataRangeMismatchException {

	HCTreeNode tree = createTree();
	HCTreeNode n;

	// get a child of the root and assert, they are different.
	n = tree.getLeftChild().getLeftChild();
	assertTrue(n != tree);

	// get root of a child node.
	n = n.getRoot();
	assertTrue(n == tree);

	// get root of a root node also.
	n = n.getRoot();
	assertTrue(n == tree);

    }

    /**
     * Tests getLeafNodeByIndex method
     */
    public void testGetLeafNodeByIndex()
        throws DataRangeMismatchException {

	HCTreeNode tree = createTree();
	HCTreeNode n;
	HCTreeNode target;

	target = tree.getRightChild(); // == 3
	n = tree.getLeafNodeByIndex(2);
	assertTrue (n == target);
	n = target.getLeafNodeByIndex(2);
	assertTrue (n == target);
	n = tree.getLeftChild().getLeafNodeByIndex(2);
	assertTrue (n == target);

	target = tree.getLeftChild().getLeftChild(); // == 1
	n = tree.getLeafNodeByIndex(0);
	assertTrue(n == target);
	target = tree.getLeftChild().getRightChild(); // == 2
	n = tree.getLeafNodeByIndex(1);
	assertTrue(n == target);

	try {

	    tree.getLeafNodeByIndex(3);
	    assertTrue(false);

	} catch (IndexOutOfBoundsException e) {

	    ;

	}

    }

    /**
     * Tests finalizeTree method.
     */
    public void testFinalizeTree()
	throws DataRangeMismatchException {

	HCTreeNode tree = createTree();
	HCTreeNode n = new HCTreeNode(1,5);

	// check if finalized tree is writable.
	tree.finalizeTree();

	try {
	    tree.setLeftChild(n);
	    assertTrue(false);
	} catch (RuntimeException e) {
	    ;
	}

	try {
	    tree.setRightChild(n);
	    assertTrue(false);
	} catch (RuntimeException e) {
	    ;
	}

	// try finalizing a tree single node tree, with non-zero index.
	n = new HCTreeNode(0,5);
	try {
	    n.finalizeTree();
	    assertTrue(false);
	} catch (Exception e) {
	    ;
	}

	// try finalizing a height mismatched tree.
	n = new HCTreeNode(1);
	n.setLeftChild(new HCTreeNode(2,2));
	n.setRightChild(new HCTreeNode(2,2));
	try {
	    n.finalizeTree();
	    assertTrue(false);
	} catch (Exception e) {
	    ;
	}

	// try finalizing a tree with node with only a left child.
	n = new HCTreeNode(1);
	n.setLeftChild(new HCTreeNode(0,0));
	try {
	    n.finalizeTree();
	    assertTrue(false);
	} catch (Exception e) {
	    ;
	}

	// try finalizing a tree with node with only a right child.
	n = new HCTreeNode(1);
	n.setRightChild(new HCTreeNode(0,1));
	try {
	    n.finalizeTree();
	    assertTrue(false);
	} catch (Exception e) {
	    ;
	}

    }

}

