/*
 * Decompiled with CFR 0.152.
 */
package fi.csc.microarray.analyser;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Date;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.log4j.Logger;

public class ProcessPool {
    static final Logger logger = Logger.getLogger(ProcessPool.class);
    private BlockingQueue<NamiProcess> availableProcesses;
    private ConcurrentMap<Integer, NamiProcess> inUseProcesses;
    private final int poolSizeMin;
    private final int poolSizeMax;
    private final int poolTimeout;
    private final int processUseCountMax;
    private final int processLifetimeMax;
    private final String rCommand;
    private File workDir;
    private static final String RECYCLE_SUCCESFUL_STRING = "recycling-succesful";

    public ProcessPool(File workDir, String command, int poolSizeMin, int poolSizeMax, int poolTimeout, int processUseCountMax, int processLifetimeMax) throws IOException {
        this.workDir = workDir;
        this.rCommand = command;
        this.poolSizeMin = poolSizeMin;
        this.poolSizeMax = poolSizeMax;
        this.poolTimeout = poolTimeout;
        this.processUseCountMax = processUseCountMax;
        this.processLifetimeMax = processLifetimeMax;
        this.availableProcesses = new LinkedBlockingQueue<NamiProcess>();
        this.inUseProcesses = new ConcurrentHashMap<Integer, NamiProcess>();
        for (int i = 0; i < poolSizeMin; ++i) {
            this.availableProcesses.add(this.createProcess());
        }
        logger.debug((Object)("R process pool initialized, processes available: " + this.availableProcesses.size()));
    }

    public Process getProcess() throws IOException, InterruptedException {
        NamiProcess nProcess = (NamiProcess)this.availableProcesses.poll();
        if (nProcess == null) {
            if (this.availableProcesses.size() + this.inUseProcesses.size() < this.poolSizeMax) {
                nProcess = this.createProcess();
            } else {
                nProcess = this.availableProcesses.poll(this.poolTimeout, TimeUnit.SECONDS);
                if (nProcess == null) {
                    throw new IOException("Timeout when getting an R process.");
                }
            }
        }
        nProcess.increaseUseCount();
        this.inUseProcesses.put(nProcess.getProcess().hashCode(), nProcess);
        return nProcess.getProcess();
    }

    public void releaseProcess(Process process, boolean recycle) throws IOException {
        NamiProcess nProcess = (NamiProcess)this.inUseProcesses.get(process.hashCode());
        if (nProcess == null) {
            throw new IOException("Trying to release an unknown process.");
        }
        boolean processOk = true;
        boolean processAlive = false;
        if (!recycle) {
            logger.debug((Object)("Process " + nProcess.getProcess().hashCode() + " not recycled as requested."));
        } else if (nProcess.getUseCount() >= this.processUseCountMax) {
            processOk = false;
            logger.debug((Object)("Process " + nProcess.getProcess().hashCode() + " has been used for " + nProcess.getUseCount() + " times and is therefore not recycled."));
        } else if (System.currentTimeMillis() - nProcess.getCreationTime().getTime() >= (long)(this.processLifetimeMax * 1000)) {
            processOk = false;
            logger.debug((Object)("Process " + nProcess.getProcess().hashCode() + " has been running over " + this.processLifetimeMax + " seconds and is therefore not recycled."));
        } else {
            try {
                process.exitValue();
            }
            catch (IllegalThreadStateException itse) {
                processAlive = true;
            }
            if (!processAlive) {
                logger.debug((Object)("Process " + nProcess.getProcess().hashCode() + " is dead and is therefore not recycled."));
            }
        }
        if (recycle && processOk && processAlive) {
            logger.debug((Object)("Recycling process " + nProcess.getProcess().hashCode() + "."));
            CountDownLatch recycleLatch = new CountDownLatch(1);
            ProcessMonitor recycleMonitor = new ProcessMonitor(process, recycleLatch);
            new Thread(recycleMonitor).start();
            BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(process.getOutputStream()));
            writer.write("rm(list=objects())");
            writer.newLine();
            writer.write("setwd(\"" + this.workDir.getAbsolutePath() + "\")");
            writer.newLine();
            writer.write("print(\"recycling-succesful\")");
            writer.newLine();
            writer.flush();
            try {
                recycleLatch.await(10L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                processOk = false;
            }
            if (!processOk || recycleLatch.getCount() > 0L || !recycleMonitor.processOk() || process.getErrorStream().available() > 0) {
                process.destroy();
                nProcess = this.createProcess();
            }
        } else {
            process.destroy();
            nProcess = this.createProcess();
        }
        if (this.availableProcesses.size() < this.poolSizeMin && this.availableProcesses.size() + this.inUseProcesses.size() <= this.poolSizeMax) {
            this.availableProcesses.add(nProcess);
        } else {
            nProcess.getProcess().destroy();
        }
        this.inUseProcesses.remove(process.hashCode());
        logger.debug((Object)("Available processes: " + this.availableProcesses.size() + ", in use: " + this.inUseProcesses.size()));
    }

    private NamiProcess createProcess() throws IOException {
        logger.debug((Object)"Creating a new R process.");
        ProcessBuilder builder = new ProcessBuilder(this.rCommand.split(" "));
        builder.directory(this.workDir);
        builder.redirectErrorStream(true);
        Process p = builder.start();
        return new NamiProcess(p);
    }

    private class ProcessMonitor
    implements Runnable {
        private CountDownLatch latch;
        private Process process;
        private boolean processOk = false;

        public ProcessMonitor(Process process, CountDownLatch latch) {
            this.process = process;
            this.latch = latch;
        }

        public boolean processOk() {
            return this.processOk;
        }

        public void run() {
            BufferedReader reader = new BufferedReader(new InputStreamReader(this.process.getInputStream()));
            boolean readMore = true;
            try {
                String line = reader.readLine();
                while (readMore) {
                    if (line == null) {
                        this.processOk = false;
                        readMore = false;
                    } else if (line.contains(ProcessPool.RECYCLE_SUCCESFUL_STRING)) {
                        this.processOk = true;
                        readMore = false;
                    }
                    line = reader.readLine();
                }
            }
            catch (IOException e) {
                this.processOk = false;
            }
            this.latch.countDown();
        }
    }

    private class NamiProcess {
        private Process process;
        private int useCount;
        private Date creationTime;

        public NamiProcess(Process p) {
            this.process = p;
            this.useCount = 0;
            this.creationTime = new Date(System.currentTimeMillis());
        }

        public Process getProcess() {
            return this.process;
        }

        public int getUseCount() {
            return this.useCount;
        }

        public Date getCreationTime() {
            return this.creationTime;
        }

        public void increaseUseCount() {
            ++this.useCount;
        }
    }
}

