
package org.jfree.chart.plot.junit;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.jfree.chart.plot.HCTreeNodeInfo;
import org.jfree.chart.plot.StandardHCClusteringInfo;
import org.jfree.data.hc.HCTreeNode;
import org.jfree.data.hc.DataRange;

/**
 * Tests for the {@link org.jfree.chart.plot.HCTreeNodeInfo} class.
 */
 
public class HCTreeNodeInfoTests extends TestCase {

    private HCTreeNode node;
    private HCTreeNode leftnode;
    private HCTreeNode rightnode;
    
    private StandardHCClusteringInfo ci;

   /**
     * Returns the tests as a test suite.
     *
     * @return The test suite.
     */
    public static Test suite() {
        return new TestSuite(HCTreeNodeInfoTests.class);
    }

    /**
     * Constructs a new set of tests.
     *
     * @param name  the name of the tests.
     */
    public HCTreeNodeInfoTests(String name) throws Exception {

        super(name);

        
        this.node = new HCTreeNode(1.0);
        this.leftnode = new HCTreeNode(1.0, 0);
        this.rightnode = new HCTreeNode(1.0, 1);


        try {
           
            this.node.setRightChild(rightnode);
            this.node.setLeftChild(leftnode);

        } catch (Exception e) {}

        this.ci =  new StandardHCClusteringInfo(
            this.node, new String[] {"names1","names2"}, 1);
            
    }

    /**
     * Checks the conststructor
     */
    public void testConstructor() {

        String s1 =
            "A HCTreeNodeInfo object must have a non-null HCTreeNode.";
        String s2 = 
            "A HCTreeNodeInfo object must have a non-null clustering info.";   
            
        try {
            new HCTreeNodeInfo(null, this.node);
            assertTrue(false);
        } catch (NullPointerException e) {
            assertTrue(s2.equals(e.getMessage()));
        }

        try {
            new HCTreeNodeInfo(this.ci, null);
            assertTrue(false);
        } catch (NullPointerException e) {
            assertTrue(s1.equals(e.getMessage()));
        }
        
        assertNotNull(new HCTreeNodeInfo(this.ci,this.node));
        assertNotNull(new HCTreeNodeInfo(this.ci,this.leftnode));
        
    }
    /**
     * Checks the setNodeOpen(boolean open)- and
     * setSubTreeOpen(boolean open)-methods
     */
    public void testNodeOpening() {

        HCTreeNodeInfo info = new HCTreeNodeInfo(this.ci, this.node);

        info.setNodeOpen(true);
        assertTrue(info.isNodeOpen());
        info.setNodeOpen(false);
        assertFalse(info.isNodeOpen());
        
        info.setSubTreeOpen(true);
        assertTrue(info.isNodeOpen());
        //leafnodes
        assertFalse(info.getLeftChild().isNodeOpen());
        assertFalse(info.getRightChild().isNodeOpen());
        
        info.setSubTreeOpen(false);
        assertFalse(info.isNodeOpen());
        //leafnodes
        assertFalse(info.getLeftChild().isNodeOpen());
        assertFalse(info.getRightChild().isNodeOpen());
    }
    
    /**
     * Checks the methods that "set and get parents and children".
     */
    public void testChildrenAndParents() {
        
        HCTreeNodeInfo info= new HCTreeNodeInfo(this.ci, this.node);

        assertNull(info.getParent());
        assertEquals(info.getLeftChild().getParent(), info);
        
        HCTreeNodeInfo rightinfo = new HCTreeNodeInfo(
            this.ci, this.rightnode);
        assertNull(rightinfo.getRightChild());
        assertNull(rightinfo.getLeftChild());
        
        String s1 = "You cannot set a parent for a HCTreeNodeInfo node that already has one.";
        String s2 = "A HCTreeNodeInfo object can only be set as a parent of another object, if that node is already its child.";
        
        HCTreeNodeInfo info2 = new HCTreeNodeInfo(
            this.ci, new HCTreeNode(1.0));
            
        //setting a parent when it is already done
        try {
            info.getRightChild().setParent(info);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
            assertTrue(s1.equals(e.getMessage()));
        }
        
        //node is not a child of this parent
        try {
            info.setParent(info2);
            assertEquals(info.getParent(), info2);
            info.setParent(info2);
            assertTrue(false);
        } catch (IllegalArgumentException e) {
            assertTrue(s2.equals(e.getMessage()));
        } 
        
        try {
            info.setParent(null);
            assertTrue(false);
       } catch (Exception e) {
           //node is not a child of this parent
       }
       
    }
    /**
     * Checks everything else
     */
    public void testNodesAndDataRanges(){

        HCTreeNodeInfo info = new HCTreeNodeInfo(this.ci, this.node);

        assertEquals(this.node, info.getNode());
        assertEquals(this.ci, info.getClusteringInfo());
        assertTrue(info.getLeftChild().toString().equals("names1"));

        assertTrue(this.ci.getRootNode().getVisibleDataRange().equals(
                     this.node.getDataRange()));

        //test getNodeByNode()-method
        try {
            assertEquals(
                info.getNodeByNode(this.node),
                info);
            assertEquals(
                info.getNodeByNode(this.leftnode),
                info.getLeftChild());
            assertEquals(
                info.getNodeByNode(this.rightnode),
                info.getRightChild());
            info.getLeftChild().getNodeByNode(this.node);
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {
        }
        try {
            info.getNodeByNode(null);
            assertTrue(false);
        }catch (NullPointerException e) {}

         //test getNodeByIndex()-method
        assertEquals(
            info.getNodeByIndex(0),
            info.getLeftChild());
        assertEquals(
            info.getNodeByIndex(1),
            info.getRightChild());
        assertEquals(
            info.getLeftChild().getNodeByIndex(0),
            info.getLeftChild());
        try {
            info.getNodeByIndex(-1);
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {}
        try {
            info.getNodeByIndex(2);
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {}

        //testing getNodeByVisibleIndex(int index)
        assertEquals(
            this.ci.getRootNode().getNodeByVisibleIndex(0),
            this.ci.getRootNode().getLeftChild());
        assertEquals(
            this.ci.getRootNode().getNodeByVisibleIndex(1),
            this.ci.getRootNode().getRightChild());
        assertEquals(
            this.ci.getRootNode().getLeftChild().getNodeByVisibleIndex(0),
            this.ci.getRootNode().getLeftChild());
        try {
            info.getNodeByVisibleIndex(-1);
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {}
        try {
            info.getNodeByVisibleIndex(2);
            assertTrue(false);
        } catch (IndexOutOfBoundsException e) {}

       
    }
    
 
    /**
     * Runs the test suite using JUnit's text-based runner.
     *
     * @param args  ignored.
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }
                        
 }   

 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
 
     
 
