/* =======================================================================
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D;
import java.awt.Point;
import java.awt.Color;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.lang.IllegalArgumentException;
import java.lang.Integer;
import java.lang.Math;
import java.util.ResourceBundle;
import java.util.TreeSet;

import javax.swing.JPanel;
import javax.swing.event.ChangeListener;
import javax.swing.event.ChangeEvent;

import org.jfree.ui.RectangleEdge;

import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartMouseListener;
import org.jfree.chart.axis.HeatMapAxis;
import org.jfree.chart.axis.CategoryLabelPositions;
import org.jfree.chart.entity.HCTreeNodeEntity;
import org.jfree.chart.entity.HeatMapBlockEntity;
import org.jfree.chart.entity.EntityCollection;
import org.jfree.chart.event.ClusteringTreeChangeEvent;
import org.jfree.chart.event.PlotChangeEvent;
import org.jfree.chart.event.SelectionChangeEvent;
import org.jfree.chart.labels.HCToolTipGenerator;
import org.jfree.chart.plot.HCMediator;
import org.jfree.chart.plot.AbstractHCClusteringInfo;
import org.jfree.chart.plot.DummyHCClusteringInfo;
import org.jfree.chart.plot.StandardHCClusteringInfo;
import org.jfree.chart.plot.HCTreeNodeInfo;
import org.jfree.chart.plot.Plot;
import org.jfree.data.hc.HCDataset;
import org.jfree.data.hc.HeatMap;
import org.jfree.data.hc.DataRange;
import org.jfree.data.hc.HCTreeNode;
import org.jfree.chart.editor.GradientColorPaletteEditor;
import org.jfree.chart.editor.HCOptionsEditor;


/**
 * A plot that displays data in the a Hierarchical clustering
 * and Heatmap visualisation. The plot uses
 * data from a {@link HCDataset} -object.
 *
 * @author viski project.
 */
public class HCPlot
    extends Plot
    implements
	ChartMouseListener,
	ChangeListener {

    private HCDataset dataset;
    private HeatMapAxis rowNames;
    private HeatMapAxis columnNames;
    private AbstractHCClusteringInfo rowClusteringInfo;
    private AbstractHCClusteringInfo columnClusteringInfo;
    private boolean columnNamesVisibility;
    private boolean rowNamesVisibility;
    private boolean columnTreeVisibility;
    private boolean rowTreeVisibility;

    private double columnTreeSize;
    private double rowTreeSize;
    private double columnNamesSize;
    private double rowNamesSize;
    private double topMarginSize;
    private double bottomMarginSize;
    private double leftMarginSize;
    private double rightMarginSize;

    private GradientColorPalette palette;
    private GradientColorPaletteEditor paletteEditor;
    private HCOptionsEditor optionsEditor;

    private Rectangle selection;

    public static final int LEFT = 0;
    public static final int TOP = 1;

    private HCToolTipGenerator toolTipGenerator;
    private boolean mouseOverHighlight;
    private int columnTreeHighlight;
    private int rowTreeHighlight;
    private boolean selectionHighlight;
    private boolean averageHighlight;

    private TreeSet closedRows;
    private TreeSet closedColumns;

    private boolean drawFromLeftToRight;
    private boolean drawFromTopToBottom;

    /**
     * Creates a new plot that will draw a Hierarchical Clustering and
     * Heatmap for the given dataset.
     *
     * @param dataset  the dataset.
     */
    public HCPlot(HCDataset dataset) {

	HCMediator rowHCMediator;
	HCMediator columnHCMediator;

	this.dataset=dataset;
	this.columnNamesVisibility = true;
	this.rowNamesVisibility = true;

	if (this.dataset.getColumnClusteringTree() != null) {

	    this.columnClusteringInfo = new StandardHCClusteringInfo(
		this.dataset.getColumnClusteringTree(),
		this.dataset.getHeatMap().getColumnNames(),
		this.TOP
	    );
	    this.columnTreeVisibility = true;

	} else {

	    this.columnClusteringInfo = new DummyHCClusteringInfo(
		this.dataset.getHeatMap().getColumnNames(),
		this.TOP
	    );
	    this.columnTreeVisibility = false;

	}

	if (this.dataset.getRowClusteringTree() != null) {

	    this.rowClusteringInfo = new StandardHCClusteringInfo(
		this.dataset.getRowClusteringTree(),
		this.dataset.getHeatMap().getRowNames(),
		this.LEFT
	    );
	    this.rowTreeVisibility = true;

	} else {

	    this.rowClusteringInfo = new DummyHCClusteringInfo(
		this.dataset.getHeatMap().getRowNames(),
		this.LEFT
	    );
	    this.rowTreeVisibility = false;

	}

	columnHCMediator = new HCMediator(this.columnClusteringInfo);
	this.columnClusteringInfo.addChangeListener(this);
	rowHCMediator = new HCMediator(this.rowClusteringInfo);
	this.rowClusteringInfo.addChangeListener(this);

	this.rowNames = new HeatMapAxis();
	this.rowNames.setPlot(rowHCMediator);
	this.rowNames.setLowerMargin(0);
	this.rowNames.setUpperMargin(0);
	this.rowNames.setMaximumCategoryLabelWidthRatio(
                (float)0.95);
	//this.rowNames.setCategoryLabelPositionOffset(4);

	this.columnNames = new HeatMapAxis();
	this.columnNames.setPlot(columnHCMediator);
	this.columnNames.setLowerMargin(0);
	this.columnNames.setUpperMargin(0);
	this.columnNames.setCategoryLabelPositions(

	    CategoryLabelPositions.UP_90
	);
	 this.columnNames.setMaximumCategoryLabelWidthRatio(
                (float)0.95);
	//this.columnNames = new HeatMapAxis();
	//this.columnNames.setPlot(columnHCMediator);
	//this.columnNames.setLowerMargin(0);
	//this.columnNames.setUpperMargin(-1);
	//this.columnNames.setCategoryLabelPositionOffset(0);
	//this.columnNames.setCategoryLabelPositions(

	this.optionsEditor = new HCOptionsEditor(this);
	this.paletteEditor = new GradientColorPaletteEditor();
	this.palette = null;
	this.setColoring(createDefaultColorMap());
	this.selection = null;

        this.closedColumns = new TreeSet();
	this.closedRows = new TreeSet();

	// these should be reasonable defaults
	double ratio = (double)(this.dataset.getHeatMap().getRowCount())
	    / this.dataset.getHeatMap().getColumnsCount();
	if ( ratio < 1 ) {

	    // less rows than columns, ratio < 1
	    this.columnTreeSize = 0.2;
	    this.rowTreeSize = 0.2*ratio;
	    this.columnNamesSize = 0.12;
	    this.rowNamesSize = 0.12*ratio;
	    this.topMarginSize = 1.0/64.0;
	    this.leftMarginSize = 1.0/64.0*ratio;

	} else {

	    // more rows than columns, ratio >= 1
	    this.rowTreeSize = 0.2;
	    this.columnTreeSize = 0.2/ratio;
	    this.rowNamesSize = 0.12;
	    this.columnNamesSize = 0.12/ratio;
	    this.leftMarginSize = 1.0/64.0;
	    this.topMarginSize = 1.0/64.0/ratio;

	}
	this.bottomMarginSize = this.topMarginSize;
	this.rightMarginSize = this.leftMarginSize;

	this.mouseOverHighlight = true;
	this.selectionHighlight = true;
	this.averageHighlight = true;
	this.columnTreeHighlight = -1;
	this.rowTreeHighlight = -1;
	
	this.drawFromLeftToRight = true;
	this.drawFromTopToBottom = true;

    }

    /**
     * Calculates min and max values of the heatmap and returns
     * a scaled palette for the heatmap.
     *
     * @return  The GradientColorPalette.
     */
    private GradientColorPalette createDefaultColorMap() {

	GradientColorPalette palette;

	double min=0;
	double max=0;
	int i;

	for (i = 0; i<this.dataset.getHeatMap().getItemCount(); i++) {

	    double value = this.dataset.getHeatMap().getItem(i).doubleValue();
	    if (value < min) min = value;
	    if (value > max) max = value;

	}

	return new GradientColorPalette(min,max);

    }

    /**
     * Returns the column clustering info
     *
     * @return The HCClusteringInfo info object.
     */
    public AbstractHCClusteringInfo getColumnClusteringInfo() {

	return this.columnClusteringInfo;

    }

    /**
     * Returns the row clustering info
     *
     * @return The HCClusteringInfo info object.
     */
    public AbstractHCClusteringInfo getRowClusteringInfo() {

	return this.rowClusteringInfo;

    }

    /**
     * Returns the dataset used with this plot.
     *
     * @return  The dataset.
     */
    public HCDataset getDataset() {

	return this.dataset;

    }

    /**
     * Returns a string describing the plot type.
     *
     * @return  The name.
     */
    public String getPlotType() {

	return "Hierarchical Clustering and Heatmap";

    }

    /**
     * Sets the row clustering tree visible.
     */
    public void showRowTree() {

	if (this.dataset.getRowClusteringTree() == null)
	    throw new NullPointerException("Cannot show a null tree.");
	this.rowTreeVisibility = true;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets the column clustering tree visible.
     */
    public void showColumnTree() {

	if (this.dataset.getColumnClusteringTree() == null)
	    throw new NullPointerException("Cannot show a null tree.");
	this.columnTreeVisibility = true;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets the row clustering tree invisible.
     */
    public void hideRowTree() {

	if (this.dataset.getRowClusteringTree() == null)
	    throw new NullPointerException("Cannot hide a null tree.");
	this.rowTreeVisibility = false;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets the column clustering tree invisible.
     */
    public void hideColumnTree() {

	if (this.dataset.getColumnClusteringTree() == null)
	    throw new NullPointerException("Cannot hide a null tree.");
	this.columnTreeVisibility = false;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns true if row clustering tree is visible, false otherwise.
     *
     * @return  A boolean.
     */
    public boolean getRowTreeVisibility() {

	if (this.dataset.getRowClusteringTree() == null)
	    return false;
	return this.rowTreeVisibility;

    }

    /**
     * Returns true if the column clustering tree is visible, false otherwise.
     *
     * @return  A boolean.
     */
    public boolean getColumnTreeVisibility() {

	if (this.dataset.getColumnClusteringTree() == null)
	    return false;
	return this.columnTreeVisibility;

    }

    /**
     * Sets column names visible.
     */
    public void showColumnNames() {

	this.columnNamesVisibility = true;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets row names visible.
     */
    public void showRowNames() {

	this.rowNamesVisibility = true;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets column names invisible.
     */
    public void hideColumnNames() {

	this.columnNamesVisibility = false;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets row names invisible.
     */
    public void hideRowNames() {

	this.rowNamesVisibility = false;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns true if the column names are visible, false otherwise.
     *
     * @return  A boolean.
     */
    public boolean getColumnNamesVisibility() {

	return this.columnNamesVisibility;

    }

    /**
     * Returns true if specified names are visible, false otherwise.
     *
     * @return  A boolean.
     */
    public boolean getRowNamesVisibility() {

	return this.rowNamesVisibility;

    }

    /**
     * Sets a palette to be used for coloring heatmap.
     *
     * @param color  the color palette.
     */
    public void setColoring(GradientColorPalette color) {

	if (this.palette != null) this.palette.removeChangeListener(this);
	this.palette = color;
	this.palette.addChangeListener(this);
	this.paletteEditor.setColoring (color);
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns the palette used for the heatmap.
     *
     * @return  The heatmap color palette.
     */
    public GradientColorPalette getColoring() {

	return this.palette;

    }

    /**
     * Sets row names-size as percentage of visualization.
     *
     * @param size  the size as proportion. 0 is 0 pixels, 1 is the whole
     * visualisation.
     */
    public void setRowNamesSize(double size) {

	if (size < 0) throw new IllegalArgumentException(
		"Size of names cannot be negative.");
	if (size > 1) throw new IllegalArgumentException(
		"Size of names cannot be more than 1.");
	this.rowNamesSize = size;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets column names-size as percentage of visualization.
     *
     * @param size  the size as proportion. 0 is 0 pixels, 1 is the whole
     * visualisation.
     */
    public void setColumnNamesSize(double size) {

	if (size < 0) throw new IllegalArgumentException(
		"Size of names cannot be negative.");
	if (size > 1) throw new IllegalArgumentException(
		"Size of names cannot be more than 1.");
	this.columnNamesSize = size;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns row names-size as percentage of visualization.
     *
     * @return  A value between [0,1] that specifies the size of names.
     */
    public double getRowNamesSize() {

	return this.rowNamesSize;

    }

    /**
     * Returns column names-size as percentage of visualization.
     *
     * @return  A value between [0,1] that specifies the size of names.
     */
    public double getColumnNamesSize() {

	return this.columnNamesSize;

    }

    /**
     * Sets row tree-size as percentage of visualization.
     *
     * @param size  the size as proportion. 0 is 0 pixels, 1 is the whole
     * visualisation.
     */
    public void setRowTreeSize(double size) {

	if (size < 0) throw new IllegalArgumentException(
		"Size of a tree cannot be negative.");
	if (size > 1) throw new IllegalArgumentException(
		"Size of a tree cannot be more than 1.");
	this.rowTreeSize = size;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets column tree-size as percentage of visualization.
     *
     * @param size  the size as proportion. 0 is 0 pixels, 1 is the whole
     * visualisation.
     */
    public void setColumnTreeSize(double size) {

	if (size < 0) throw new IllegalArgumentException(
		"Size of a tree cannot be negative.");
	if (size > 1) throw new IllegalArgumentException(
		"Size of a tree cannot be more than 1.");
	this.columnTreeSize = size;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns the size of the row tree as percentage of visualisation size.
     *
     * @return  A value between [0,1] that specifies the size of the tree.
     */
    public double getRowTreeSize() {

	return this.rowTreeSize;

    }

    /**
     * Returns the size of the row tree as percentage of visualisation size.
     *
     * @return  A value between [0,1] that specifies the size of the tree.
     */
    public double getColumnTreeSize() {

	return this.columnTreeSize;

    }

    /**
     * Sets the selection.
     *
     * @param selection  the rectangle specifying which blocks of the visible
     * heatmap belong to the selection. Indexing begins from 0.
     *
     */
    public void setSelection(Rectangle selection)
	throws IndexOutOfBoundsException {

	if (selection.getMinX() < 0) throw new IndexOutOfBoundsException();
	if (selection.getMinY() < 0) throw new IndexOutOfBoundsException();
	if (selection.getMaxX() > this.dataset.getHeatMap().getColumnsCount())
	    throw new IndexOutOfBoundsException();
	if (selection.getMaxY() > this.dataset.getHeatMap().getRowCount())
	    throw new IndexOutOfBoundsException();

	Rectangle oldSelection = this.selection;
	this.selection = selection;
        notifyListeners(new SelectionChangeEvent(this,oldSelection));

    }

    /**
     * Returns the current selection
     *
     * @return  The rectangle specifying which blocks of the visible
     * heatmap belong to the selection. Indexing begins from 0.
     */
    public Rectangle getSelection() {

	return this.selection;
    }

    /**
     * Turns on or off the feature, that makes
     * branches of trees highlighted when the mouse cursor is moved
     * on the corresponding heatmap blocks.
     *
     * @param highlight  A boolean.
     */
    public void setMouseOverHighlight(boolean highlight) {

	this.mouseOverHighlight = highlight;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns the state of the feature, that makes
     * branches of trees highlighted, when mouse cursor is moved
     * on corresponding heatmap blocks.
     *
     * @return  True, if the feature is turned on,
     * false otherwise.
     */
    public boolean getMouseOverHighlight() {

	return this.mouseOverHighlight;

    }

    /**
     * Turns on or off the feature, that makes the
     * heatmap bloks highlighted, when the mouse is clicked
     * on the corresponding heatmapblocks or tree nodes.
     *
     * @param highlight  a boolean.
     */
    public void setSelectionHighlight(boolean highlight) {

        this.selectionHighlight = highlight;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns the state of the feature, that makes
     * heatmap blocks highlighted, when mouse is clicked
     * on corresponding heatmap blocks or tree nodes.
     *
     * @return  True, if the feature is turned on,
     * false otherwise.
     */
    public boolean getSelectionHighlight() {

        return this.selectionHighlight;

    }

    /**
     * Turns on or off the feature, that makes
     * heatmap blocks highlighted, when a corresponding clustering tree
     * node is closed by a double click.
     *
     * @param highlight  A boolean.
     */
    public void setAverageHighlight(boolean highlight) {

        this.averageHighlight = highlight;
        notifyListeners(new PlotChangeEvent(this));

    }


    /**
     * Returns the state of the feature, that makes
     * heatmap blocks highlighted, when a corresponding clustering tree
     * node is closed by double click.
     *
     * @return  True, if the feature is turned on,
     * false otherwise.
     */
    public boolean getAverageHighlight() {

        return this.averageHighlight;

    }

    /**
     * Sets specified branch on the row tree highlighted.
     *
     * @param index  The index of the heatmap row that
     * specifies the branch to highlight. Negative
     * values turn highlight off.
     */
    public void setRowTreeHighlight(int index) {
	
	this.rowTreeHighlight = index;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Sets specified branch on the column tree highlighted.
     *
     * @param index  The index of the heatmap column that
     * specifies the branch to highlight. Negative
     * values turn highlight off.
     */
    public void setColumnTreeHighlight(int index) {
	
	this.columnTreeHighlight = index;
        notifyListeners(new PlotChangeEvent(this));

    }

    /**
     * Returns the index of the highlighted branch.
     *
     * @return  The index of the heatmap row that
     * specifies the currently highlighted clustering tree branch.
     */
    public int getRowTreeHighlight() {
	
	return this.rowTreeHighlight;

    }

    /**
     * Returns the index of the highlighted branch.
     *
     * @return  The index of the heatmap column that
     * specifies the currently highlighted clustering tree branch.
     */
    public int getColumnTreeHighlight() {
	
	return this.columnTreeHighlight;

    }

    /**
     * Returns the number of visible rows on the heatmap.
     *
     * @return  The number of visible rows on the heatmap.
     */
    public int getHeatMapRowCount() {

	return this.rowClusteringInfo.getNumberOfVisibleItems();

    }

    /**
     * Returns the number of visible columns on the heatmap.
     *
     * @return  The number of visible columns on heatmap.
     */
    public int getHeatMapColumnsCount() {

	return this.columnClusteringInfo.getNumberOfVisibleItems();

    }

    /**
     * Returns the indexes of the rows of the dataset
     * that corresponds to the specified row of the
     * visible heatmap.
     *
     * @param row  the row.
     * @return  The datarange specifying the rows.
     */
    public DataRange getDatasetRowsAtHeatMapRow(int row) {

	return this.rowClusteringInfo.getDataRangeForVisibleIndex(row);

    }

    /**
     * Returns the indexes of the columns of the dataset
     * that corresponds to the specified column of the
     * visible heatmap.
     *
     * @param column  the column.
     * @return  The datarange specifying the columns.
     */
    public DataRange getDatasetColumnsAtHeatMapColumn(int column) {

	return this.columnClusteringInfo.getDataRangeForVisibleIndex(column);

    }

    /**
     * Draws a subtree of the clustering tree rooted at specified node.
     * This method also creates HCTreeNodeEntities of the nodes drawn.
     *
     * @param g2  the Graphics2D object to draw the subtree on
     * @param state  a state object containing information about the geometry
     * of the plot.
     * @param node  the node the subtree is rooted at.
     * @param area  the area where the subtree has to fit.
     * @param edge  edge of the heatmap the subtree is drawn to.
     */
    private HCTreeNodeEntity drawSubTree(
	    Graphics2D g2,
	    HCPlotState state,
	    HCTreeNodeInfo node,
	    Rectangle2D area,
	    RectangleEdge edge) {

	HCTreeNodeEntity leftEntity;
	HCTreeNodeEntity rightEntity;
	HCTreeNodeEntity thisEntity;
	Rectangle thisArea;
	Rectangle2D areaForRightSubTree;
	Point center;
	EntityCollection entities;
	String tip;

	// first we calculate needed areas and draw subtrees.

	if (!node.isNodeOpen()) {

	    // closed node
	    thisArea = state.calculateClosedNodeArea(
		area,
		node,
		edge
	    );
	    center = state.calculateClosedNodeCenter(
		thisArea,
		node.getNode().getHeight(),
		edge
	    );


	} else if (
	    (node.getLeftChild() == null) ||
	    (node.getRightChild() == null)
	) {

	    // open leaf node
	    // Currently (2006-08-20), this block should be unreachable.
	    thisArea = state.calculateLeafNodeArea(area,node,edge);
	    center = state.calculateLeafNodeCenter(thisArea,edge);

	} else {

	    // open branch node
	    leftEntity = drawSubTree (
		g2,
		state,
		node.getLeftChild(),
		area,
		edge
	    );

	    areaForRightSubTree = state.calculateSubTreeArea(
		area,
		leftEntity.getSubTreeArea(),
		edge
	    );

	    rightEntity = drawSubTree (
		g2,
		state,
		node.getRightChild(),
		areaForRightSubTree,
		edge
	    );
	    thisArea = state.calculateBranchNodeArea(
		node.getNode().getHeight(),
		leftEntity,
		rightEntity,
		area,
		edge
	    );
	    center = state.calculateBranchNodeCenter(
		node.getNode().getHeight(),
		leftEntity,
		rightEntity,
		area,
		edge
	    );

	}

	// the only thing left to do is to create the entity and actually
	// draw this node.
	if (this.toolTipGenerator != null) {

	    tip = this.toolTipGenerator.generateToolTip(node);

	} else tip = null;
	thisEntity = new HCTreeNodeEntity(
	    new Rectangle(
		(int)center.getX()-5,
		(int)center.getY()-5,
		11,
		11
	    ),
	    tip,
	    null,
	    center,
	    thisArea,
	    node
	);
	entities = state.getEntityCollection();
	if (entities != null) {

		entities.add(thisEntity);

	}
	redrawNode(g2, state, thisEntity, edge);

	return thisEntity;

    }

    /**
     * Draws a single node of a clustering tree.
     * This does not create HCTreeNodeEntities so it can be used repeatedly.
     *
     * @param g2  the Graphics2D object to draw the subtree on
     * @param state  a state object containing information about the geometry
     * of the plot.
     * @param entity  the entity describing the node being drawn.
     * @param edge  edge of the heatmap the subtree is drawn to.
     */
    private void redrawNode(
	    Graphics2D g2,
	    HCPlotState state,
	    HCTreeNodeEntity entity,
	    RectangleEdge edge) {

	int centerX = (int)entity.getCenter().getX();
	int centerY = (int)entity.getCenter().getY();
	Rectangle subTreeArea = entity.getSubTreeArea();
	int width = (int)(subTreeArea.getWidth());
	int height = (int)(subTreeArea.getHeight());
	int minX = (int)(subTreeArea.getMinX());
	int minY = (int)(subTreeArea.getMinY());
	int maxX = (int)(subTreeArea.getMinX()+width);
	int maxY = (int)(subTreeArea.getMinY()+height);
	HCTreeNode node = entity.getHCTreeNodeInfo().getNode();
	double height1;
	double height2;

	g2.setPaint(getOutlinePaint());

	if (entity.getHCTreeNodeInfo().isNodeOpen()) {

	    // open nodes
	    if ((node.getLeftChild() != null) &&
		(node.getRightChild() != null)
	    ) {

		// Root and branch nodes
		height1 = node.getLeftChild().getHeight();
		height2 = node.getRightChild().getHeight();
		if (edge == RectangleEdge.TOP) {

		    g2.drawLine(
			minX,
			centerY,
			minX,
			maxY-(int)(
			    height1*state.getColumnTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			));
		    g2.drawLine(
			maxX,
			centerY,
			maxX,
			maxY-(int)(
			    height2*state.getColumnTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			));
		    g2.drawLine(
			minX,
			centerY,
			maxX,
			centerY);

		} else if (edge == RectangleEdge.BOTTOM) {

		    // Currently (2006-08-20), this block should be unreachable.

		    g2.drawLine(
			minX,
			centerY,
			minX,
			minY+(int)(
			    height1*state.getColumnTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			));
		    g2.drawLine(
			maxX,
			centerY,
			maxX,
			minY+(int)(
			    height2*state.getColumnTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			));
		    g2.drawLine(
			minX,
			centerY,
			maxX,
			centerY);

		} else if(edge == RectangleEdge.LEFT) {

		    g2.drawLine(
			centerX,
			minY,
			maxX-(int)(
			    height1*state.getRowTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			),
			minY);
		    g2.drawLine(
			centerX,
			maxY,
			maxX-(int)(
			    height2*state.getRowTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			),
			maxY);
		    g2.drawLine(
			centerX,
			minY,
			centerX,
			maxY);

		} else if(edge == RectangleEdge.RIGHT) {

		    // Currently (2006-08-20), this block should be unreachable.

		    g2.drawLine(
			centerX,
			minY,
			minX+(int)(
			    height1*state.getRowTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			),
			minY);
		    g2.drawLine(
			centerX,
			maxY,
			minX+(int)(
			    height2*state.getRowTreeHeightUnitInPixels()
			    +state.getSizeOfNodeSymbol()/2
			),
			maxY);
		    g2.drawLine(
			centerX,
			minY,
			centerX,
			maxY);

		} else throw new IllegalArgumentException("Invalid edge.");

	    } else {
		
		; //throw new Exception("leaf nodes should always be closed.");

	    }
	    Rectangle r = new Rectangle(
		    centerX-state.getSizeOfNodeSymbol()/2,
		    centerY-state.getSizeOfNodeSymbol()/2,
		    state.getSizeOfNodeSymbol(),
		    state.getSizeOfNodeSymbol());
	    g2.fill(r);

	} else {

	    // closed node.
	    Rectangle r;

	    if ((node.getLeftChild() == null) ||
		(node.getRightChild() == null)
	    ) {

		// closed leaf node.
		if (edge == RectangleEdge.TOP) {

		    r = new Rectangle(
			centerX-state.getSizeOfNodeSymbol()/2,
			centerY-state.getSizeOfNodeSymbol()/2,
			state.getSizeOfNodeSymbol(),
			state.getSizeOfNodeSymbol()/2);

		} else if (edge == RectangleEdge.BOTTOM) {

		    // Currently (2006-08-20), this block should be unreachable.
		    r = new Rectangle(
			centerX-state.getSizeOfNodeSymbol()/2,
			centerY,
			state.getSizeOfNodeSymbol(),
			state.getSizeOfNodeSymbol()/2);

		} else if (edge == RectangleEdge.LEFT) {

		    r = new Rectangle(
			centerX-state.getSizeOfNodeSymbol()/2,
			centerY-state.getSizeOfNodeSymbol()/2,
			state.getSizeOfNodeSymbol()/2,
			state.getSizeOfNodeSymbol());

		} else if (edge == RectangleEdge.RIGHT) {

		    // Currently (2006-08-20), this block should be unreachable.
		    r = new Rectangle(
			centerX,
			centerY-state.getSizeOfNodeSymbol()/2,
			state.getSizeOfNodeSymbol()/2,
			state.getSizeOfNodeSymbol());

		} else throw new IllegalArgumentException("Invalid edge.");

	    } else {

		// closed branch  node.
	    	r = new Rectangle(
		    centerX-(state.getSizeOfNodeSymbol())/2,
		    centerY-(state.getSizeOfNodeSymbol())/2,
		    state.getSizeOfNodeSymbol(),
		    state.getSizeOfNodeSymbol());

	    }
	    if (edge == RectangleEdge.TOP) {

		g2.drawLine( centerX, (int)r.getMaxY(), centerX, maxY );

	    } else if (edge == RectangleEdge.BOTTOM) {

		// Currently (2006-08-20), this block should be unreachable.
		g2.drawLine( centerX, (int)r.getMinY(), centerX, minY );

	    } else if(edge == RectangleEdge.LEFT) {

		g2.drawLine( (int)r.getMaxX(),centerY,maxX,centerY);

	    } else if(edge == RectangleEdge.RIGHT) {

		// Currently (2006-08-20), this block should be unreachable.
		g2.drawLine( (int)r.getMinX(),centerY,minX,centerY);

	    } else throw new IllegalArgumentException("Invalid edge.");

	    g2.draw(r);
	}

    }


    /**
     * Draws a single block of heatmap.
     *
     * @param g2  the Graphics2D object to draw the subtree on.
     * @param area  the area, where the heatmap is being drawn.
     * @param state  a state object containing information about the geometry
     * of the plot.
     * @param row  the row of the heatmap block.
     * @param column  the column of the heatmap block.
     */
    private void drawBlock (
	    Graphics2D g2,
	    Rectangle2D area,
	    HCPlotState state,
	    int row,
	    int column) {
	DataRange columns;
	DataRange rows;
	HeatMapBlockEntity entity;
	EntityCollection entities;
	Rectangle r;
	int minRow;
	int maxRow;
	int minColumn;
	int maxColumn;
	int rowCounter;
	int columnCounter;
	int blockCount;
	double averageValue;
	String tip;

	// O(lg sqrt(n))
	columns = this.columnClusteringInfo.getDataRangeForVisibleIndex(column);
	rows = this.rowClusteringInfo.getDataRangeForVisibleIndex(row);

	try {
	    minColumn = columns.getLeftBound();
	    maxColumn = columns.getRightBound();
	    minRow = rows.getLeftBound();
	    maxRow = rows.getRightBound();
	} catch (Exception e) {
	    return; // trying to draw something that doesn't exist. throw?
	}
	// any reason to cache these? don't think so.
	for (
	    averageValue = 0, blockCount = 0, rowCounter = minRow;
	    rowCounter <= maxRow;
	    rowCounter++
	) {
	    for (
		columnCounter = minColumn;
		columnCounter <= maxColumn;
		columnCounter++, blockCount++
	    ) {
		averageValue += this.dataset.getHeatMap()
		    .get(rowCounter,columnCounter);
	    }
	}
	averageValue = averageValue/blockCount;

	g2.setColor(this.palette.getColor(averageValue));

        // we need to calculate width and height stupidly to
	// handle rounding problems.
	r = new Rectangle(
	    state.getHeatMapXCoordinate(column),
	    state.getHeatMapYCoordinate(row),
	    state.getHeatMapXCoordinate(column+1) - 
		state.getHeatMapXCoordinate(column),
	    state.getHeatMapYCoordinate(row+1) -
		state.getHeatMapYCoordinate(row)
	);

        g2.fill(r);

        if (blockCount>1 && getAverageHighlight()) {

            if ( rows.getWidth() > 1) {
               Integer introw = new Integer(row);
               closedRows.add(introw);
            }
            if ( columns.getWidth() > 1) {
               Integer intcolumn = new Integer(column);
               closedColumns.add(intcolumn);
            }

        }

	if (this.toolTipGenerator != null) {
	    tip = this.toolTipGenerator.generateToolTip(
		    this.dataset.getHeatMap(),
		    rows,
		    columns
	    );
	}
	else tip = null;
	entity = new HeatMapBlockEntity(
	    r,
	    tip,
	    null,
	    row,
	    column
	);
	entities = state.getEntityCollection();
	if (entities != null) {
	    entities.add(entity);
	}

    }


    /**
     * Draws the rectangles indicating selections in the heatmap.
     *
     * @param g2  the Graphics2D object to draw the heatmap to.
     * @param state  a state object containing information about the geometry
     * of the plot.
     * @param color  color to use.
     * @param r  the rectangle specifying the selection.
     */
    private void drawSelection(Graphics2D g2, HCPlotState state, Color color, Rectangle r) {

            g2.setColor(color);

	    int column = (int)r.getX();
            int row = (int)r.getY();
            int lastrow = row + (int)r.getHeight()-1;
            int lastcolumn = column + (int)r.getWidth()-1;

	    g2.draw(new Rectangle(
		state.getHeatMapXCoordinate(column),
		state.getHeatMapYCoordinate(row),
		state.getHeatMapXCoordinate(lastcolumn+1)
                    -state.getHeatMapXCoordinate(column)-1,
                state.getHeatMapYCoordinate(lastrow+1)
                    -state.getHeatMapYCoordinate(row)-1
	    ));
    }

    /**
     * Draws the heatmap.
     *
     * @param g2  the object to draw the heatmap on.
     * @param area  where to exactly draw heatmap.
     */
    private void drawHeatMap(
	    Graphics2D g2,
	    HCPlotState state,
	    Rectangle2D area,
	    int visibleColumns,
	    int visibleRows
	    ) {

	int height;
	int width;
	int row;
	int column;
	int minRow;
	int minColumn;
	int maxRow;
	int maxColumn;
	int count = 0;

	Rectangle clip = g2.getClip().getBounds();

	// only draw as many blocks as there are in the clipping region.
	minColumn = (int)Math.floor(
		(clip.getMinX() - area.getMinX()) / state.getBlockWidth()
		);
	if (minColumn < 0) minColumn = 0;

	minRow = (int)Math.floor(
		(clip.getMinY() - area.getMinY()) / state.getBlockHeight()
		);
	if (minRow < 0) minRow = 0;

	maxColumn = (int)Math.ceil(
		(clip.getMaxX() - area.getMinX()) / state.getBlockWidth()
		);
	if (maxColumn > visibleColumns) maxColumn = visibleColumns;

	maxRow = (int)Math.ceil(
		(clip.getMaxY() - area.getMinY()) / state.getBlockHeight()
		);
	if (maxRow > visibleRows) maxRow = visibleRows;

	for (row=minRow; row < maxRow; row ++) {

	    for (column=minColumn; column < maxColumn; column ++) {

		drawBlock (g2,area,state,row,column);
		count++;

	    }

	}

    }

    /**
     * Draws the plot.
     *
     * @param g2  the graphics context.
     * @param area  the area.
     * @param anchor  the anchor.
     * @param parentState   the parentstate.
     * @param info  the info.
     */
    public void draw(
	    Graphics2D g2,
	    Rectangle2D area,
	    Point2D anchor,
	    PlotState parentState,
	    PlotRenderingInfo info) {

	// Most of this method is just calculating exactly where to draw
	// what. The names of the variables should be descriptive
	// enough for anybody to follow.
	int visibleColumns = this.getHeatMapColumnsCount();
	int visibleRows = this.getHeatMapRowCount();
	int totalColumns = this.dataset.getHeatMap().getColumnsCount();
	int totalRows = this.dataset.getHeatMap().getRowCount();

	int topMargin = (int)(area.getHeight() * this.topMarginSize);
	int bottomMargin = (int)(area.getHeight() * this.bottomMarginSize);
	int leftMargin = (int)(area.getWidth() * this.leftMarginSize);
	int rightMargin = (int)(area.getWidth() * this.rightMarginSize);
	int minX = (int)area.getMinX()+leftMargin;
	int minY = (int)area.getMinY()+topMargin;
	int width = (int)area.getWidth()-leftMargin-rightMargin;
	int height = (int)area.getHeight()-bottomMargin-topMargin;
	int columnTreeHeight = (int)(height *this.columnTreeSize);
	int rowTreeWidth = (int)(width *this.rowTreeSize);
	if (!this.rowTreeVisibility) rowTreeWidth = 0;
	if (!this.columnTreeVisibility) columnTreeHeight = 0;
	int columnNamesInitialHeight = (int)(height * this.columnNamesSize);
	int rowNamesInitialWidth = (int)(width * this.rowNamesSize);
	int heatMapWidth = (int)((double)
	    ( area.getWidth() - rowTreeWidth - rowNamesInitialWidth)
	    * visibleColumns / totalColumns);
	int heatMapHeight = (int)((double)
	    ( area.getHeight() - columnTreeHeight - columnNamesInitialHeight)
	    * visibleRows / totalRows);
	int columnNamesHeight = height - heatMapHeight - columnTreeHeight;
	int rowNamesWidth = width - heatMapWidth - rowTreeWidth;
	if (!this.rowNamesVisibility) rowNamesWidth = 0;
	if (!this.columnNamesVisibility) columnNamesHeight = 0;
	int maxX = minX+width;
	int maxY = minY+height;
	int heatMapMinX;
	int heatMapMinY;
	int heatMapMaxX;
	int heatMapMaxY;
	RectangleEdge columnNamesLocation;
	RectangleEdge rowNamesLocation;
	RectangleEdge columnTreeLocation;
	RectangleEdge rowTreeLocation;

	Rectangle columnTreeArea;
	Rectangle rowTreeArea;
	Rectangle columnNamesArea;
	Rectangle rowNamesArea;
	Rectangle heatMapArea;

	if (drawFromLeftToRight) {
	    
	    heatMapMinX = minX+rowTreeWidth;
	    rowTreeLocation = RectangleEdge.LEFT;
	    rowNamesLocation = RectangleEdge.RIGHT;

	} else {
	    
        // Currently (2006-08-16), this block is unreachable.
	    heatMapMinX = minX+rowNamesWidth;
	    rowTreeLocation = RectangleEdge.RIGHT;
	    rowNamesLocation = RectangleEdge.LEFT;

	}

	if (drawFromTopToBottom) {
	    
	    heatMapMinY = minY+columnTreeHeight;
	    columnTreeLocation = RectangleEdge.TOP;
	    columnNamesLocation = RectangleEdge.BOTTOM;

	} else {
	    
        // Currently (2006-08-16), this block is unreachable.
	    heatMapMinY = minY+columnNamesHeight;
	    columnTreeLocation = RectangleEdge.BOTTOM;
	    columnNamesLocation = RectangleEdge.TOP;

	}

	heatMapMaxX = heatMapMinX + heatMapWidth;
	heatMapMaxY = heatMapMinY + heatMapHeight;

	if (drawFromTopToBottom) {
	    
	    columnTreeArea = new Rectangle(
		heatMapMinX, minY, heatMapWidth, columnTreeHeight
	    );
	    columnNamesArea = new Rectangle(
		heatMapMinX, heatMapMaxY, heatMapWidth, maxY-heatMapMaxY
	    );

	} else {
	    
        // Currently (2006-08-16), this block is unreachable.
	    columnTreeArea = new Rectangle(
		heatMapMinX, heatMapMaxY, heatMapWidth, columnTreeHeight
	    );
	    columnNamesArea = new Rectangle(
		heatMapMinX, minY, heatMapWidth, maxY-heatMapMaxY
	    );

	}

	if (drawFromLeftToRight) {

	    rowTreeArea = new Rectangle(
		minX, heatMapMinY, rowTreeWidth, heatMapHeight
	    );
	    rowNamesArea = new Rectangle(
		heatMapMaxX, heatMapMinY, maxX-heatMapMaxX, heatMapHeight
	    );
	    
	} else {
	    
        // Currently (2006-08-16), this block is unreachable.
	    rowTreeArea = new Rectangle(
		heatMapMaxX, heatMapMinY, rowTreeWidth, heatMapHeight
	    );
	    rowNamesArea = new Rectangle(
		minX, heatMapMinY, maxX-heatMapMaxX, heatMapHeight
	    );

	}

	heatMapArea = new Rectangle(
	    heatMapMinX, heatMapMinY, heatMapWidth, heatMapHeight
	);

	int sizeOfNodeSymbol;
	double getColumnTreeHeightUnitInPixels;
	double getRowTreeHeightUnitInPixels;
	double blockWidth = columnTreeArea.getWidth()/visibleColumns;
	double blockHeight = rowTreeArea.getHeight()/visibleRows;

	if (blockWidth < blockHeight)
	    sizeOfNodeSymbol = (int)(3 + blockWidth/5);
	else 
	    sizeOfNodeSymbol = (int)(3 + blockHeight/5);

	if (this.dataset.getColumnClusteringTree() != null)
	    getColumnTreeHeightUnitInPixels = columnTreeArea.getHeight() /
		this.dataset.getColumnClusteringTree().getHeight();
	else
	    getColumnTreeHeightUnitInPixels = 0;

	if (this.dataset.getRowClusteringTree() != null)
	    getRowTreeHeightUnitInPixels = rowTreeArea.getWidth() /
		this.dataset.getRowClusteringTree().getHeight();
	else
	    getRowTreeHeightUnitInPixels = 0;

	HCPlotState state = new HCPlotState(
	    info,
	    (int)(heatMapArea.getMinX()),
	    (int)(heatMapArea.getMinY()),
	    sizeOfNodeSymbol,
	    blockWidth,
	    blockHeight,
	    getColumnTreeHeightUnitInPixels,
	    getRowTreeHeightUnitInPixels);

	if (info != null) {

	    info.setPlotArea(area);
	    info.setDataArea(heatMapArea);

	}

	// ok, the main geometry is calculated now. We begin
	// drawing here.

	drawBackground(g2,area);

	drawHeatMap(g2,state,heatMapArea,visibleColumns,visibleRows);

	if ( (getColumnTreeVisibility ()) &&
	    (g2.getClip().intersects(columnTreeArea))) {

	    drawSubTree(
		g2,
		state,
		this.columnClusteringInfo.getRootNode(),
		columnTreeArea,
		columnTreeLocation
	    );

	}
	if ( (getRowTreeVisibility ()) &&
	    (g2.getClip().intersects(rowTreeArea))) {

	    drawSubTree(
		g2,
		state,
		this.rowClusteringInfo.getRootNode(),
		rowTreeArea,
		rowTreeLocation
	    );

	}
	if (getRowNamesVisibility ()) {

	    this.rowNames.draw(
		g2,
		RectangleEdge.coordinate(heatMapArea,rowNamesLocation),
		area,
		rowNamesArea,
		rowNamesLocation,
		info

	    );

	}
	if (getColumnNamesVisibility ()) {

	    this.columnNames.draw(
		g2,
		RectangleEdge.coordinate(heatMapArea,columnNamesLocation),
		area,
		columnNamesArea,
		columnNamesLocation,
		info
	    );
	}

        if ( !closedRows.isEmpty() ) {

           int rowcount = closedRows.size();
           Integer item = null;
           Color color = new Color(0,0,0);

           int i=0;
           while (i < rowcount) {
               i++;
               item = (Integer)closedRows.first();
               closedRows.remove(item);
               Rectangle r = new Rectangle(
                         0,
                         item.intValue(),
                         visibleColumns,
                         1
               );
               drawSelection(g2, state, color, r);
           }
        }

        if ( !closedColumns.isEmpty() ) {

           int columncount = closedColumns.size();
           Integer item = null;
           Color color = new Color(0,0,0);

           int i=0;
           while (i < columncount) {
               i++;
               item = (Integer)closedColumns.first();
               closedColumns.remove(item);
               Rectangle r = new Rectangle(
                         item.intValue(),
                         0,
                         1,
                         visibleRows
               );
               drawSelection(g2, state, color, r);
           }
        }

        if ( this.selection != null  && getSelectionHighlight()) {
            Color color = new Color(255,255,255);
            drawSelection(g2, state, color, this.selection);
        }

    }

    /**
     * Implements the ChartMouseListener interface.
     *
     * @param event  the mouse event.
     */
    public void chartMouseMoved(ChartMouseEvent event) {

	if ((this.mouseOverHighlight) && (event.getEntity() != null)) {

	    if (event.getEntity() instanceof HeatMapBlockEntity) {

		HeatMapBlockEntity entity =
		    (HeatMapBlockEntity)event.getEntity();
		int oldRow = getRowTreeHighlight ();
		int oldColumn = getColumnTreeHighlight ();

		if ((oldRow != entity.getRow())
		    || (oldColumn != entity.getColumn())) {

		    setRowTreeHighlight (entity.getRow());
		    setColumnTreeHighlight (entity.getColumn());

		    //redrawHighlightedNodes();

		}

	    } else if (event.getEntity() instanceof HCTreeNodeEntity) {

		;

	    }

        }

    }

    /**
     * Implements the ChartMouseListener interface.
     *
     * @param event  the mouse event.
     */
    public void chartMouseClicked(ChartMouseEvent event) {

	MouseEvent e = event.getTrigger();
        if (event.getEntity() != null) {

	    if (event.getEntity() instanceof HeatMapBlockEntity &&

                getSelectionHighlight()) {

		HeatMapBlockEntity entity =
		    (HeatMapBlockEntity)event.getEntity();

		Rectangle selectionCandidate = new Rectangle (
		    entity.getColumn(),
		    entity.getRow(),
		    1,
		    1
		);
		if (selectionCandidate.equals(this.selection)) {

		    this.selection=null;

		} else {

		    setSelection ( selectionCandidate );

                }

	    } else if (event.getEntity() instanceof HCTreeNodeEntity) {

            	HCTreeNodeEntity entity =
		    (HCTreeNodeEntity)event.getEntity();
		HCTreeNodeInfo node = entity.getHCTreeNodeInfo();
		boolean alt = event.getTrigger().isAltDown();
		boolean shift = event.getTrigger().isShiftDown();
		boolean control = event.getTrigger().isControlDown();
		if (shift) {

		    node.setSubTreeOpen(!node.isNodeOpen());

		} else if (!control) {

                    this.selection = null;
		    node.setNodeOpen(!node.isNodeOpen());

		} else if (control) {

                    handleClick(node);

                }

            }
	    

        }

    }
    

    /**
     * Handels the click.
     *
     * @param node  the node that is clicked.
     */
    private void handleClick(HCTreeNodeInfo node) {

        int treelocation = node.getClusteringInfo().getLocation();
        int leftbound=0;
        int rightbound=0;
        Rectangle r;

        try {
            leftbound = node.getVisibleDataRange().getLeftBound();
            rightbound = node.getVisibleDataRange().getRightBound();
        } catch (Exception e){
            return;
        }

        if (treelocation == this.LEFT) {
            r = new Rectangle(
                0,
                leftbound,
                this.getHeatMapColumnsCount(),
                rightbound-leftbound+1
            );
        }
        else if (treelocation == this.TOP) {
            r = new Rectangle(
                leftbound,
                0,
                rightbound-leftbound+1,
                this.getHeatMapRowCount()


            );

        }
        else r=null; // Currently (2006-08-21), this block is unreachable.

        if (this.selection != null && this.selection.equals(r))
            this.selection = null;
        else
            this.setSelection(r);


    }


    /**
     * Implements the ChangeListener interface.
     *
     * @param event  the state change event.
     */
    public void stateChanged(ChangeEvent event) {
 
	Object source = event.getSource();
	if (source instanceof HCTreeNodeInfo) {

	    // TODO: shouldn't the received event already be a
	    // ClusteringTreeChangeEvent in the first place?
	    HCTreeNodeInfo info = (HCTreeNodeInfo)(event.getSource());
	    this.notifyListeners(new ClusteringTreeChangeEvent (this, info));

	}

    }

    /**
     * Returns the tooltip generator of a plot.
     *
     * @return  The tooltip generator.
     */
    public HCToolTipGenerator getToolTipGenerator() {

        return this.toolTipGenerator;

    }

    /**
     * Sets the tooltip generator of a plot.
     *
     * @param toolTipGenerator  the tooltip generator.
     */
    public void setToolTipGenerator(HCToolTipGenerator toolTipGenerator) {

        this.toolTipGenerator = toolTipGenerator;

    }

    /**
     * Creates a new JPanel for adjusting visualisation settings.
     *
     * @return  The panel.
     */
    public JPanel getOptionsPanel() {

	return this.optionsEditor.getPanel();

    }

    /**
     * Creates a new JPanel for adjusting heatmap colors.
     *
     * @return  The panel.
     */
    public JPanel getPalettePanel() {

	return this.paletteEditor.getPanel();

    }

    /**
     * Returns a name for the visualisation options panel.
     *
     * @return  Panel name as a string.
     */
    public String getOptionsPanelName() {

        ResourceBundle lr = ResourceBundle.getBundle("org.jfree.chart.editor.LocalizationBundle");
        return lr.getString("HC_options_panel");

    }
    
    /**
     * Returns a name for the heatmap coloring panel.
     *
     * @return  Panel name as a string.
     */
    public String getPalettePanelName() {

        ResourceBundle lr = ResourceBundle.getBundle("org.jfree.chart.editor.LocalizationBundle");
        return lr.getString("HC_palette_panel");

    }

}

