/* =======================================================================
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jfree.data.hc.HCTreeNode;
import org.jfree.data.hc.DataRange;
import org.jfree.data.hc.DataRangeMismatchException;

/**
 * A class that stores information about the state of {@link HCTreeNode}-nodes
 * used in {@link org.jfree.chart.plot.HCPlot}. This is organized as a tree with the same topology
 * as the original clustering tree. 
 *
 * @author  viski project
 */
public class HCTreeNodeInfo {

    private boolean open;
    private HCTreeNode node;
    private HCTreeNodeInfo parent;
    private HCTreeNodeInfo leftChild;
    private HCTreeNodeInfo rightChild;
    private DataRange visibleDataRange;
    private StandardHCClusteringInfo clusteringInfo;

    /**
     * Creates a new HCTreeNodeInfo reflecting data in HCTreeNode.
     * The constructor also creates objects for all the children
     * of HCTreeNode.
     * After creating the HCTreeNode-tree, the objects are not yet complete.
     * You still have to call updateVisibleDataRange() on the root
     * node to make visible data range related functionality working ok.
     *
     * @param clusteringInfo  column or leaf tree?
     * @param node the root node of a clustering tree for which
     * HCTreeNodeInfo-objects are created for.
     */
    public HCTreeNodeInfo(
	    StandardHCClusteringInfo clusteringInfo,
	    HCTreeNode node
    ) {

	if (node == null) throw new NullPointerException(
	    "A HCTreeNodeInfo object must have a non-null HCTreeNode."
	);
	if (clusteringInfo == null) throw new NullPointerException(
	    "A HCTreeNodeInfo object must have a non-null clustering info."
	);

	this.open = false;
	this.node = node;
	this.parent = null;
	this.clusteringInfo = clusteringInfo;
	this.visibleDataRange = new DataRange(0,0);

	if (node.getLeftChild() == null) this.leftChild = null;
	else {

	    this.leftChild = new HCTreeNodeInfo(
		clusteringInfo,
		node.getLeftChild()
	    );
	    this.leftChild.setParent(this);
	    this.open = true;

	}
	if (node.getRightChild() == null) this.rightChild = null;
	else {

	    this.rightChild = new HCTreeNodeInfo(
		clusteringInfo,
		node.getRightChild()
	    );
	    this.rightChild.setParent(this);
	    this.open = true;

	}

    }

    /**
     * Open or close a single node.
     *
     * @param open  true, if we wish to open the node or false otherwise.
     */
    public void setNodeOpen(boolean open) {

	if ((this.leftChild != null) && (this.rightChild != null)) {

	    //throw new IllegalArgumentException();

	   this.open = open;
	   this.updateVisibleDataRange(); // could be optimized
	   this.clusteringInfo.notifyChangeListeners(new ChangeEvent(this));

	}

    }

    /**
     * The recursive part of setSubTreeOpen().
     * This should not be invoked by any other method.
     *
     * @param open  true, if we wish to open the nodes or false otherwise.
     */
    private void setSubTreeOpenRecursively(boolean open) {

	if ((this.leftChild != null) && (this.rightChild != null)) {

	    this.leftChild.setSubTreeOpenRecursively(open);
	    this.rightChild.setSubTreeOpenRecursively(open);
	    this.open = open;

	}

    }

    /**
     * Open or close each node in the whole subtree.
     *
     * @param open  true, if we wish to open the nodes or false otherwise.
     */
    public void setSubTreeOpen(boolean open) {

	if ((this.leftChild != null) && (this.rightChild != null)) {

	    //throw new IllegalArgumentException();

	    this.setSubTreeOpenRecursively(open);
	    this.updateVisibleDataRange(); // could be optimized
	    this.clusteringInfo.notifyChangeListeners(
		    new ChangeEvent(this));

	}

    }

    /**
     * Is this node open or closed.
     * It is arranged so that leaf nodes are always closed.
     *
     * @return true for nodes that are open and false, if the node is closed.
     */
    public boolean isNodeOpen() {

	return this.open;

    }

    /**
     * Returns the dataset node corresponding to this node.
     *
     * @return  HCTreeNode object.
     */
    public HCTreeNode getNode() {

	return this.node;

    }

    /**
     * Returns the left child of this node.
     *
     * @return a child object or null, if this node has no left child.
     */
    public HCTreeNodeInfo getLeftChild() {

	return this.leftChild;

    }

    /**
     * Returns the right child of this node.
     *
     * @return a child object or null, if this node has no right child.
     */
    public HCTreeNodeInfo getRightChild() {

	return this.rightChild;

    }

    /**
     * Returns the parent of this node.
     *
     * @return a parent object or null, if this is the root node.
     */
    public HCTreeNodeInfo getParent() {

	return this.parent;

    }

    /**
     * Sets the parent of this node to be a specified node.
     * This is only intended to be invoked by setLeftChild and
     * setRightChild methods.
     *
     * @param node  specifies the parent.
     *
     * @throws IllegalArgumentException  if the parent does not report
     * this as a child already.
     * @throws IllegalArgumentException  if this node already has a parent.
     */
    public void setParent(HCTreeNodeInfo node) {

	
	if (
	    (parent != null) &&
	    ((parent.getLeftChild() != this) &&
	    (parent.getRightChild() != this))
	) throw new IllegalArgumentException(
	    "A HCTreeNodeInfo object can only be set as a parent of another " +
	    "object, if that node is already its child.");
        if (this.parent != null) throw new IllegalArgumentException(
	    "You cannot set a parent for a HCTreeNodeInfo node that already " +
	    "has one.");
	this.parent = node;

    }

    /**
     * Returns the name of this node as a string.
     *
     * @return  the name of this node.
     */
    public String toString() {

	return this.clusteringInfo.getName(this.getNode().getDataRange());

    }

    /**
     * Returns the ClustringInfo object associated with this tree.
     *
     * @return  a StandardHCClusteringInfo object.
     */
    public StandardHCClusteringInfo getClusteringInfo() {
      
        return this.clusteringInfo;

    }

    /**
     * Returns the visible data range of this subtree.
     *
     * @return  a DataRange object.
     */
    public DataRange getVisibleDataRange() {

	return this.visibleDataRange;

    }

    /**
     * Returns a node representing the specified dataset node.
     *
     * @param node  the dataset node specifying a HCTreeNodeInfo node.
     *
     * @throws IndexOutOfBoundsException  if the specified node
     * can't be found from this subtree.
     *
     * @return HCTreeNodeInfo object.
     */
    public HCTreeNodeInfo getNodeByNode(HCTreeNode node) {

	DataRange dr = node.getDataRange();

	if (dr.equals(this.getNode().getDataRange())) return this;
	if ((this.getLeftChild() != null) &&
	    (this.getLeftChild().getNode().getDataRange().contains(dr)))
	    return this.getLeftChild().getNodeByNode(node);
	if ((this.getRightChild() != null) &&
	    (this.getRightChild().getNode().getDataRange().contains(dr)))
	    return this.getRightChild().getNodeByNode(node);

	throw new IndexOutOfBoundsException();

    }

    /**
     * Returns a node representing the specified dataset row/column.
     *
     * @param index  the index of the dataset row/column.
     *
     * @throws IndexOutOfBoundsException  if the specified node
     * can't be found from this subtree.
     *
     * @return HCTreeNodeInfo object.
     */
    public HCTreeNodeInfo getNodeByIndex(int index) {

	if (
	    (this.leftChild != null) &&
	    (this.leftChild.getNode().getDataRange().contains(index))) {

	    return this.leftChild.getNodeByIndex(index);

	}
	if (
	    (this.rightChild != null) &&
	    (this.rightChild.getNode().getDataRange().contains(index))) {

	    return this.rightChild.getNodeByIndex(index);

	}
	if (this.getNode().getDataRange().equals(new DataRange(index,index))) {

	    return this;

	}
	throw new IndexOutOfBoundsException(
	    "There is no node " + index + "under this node.");

    }

    /**
     * Returns a node representing the specified visible row/column.
     *
     * @param index  the index of the visible row/column.
     *
     * @throws IndexOutOfBoundsException  if the specified node
     * can't be found from this subtree.
     *
     * @return HCTreeNodeInfo object.
     */
    public HCTreeNodeInfo getNodeByVisibleIndex(int index) {

	if (
	    (this.leftChild != null) &&
	    (this.leftChild.getVisibleDataRange().contains(index)) &&
	    (this.open)) {

	    return this.leftChild.getNodeByVisibleIndex(index);

	}
	if (
	    (this.rightChild != null) &&
	    (this.rightChild.getVisibleDataRange().contains(index)) &&
	    (this.open)) {

	    return this.rightChild.getNodeByVisibleIndex(index);

	}
	if (this.visibleDataRange.equals(new DataRange(index,index))) {

	    return this;

	}
	throw new IndexOutOfBoundsException(
	    "There is no visible node " + index + "under this node.");

    }

    /**
     * Updates visible data range of the subtree rooted at this.
     * This is only intended to be invoked by updateVisibleDataRange().
     * This can break the tree, so don't use it.
     *
     * @param index  index for the leftmost leaf or closed node of this
     * subtree.
     *
     * @return index for the other subtrees of any of the grandparents
     * of this node.
     */
    private int updateVisibleDataRange(int index) {

	if (
	    (this.leftChild == null) ||
	    (this.rightChild == null) ||
	    (!this.open)
	) {

	    this.visibleDataRange = new DataRange(index,index);
	    return index+1;

	} else {

	    index = this.getLeftChild().updateVisibleDataRange(index);
	    index = this.getRightChild().updateVisibleDataRange(index);
	    this.visibleDataRange = (DataRange)
		(this.getLeftChild().getVisibleDataRange().clone());
	    try {
	    	this.visibleDataRange.add(
		    this.getRightChild().getVisibleDataRange()
	    	);
	    } catch (DataRangeMismatchException e) {
		// this never happens if used properly.
	    }
	    return index;

	}

    }

    /**
     * Updates the visibleDataRange attributes of the tree.
     * This needs to be explicitly invoked only after creating a new
     * tree. Otherwise, it is invoked automatically.
     */
    public void updateVisibleDataRange() {

	this.clusteringInfo.getRootNode().updateVisibleDataRange(0);

    }

}

