/* ======================================================================= 
 * A visualisation library extension for JFreeChart. Please see JFreeChart
 * for further information.
 * =======================================================================
 * Copyright (C) 2006  University of Helsinki, Department of Computer Science
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * -----------------------------
 * Contact:  ohtu@cs.helsinki.fi
 * -----------------------------
 *
 */

package org.jfree.chart.plot;

import java.awt.Color;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeMap;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * A class implementing mapping of colors to values. Used in {@link HCPlot}.
 *
 * @author  viski project
 */
public class GradientColorPalette
    implements Cloneable {

    private TreeMap keyColorMap;
    private boolean linearMapping;

    private LinkedList listeners;


    /**
     * Creates a default color palette.
     * The default palette is the following:
     * value < -1	red
     * -1 < value < 0   red to black
     *  value = 0	black
     *  0 < value < 1   black to green
     *  1 < value	green
     */
    public GradientColorPalette() {

	this.keyColorMap = new TreeMap();
	this.linearMapping = true;
	this.listeners = new LinkedList();
	addKeyColor (-1.0, new Color(255,  0,  0));
	addKeyColor ( 0.0, new Color(  0,  0,  0));
	addKeyColor ( 1.0, new Color(  0,255,  0));
	return;

    }

    /**
     * Creates a default scaled color palette.
     * The default palette is the following:
     * value < -min	red
     * -min < value < 0	red to black
     *  value = 0	black
     *  0 < value < max	black to green
     *  max < value	green
     */
    public GradientColorPalette(double min, double max) {

	this.keyColorMap = new TreeMap();
	this.linearMapping = true;
	this.listeners = new LinkedList();
	addKeyColor (min, new Color(255,  0,  0));
	addKeyColor ( 0.0, new Color(  0,  0,  0));
	addKeyColor (max, new Color(  0,255,  0));
	return;

    }

    /**
     * Creates a custom color palette.
     *
     * @param keyColorMap  a mapping between values of type double and
     * a color to be used as key colors.
     */
    public GradientColorPalette(Map keyColorMap) {
	this.keyColorMap = new TreeMap(keyColorMap);
	this.linearMapping = true;
	this.listeners = new LinkedList();
    }

    /**
     * Sets a new color for an already mapped value.
     *
     * @param value  value that maps to this color.
     * @param color  the color the specified value is mapped to.
     *
     * @throws  NullPointerException, if the specified color is null.
     */
    public Color setKeyColor(double value, Color color) {

	if (color == null) throw new NullPointerException();
	Color old = (Color)this.keyColorMap.remove((Object)new Double(value));
	this.keyColorMap.put ((Object)new Double(value), (Object)color);
	notifyChangeListeners(new ChangeEvent(this));
	return old;

    }

    /**
     * Adds a new keycolor.
     *
     * @param value  value that maps to this color. This can be any
     *     double. If a keycolor with this value already exists, it
     *     its color-value is reset.
     * @param r  the value of the red component of the color as a
     *     integer from 0 to 255.
     * @param g  the value of the green component of the color as a
     *     integer from 0 to 255.
     * @param b  the value of the blue component of the color as a
     *     integer from 0 to 255.
     *
     * @throws IllegalArgumentException if either r, g or b is not
     *     in the range [0,255]
     */
    public void addKeyColor(double value, int r, int g, int b)
	throws IllegalArgumentException {

	addKeyColor(value, new Color(r,g,b));

    }

    /**
     * Adds a new keycolor.
     *
     * @param value  value that maps to this color. This can be any
     *     double. If a keycolor with this value already exists, it
     *     its color-value is reset.
     * @param color  the color the specified value is mapped to.
     *
     * @throws  NullPointerException, if the specified color is null.
     */
    public void addKeyColor(double value, Color color) 
        throws NullPointerException {

	if (color == null) throw new NullPointerException();
	this.keyColorMap.put ((Object)new Double(value), (Object)color);
	notifyChangeListeners(new ChangeEvent(this));

    }

    /**
     * Adds a change listener
     *
     * @param listener  the listener.
     */
    public void addChangeListener(ChangeListener listener) {

	this.listeners.add((Object)listener);

    }

    /**
     * Removes a change listener
     *
     * @param listener  the listener.
     */
    public void removeChangeListener(ChangeListener listener) {

	this.listeners.remove((Object)listener);

    }

    /**
     * Notifies listeners of an event
     *
     * @param event  the event.
     */
    private void notifyChangeListeners(ChangeEvent event) {

	Iterator iterator = listeners.iterator();
	ChangeListener listener;

	while(iterator.hasNext()) {

	    listener = (ChangeListener)(iterator.next());
	    listener.stateChanged (event);

	}

    }

    /**
     * Returns the keycolor map.
     *
     * @return  The keycolor map.
     */
    public Set getKeyColors() {

	return this.keyColorMap.keySet();

    }

    /**
     * Returns a specified key color.
     *
     * @param value  a value specifying a key color.
     *
     * @return  The keycolor, or null, if the specified key color does not
     * exist.
     *
     */
    public Color getKeyColor(double value) {

	return (Color)this.keyColorMap.get((Object)new Double(value));

    }

    /**
     * Removes a specified key color.
     *
     * @param value  a value specifying a key color.
     *
     * @return  The removed keycolor, or null, if the specified key color
     * does not exist.
     */
    public Color removeKeyColor(double value) {

	Color old = (Color)this.keyColorMap.remove((Object)new Double(value));
	notifyChangeListeners(new ChangeEvent(this));
	return old;

    }

    /**
     * Returns the keycolor map.
     *
     * @return  The keycolor map.
     */
    public Color getColor(double value) {

	double proportion;
	Color color1;
	Color color2;
	Color color;
	Double valueObject = new Double(value);
	Double value1;
	Double value2;

	// if the value maps directly to a color, return it.
	color = (Color)this.keyColorMap.get((Object)(valueObject));
	if (color != null) return color;

	// get the closest keyColor < value
	try {
	    value1 = (Double)this.keyColorMap.headMap(valueObject).lastKey();
	} catch (NoSuchElementException e) {
	    // or if it doesn't exist, return the closest keyColor > value
	    return (Color)this.keyColorMap.get((Object)
		(this.keyColorMap.tailMap(valueObject).firstKey())
	    );
	}

	// get the closest keyColor > value
	try {
	    value2 = (Double)this.keyColorMap.tailMap(valueObject).firstKey();
	} catch (NoSuchElementException e) {
	    // or if it doesn't exist, return the closest keyColor < value
	    return (Color)this.keyColorMap.get((Object)
		(this.keyColorMap.headMap(valueObject).lastKey())
	    );
	}

	color1 = (Color)this.keyColorMap.get((Object)value1);
	color2 = (Color)this.keyColorMap.get((Object)value2);

	// map the color between color1 and color2
	if (this.linearMapping) {

	    proportion = (value-value1.doubleValue())
		/ (value2.doubleValue()-value1.doubleValue());

	} else {

	    // TODO: how does one calculate this?
	    double logvalue;
	    double logvalue1;
	    double logvalue2;
	    if (value<0) {

		// there has to be keyValue 0 if we are using
		// logarithmic scale.
		// if not, just do something.
		if (value2.doubleValue() > 0)
		    return new Color(0,0,0);
		    // this is unreachable.

		// kludge
		if (value2.doubleValue() == 0) value2 = new Double(value/100);

		logvalue = Math.log(-value);
		logvalue1 = Math.log(-value1.doubleValue());
		logvalue2 = Math.log(-value2.doubleValue());
		proportion = (logvalue-logvalue1)
		    / (logvalue2-logvalue1);

	    } else {

		// there has to be keyValue 0 if we are using
		// logarithmic scale.
		// if not, just do something.
		if (value1.doubleValue() < 0)
		    return new Color(0,0,0);
		    // this is unreachable.

		// kludge
		if (value1.doubleValue() == 0) value1 = new Double(value/100);

		logvalue = Math.log(value);
		logvalue1 = Math.log(value1.doubleValue());
		logvalue2 = Math.log(value2.doubleValue());

		proportion = (logvalue-logvalue1)
		    / (logvalue2-logvalue1);

	    }

	}

	return new Color(
	    (int)(color2.getRed()*proportion +
		  color1.getRed()*(1-proportion)),
	    (int)(color2.getGreen()*proportion +
		  color1.getGreen()*(1-proportion)),
	    (int)(color2.getBlue()*proportion +
		  color1.getBlue()*(1-proportion))
	);
	
    }

    /**
     * sets linear or logarithmic value mapping.
     *
     * @param linear  true to set linear mapping, false, to set logarithmic
     * mapping.
     */
    public void setLinear(boolean linear) {

	this.linearMapping = linear;
	notifyChangeListeners (new ChangeEvent(this));

    }

    /**
     * Returns a value indicating whether linear or logarithmic mapping
     * is being used.
     *
     * @return  true, if linear mapping is in use, false otherwise.
     */
    public boolean isLinear() {

	return this.linearMapping;

    }

    /**
     * Clones this object.
     *
     * @return  a clone of this object.
     */
    public Object clone() {

	Iterator iterator = listeners.iterator();
	ChangeListener listener;

	GradientColorPalette clone = new GradientColorPalette(this.keyColorMap);
	clone.setLinear(this.linearMapping);

	while(iterator.hasNext()) {

	    listener = (ChangeListener)(iterator.next());
	    clone.addChangeListener(listener);

	}

	return (Object)clone;

    }

    /**
     * Checks whether this object is equal to a specified object.
     *
     * @param obj  an object to check equality against.
     *
     * @return  true, if the objects are equal, false otherwise.
     */
    public boolean equals(Object obj) {

	GradientColorPalette that;
	Set thisKeyColors;
	Set thatKeyColors;
	LinkedList listeners;
	Iterator iterator;

	if (!(obj instanceof GradientColorPalette)) return false;
	that = (GradientColorPalette)obj;
	if (that.isLinear() != this.linearMapping) return false;

	thisKeyColors = this.getKeyColors();
	thatKeyColors = that.getKeyColors();
	if (!thisKeyColors.equals(thatKeyColors)) return false;
	iterator = thisKeyColors.iterator();
	while (iterator.hasNext()) {
	    double value = ((Double)iterator.next()).doubleValue();
	    if (!this.getKeyColor(value).equals(that.getKeyColor(value)))
		return false;
	}

	return true;
    }

}
