
package org.jfree.chart.plot.junit;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import javax.swing.event.ChangeEvent;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartMouseEvent;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.ChartRenderingInfo;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.entity.SOMItemEntity;
import org.jfree.chart.labels.SOMToolTipGenerator;
import org.jfree.chart.labels.StandardSOMToolTipGenerator;
import org.jfree.chart.plot.SOMPlot;
import org.jfree.data.som.SOMDataset;



/**
 * Tests for the {@link SOMPlot} class.
 */
public class SOMPlotTests extends TestCase {

    private SOMDataset dataset;
    /**
     * Returns the tests as a test suite.
     *
     * @return The test suite.
     */
    public static Test suite() {
        return new TestSuite(SOMPlotTests.class);
    }

    /**
     * Constructs a new set of tests.
     *
     * @param name  the name of the tests.
     */
    public SOMPlotTests(String name) {
        super(name);
    }
    
    protected void setUp() throws Exception {
        this.dataset = new SOMDataset(2, 3);
        dataset.addValue(0, 0, Color.BLACK,  new String[] { "A" }, new double[] { 1.0 });
        dataset.addValue(1, 0, Color.RED,  new String[] { "B" }, new double[] { 2.0 });
        dataset.addValue(0, 1, Color.GREEN,  new String[] { "C" }, new double[] { 3.0 });
        dataset.addValue(1, 1, Color.BLUE,  new String[] { "D" }, new double[] { 1.0, 0.9 });
        dataset.addValue(0, 2, Color.WHITE,  new String[] { "E", "F" }, new double[] { 1.0, 4.9, 2.2 });
        dataset.addValue(1, 2, new Color(139,69,16),  new String[] { "G", "H" }, new double[] { 1.0, 0.3 });
    }
    
    public void testConstructor() {
        try {
            new SOMPlot(null);
            assertTrue(false);
        } catch (NullPointerException e) {}
        
        assertNotNull(new SOMPlot(this.dataset));
    }
    
    public void testGetSetDataset() {
        assertEquals(new SOMPlot(this.dataset).getDataset(), this.dataset);
        SOMPlot plot = new SOMPlot(this.dataset);
        plot.setDataset(this.dataset);
        assertEquals(plot.getDataset(), this.dataset);
        try {
            new SOMPlot(this.dataset).setDataset(null);
            assertTrue(false);
        } catch (NullPointerException e) {}
    }
    
    public void testGetSetTooltipGenerator() {
        assertNull(new SOMPlot(this.dataset).getToolTipGenerator());
        try {
            new SOMPlot(this.dataset).setToolTipGenerator(null);
        } catch (Exception e) {
            assertTrue(false);
        }
        
        SOMToolTipGenerator generator = new StandardSOMToolTipGenerator();
        SOMPlot plot = new SOMPlot(this.dataset);
        plot.setToolTipGenerator(generator);
        assertEquals(plot.getToolTipGenerator(), generator);
    }
    
    public void testGetPlotType() {
        assertNotNull(new SOMPlot(this.dataset).getPlotType());
        assertTrue(new SOMPlot(this.dataset).getPlotType().length() != 0);
    }
    
    public void testGetSetDescriptionFont() {
        assertNotNull(new SOMPlot(this.dataset).getDescriptionFont());
        try {
            new SOMPlot(this.dataset).setDescriptionFont(null);
            assertTrue(false);
        } catch (NullPointerException e) {}
        
        SOMPlot plot = new SOMPlot(this.dataset);
        Font font = new Font("SansSerif", Font.PLAIN, 14);
        plot.setDescriptionFont(font);
        assertEquals(plot.getDescriptionFont(), font);
    }
    
    public void testGetpanel() {
        assertNotNull(new SOMPlot(this.dataset).getPanel());
        
        JFreeChart chart = ChartFactory.createSOMChart(
                "SOM Chart Demo",         // chart title
                this.dataset,                  // data
                true,                     // tooltips?
                false                     // URLs?
                );
        
        ChartPanel panel = new ChartPanel(chart);
        SOMPlot plot = (SOMPlot)chart.getPlot();
        SOMItemEntity upperLeftCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 0, 0, "A", "B");
        SOMItemEntity lowerRightCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 1, 1, "A", "B");
        
        // Select one cell. The 'select neighbors' slider will be active if one cell is selected.
        MouseEvent mouseE = new MouseEvent(panel, 0, 0, InputEvent.BUTTON1_MASK, 1, 1, 1, false);
        ChartMouseEvent chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
        plot.chartMouseClicked(chartMouseE);
        assertNotNull(plot.getPanel());
    }
    
    public void testGetpanelName() {
        assertNotNull(new SOMPlot(this.dataset).getPanelName());
        assertTrue(new SOMPlot(this.dataset).getPanelName().length() != 0);
    }
    
    public void testDraw() {
        JFreeChart chart = ChartFactory.createSOMChart(
                "SOM Chart Demo",         // chart title
                this.dataset,                  // data
                true,                     // tooltips?
                false                     // URLs?
                );
        
        // Normal case, valid parameters.
        try {
            this.dataset.getValue(1, 1).setSelected(true);
            BufferedImage image = new BufferedImage(200, 200, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            ChartRenderingInfo chartInfo = new ChartRenderingInfo();
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);
        } catch (Exception e) {
            assertTrue(false);
        }
        
        // Normal case, different background color.
        // This is done to achieve full coverage for the contrastingColor()-method.
        try {
            this.dataset.getValue(1, 1).setSelected(true);
            BufferedImage image = new BufferedImage(200, 200, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            ChartRenderingInfo chartInfo = new ChartRenderingInfo();
            chart.getPlot().setBackgroundPaint(Color.BLACK);
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), chartInfo);
            assertTrue(image != null);
        } catch (Exception e) {
            assertTrue(false);
        }
        
        // ChartRenderingInfo == null, should still work
        try {
            BufferedImage image = new BufferedImage(200, 200, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            chart.draw(g, new Rectangle2D.Double(0, 0, 200, 200), null);
        } catch (Exception e) {
            assertTrue(false);
        }
        
        // Area == null
        try {
            BufferedImage image = new BufferedImage(200, 200, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = image.createGraphics();
            ChartRenderingInfo chartInfo = new ChartRenderingInfo();
            chart.draw(g, null, chartInfo);
            assertTrue(false);
        } catch (Exception e) {
        }
    }
    
    public void testChartMouseClicked() {
        try {
            JFreeChart chart = ChartFactory.createSOMChart(
                    "SOM Chart Demo",         // chart title
                    this.dataset,                  // data
                    true,                     // tooltips?
                    false                     // URLs?
                    );
            
            ChartPanel panel = new ChartPanel(chart);
            SOMPlot plot = (SOMPlot)chart.getPlot();
            SOMItemEntity upperLeftCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 0, 0, "A", "B");
            SOMItemEntity lowerRightCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 1, 2, "A", "B");
            
            MouseEvent mouseE = new MouseEvent(panel, 0, 0, InputEvent.BUTTON1_MASK, 1, 1, 1, false);
            ChartMouseEvent chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            // Normal mouse click on upper left cell should select the cell, but not others
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            // Normal click on the same cell should deselect it.
            plot.chartMouseClicked(chartMouseE);
            assertFalse(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0,InputEvent.BUTTON1_MASK | InputEvent.CTRL_MASK,1,1,1,false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            chartMouseE = new ChartMouseEvent(chart, mouseE,  lowerRightCorner);
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertTrue(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0, InputEvent.BUTTON1_MASK, 1, 1, 1, false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseClicked(chartMouseE);
            assertFalse(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0,InputEvent.BUTTON1_MASK | InputEvent.SHIFT_MASK,1,1,1,false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  lowerRightCorner);
            // No cell previously selected. Shift+click on the lower right cell should select all cells.
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertTrue(this.dataset.getValue(1, 0).isSelected());
            assertTrue(this.dataset.getValue(0, 1).isSelected());
            assertTrue(this.dataset.getValue(1, 1).isSelected());
            assertTrue(this.dataset.getValue(0, 2).isSelected());
            assertTrue(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0, InputEvent.BUTTON1_MASK, 1, 1, 1, false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseClicked(chartMouseE);
            assertFalse(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0,InputEvent.BUTTON1_MASK | InputEvent.SHIFT_MASK,1,1,1,false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  lowerRightCorner);
            plot.chartMouseClicked(chartMouseE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertTrue(this.dataset.getValue(1, 0).isSelected());
            assertTrue(this.dataset.getValue(0, 1).isSelected());
            assertTrue(this.dataset.getValue(1, 1).isSelected());
            assertTrue(this.dataset.getValue(0, 2).isSelected());
            assertTrue(this.dataset.getValue(1, 2).isSelected());
            
            mouseE = new MouseEvent(panel, 0, 0,InputEvent.BUTTON1_MASK | InputEvent.CTRL_MASK,1,1,1,false);
            chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseClicked(chartMouseE);
            assertFalse(this.dataset.getValue(0, 0).isSelected());
            assertTrue(this.dataset.getValue(1, 0).isSelected());
            assertTrue(this.dataset.getValue(0, 1).isSelected());
            assertTrue(this.dataset.getValue(1, 1).isSelected());
            assertTrue(this.dataset.getValue(0, 2).isSelected());
            assertTrue(this.dataset.getValue(1, 2).isSelected());
            
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    public void testChartMouseMoved() {
        try {
            JFreeChart chart = ChartFactory.createSOMChart(
                    "SOM Chart Demo",         // chart title
                    this.dataset,                  // data
                    true,                     // tooltips?
                    false                     // URLs?
                    );
            
            ChartPanel panel = new ChartPanel(chart);
            SOMPlot plot = (SOMPlot)chart.getPlot();
            SOMItemEntity upperLeftCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 0, 0, "A", "B");
            SOMItemEntity lowerRightCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 1, 2, "A", "B");
            
            MouseEvent mouseE = new MouseEvent(panel, 0, 0, InputEvent.CTRL_MASK, 1, 1, 1, false);
            ChartMouseEvent chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseMoved(chartMouseE);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    public void testStateChanged() {
        JFreeChart chart = ChartFactory.createSOMChart(
                "SOM Chart Demo",         // chart title
                this.dataset,                  // data
                true,                     // tooltips?
                false                     // URLs?
                );
        
        try {
            ChartPanel panel = new ChartPanel(chart);
            SOMPlot plot = (SOMPlot)chart.getPlot();
            SOMItemEntity upperLeftCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 0, 0, "A", "B");
            SOMItemEntity lowerRightCorner = new SOMItemEntity(new Rectangle2D.Double(0, 0, 20, 20), dataset, 1, 2, "A", "B");
            
            MouseEvent mouseE = new MouseEvent(panel, 0, 0, InputEvent.BUTTON1_MASK, 1, 1, 1, false);
            ChartMouseEvent chartMouseE = new ChartMouseEvent(chart, mouseE,  upperLeftCorner);
            plot.chartMouseClicked(chartMouseE);
            
            plot.getColorHueSlider().setValue(100);
            ChangeEvent changeE = new ChangeEvent(plot.getColorHueSlider());
            plot.stateChanged(changeE);
            
            plot.getDistanceSlider().setValue(plot.getDistanceSlider().getMaximum());
            changeE = new ChangeEvent(plot.getDistanceSlider());
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertFalse(this.dataset.getValue(1, 0).isSelected());
            assertFalse(this.dataset.getValue(0, 1).isSelected());
            assertFalse(this.dataset.getValue(1, 1).isSelected());
            assertFalse(this.dataset.getValue(0, 2).isSelected());
            assertFalse(this.dataset.getValue(1, 2).isSelected());
            plot.stateChanged(changeE);
            assertTrue(this.dataset.getValue(0, 0).isSelected());
            assertTrue(this.dataset.getValue(1, 0).isSelected());
            assertTrue(this.dataset.getValue(0, 1).isSelected());
            assertTrue(this.dataset.getValue(1, 1).isSelected());
            assertTrue(this.dataset.getValue(0, 2).isSelected());
            assertTrue(this.dataset.getValue(1, 2).isSelected());
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    /**
     * Test the equals() method.
     */
    public void testEquals() {
        
        SOMPlot plot1 = new SOMPlot(this.dataset);
        SOMPlot plot2 = new SOMPlot(this.dataset);
        assertTrue(plot1.equals(plot1));
        assertTrue(plot1.equals(plot2));
        assertTrue(plot2.equals(plot1));
        assertFalse(plot2.equals(new SOMDataset(1, 1)));
        
        plot1.setDescriptionFont(new Font("SansSerif", Font.ITALIC, 10));
        assertFalse(plot1.equals(plot2));
        
        plot1.setToolTipGenerator(new StandardSOMToolTipGenerator());
        assertFalse(plot1.equals(plot2));
        assertFalse(plot2.equals(plot1));
        
        plot1 = new SOMPlot(this.dataset);
        plot2 = new SOMPlot(this.dataset);
        plot2.getColorHueSlider().setValue(100);
        ChangeEvent changeE = new ChangeEvent(plot2.getColorHueSlider());
        plot2.stateChanged(changeE);
        assertFalse(plot1.equals(plot2));
    }

    /**
     * Some basic checks for the clone() method.
     */
    public void testCloning() {
        SOMPlot p1 = new SOMPlot(this.dataset);
        SOMPlot p2 = null;
        try {
            p2 = (SOMPlot) p1.clone();
        }
        catch (CloneNotSupportedException e) {
            e.printStackTrace();
            System.err.println("Failed to clone.");
        }
        assertTrue(p1 != p2);
        assertTrue(p1.getClass() == p2.getClass());
        assertTrue(p1.equals(p2));
    }

    /**
     * Serialize an instance, restore it, and check for equality.
     */
    public void testSerialization() {
        SOMPlot p1 = new SOMPlot(this.dataset);
        SOMPlot p2 = null;
        try {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            ObjectOutput out = new ObjectOutputStream(buffer);
            out.writeObject(p1);
            out.close();

            ObjectInput in = new ObjectInputStream(
                new ByteArrayInputStream(buffer.toByteArray())
            );
            p2 = (SOMPlot) in.readObject();
            in.close();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        assertEquals(p1, p2);
    }

    /**
     * Runs the test suite using JUnit's text-based runner.
     * 
     * @param args  ignored.
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }

}
