/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.ReservedTagConstants;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.fastq.FastqConstants;
import htsjdk.samtools.fastq.FastqReader;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.util.FastqQualityFormat;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Iso8601Date;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import htsjdk.samtools.util.QualityEncodingDetector;
import htsjdk.samtools.util.SolexaQualityConverter;
import htsjdk.samtools.util.StringUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.SamOrBam;

@CommandLineProgramProperties(usage="Extracts read sequences and qualities from the input fastq file and writes them into the output file in unaligned BAM format. Input files can be in GZip format (end in .gz).\n", usageShort="Converts a fastq file to an unaligned BAM or SAM file", programGroup=SamOrBam.class)
public class FastqToSam
extends CommandLineProgram {
    private static final Log LOG = Log.getInstance(FastqToSam.class);
    @Option(shortName="F1", doc="Input fastq file (optionally gzipped) for single end data, or first read in paired end data.")
    public File FASTQ;
    @Option(shortName="F2", doc="Input fastq file (optionally gzipped) for the second read of paired end data.", optional=true)
    public File FASTQ2;
    @Option(doc="Use sequential fastq files with the suffix <prefix>_###.fastq or <prefix>_###.fastq.gz", optional=true)
    public boolean USE_SEQUENTIAL_FASTQS = false;
    @Option(shortName="V", doc="A value describing how the quality values are encoded in the fastq.  Either Solexa for pre-pipeline 1.3 style scores (solexa scaling + 66), Illumina for pipeline 1.3 and above (phred scaling + 64) or Standard for phred scaled scores with a character shift of 33.  If this value is not specified, the quality format will be detected automatically.", optional=true)
    public FastqQualityFormat QUALITY_FORMAT;
    @Option(doc="Output SAM/BAM file. ", shortName="O")
    public File OUTPUT;
    @Option(shortName="RG", doc="Read group name")
    public String READ_GROUP_NAME = "A";
    @Option(shortName="SM", doc="Sample name to insert into the read group header")
    public String SAMPLE_NAME;
    @Option(shortName="LB", doc="The library name to place into the LB attribute in the read group header", optional=true)
    public String LIBRARY_NAME;
    @Option(shortName="PU", doc="The platform unit (often run_barcode.lane) to insert into the read group header", optional=true)
    public String PLATFORM_UNIT;
    @Option(shortName="PL", doc="The platform type (e.g. illumina, solid) to insert into the read group header", optional=true)
    public String PLATFORM;
    @Option(shortName="CN", doc="The sequencing center from which the data originated", optional=true)
    public String SEQUENCING_CENTER;
    @Option(shortName="PI", doc="Predicted median insert size, to insert into the read group header", optional=true)
    public Integer PREDICTED_INSERT_SIZE;
    @Option(shortName="PG", doc="Program group to insert into the read group header.", optional=true)
    public String PROGRAM_GROUP;
    @Option(shortName="PM", doc="Platform model to insert into the group header (free-form text providing further details of the platform/technology used)", optional=true)
    public String PLATFORM_MODEL;
    @Option(doc="Comment(s) to include in the merged output file's header.", optional=true, shortName="CO")
    public List<String> COMMENT = new ArrayList<String>();
    @Option(shortName="DS", doc="Inserted into the read group header", optional=true)
    public String DESCRIPTION;
    @Option(shortName="DT", doc="Date the run was produced, to insert into the read group header", optional=true)
    public Iso8601Date RUN_DATE;
    @Option(shortName="SO", doc="The sort order for the output sam/bam file.")
    public SAMFileHeader.SortOrder SORT_ORDER = SAMFileHeader.SortOrder.queryname;
    @Option(doc="Minimum quality allowed in the input fastq.  An exception will be thrown if a quality is less than this value.")
    public int MIN_Q = 0;
    @Option(doc="Maximum quality allowed in the input fastq.  An exception will be thrown if a quality is greater than this value.")
    public int MAX_Q = 93;
    @Option(doc="If true and this is an unpaired fastq any occurance of '/1' will be removed from the end of a read name.")
    public Boolean STRIP_UNPAIRED_MATE_NUMBER = false;
    @Option(doc="Allow (and ignore) empty lines")
    public Boolean ALLOW_AND_IGNORE_EMPTY_LINES = false;
    private static final SolexaQualityConverter solexaQualityConverter = SolexaQualityConverter.getSingleton();

    public static FastqQualityFormat determineQualityFormat(FastqReader reader1, FastqReader reader2, FastqQualityFormat expectedQuality) {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        if (reader2 == null) {
            detector.add(10000L, new FastqReader[]{reader1});
        } else {
            detector.add(10000L, new FastqReader[]{reader1, reader2});
            reader2.close();
        }
        reader1.close();
        FastqQualityFormat qualityFormat = detector.generateBestGuess(QualityEncodingDetector.FileContext.FASTQ, expectedQuality);
        if (detector.isDeterminationAmbiguous()) {
            LOG.warn(new Object[]{"Making ambiguous determination about fastq's quality encoding; more than one format possible based on observed qualities."});
        }
        LOG.info(new Object[]{String.format("Auto-detected quality format as: %s.", qualityFormat)});
        return qualityFormat;
    }

    public static void main(String[] argv) {
        System.exit(new FastqToSam().instanceMain(argv));
    }

    protected static List<File> getSequentialFileList(File baseFastq) {
        ArrayList<File> files = new ArrayList<File>();
        files.add(baseFastq);
        FastqConstants.FastqExtensions fastqExtensions = null;
        String suffix = null;
        for (FastqConstants.FastqExtensions ext : FastqConstants.FastqExtensions.values()) {
            suffix = "_001" + ext.getExtension();
            if (!baseFastq.getAbsolutePath().endsWith(suffix)) continue;
            fastqExtensions = ext;
            break;
        }
        if (null == fastqExtensions) {
            throw new PicardException(String.format("Could not parse the FASTQ extension (expected '_001' + '%s'): %s", FastqConstants.FastqExtensions.values().toString(), baseFastq));
        }
        int idx = 2;
        while (true) {
            String fastq = baseFastq.getAbsolutePath();
            fastq = String.format("%s_%03d%s", fastq.substring(0, fastq.length() - suffix.length()), idx, fastqExtensions.getExtension());
            try {
                IOUtil.assertFileIsReadable((File)new File(fastq));
            }
            catch (SAMException e) {
                break;
            }
            files.add(new File(fastq));
            ++idx;
        }
        return files;
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.FASTQ);
        if (this.FASTQ2 != null) {
            IOUtil.assertFileIsReadable((File)this.FASTQ2);
        }
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        SAMFileHeader header = this.createSamFileHeader();
        SAMFileWriter writer = new SAMFileWriterFactory().makeSAMOrBAMWriter(header, false, this.OUTPUT);
        this.QUALITY_FORMAT = FastqToSam.determineQualityFormat(this.fileToFastqReader(this.FASTQ), this.FASTQ2 == null ? null : this.fileToFastqReader(this.FASTQ2), this.QUALITY_FORMAT);
        ArrayList<FastqReader> readers1 = new ArrayList<FastqReader>();
        ArrayList<FastqReader> readers2 = new ArrayList<FastqReader>();
        if (this.USE_SEQUENTIAL_FASTQS) {
            for (File fastq : FastqToSam.getSequentialFileList(this.FASTQ)) {
                readers1.add(this.fileToFastqReader(fastq));
            }
            if (null != this.FASTQ2) {
                for (File fastq : FastqToSam.getSequentialFileList(this.FASTQ2)) {
                    readers2.add(this.fileToFastqReader(fastq));
                }
                if (readers1.size() != readers2.size()) {
                    throw new PicardException(String.format("Found %d files for FASTQ and %d files for FASTQ2.", readers1.size(), readers2.size()));
                }
            }
        } else {
            readers1.add(this.fileToFastqReader(this.FASTQ));
            if (this.FASTQ2 != null) {
                readers2.add(this.fileToFastqReader(this.FASTQ2));
            }
        }
        for (int idx = 0; idx < readers1.size(); ++idx) {
            this.makeItSo((FastqReader)readers1.get(idx), readers2.isEmpty() ? null : (FastqReader)readers2.get(idx), writer);
        }
        for (FastqReader reader : readers1) {
            reader.close();
        }
        for (FastqReader reader : readers2) {
            reader.close();
        }
        writer.close();
        return 0;
    }

    public void makeItSo(FastqReader reader1, FastqReader reader2, SAMFileWriter writer) {
        int readCount = reader2 == null ? this.doUnpaired(reader1, writer) : this.doPaired(reader1, reader2, writer);
        LOG.info(new Object[]{"Processed " + readCount + " fastq reads"});
    }

    protected int doUnpaired(FastqReader freader, SAMFileWriter writer) {
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader.hasNext()) {
            FastqRecord frec = freader.next();
            SAMRecord srec = this.createSamRecord(writer.getFileHeader(), this.getReadName(frec.getReadHeader(), false), frec, false);
            srec.setReadPairedFlag(false);
            writer.addAlignment(srec);
            progress.record(srec);
            ++readCount;
        }
        return readCount;
    }

    protected int doPaired(FastqReader freader1, FastqReader freader2, SAMFileWriter writer) {
        int readCount = 0;
        ProgressLogger progress = new ProgressLogger(LOG);
        while (freader1.hasNext() && freader2.hasNext()) {
            FastqRecord frec1 = freader1.next();
            FastqRecord frec2 = freader2.next();
            String frec1Name = this.getReadName(frec1.getReadHeader(), true);
            String frec2Name = this.getReadName(frec2.getReadHeader(), true);
            String baseName = this.getBaseName(frec1Name, frec2Name, freader1, freader2);
            SAMRecord srec1 = this.createSamRecord(writer.getFileHeader(), baseName, frec1, true);
            srec1.setFirstOfPairFlag(true);
            srec1.setSecondOfPairFlag(false);
            writer.addAlignment(srec1);
            progress.record(srec1);
            SAMRecord srec2 = this.createSamRecord(writer.getFileHeader(), baseName, frec2, true);
            srec2.setFirstOfPairFlag(false);
            srec2.setSecondOfPairFlag(true);
            writer.addAlignment(srec2);
            progress.record(srec2);
            ++readCount;
        }
        if (freader1.hasNext() || freader2.hasNext()) {
            throw new PicardException("Input paired fastq files must be the same length");
        }
        return readCount;
    }

    private FastqReader fileToFastqReader(File file) {
        return new FastqReader(file, this.ALLOW_AND_IGNORE_EMPTY_LINES.booleanValue());
    }

    private SAMRecord createSamRecord(SAMFileHeader header, String baseName, FastqRecord frec, boolean paired) {
        SAMRecord srec = new SAMRecord(header);
        srec.setReadName(baseName);
        srec.setReadString(frec.getReadString());
        srec.setReadUnmappedFlag(true);
        srec.setAttribute(ReservedTagConstants.READ_GROUP_ID, (Object)this.READ_GROUP_NAME);
        byte[] quals = StringUtil.stringToBytes((String)frec.getBaseQualityString());
        this.convertQuality(quals, this.QUALITY_FORMAT);
        for (byte qual : quals) {
            int uQual = qual & 0xFF;
            if (uQual >= this.MIN_Q && uQual <= this.MAX_Q) continue;
            throw new PicardException("Base quality " + uQual + " is not in the range " + this.MIN_Q + ".." + this.MAX_Q + " for read " + frec.getReadHeader());
        }
        srec.setBaseQualities(quals);
        if (paired) {
            srec.setReadPairedFlag(true);
            srec.setMateUnmappedFlag(true);
        }
        return srec;
    }

    public SAMFileHeader createSamFileHeader() {
        SAMReadGroupRecord rgroup = new SAMReadGroupRecord(this.READ_GROUP_NAME);
        rgroup.setSample(this.SAMPLE_NAME);
        if (this.LIBRARY_NAME != null) {
            rgroup.setLibrary(this.LIBRARY_NAME);
        }
        if (this.PLATFORM != null) {
            rgroup.setPlatform(this.PLATFORM);
        }
        if (this.PLATFORM_UNIT != null) {
            rgroup.setPlatformUnit(this.PLATFORM_UNIT);
        }
        if (this.SEQUENCING_CENTER != null) {
            rgroup.setSequencingCenter(this.SEQUENCING_CENTER);
        }
        if (this.PREDICTED_INSERT_SIZE != null) {
            rgroup.setPredictedMedianInsertSize(this.PREDICTED_INSERT_SIZE);
        }
        if (this.DESCRIPTION != null) {
            rgroup.setDescription(this.DESCRIPTION);
        }
        if (this.RUN_DATE != null) {
            rgroup.setRunDate((Date)this.RUN_DATE);
        }
        if (this.PLATFORM_MODEL != null) {
            rgroup.setPlatformModel(this.PLATFORM_MODEL);
        }
        if (this.PROGRAM_GROUP != null) {
            rgroup.setProgramGroup(this.PROGRAM_GROUP);
        }
        SAMFileHeader header = new SAMFileHeader();
        header.addReadGroup(rgroup);
        for (String comment : this.COMMENT) {
            header.addComment(comment);
        }
        header.setSortOrder(this.SORT_ORDER);
        return header;
    }

    void convertQuality(byte[] quals, FastqQualityFormat version) {
        switch (version) {
            case Standard: {
                SAMUtils.fastqToPhred((byte[])quals);
                break;
            }
            case Solexa: {
                solexaQualityConverter.convertSolexaQualityCharsToPhredBinary(quals);
                break;
            }
            case Illumina: {
                solexaQualityConverter.convertSolexa_1_3_QualityCharsToPhredBinary(quals);
            }
        }
    }

    String getBaseName(String readName1, String readName2, FastqReader freader1, FastqReader freader2) {
        String[] toks = this.getReadNameTokens(readName1, 1, freader1);
        String baseName1 = toks[0];
        String num1 = toks[1];
        toks = this.getReadNameTokens(readName2, 2, freader2);
        String baseName2 = toks[0];
        String num2 = toks[1];
        if (!baseName1.equals(baseName2)) {
            throw new PicardException(String.format("In paired mode, read name 1 (%s) does not match read name 2 (%s)", baseName1, baseName2));
        }
        boolean num1Blank = StringUtil.isBlank((String)num1);
        boolean num2Blank = StringUtil.isBlank((String)num2);
        if (num1Blank || num2Blank) {
            if (!num1Blank) {
                throw new PicardException(this.error(freader1, "Pair 1 number is missing (" + readName1 + "). Both pair numbers must be present or neither."));
            }
            if (!num2Blank) {
                throw new PicardException(this.error(freader2, "Pair 2 number is missing (" + readName2 + "). Both pair numbers must be present or neither."));
            }
        } else {
            if (!num1.equals("1")) {
                throw new PicardException(this.error(freader1, "Pair 1 number must be 1 (" + readName1 + ")"));
            }
            if (!num2.equals("2")) {
                throw new PicardException(this.error(freader2, "Pair 2 number must be 2 (" + readName2 + ")"));
            }
        }
        return baseName1;
    }

    private String[] getReadNameTokens(String readName, int pairNum, FastqReader freader) {
        if (readName.equals("")) {
            throw new PicardException(this.error(freader, "Pair read name " + pairNum + " cannot be empty: " + readName));
        }
        int idx = readName.lastIndexOf("/");
        String[] result = new String[2];
        if (idx == -1) {
            result[0] = readName;
            result[1] = null;
        } else {
            result[1] = readName.substring(idx + 1, readName.length());
            if (!result[1].equals("1") && !result[1].equals("2")) {
                result[0] = readName;
                result[1] = null;
            } else {
                result[0] = readName.substring(0, idx);
            }
        }
        return result;
    }

    private String error(FastqReader freader, String str) {
        return str + " at line " + freader.getLineNumber() + " in file " + freader.getFile().getAbsolutePath();
    }

    private String getReadName(String fastqHeader, boolean paired) {
        String readName;
        int idx = fastqHeader.indexOf(" ");
        String string = readName = idx == -1 ? fastqHeader : fastqHeader.substring(0, idx);
        while (this.STRIP_UNPAIRED_MATE_NUMBER.booleanValue() && !paired && (readName.endsWith("/1") || readName.endsWith("/2"))) {
            readName = readName.substring(0, readName.length() - 2);
        }
        return readName;
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.MIN_Q < 0) {
            return new String[]{"MIN_Q must be >= 0"};
        }
        if (this.MAX_Q > 93) {
            return new String[]{"MAX_Q must be <= 93"};
        }
        return null;
    }
}

