`corna.test.fun` <-
function(x, y, z, min.pop=-1, min.sam=-1,
         hypergeometric=TRUE, hyper.lower.tail=FALSE, 
         fisher=FALSE, fisher.alternative="two.sided", 
         chi.square=FALSE,
         p.adjust.method="none", label=FALSE, sort="hypergeometric", desc=NULL){
                           
  sam <- data.frame(t=unique(x))                      # data frame of sample transcripts
  pop <- data.frame(t=unique(y))                      # data frame of population transcripts
  
  sam <- merge(sam, pop)                              # sample must be in population
  
  map <- data.frame(t=z[,1], m=z[,2])                 # data frame of tran-mir pairs
    
  s2m <- unique(merge(map, sam, sort=F))              # sample tran-mir pairs     
  p2m <- unique(merge(map, pop, sort=F))              # population tran-mir pairs
  stn <- nrow(sam)                                    # total number of tran in sample
  ptn <- length(y)                                    # total number of tran in population
  rat <- stn/ptn                                      # ratio of sample and population 
      
  mv <- as.vector(na.omit(unique(s2m[, "m"])))        # all of mir associated with tran in sample
  pv <- vector()                                      # number of tran associated with this miRNA in population
  sv <- vector()                                      # number of tran associated with this miRNA in sample
  ev <- vector()                                      # expected frequency if the sample was randomly selected
  hv <- vector()                                      # hypergeometric test p-value
  fv <- vector()                                      # fisher's exact test p-value 
  cv <- vector()                                      # chi-squared test p-value 
  
  for(m in mv){                                       # for each miRNA associated with some sample gene
    san <- length(unique(s2m$t[s2m$m==m]))            # number of tran associated with this miRNA in sample
    pan <- length(unique(p2m$t[p2m$m==m]))            # number of tran associated with this miRNA in population
    ean <- round(pan*rat )                            # calculate expectation 
    
    pv <- append(pv, pan)
    sv <- append(sv, san) 
    ev <- append(ev, ean)
    
    # build 2 x 2 contingency table according to 2 variables
    # variable 1 is data category: "associated with this mir" or "not associated"
    # varialbe 2 is result category: "being selected in sample" or "not selected"  
     
    n1 <- san                                         # associated     and selected 
    n2 <- stn-san                                     # not associated and selected
    n3 <- pan-san                                     # associated     and not selected
    n4 <- ptn-stn-pan+san                             # not associated and not selected
    mt <- matrix(c(n1, n2, n3, n4), nrow=2)           # create matrix Fisher's exact test
    
    if(hypergeometric==TRUE){                         # hypergeometric test (default)
      if(hyper.lower.tail==FALSE){
        san <- san-1  
      }
      hp <- phyper(san, pan, ptn-pan, stn, lower.tail=hyper.lower.tail) 
      hv <- append(hv, hp)                                 
    }
    if(fisher==TRUE){                                 # Fisher's exact test (optional)
      fp <- fisher.test(mt, alternative=fisher.alternative)$p.v  
      fv <- append(fv, fp)
    }
    if(chi.square==TRUE){                              # Chi-square test (optional)
      cp <- tryCatch(chisq.test(mt)$p.value, warning = function(x) NA)                    
      cv <- append(cv, cp)
    }
  }
        
  d <- data.frame(row.names=mv,                       # data frame for the test results
                  total=pv, 
                  expectation=ev, 
                  observation=sv)       
                  
  if(hypergeometric==TRUE){d[, "hypergeometric"] = hv} 
  if(fisher==TRUE){d[, "fisher"] = fv}
  if(chi.square==TRUE){d[, "chi.square"] = cv}
  
  d <- d[order(d[, sort]), ]                          # sort by p-value (default hypergeometric) 

  for(i in 4:length(d)){                              # adjust p-values
    d[, i] <- as.numeric(p.adjust(as.numeric(d[, i]), p.adjust.method))
    if(label==TRUE){                                  # label significant ones 
      d[d[, i] <= 0.05, i] <- paste(d[d[, i] <= 0.05, i], "*", sep="")     
      d[d[, i] <= 0.01, i] <- paste(d[d[, i] <= 0.01, i], "*", sep="") 
    }
  }
    
  if(is.data.frame(desc)){
    v <- vector()
    for(i in rownames(d)){
      j <- corna.map.fun(desc, i, 1, 2)
      t <- ""
      for(k in j){
        t <- paste(t, k, sep=":")
      }
      t <- sub(":", "", t)
      v <- append(v, t)
    }
    d[, "description"] <- v
  }
 
  if(min.pop!=-1){d <- d[d[, "total"]>=min.pop, ]}
  if(min.sam!=-1){d <- d[d[, "observation"]>=min.sam, ]}
 
  return(d)
}                                                             
  