#ifndef PSSM_H
#define PSSM_H

// referred to by mhmm-state.h
typedef struct pssm PSSM_T;

#include "mhmm-state.h"

//
// Range of integral score values for a PSSM column.
//
#define PSSM_RANGE 100

//
// Macros to convert between scaled score and raw score.
//
#define scaled_to_raw(x,w,scale,offset) (((x)/(scale)) + ((w)*(offset)))
#define raw_to_scaled(x,w,scale,offset) (nint(((x) - ((w)*offset)) * (scale)))

#define get_pssm_w(pssm) ((pssm)->w)
#define get_pssm_alphsize(pssm) ((pssm)->alphsize)
#define get_pssm_scale(pssm) ((pssm)->scale)
#define get_pssm_offset(pssm) ((pssm)->offset)
#define get_pssm_pv_length(pssm) (get_array_length((pssm)->pv))
#define get_pssm_pv(score, pssm) (get_array_item(score, (pssm)->pv))
#define get_pssm_score(row, col, pssm) (get_matrix_cell((row), (col), (pssm)->matrix))

//
// PSSM
//
// This object was created for AMA because the "scale" and "offset"
// parameters need to be stored with each PSSM, but were 
// globals in pssm.c.  This object should be used in all the programs
// that use PSSMs, since their scale and offset can differ, and their
// cdfs and pdfs should be kept with them.
//
struct pssm {
  MATRIX_T *matrix;	// The PSSM score matrix.
  int w;		// Width of PSSM.
  int alphsize;		// Wize of PSSM alphabet.
  BOOLEAN_T matrix_is_log;	// True if matrix is log likelihood ratio.
  BOOLEAN_T matrix_is_scaled;	// True if matrix is scaled.
  double scale;		// Scale factor for scores.
  double offset;	// Offset for scores.
  int range;		// Scaled scores in range [0..range].
  ARRAY_T *pv;		// P-value lookup table for scores.
  int min_score;	// Smallest index with non-zero pdf.
  int max_score;	// Largest index with non-zero pdf.
};

//
// PSSM_PAIR
//
// PSSMs for the negative and positive DNA motifs.
//
typedef struct pssm_pair {
  PSSM_T* pos_pssm;		// positive strand PSSM
  PSSM_T* neg_pssm;		// negative strand PSSM 
  // Stuff below here is for AMA:
  // The pv lookup table for the average of n scores will be
  // in row log_2(n), for n=1, 2, 4, ...
  MATRIX_T* n_pv_lookup;	// pv[nsamples, score] lookup table
  ARRAY_T* scaled_to_ama;	// for speed
} PSSM_PAIR_T;

void set_up_pssms_and_pvalues (
  BOOLEAN_T motif_scoring,		// Motif scoring?
  double p_threshold,			// Scale/offset PSSM and create table if > 0
  BOOLEAN_T use_both_strands,           // Compute PSSM for negative strand, too?
  BOOLEAN_T allow_weak_motifs,		// Allow motifs with min p-value < p_threshold?
  MHMM_T*  the_hmm                      // The HMM.
);

void compute_motif_score_matrix
  (BOOLEAN_T  use_pvalues,	// Returns scores as p-values, not log-odds.
   double     p_threshold,	// Divide p-values by this.
   int*       int_sequence,
   int        seq_length,
   MHMM_T*    the_hmm,
   MATRIX_T** motif_score_matrix);

void scale_pssm(
  PSSM_T *pssm,			// The PSSM. (IN/OUT)
  int range			// The desired range. (IN)   
);

void get_pv_lookup_pos_dep(
  PSSM_T* pssm,			// The PSSM.
  MATRIX_T* background_matrix 	// The background model PSSM matrix.
);

void get_pv_lookup(
  PSSM_T* pssm,			// The PSSM.
  ARRAY_T* background		// The background model.
);

double get_unscaled_pssm_score(
  double score,
  PSSM_T* pssm
);

double get_scaled_pssm_score(
  double score,
  PSSM_T* pssm
);

PSSM_T* build_motif_pssm(
  MOTIF_T* motif,       // motif frequencies p_ia (IN)
  ARRAY_T* bg_freqs,    // background frequencies b_a for motif (IN)
  ARRAY_T* pv_bg_freqs, // background frequencies b_a for p-values (IN)
  int range,            // range of scaled scores is [0..w*range]
  BOOLEAN_T get_pdf     // get the PDF rather than the p-value lookup table
);

PSSM_T* build_matrix_pssm(
  MATRIX_T* matrix,     // pssm matrix (IN)
  ARRAY_T* bg_freqs,    // background frequencies b_a (IN)
  int range             // range of scaled scores is [0..w*range] (IN)
);

double get_ama_pv(
  double ama_score,                     // average likelihood ratio score
  int seqlen,                           // length of sequence scanned
  PSSM_PAIR_T* pssm_pair                // pssms for pos and neg motifs
);

PSSM_PAIR_T* create_pssm_pair(
  PSSM_T* pos_pssm,		// positive strand pssm
  PSSM_T* neg_pssm 		// negative strand pssm
);

void free_pssm_pair(
  PSSM_PAIR_T *pssm_pair
);

PSSM_T* allocate_pssm(
  int w, 
  int alphsize
);

void free_pssm(
  PSSM_T* pssm
);
#endif
