;;; -*- Mode:Lisp; Package:Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			      General Representation
;;; ===========================================================================
;;; (c) Copyright 1989, 1991 Cornell University

;;; $Id: general.lisp,v 1.34 1992/12/19 19:17:47 rz Exp $

(in-package "WEYLI")

(defclass has-memoization ()
     ((memos :initform (make-hash-table :test #'eq))))

(defmethod set-memoization ((domain has-memoization) key value)
  (with-slots (memos) domain
    (setf (gethash key memos) value)
    value))

(defmethod get-memoization ((domain has-memoization) key)
  (with-slots (memos) domain
    (gethash key memos)))

(defsetf get-memoization set-memoization)

(defmacro memoize (domain expression &body body)
  `(let ((.expr. ,expression))
    (with-slots (memos) ,domain
       (multiple-value-bind (value found?) (gethash .expr. memos)
	 (if found? value
	     (setf (get-memoization ,domain .expr.) (progn ,@body)))))))

;; Classes for General expressions

(defclass general-expressions (domain has-memoization non-strict-domain)
     ((variables :initform ()
		 :accessor ge-variables)
      (context :initform ()
	       :accessor ge-context)))

(defclass general-expression (domain-element)
     ((simplified? :initform nil
		   :accessor simplified?)))

(defclass variable (general-expression has-property-list)
     ((symbol :initarg :symbol
	      :accessor symbol-of)
      (string :initarg :string
	      :accessor string-of)))

(defclass ge-1-ary (general-expression)
     ((arg :initarg :arg
	   :accessor argument-of)))

;; N-ary operators are built from this class 
(defclass ge-nary (general-expression)
     ((terms :initform nil
	     :initarg :terms
	     :accessor terms-of)))

(defsubst ge-nary? (x) (typep x 'ge-nary))

(defclass ge-plus (ge-nary)
     ())

(defsubst ge-plus? (x) (typep x 'ge-plus))

(defclass ge-times (ge-nary)
     ())

(defsubst ge-times? (x) (typep x 'ge-times))

(defclass ge-expt (general-expression)
     ((base :initarg :base
	    :accessor base-of)
      (exp :initarg :exp
	   :accessor exponent-of)))

(defsubst ge-expt? (x) (typep x 'ge-expt))

(defclass ge-function
     (general-expression)
     ((funct :initarg :funct
	     :accessor funct-of)
      (args :initarg :args
	    :accessor args-of)))

(defsubst ge-function? (x) (typep x 'ge-function))

(defclass ge-deriv (general-expression)
     ((argument :initarg :argument
		:accessor argument-of)
      (varlist :initarg :varlist
	       :accessor varlist-of)))

(defsubst ge-deriv? (x) (typep x 'ge-deriv))

(defclass ge-equation (general-expression)
     ((lhs :initarg :lhs
	   :accessor lhs-of)
      (rhs :initarg :rhs
	   :accessor rhs-of)))

(defclass ge-eqn= (ge-equation)
     ())

(defsubst ge-eqn=? (exp)
  (typep exp 'ge-eqn=))

(defclass ge-eqn> (ge-equation)
     ())

(defsubst ge-eqn>? (exp)
  (typep exp 'ge-eqn>))

(defclass ge-eqn>= (ge-equation)
     ())

(defsubst ge-eqn>=? (exp)
  (typep exp 'ge-eqn>=))

(defmethod display ((ge general-expression) &optional stream &rest ignore)
  (declare (ignore ignore))
  (princ ge stream))

(defmethod simplify ((ge general-expression))
  ge)

(defmethod ge-equal ((x general-expression) (y general-expression))
  nil)

(defmethod ge-great ((x general-expression) (y general-expression))
  nil)


;; Numbers and Variables

(defmethod make-quotient-element
    ((domain general-expressions) (x integer) (y integer))
  (make-instance 'rational-number :domain domain
		 :numerator x :denominator y))

(defmethod make-element ((domain general-expressions) (x integer) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-integer :domain domain :value x))

(defmethod make-element ((domain general-expressions) (x ratio) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'rational-number :domain domain
		 :numerator (lisp:numerator x)
		 :denominator (lisp:denominator x)))

(defmethod make-element ((domain general-expressions) (x float) &rest ignore)
  (declare (ignore ignore))
  (make-instance 'floating-point-number :domain domain
		 :value x))

(defmethod make-element ((domain general-expressions) (x lisp:complex)
			 &rest ignore)
  (declare (ignore ignore))
  (make-instance 'complex-number :domain domain
		 :realpart (lisp:realpart x)
		 :imagpart (lisp:imagpart x)))

(defmethod coerce ((num number) (domain general-expressions))
  (make-element domain num))

(defmethod coerce ((num bigfloat) (domain general-expressions))
  (make-bigfloat domain (bigfloat-mantissa num) (bigfloat-exponent num)))

(defmethod simplify ((x number))
  (make-element *general* x))

(defmethod simplify ((x numeric))
  x)

(defmethod ge-equal ((x (or number numeric)) (y (or number numeric)))
  (= x y))

(defmethod ge-great ((x (or number numeric)) (y (or number numeric)))
  (> x y))

;; Variables

(defmethod initialize-instance :after ((var variable) &rest ignore)
  (declare (ignore ignore))
  (reparse-print-string var))

(defmethod reparse-print-string ((var variable))
  (let ((string (cond ((atom (symbol-of var))
		       (string-downcase (symbol-of var)))
		      (t (format nil "[~A]" (symbol-of var)))))
	temp)
    (when (setq temp (getf var :subscripts))
      (setq string 
	    (format nil "~A(~S~{,~S~})"
		    string (first temp) (rest temp))))
    (setf (string-of var) string)))

(defmethod coerce ((var symbol) (domain general-expressions))
  (make-variable domain var))

(defmethod make-variable ((domain general-expressions) var)
  (loop for v in (ge-variables domain)
	do (when (equal (symbol-of v) var)
	     (return v))
	finally
         (setq var (make-instance 'variable :domain domain :symbol var))
	 (push var (ge-variables domain))
	 (return var)))

(defmethod print-object ((var variable) stream)
  (let ((sym (string-of var)))
    (cond ((and (not (null sym)) (atom sym))
	   #+Genera
	   (format stream "~'i~A~" sym)
	   #-Genera
	   (princ sym stream))
	  (t (princ (symbol-of var) stream)))))

;; This function is only to be applied to general expressions. 
(defsubst ge-variable? (x)
  (typep x 'variable))

(defmethod add-subscripts ((var symbol) &rest subscripts)
  (%apply #'add-subscripts (coerce var *general*) subscripts))

(defmethod add-subscripts ((var variable) &rest subscripts)
  (setq var (coerce var *general*))
  (let* ((symbol (symbol-of var))
	 (subscripts (append (getf var :subscripts) (copy-list subscripts)))
	 (canonical-var 
	  (member symbol (ge-variables *general*)
		  :test (lambda (a b)
			  (and (equal a (symbol-of b))
			       (equal subscripts (getf b :subscripts)))))))
    (cond (canonical-var
	   (first canonical-var))
	  (t (setq var (make-instance 'variable :domain (domain-of var)
				      :symbol symbol))
	     (setf (getf var :subscripts) subscripts)
	     (reparse-print-string var)
	     (push var (ge-variables *general*))
	     var))))

(defmethod ge-equal ((x variable) (y variable))
  (eql x y))

(defmethod ge-great ((x variable) (y variable))
  (string-greaterp (string-of x) (string-of y)))

(defmethod ge-great ((x variable) (y (or ge-plus ge-times)))  
  (loop for w in (terms-of y)
	unless (ge-great x w)
	  do (return nil)
	finally (return t)))

(defmethod ge-great ((x (or ge-plus ge-times)) (y variable))  
  (loop for w in (terms-of x)
	unless (ge-great w y)
	  do (return t)
	finally (return nil)))

;; Contexts

(defun initialize-contexts ()
  (setq *general* (make-instance 'general-expressions)))

(defmacro with-new-context (&body body)
  `(let ((*general* (make-instance 'general-expressions)))
     ,@body))

(defmacro check-point-context (&body body)
  `(let ((.old-variables. (ge-variables *general*))
	 (.old-context. (ge-context *general*)))
     (unwind-protect (progn ,@body)
       (setf .old-variables. (ge-variables *general*))
       (setf .old-context. (ge-context *general*)))))

(defmethod coerce ((exp list) (domain general-expressions))
  (flet ((coerce-obj (x)
	   (coerce x domain)))
    (cond ((eql (first exp) '+)
	   (make-ge-plus domain
			 (mapcar #'coerce-obj (rest exp))))
	  ((eql (first exp) '*)
	   (make-ge-plus domain
			 (mapcar #'coerce-obj (rest exp))))
	  ((eql (first exp) '-)
	   (if (null (rest (rest exp)))
	       (make-ge-times domain (list -1 (coerce-obj (second exp))))
	       (make-ge-plus domain
		 (list (coerce-obj (second exp))
		       (make-ge-times domain
			 (cons (make-element domain -1)
			       (mapcar #'coerce-obj (rest (rest exp)))))))))
	  ((eql (first exp) '/)
	   (make-ge-times domain
	     (list (coerce-obj (second exp))
		   (make-ge-expt domain
		     (make-ge-times domain
		       (mapcar #'coerce-obj (rest (rest exp))))
		     (make-element domain -1)))))
	  (t (error "Don't know how to coerce ~S into ~S"
		    exp domain)))))			     
			     
(defmethod get-variable-property ((domain domain) (var variable) key)
  (loop for var-prop in (ge-context domain)
	do (when (eql (first var-prop) var)
	     (return (%getf (rest var-prop) key)))
	finally (progn 
		  (push (list var) (ge-context domain))
		  (return nil))))

(defmethod set-variable-property (domain (var variable) key value)
  (loop for var-prop in (ge-context domain)
	do (when (eql (first var-prop) var)
	     (setf (%getf (rest var-prop) key) value)
	     (return value))	
	finally (progn 
		  (push (list var key value) (ge-context domain))
		  (return value))))

(defsetf get-variable-property set-variable-property)

(defun parenthesized-display (expr stream)
  (princ "(" stream)
  (print-object expr stream)
  (princ ")" stream))

(defun safe-display (expr stream)
  (if (or (number? expr) 
	  (ge-variable? expr)
	  (ge-expt? expr))
      (print-object expr stream)      
      (parenthesized-display expr stream)))

;; Display a list of objects, paying attention to *print-length*.  No
;; surrounding delimiters.   This is a method so that we can define
;; similar functions for sets of objects embedded in arrays.
(defmethod display-list
    ((objects list) &optional (stream *standard-output*))
  (when objects
    (let ((cnt (or *print-length* -1)))
      (declare (fixnum cnt))
      (print-object (first objects) stream)
      (lisp:decf cnt)
      (loop for var in (rest objects)
	    do (princ ", " stream)
	       (when (lisp:zerop cnt)
		 (princ "..." stream)
		 (return))
	       (print-object var stream)
	       (lisp:decf cnt)))))


;; Ordering functions for general expressions

;; Some operators may choose to ignore various parameters here.

(defun ge-lequal (x y)
  (loop
    (when (and (null x) (null y))
      (return-from ge-lequal t))
    (when (or (null x) (null y)
	      (not (ge-equal (first x) (first y))))
      (return-from ge-lequal nil))
    (pop x) (pop y)))

(defun ge-lgreat (x y)
  (loop 
    (cond ((null x)
	   (return nil))
	  ((null y)
	   (return t))
	  ((ge-equal (first x) (first y)))
	  ((ge-great (first x) (first y))
	   (return t))
	  (t (return nil)))
    (pop x) (pop y)))


;; x is assumed to be a variable


(defmethod real? ((x number))
  (not (lisp:complexp x)))

(defmethod real? ((x bigfloat))
  t)

(defmethod real? ((x numeric))
  (not (typep x 'complex-number)))

(defmethod minus? ((x t))
  nil)

;; For compatibility with Common Lisp
(defun minusp (x) (minus? x))
(defun plusp (x) (plus? x))
(defun zerop (x) (0? x))

(defun ge-minus? (x)
  (cond ((and (number? x) (real? x)) (minus? x))
	((ge-times? x)
	 (let ((lead-term (first (terms-of x))))
	   (and (and (number? lead-term)
		     (real? lead-term)
		     (minus? lead-term)))))
	(t nil)))

;; This works by converting the sum into a list of dotted pairs.  The
;; first element of the list is a number, while the second is a list
;; of product terms.  This makes combining new elements quite easy.
;; After the combination, everything is converted back to the standard
;; representation. 

(defmacro merge-terms-in-sum (terms &body body)
  `(let ((,terms (list nil)))
     (labels ((add-term (base order) 
		(loop with terms = ,terms do
		  (cond ((or (null (rest terms))
			     (ge-lgreat base (rest (second terms))))
			 (push (cons order base) (rest terms))
			 (return t))
			((ge-lequal base (rest (second terms)))
			 (setf (first (second terms))
			       (+ (first (second terms)) order))
			 (when (0? (first (second terms)))
			   (setf (rest terms) (rest (rest terms))))
			 (return t)))
		  (pop terms))))
       ,@body)))


(defmethod make-ge-plus ((domain general-expressions) terms)
  (make-instance 'ge-plus :domain domain :terms terms))



(defmethod print-object ((sum ge-plus) stream)
  (let ((terms (terms-of sum)))
    (print-object (first terms) stream)
    (loop for x in (rest terms)
	  do (cond ((and (number? x) (real? x))
		    (if (plus? x)
			(format stream " + ~S" x)
			(format stream " - ~S" (minus x))))
		   ((ge-minus? x)
		    (princ " - " stream)
		    (safe-display 
		     (simp-times-terms (domain-of sum) (list -1 x))
		     stream))
		   (t (princ " + " stream)
		      (print-object x stream))))))

(defmethod simplify ((x ge-plus))
  (simp-plus-terms (domain-of x) (terms-of x)))
  
(defun simp-plus-terms (domain old-terms)
  (merge-terms-in-sum terms
    (let ((const 0))
      (labels ((loop-over-terms (terms)
		 (loop for term in terms
		       do (setq term (simplify term))
			  (cond ((number? term) 
				 (setq const (+ const term)))
				((ge-plus? term)
				 (loop-over-terms (terms-of term)))
				((ge-times? term)
				 (setq term (terms-of term))
				 (cond ((number? (first term))
					(add-term (rest term) (first term)))
				       (t (add-term term 1))))
				(t (add-term (list term) 1))))))
	(loop-over-terms old-terms)
	(setq terms (loop for (c . term-l) in (rest terms)
			  collect
			  (if (or (eql c 1) (eql c 1.0))
			      (if (null (rest term-l))
				  (first term-l)
				  (simplify
				   (make-ge-times domain term-l)))
			      (simplify
			       (make-ge-times domain (cons c term-l)))))) 
	(cond ((not (0? const))
	       (if (null terms) const
		   (make-ge-plus domain (cons const terms))))
	      ((null terms)
	       (make-element domain 0))
	      ((null (rest terms))
	       (first terms))
	      (t (make-ge-plus domain terms)))))))

(defmethod ge-equal ((x ge-plus) (y ge-plus))
  (ge-lequal (terms-of x) (terms-of y)))

(defmethod ge-great ((x ge-plus) (y ge-plus))
  (ge-lgreat (terms-of x) (terms-of y)))


(defmethod make-ge-times ((domain general-expressions) terms)
  (make-instance 'ge-times :domain domain :terms terms))

(defmethod print-object ((x ge-times) stream)
  (let ((terms (terms-of x)))
    (safe-display (first terms) stream)
    (loop for x in (rest terms)
	  do (princ " " stream)
	     (safe-display x stream))))

(defmethod simplify ((x ge-times)) 
  (simp-times-terms (domain-of x) (terms-of x)))

(defun simp-times-terms (domain old-terms)
  (merge-terms-in-sum terms 
    (let ((const 1))
      (labels ((loop-over-terms (terms) 
		 (loop for term in terms do
		   (setq term (simplify term))
		   (cond ((number? term)
			  (when (0? term)
			    (return-from simp-times-terms
			      (make-element domain 0)))
			  (setq const (* const term)))
			 ((ge-times? term)
			  (loop-over-terms (terms-of term)))
			 ((ge-expt? term)
			  (let ((exp (exponent-of term))
				(base (base-of term)))
			    (cond ((number? (exponent-of term))
				   (add-term (list base) exp))
				  (t (add-term (list base)
					       (make-element domain 1))))))
			 (t (add-term (list term) 1))))))
	(loop-over-terms old-terms)
	(setq terms (loop for (exp base) in (rest terms)
			  collect
			  (if (1? exp) base
			      (make-ge-expt domain base exp))))
	(cond ((not (1? const))
	       (if (null terms)
		   const
		   (make-ge-times domain (cons const terms))))
	      ((null terms)
	       (make-element domain 1))
	      ((null (rest terms))
	       (first terms))
	      (t (make-ge-times domain terms)))))))

(defmethod ge-equal ((x ge-times) (y ge-times))
  (ge-lequal (terms-of x) (terms-of y)))

(defmethod ge-great ((x ge-times) (y ge-times))
  (ge-lgreat (terms-of x) (terms-of y)))


(defmethod make-ge-expt ((domain general-expressions) base exp)
  (make-instance 'ge-expt :domain domain :base base :exp exp))

(defmethod simplify ((x ge-expt))
  (let ((exp (simplify (exponent-of x)))
	(base (base-of x)))
    (cond ((0? exp) 1)
	  ((1? exp) (simplify base))
	  ((and (number? (setq base (simplify base)))
		(number? exp))
	   (expt base exp))
	  ((ge-expt? base)
	   (simplify 
	    (make-ge-expt (domain-of x) (base-of base)
			  (* (exponent-of base) exp))))
	  (t (make-ge-expt (domain-of x) (simplify (base-of x)) exp)))))

(defmethod print-object ((expr ge-expt) stream)
  (safe-display (base-of expr) stream)
  (princ "^" stream)
  (safe-display (exponent-of expr) stream))

(defmethod ge-equal ((x ge-expt) (y ge-expt))
  (and (ge-equal (base-of x) (base-of y))
       (ge-equal (exponent-of x) (exponent-of y))))

(defmethod ge-great ((x ge-expt) (y ge-expt))
  (cond ((ge-great (base-of x) (base-of y))
	 t)
	((ge-equal (base-of x) (base-of y))
	 (ge-great (exponent-of x) (exponent-of y)))
	(t nil)))


(defmethod make-ge-funct ((domain general-expressions) funct &rest args)
  (make-instance 'ge-function :domain domain
		 :funct funct :args (copy-list args)))

(defmacro funct (function &rest args)
  `(make-ge-funct *general* ',function
      ,@(mapcar (lambda (q) `(coerce ,q *general*))
		args)))

(defmethod print-object ((x ge-function) stream)
  (print-object (funct-of x) stream)
  (write-char #\( stream)
  (display-list (args-of x) stream)
  (write-char #\) stream))

(defmethod ge-equal :around ((x ge-1-ary) (y ge-1-ary))
  (if (and (eql (class-of x) (class-of y))
	   (ge-equal (argument-of x) (argument-of y)))
      t
      (call-next-method x y)))

(defmethod ge-equal ((x ge-function) (y ge-function))
  (and (equal (funct-of x) (funct-of y))
       (ge-lequal (args-of x) (args-of y))))

(defmacro def-ge-1oper (name (arg))
  (let ((class-name (intern (format nil "GE-~A" (string name))))
	(maker-name (intern (format nil "MAKE-GE-~A" (string name)))))
    `(progn
       (defclass ,class-name (ge-1-ary)
	    ())
       (defsubst ,(intern (format nil "GE-~A?" (string name))) (exp)
	 (typep exp ',class-name))
       (defmethod print-object ((exp ,class-name) stream)
	 (princ ,(format nil "~A(" (string-downcase (string name)))
		stream)
	 (print-object (argument-of exp) stream)
	 (write-char #\) stream))
       (defun ,(intern (format nil "MAKE-GE-~A" (string name))) (domain ,arg)
	 (make-instance ',class-name :domain domain :arg ,arg))
       (defmethod ,name ((,arg symbol))
	 (simplify (,maker-name *general* (coerce ,arg *general*))))
       (defmethod ,name ((,arg general-expression))
	 (simplify (,maker-name (domain-of ,arg) ,arg)))
       )))

(def-ge-1oper LOG (x))
(defmethod simplify ((x ge-log))
  (let ((exp (simplify (argument-of x))))
    (cond ((lisp::floatp exp)
	   (make-element (domain-of x) (lisp:log exp)))
	  ((ge-expt? exp)
	   (simplify
	    (make-ge-times (domain-of x)
			   `(,(exponent-of exp)
			      ,(make-ge-log (domain-of exp) (base-of exp))))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-log) (var variable))
  (/ (ge-deriv (argument-of exp) var)
     (argument-of exp)))

(def-ge-1oper SIN (x))
(defmethod simplify ((x ge-sin))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:sin exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-sin domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-sin) (var variable))
  (* (ge-deriv (argument-of exp) var)
     (cos (argument-of exp))))

(def-ge-1oper COS (x))
(defmethod simplify ((x ge-cos))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:cos exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 1))
	  ((ge-minus? exp)
	   (make-ge-cos domain (- exp)))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-cos) (var variable))
  (* (- (ge-deriv (argument-of exp) var))
     (sin (argument-of exp))))

(def-ge-1oper TAN (x))
(defmethod simplify ((x ge-tan))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:tan exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-tan domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-tan) (var variable))
  (/ (- (ge-deriv (argument-of exp) var))
     (expt (cos (argument-of exp)) 2)))

(def-ge-1oper ASIN (x))
(defmethod simplify ((x ge-asin))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp:floatp exp)
	   (make-element domain (lisp:asin exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-asin domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-asin) (var variable))
  (* (- (ge-deriv (argument-of exp) var))
     (expt (- 1 (expt (argument-of exp) 2)) -1/2)))

(def-ge-1oper ACOS (x))
(defmethod simplify ((x ge-acos))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:acos exp)))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-acos) (var variable))
  (* (- (ge-deriv (argument-of exp) var))
     -1
     (expt (- 1 (expt (argument-of exp) 2)) -1/2)))

#+FIXTHIS
(def-ge-1oper ATAN (x))
#+FIXTHIS
(defun simplify-atan (x)
  (let ((exp (simplify (argument-of x))))
    (cond ((lisp::floatp exp) (lisp:atan exp))
	  (t `(atan ,exp)))))

#+FIXTHIS
(defmethod-sd ge-deriv ((exp ge-atan) (var variable))
 (/ (ge-deriv (argument-of exp) var)
    (+ 1 (expt (argument-of exp) 2))))

(def-ge-1oper SINH (x))
(defmethod simplify ((x ge-sinh))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:sinh exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-sinh domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-sinh) (var variable))
  (* (- (ge-deriv (argument-of exp) var))
     (cosh (argument-of exp))))

(def-ge-1oper COSH (x))
(defmethod simplify ((x ge-cosh))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:cosh exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 1))
	  ((ge-minus? exp)
	   (make-ge-cosh domain (- exp)))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-cosh) (var variable))
  (* (ge-deriv (argument-of exp) var)
     (sinh (argument-of exp))))

(def-ge-1oper TANH (x))
(defmethod simplify ((x ge-tanh))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:tanh exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-tanh domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-tanh) (var variable))
  (/ (ge-deriv (argument-of exp) var)
     (expt (cosh (argument-of exp)) 2)))

(def-ge-1oper ASINH (x))
(defmethod simplify ((x ge-asinh))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp:floatp exp)
	   (make-element domain (lisp:asinh exp)))
	  ((and (number? exp) (0? exp))
	   (make-element domain 0))
	  ((ge-minus? exp)
	   (- (make-ge-asinh domain (- exp))))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-asinh) (var variable))
  (* (ge-deriv (argument-of exp) var)
     (expt (+ 1 (expt (argument-of exp) 2)) -1/2)))

(def-ge-1oper ACOSH (x))
(defmethod simplify ((x ge-acosh))
  (let ((exp (simplify (argument-of x)))
	(domain (domain-of x)))
    (cond ((lisp::floatp exp)
	   (make-element domain (lisp:acosh exp)))
	  (t x))))

(defmethod-sd ge-deriv ((exp ge-acosh) (var variable))
  (* (ge-deriv (argument-of exp) var)
     (expt (+ (expt (argument-of exp) 2) -1) -1/2)))

#+FIXTHIS
(def-ge-1oper ATANH (x))
#+FIXTHIS
(defun simplify-atanh (x)
  (let ((exp (simplify (argument-of x))))
    (cond ((lisp::floatp exp) (lisp:atanh exp))
	  (t `(atanh ,exp)))))

#+FIXTHIS
(defmethod-sd ge-deriv ((exp ge-atanh) (var variable))
  (/ (ge-deriv (argument-of exp) var)
     (- 1 (expt (argument-of exp) 2))))

;; Derivatives

(defmethod make-ge-deriv ((domain general-expressions) argument varlist)
  (make-instance 'ge-deriv :domain domain
		 :argument argument :varlist varlist))

(defmethod print-object ((expr ge-deriv) stream)
  (princ "D{" stream)
  (print-object (argument-of expr) stream)
  (let ((derivs (varlist-of expr)))
    (cond ((numberp derivs)
	   (format stream ", ~D}" derivs))
	  ((and (null (rest derivs))
		(eql 1 (second (first derivs))))
	   (princ ", " stream)
	   (print-object (first (first derivs)) stream)
	   (princ "}" stream))
	  (t (princ ", {" stream)
	     (loop for (var order) in derivs 
		   and first? = t then nil do 
		     (unless first?
		       (princ ", " stream))
		     (cond ((eql order 1)
			    (print-object var stream))
			   (t (print-object var stream)
			      (format stream "^~D" order))))
	     (princ "}}" stream)))))

(defmethod simplify ((x ge-deriv))
  (let ((arg (simplify (argument-of x)))
	(domain (domain-of x)))
    (merge-terms-in-sum derivs
      (loop for (var order) in (varlist-of x) do
	(add-term (list var) order))
      (when (ge-deriv? arg)
	(loop for (var order) in (varlist-of arg) do
	  (add-term (list var) order))			 
	(setq arg (second arg)))
      (make-ge-deriv domain arg 
		     (loop for (order base) in (rest derivs)
			   collect (list base order))))))

(defmethod ge-equal ((x ge-deriv) (y ge-deriv))
  (let ((x-vars (varlist-of x))
	(y-vars (varlist-of y)))
    (and (ge-equal (argument-of x) (argument-of y))
	 (equal (length x-vars) (length y-vars))
	 (loop for (x-var x-order) in x-vars
	       and (y-var y-order) in y-vars
	       unless (and (ge-equal x-var y-var)
			   (ge-equal x-order y-order))
		 do (return nil)
	       finally (return t)))))

(defmethod ge-great ((x ge-deriv) (y ge-deriv))
  (let ((x-vars (varlist-of x))
	(y-vars (varlist-of y)))
    (cond ((ge-great (argument-of x) (argument-of y)) t)
	  ((ge-great (argument-of y) (argument-of x)) nil)
	  (t (loop for (x-var x-order) in x-vars
		   and (y-var y-order) in y-vars
		   do (cond ((ge-great x-var y-var) (return t))
			    ((ge-equal x-var y-var)
			     (cond ((ge-great x-order y-order)
				    (return t))
				   ((ge-great y-order x-order)
				    (return nil))))
			    (t (return nil))))))))

(defmethod deriv ((exp number) &rest vars)
  (if (null vars)
      (make-element *general* exp)
      (make-element *general* 0)))

(defmethod deriv ((exp numeric) &rest vars)
  (if (null vars) exp
      (make-element (domain-of exp) 0)))

(defmethod deriv ((exp (or symbol general-expression)) &rest vars)
  (setq exp (coerce exp *general*))
  (loop for v in vars
	do (setq exp (ge-deriv exp (coerce v *general*))))
  exp)

(defmethod ge-deriv (exp var)
  (error "Don't know how to take the derivative of ~S wrt ~S"
	 exp var))

(defmethod ge-deriv ((exp general-expression) (var symbol))
  (ge-deriv exp (coerce var (domain-of exp))))

(defmethod-sd ge-deriv ((exp numeric) (var variable))
  (make-element domain 0))

(defmethod-sd ge-deriv ((exp variable) (var variable))
  (cond ((ge-equal exp var) (make-element domain 1))
	((depends-on? exp var)
	 (make-ge-deriv domain exp `((,var 1))))
	(t (make-element domain 0))))

(defmethod-sd ge-deriv ((exp ge-plus) (var variable))
  (simplify
   (make-ge-plus domain (loop for x in (terms-of exp)
			      collect (ge-deriv x var)))))

(defmethod-sd ge-deriv ((exp ge-times) (var variable))
  (let ((terms (terms-of exp)))
    (simplify
     (make-ge-plus domain
       (loop for x in terms
	     collect
	     (simplify
	       (make-ge-times domain
		 (cons (ge-deriv x var) (remove x terms)))))))))

(defmethod-sd ge-deriv ((exp ge-expt) (var variable))
  (let ((base (base-of exp))
	(power (exponent-of exp)))
    (cond ((depends-on? power var)
	   (error "Not yet implemented"))
	  ((and (number? power) (= power 2))
	   (* 2 base (ge-deriv base var)))
	  (t (* power (expt base (- power 1)))))))


(defmethod-sd ge-deriv ((exp ge-deriv) (var variable))  
  (labels ((deriv (l)
	     (cond ((null l) (list (list var 1)))
		   ((ge-equal var (first (first l)))
		    (cons (list var (1+ (second (first l))))
			  (rest l)))
		   (t (cons (first l) (deriv (rest l)))))))
    (if (depends-on? (argument-of exp) var)
	(make-ge-deriv domain (argument-of exp) (deriv (varlist-of exp)))
	(make-element domain 0))))

(defmethod-sd ge-deriv ((exp ge-function) (var variable))
  (make-ge-deriv domain exp (list (list var 1))))

(defmethod make-ge-eqn= ((domain general-expressions) lhs rhs)
  (make-instance 'ge-eqn= :domain domain :rhs rhs :lhs lhs))

(defmethod print-object ((eqn ge-eqn=) stream)
  (print-object (lhs-of eqn) stream)
  (princ " = " stream)
  (print-object (rhs-of eqn) stream))

(defmethod eqn= (lhs rhs)
  (make-ge-eqn= *general*
		(simplify (coerce lhs *general*))
		(simplify (coerce rhs *general*))))

(defmethod simplify ((eqn ge-eqn=))
  (make-ge-eqn= (domain-of eqn)
		(simplify (lhs-of eqn))
		(simplify (rhs-of eqn))))

(defmethod-sd ge-deriv ((eqn ge-eqn=) (var variable))
  (make-ge-eqn= domain
		(ge-deriv (lhs-of eqn) var)
		(ge-deriv (rhs-of eqn) var)))

(defmethod make-ge-eqn> ((domain general-expressions) lhs rhs)
  (make-instance 'ge-eqn= :domain domain :rhs rhs :lhs lhs))

(defmethod print-object ((eqn ge-eqn>) stream)
  (print-object (lhs-of eqn) stream)
  (princ " > " stream)
  (print-object (rhs-of eqn) stream))

(defmethod eqn> (lhs rhs)
  (make-ge-eqn> *general*
		(simplify (coerce lhs *general*))
		(simplify (coerce rhs *general*))))

(defmethod simplify ((eqn ge-eqn>))
  (make-ge-eqn> (domain-of eqn)
		(simplify (lhs-of eqn))
		(simplify (rhs-of eqn))))

(defmethod-sd ge-deriv ((eqn ge-eqn>) (var variable))
  (make-ge-eqn> domain
		(ge-deriv (lhs-of eqn) var)
		(ge-deriv (rhs-of eqn) var)))

(defmethod make-ge-eqn>= ((domain general-expressions) lhs rhs)
  (make-instance 'ge-eqn>= :domain domain :rhs rhs :lhs lhs))

(defmethod print-object ((eqn ge-eqn>=) stream)
  (print-object (lhs-of eqn) stream)
  (princ " >= " stream)
  (print-object (rhs-of eqn) stream))

(defmethod eqn>= (lhs rhs)
  (make-ge-eqn>= *general*
		 (simplify (coerce lhs *general*))
		 (simplify (coerce rhs *general*))))

(defmethod simplify ((eqn ge-eqn>=))
  (make-ge-eqn>= (domain-of eqn)
		 (simplify (lhs-of eqn))
		 (simplify (rhs-of eqn))))

(defmethod-sd ge-deriv ((eqn ge-eqn>=) (var variable))
  (make-ge-eqn>= domain
		 (ge-deriv (lhs-of eqn) var)
		 (ge-deriv (rhs-of eqn) var)))

(defmethod plus ((x (or number symbol)) (y symbol))
  (+ (coerce x *general*) (coerce y *general*)))

(defmethod plus ((x symbol) (y number))
  (+ (coerce x *general*) (coerce y *general*)))

(defmethod plus ((x general-expression) (y (or number symbol)))
  (+ x (coerce y (domain-of x))))

(defmethod plus ((x (or number symbol)) (y general-expression))
  (+ (coerce x (domain-of y)) y))

(defmethod-sd plus ((x (or numeric general-expression)) (y general-expression))
  (simplify (make-ge-plus domain (list x y))))

(defmethod-sd plus ((x general-expression) (y numeric))
  (simplify (make-ge-plus domain (list x y))))

(defmethod-sd plus ((eq1 ge-eqn=) (eq2 ge-eqn=))
  (make-ge-eqn= domain
		(+ (lhs-of eq1) (lhs-of eq2))
		(+ (rhs-of eq1) (rhs-of eq2))))

(defmethod-sd plus ((eq1 ge-eqn=) (exp (or numeric general-expression)))
  (make-ge-eqn= domain (+ (lhs-of eq1) exp) (+ (rhs-of eq1) exp)))

(defmethod-sd plus ((exp (or numeric general-expression)) (eq1 ge-eqn=))
  (make-ge-eqn= domain (+ (lhs-of eq1) exp) (+ (rhs-of eq1) exp)))


(defmethod difference ((x (or number symbol)) (y symbol))
  (- (coerce x *general*) (coerce y *general*)))

(defmethod difference ((x symbol) (y number))
  (- (coerce x *general*) (coerce y *general*)))

(defmethod difference ((x general-expression) (y (or number symbol)))
  (- x (coerce y (domain-of x))))

(defmethod difference ((x (or number symbol)) (y general-expression))
  (- (coerce x (domain-of y)) y))

(defmethod-sd difference
    ((x (or numeric general-expression)) (y general-expression))
  (simplify (make-ge-plus domain
	      (list x (make-ge-times domain
				     (list (make-element domain -1) y))))))

(defmethod-sd difference ((x general-expression) (y numeric))
  (simplify (make-ge-plus domain
	      (list x (make-ge-times domain
				     (list (make-element domain -1) y))))))

(defmethod-sd difference ((eq1 ge-eqn=) (eq2 ge-eqn=))
  (make-ge-eqn= domain
		(- (lhs-of eq1) (lhs-of eq2))
		(- (rhs-of eq1) (rhs-of eq2))))

(defmethod-sd difference ((eq1 ge-eqn=) (exp (or numeric general-expression)))
  (make-ge-eqn= domain (- (lhs-of eq1) exp) (- (rhs-of eq1) exp)))

(defmethod-sd difference ((exp (or numeric general-expression)) (eq1 ge-eqn=))
  (make-ge-eqn= domain (- exp (lhs-of eq1)) (- exp (rhs-of eq1))))


(defmethod minus ((x symbol))
  (- (coerce x *general*)))

(defmethod minus ((x general-expression))
  (let ((domain (domain-of x)))
    (simplify
     (make-ge-times domain (list (make-element domain -1) x)))))

(defmethod minus ((eq1 ge-eqn=))
  (make-ge-eqn= (domain-of eq1) (- (lhs-of eq1)) (- (rhs-of eq1))))
							  

(defmethod times ((x (or number symbol)) (y symbol))
  (* (coerce x *general*) (coerce y *general*)))

(defmethod times ((x symbol) (y number))
  (* (coerce x *general*) (coerce y *general*)))

(defmethod times ((x general-expression) (y (or number symbol)))
  (* x (coerce y (domain-of x))))

(defmethod times ((x (or number symbol)) (y general-expression))
  (* (coerce x (domain-of y)) y))

(defmethod-sd times
    ((x (or numeric general-expression)) (y general-expression))
  (simplify (make-ge-times domain (list x y))))

(defmethod-sd times ((x general-expression) (y numeric))
  (simplify (make-ge-times domain (list x y))))

(defmethod-sd times ((eq1 ge-eqn=) (eq2 ge-eqn=))
  (error "Can't multiply two equations"))

(defmethod-sd times ((eq1 ge-eqn=) (exp (or numeric general-expression)))
  (make-ge-eqn= domain (* (lhs-of eq1) exp) (* (rhs-of eq1) exp)))

(defmethod-sd times ((exp (or numeric general-expression)) (eq1 ge-eqn=))
  (make-ge-eqn= domain (* (lhs-of eq1) exp) (* (rhs-of eq1) exp)))


(defmethod quotient ((x (or number symbol)) (y symbol))
  (/ (coerce x *general*) (coerce y *general*)))

(defmethod quotient ((x symbol) (y number))
  (/ (coerce x *general*) (coerce y *general*)))

(defmethod quotient ((x general-expression) (y (or number symbol)))
  (/ x (coerce y (domain-of x))))

(defmethod quotient ((x (or number symbol)) (y general-expression))
  (/ (coerce x (domain-of y)) y))

(defmethod-sd quotient
    ((x (or numeric general-expression)) (y general-expression))
  (simplify (make-ge-times domain
	      (list x (make-ge-expt domain y (make-element domain -1))))))

(defmethod-sd quotient ((x general-expression) (y numeric))
  (simplify (make-ge-times domain
	      (list x (make-ge-expt domain y (make-element domain -1))))))

(defmethod-sd quotient ((eq1 ge-eqn=) (eq2 ge-eqn=))
  (error "Can't divide two equations"))

(defmethod-sd quotient ((eq1 ge-eqn=) (exp (or numeric general-expression)))
  (make-ge-eqn= domain (/ (lhs-of eq1) exp) (/ (rhs-of eq1) exp)))

(defmethod-sd quotient ((exp general-expression) (eq1 ge-eqn=))
  (error "Can't divide by an equation"))


(defmethod recip ((x symbol))
  (recip (coerce x *general*)))

(defmethod recip ((x general-expression))
  (let ((domain (domain-of x)))
    (simplify (make-ge-expt domain x (make-element domain -1)))))

(defmethod recip ((eq1 ge-eqn=))
  (make-ge-eqn= (domain-of eq1) (/ (lhs-of eq1)) (/ (rhs-of eq1))))


(defmethod expt ((x (or number symbol)) (y symbol))
  (expt (coerce x *general*) (coerce y *general*)))

(defmethod expt ((x symbol) (y number))
  (expt (coerce x *general*) (coerce y *general*)))

(defmethod expt ((x general-expression) (y (or number symbol)))
  (expt x (coerce y (domain-of x))))

(defmethod-sd expt ((x (or numeric general-expression)) (y general-expression))
  (simplify (make-ge-expt domain x y)))

(defmethod-sd expt ((x general-expression) (y numeric))
  (simplify (make-ge-expt domain x y)))

(defmethod-sd expt ((eq1 ge-eqn=) (eq2 ge-eqn=))
  (error "Can't exponentiate two equations"))

(defmethod-sd expt ((eq1 ge-eqn=) (exp (or numeric general-expression)))
  (make-ge-eqn= domain (expt (lhs-of eq1) exp) (expt (rhs-of eq1) exp)))

(defmethod-sd expt ((exp (or numeric general-expression)) (eq1 ge-eqn=))
  (error "Can't put an equation in an exponent"))

;; The following is kludge until we put in more logical connectives.

;; The expression must be general-expression
(defclass universal-quantified-set (set)
     ((exprs :initarg :expressions
	    :accessor exprs-of)
      (var :initarg :variable
	   :accessor var-of)
      (var-domain :initarg :variable-domain
		  :accessor var-domain-of)))

(defmethod print-object ((set universal-quantified-set) stream)
  (format stream "{ForAll ~S in ~S . ~S"
	  (var-of set) (var-domain-of set) (first (exprs-of set)))
  (dolist (x (rest (exprs-of set)))
    (format stream ", ~S" x))
  (format stream "}"))

(defmethod make-union ((var symbol) (set set) (expr general-expression)
		       &rest rest-exprs)
  (apply #'make-union (coerce var (domain-of expr)) set expr rest-exprs))

;; This method is here to support (forall x  (forall y .. ))
(defmethod make-union ((var symbol) (set set) (expr set)
		       &rest rest-exprs)
  (apply #'make-union (coerce var *general*) set expr rest-exprs))

(defmethod make-union ((var variable) (set set)
		       (expr (or set general-expression))
		       &rest rest-exprs)
  ;;; Make sure that the union is sensible.
  (dolist (x rest-exprs)
    (if (not (eq (domain-of expr) (domain-of x)))
	(error "Can't union incompatible domains.")))
  (make-instance 'universal-quantified-set
		 :variable var
		 :variable-domain set
		 :expressions (cons expr rest-exprs)))

(defmethod simplify ((set universal-quantified-set))
  (apply #'make-union (var-of set) (var-domain-of set)
	 (mapcar #'simplify (exprs-of set))))

;; Variable dependencies and DEPENDS-ON? 

(defmethod declare-dependencies ((var variable) &rest vars)
  (let ((depends (get-variable-property (domain-of var) var :dependencies))
	(domain (domain-of var)))
    (loop for v in vars
	  do (setq v (coerce v domain))
	     (unless (member v depends :test #'ge-equal)
	       (push v depends)))
    (setf (get-variable-property (domain-of var) var :dependencies) depends)))

(defmethod depends-on? ((exp numeric) &rest vars)
  (declare (ignore vars))
  nil)

(defmethod depends-on? ((exp variable) &rest vars)
  (or (member exp vars :test #'ge-equal)
      (loop for var in (get-variable-property (domain-of exp)
			   exp :dependencies)
	    do (when (member var vars :test #'ge-equal)
		 (return t))
	    finally (return nil))))

(defmethod depends-on? ((exp ge-plus) &rest vars)
  (depends-on-terms? (terms-of exp) vars))

(defmethod depends-on? ((exp ge-times) &rest vars)
  (depends-on-terms? (terms-of exp) vars))

(defmethod depends-on? ((exp ge-expt) &rest vars)
  (or (apply #'depends-on? (base-of exp) vars)
      (apply #'depends-on? (exponent-of exp) vars)))

(defun depends-on-terms? (terms vars)  
  (loop for term in terms
	do (when (apply #'depends-on? term vars)
	     (return t))
	finally (return nil)))

(defmethod depends-on? ((exp ge-deriv) &rest vars)
  (apply #'depends-on? (argument-of exp) vars))

(defmethod depends-on? ((exp ge-function) &rest vars)
  (depends-on-terms? (args-of exp) vars))

;; Different kernels

(defmethod different-kernels ((exp numeric) (kernels list))
  kernels)

;; If we don't know anything about the object, then its a kernel.
(defmethod different-kernels (exp (kernels list))
  (if  (member exp kernels :test #'ge-equal)
       kernels
       (cons exp kernels)))

(defmethod different-kernels ((exp ge-plus) (kernels list))
  (loop for term in (terms-of exp)
	do (setq kernels (different-kernels term kernels)))
  kernels)

(defmethod different-kernels ((exp ge-times) (kernels list))
  (loop for term in (terms-of exp)
	do (setq kernels (different-kernels term kernels)))
  kernels)

(defmethod different-kernels :around ((exp ge-expt) (kernels list))
  (if (lisp:integerp (exponent-of exp))
      (different-kernels (base-of exp) kernels)
      (call-next-method exp kernels)))


(defmethod substitute
    ((value general-expression) (var variable) (expr (or number numeric))
     &rest ignore)
  (declare (ignore ignore))
  expr)

(defmethod substitute
    ((value general-expression) (var variable) (expr variable) &rest ignore)
  (declare (ignore ignore))
  (if (eql var expr) value expr))

(defmethod substitute
    ((value general-expression) (var variable) (expr ge-plus) &rest ignore)
  (declare (ignore ignore))
  (simplify
   (make-ge-plus (domain-of expr)
		 (mapcar (lambda (q) (substitute value var q))
			 (terms-of expr)))))

(defmethod substitute
    ((value general-expression) (var variable) (expr ge-times) &rest ignore)
  (declare (ignore ignore))
  (simplify
   (make-ge-times (domain-of expr)
		 (mapcar (lambda (q) (substitute value var q))
			 (terms-of expr)))))

(defmethod substitute
    ((value general-expression) (var variable) (expr ge-expt) &rest ignore)
  (declare (ignore ignore))
  (simplify
   (make-ge-expt (domain-of expr)
		 (substitute value var (base-of expr))
		 (substitute value var (exponent-of expr)))))

(defmethod substitute
    ((value general-expression) (var variable) (expr ge-function) &rest ignore)
  (declare (ignore ignore))
  (simplify
   (make-instance 'ge-function :domain (domain-of expr)
		  :funct (funct-of expr)
		  :args (mapcar (lambda (q) (substitute value var q))
				(args-of expr)))))

(defmethod substitute
    ((value general-expression) (var variable) (expr ge-deriv) &rest ignore)
  (declare (ignore ignore))
  (simplify
   (make-ge-deriv (domain-of expr)
		  (substitute value var (argument-of expr))
		  (varlist-of expr))))

