;;; -*- Mode:Lisp; Package:Weyli; Base:10; Lowercase:T; Syntax:Common-Lisp -*-
;;; ===========================================================================
;;;			      Fourier Transforms
;;; ===========================================================================
;;; (c) Copyright 1989, 1992 Cornell University

;;; $Id: fourier.lisp,v 1.1 1992/06/05 17:59:54 rz Exp $

(in-package "WEYLI")

(defclass ge-fourier (general-expression)
     ((argument :initarg :argument
		:accessor argument-of)
      (space-var :initarg :space-var
		 :accessor space-var-of)
      (freq-var :initarg :freq-var
		:accessor freq-var-of)))

(defsubst ge-fourier? (x) (typep x 'ge-fourier))

(defmethod make-ge-fourier ((domain general-expressions) argument svar fvar)
  (make-instance 'ge-fourier :domain domain :argument argument
		 :space-var svar :freq-var fvar))

(defmethod print-object ((expr ge-fourier) stream)
  (format stream "Four{~S, ~S->~S}"
	  (argument-of expr) (space-var-of expr) (freq-var-of expr)))

(defmethod ge-equal ((x ge-fourier) (y ge-fourier))
  (and (ge-equal (argument-of x) (argument-of y))
       (ge-equal (space-var-of x) (space-var-of y))
       (ge-equal (freq-var-of x) (freq-var-of y))))

(defmethod ge-great ((x ge-fourier) (y ge-fourier))
  (cond ((ge-great (argument-of x) (argument-of y)) t)
	((ge-great (argument-of y) (argument-of x)) nil)
	((ge-great (space-var-of x) (space-var-of y)) t)	
	((ge-great (space-var-of y) (space-var-of x)) nil)
	((ge-great (freq-var-of x) (freq-var-of y)) t)))

(defmethod fourier ((exp number) &rest vars)
  (declare (ignore vars))
  (make-element *general* exp))

(defmethod fourier ((exp numeric) &rest vars)
  (declare (ignore vars))
  exp)

(defmethod fourier ((exp (or symbol general-expression)) &rest vars)
  (setq exp (coerce exp *general*))
  (loop for (sv fv) on vars by #'cddr
	do (setq exp (ge-fourier exp (coerce sv *general*)
				     (coerce fv *general*))))
  exp)

(defmethod ge-fourier (exp svar fvar)
  (declare (ignore fvar))
  (error "Don't know how to take the Fourier transform of ~S wrt ~S"
	 exp svar))

(defmethod ge-fourier ((exp general-expression) (svar symbol) (fvar symbol))  
  (ge-fourier exp (coerce svar (domain-of exp)) (coerce fvar (domain-of exp))))

(defmethod ge-fourier (exp (svar variable) (fvar variable))
  (make-ge-fourier (domain-of svar) (coerce exp (domain-of svar)) svar fvar))

(defmethod ge-fourier ((exp numeric) (svar variable) (fvar variable))
  exp)

(defmethod ge-fourier :around ((exp variable) (svar variable) (fvar variable))
  (let ((domain (domain-of exp)))
    (unless (and (eql domain (domain-of svar))
		 (eql domain (domain-of fvar)))
      (error "Taking Fourier transform from different domains"))
    (cond ((ge-equal exp svar) fvar)
	  ((depends-on? exp svar)
	   (make-ge-fourier domain exp svar fvar))
	  (t exp))))

(defmethod ge-fourier :around ((exp ge-plus) (svar variable) (fvar variable))
  (let ((domain (domain-of exp)))
    (cond ((and (eql domain (domain-of svar))
		(eql domain (domain-of fvar)))
	   (call-next-method exp svar fvar))
	  (t (simplify
	      (make-ge-plus domain
		(loop for x in (terms-of exp)
		      collect (ge-fourier x svar fvar))))))))

(defmethod ge-fourier ((exp ge-times) (svar variable) (fvar variable))
  (let ((domain (domain-of exp))
	terms depend-term free-terms)
    (unless (and (eql domain (domain-of svar))
		 (eql domain (domain-of fvar)))
      (error "Taking Fourier transform from different domains"))
  (setq terms (terms-of exp))
  (loop for term in terms
	do (when (depends-on? term svar)
	     (cond ((null depend-term)
		    (setq depend-term term))
		   (t (return (setq free-terms :non-linear)))))
		 finally (setq free-terms
			       (remove depend-term terms)))
  (cond ((eql free-terms :non-linear)
	 (make-ge-fourier domain exp svar fvar))
	((null depend-term)
	 exp)
	(t (simplify
	    (make-ge-times domain
	      (cons (ge-fourier depend-term svar fvar)
		    free-terms)))))))

(defmethod ge-fourier ((exp ge-deriv) (svar variable) (fvar variable))
  (let ((domain (domain-of exp)))
    (unless (and (eql domain (domain-of svar))
		 (eql domain (domain-of fvar)))
      (error "Taking Fourier transform from different domains"))
  (loop for entry in (varlist-of exp)
	with varlist
	do (when (ge-equal svar (first entry))
	     (setq varlist (remove entry (varlist-of exp)))
	     (return
	       (simplify
		(* (expt fvar (second entry))
		   (if (null varlist)
		       (ge-fourier (argument-of exp) svar fvar)
		       (make-ge-deriv domain
				      (ge-fourier (argument-of exp) svar fvar)
				      varlist))))))
	   finally
	(return 
	  (simplify
	   (make-ge-deriv domain
			  (ge-fourier exp svar fvar)
			  (varlist-of exp)))))))

;; Inverse Fourier Transforms

(defclass ge-ifourier (general-expression)
     ((argument :initarg :argument
		:accessor argument-of)
      (space-var :initarg :space-var
		 :accessor space-var-of)
      (freq-var :initarg :freq-var
		:accessor freq-var-of)))

(defsubst ge-ifourier? (x) (typep x 'ge-ifourier))

(defmethod make-ge-ifourier ((domain general-expressions) argument svar fvar)
  (make-instance 'ge-ifourier :domain domain :argument argument
		 :space-var svar :freq-var fvar))

(defmethod print-object ((expr ge-ifourier) stream)
  (format stream "IFour{~S, ~S->~S}"
	  (argument-of expr) (space-var-of expr) (freq-var-of expr)))
