#############################################################################
##
#A  cthecke.g           CHEVIE library          Meinolf Geck & G"otz Pfeiffer
##
#A  $Id: cthecke.g,v 1.4 1993/10/28 09:53:32 goetz Exp $
##
#Y  Copyright 1992-1993,  Lehrstuhl D fuer Mathematik,    RWTH Aachen,   and 
#Y                        IWR   der   Universit"at    Heidelberg,    Germany.
##
##  This file contains  GAP functions  for  working with character tables of
##  Hecke algebras. Such  a record is  similar  to  the records for ordinary
##  character tables of finite groups,  except  that  it  has  the following
##  additional components:
##
##  'cartan'    : the Cartan matrix of the underlying Weyl group
##  'parameter' : the parameters of the Hecke algebra
##  'classtext' : representatives of minimal length of the conjugacy classes
##                given  as  reduced  expressions in the standard generators
##
##  It may also have additional components like  'irreddinfo',  'classtext',
##  'classnames' containing information about  labels  for  the  irreducible
##  characters and the conjugacy classes.
##
#H  $Log: cthecke.g,v $
#H  Revision 1.4  1993/10/28  09:53:32  goetz
#H  added function 'SchurElements' and other slight changes.
#H
#H  Revision 1.3  1993/10/20  20:40:28  goetz
#H  another improvement of 'HeckeReducedChar'.
#H
#H  Revision 1.2  1993/10/05  13:28:10  goetz
#H  Correction of 'HeckeReducedChar'.
#H
#H  Revision 1.1  1993/10/05  13:18:12  goetz
#H  Initial revision
#H
##

Print("\n\n                       WELCOME to the GAP part of CHEVIE\n\n",
      "                              Version 2 Release 0\n\n",
      "    Meinolf Geck, Gerhard Hiss, Frank Luebeck, ",
      "Gunter Malle, Goetz Pfeiffer\n\n",
      "  Lehrstuhl D fuer Mathematik, RWTH Aachen and IWR der ",
      "Universitaet Heidelberg\n\n\n");

RequirePackage("weyl");

############################################################################
##
#F  HeckeCharTable( <type>, <rank>, <parameter> ) . . . . .  character table
##  of  an Iwahori Hecke algebra  with  given  type,  rank,  and  parameters
##
##  Note that, for type $I_2(m)$,  $G_2$,  $H_3$,  $H_4$,  $E_7$,  $E_8$ the 
##  parameters of  the  Hecke  algebra are  the  squares  of the elements in 
##  <parameter>.
##
##  Note that for type $I_2(m)$, <type> is "I2" and <rank> is $m$.
##
HeckeCharTable:=function(type,rank,parameter)
  if not IsString(type) or not IsInt(rank) then
    return false;
  elif rank<=0 then
    Print("#I  rank too small \n");
    return false;
  elif rank=1 then
    return HeckeCharTableA(rank,parameter);
  elif type="A" then
    return HeckeCharTableA(rank,parameter);
  elif type="B" then
    return HeckeCharTableB(rank,parameter);
  elif type="D" then
    if rank=2 then
      Print("#I  this is type A1 x A1 \n");
      return false;
    elif rank=3 then
      Print("#I  this is type A3 with labelling: 1 - 3 - 2 \n");
      return false;
    else
      return HeckeCharTableD(rank,parameter);
    fi;
  elif type="G" and rank=2 then
    return HeckeCharTableG2(parameter); 
  elif type="F" and rank=4 then
    return HeckeCharTableF4(parameter); 
  elif type="E" and rank=6 then
    return HeckeCharTableE6(parameter); 
  elif type="E" and rank=7 then
    return HeckeCharTableE7(parameter); 
  elif type="E" and rank=8 then
    return HeckeCharTableE8(parameter); 
  elif type="I2" then
    return HeckeCharTableDihedral(rank,parameter); 
  elif type="H" and rank=3 then
    return HeckeCharTableH3(parameter); 
  elif type="H" and rank=4 then
    return HeckeCharTableH4(parameter); 
  else
    Print("#I  funny type or rank \n");
    return false;
  fi;
end;

############################################################################
##
#F  SchurElements( <type>, <rank>, <parameter> ) . . . . . .  Schur elements
##
##  'SchurElements' returns the list of elements $P/D_{\phi}$ where  $P$  is 
##  the Poincare polynomial of the Hecke algebra $H$  of  type  <type>, rank
##  <rank> and parameters determined by <parameters>,  and  $\phi$ runs over 
##  the irreducible characters of $H$.
##
##  Note that, for type $I_2(m)$,  $G_2$,  $H_3$,  $H_4$,  $E_7$,  $E_8$ the 
##  parameters of  the  Hecke  algebra are  the  squares  of the elements in 
##  <parameter>.
##
##  Note that for type $I_2(m)$, <type> is "I2" and <rank> is $m$.
##
SchurElements:=function(type,rank,parameter)
  if not IsString(type) or not IsInt(rank) then
    return false;
  elif rank<=0 then
    Print("#I  rank too small \n");
    return false;
  elif type="A" or (rank=1 and type in ["B","D"]) then
    Print("#I  not yet implemented \n");
    return false;
  elif type="B" then
    Print("#I  not yet implemented \n");
    return false;
  elif type="D" then
    if rank=2 then
      Print("#I  this is type A1 x A1 \n");
      return false;
    elif rank=3 then
      Print("#I  this is type A3 with labelling: 1 - 3 - 2 \n");
      return false;
    else
      Print("#I  not yet implemented \n");
      return false;
    fi;
  elif type="G" and rank=2 then
    return SchurElementsG2(parameter); 
  elif type="F" and rank=4 then
    return SchurElementsF4(parameter); 
  elif type="E" and rank=6 then
    return SchurElementsE6(parameter); 
  elif type="E" and rank=7 then
    return SchurElementsE7(parameter); 
  elif type="E" and rank=8 then
    return SchurElementsE8(parameter); 
  elif type="I2" then
    Print("#I  not yet implemented \n");
    return false;
  elif type="H" and rank=3 then
    return SchurElementsH3(parameter); 
  elif type="H" and rank=4 then
    return SchurElementsH4(parameter); 
  else
    Print("#I  funny type or rank \n");
    return false;
  fi;
end;

############################################################################
##
#F  HeckeFusion( <tr>, <ti>, <J> ) . . . . . . . . . . .  compute fusion map
##
##  'HeckeFusion' returns a  record  describing the fusion of classes of the
##  character table record  <tr> into  the  character table record <ti>. The
##  argument  <J>  is a  list such that <J>[s]  is the standard generator of
##  <ti>  to  which the standard generator  s  of  <tr>  is mapped under the
##  chosen embedding. The returned record has components 'name', 'map', 'J'.
##
HeckeFusion:=function(tr,ti,J)
  local G, W, fus, cc, rep;
  W:= Weyl(ti.cartan);
  G:= Group(W.permgens, ());
  cc:= [];
  for rep in ti.classtext do
    Add(cc, ConjugacyClass(G, PermWeylWord(W, rep)));
  od;
  fus:= [];
  for rep in tr.classtext do
    Add(fus, Position(cc, ConjugacyClass(G, PermWeylWord(W, J{rep}))));
  od;
  return rec(name:= ti.name, map:= fus, J:= J);
end;

############################################################################
##
#F  HeckePowermap( <ct> ) . . . . . . . . . . . . . . . .  compute powermaps
##
##  'HeckePowermap'  returns  the list of powermaps of  the  Hecke character
##  table record <ct>.
##
HeckePowermap:=function(W,r)
  local G,f,pg,pp,p,i,j,pow,l;
  l:=[];
  pg:=List(r,i->PermWeylWord(W,i));
  G:=Group(W.permgens,());
  for p in Set(Factors(Size(G))) do
    pp:=List(pg,i->i^p);
    pow:=[];
    for i in pp do
      j:=1;
      while not IsConjugate(G,i,pg[j]) do 
        j:=j+1;
      od; 
      Add(pow,j);
    od;
    l[p]:=pow;
  od;
  return l;
end;

############################################################################
##
#F  HeckeCharTableDirectProduct( <t1>, <t2> )  . . . . . . . . . . . . . . .
#F  . . . . . . . . . . . . compute direct procuct of Hecke character tables
##
##  'HeckeCharTableDirectProduct'  returns  the record corresponding to  the 
##  tensor products of the Hecke character tables <t1>, <t2>.
##
HeckeCharTableDirectProduct:=function(t1,t2)
  local d,r,w1,w2,s,x,t;
  d:=CharTableDirectProduct(t1,t2);
  if IsBound(t1.name) and IsBound(t2.name) then
    d.name:=ConcatenationString(t1.name," x ",t2.name);
  fi;
  d.cartan:=DirectSumCartanMat(t1.cartan,t2.cartan);
  d.parameter:=Concatenation(t1.parameter,t2.parameter);
  d.classtext:=[];
  d.irredinfo:=[];
  for w1 in [1..Length(t1.classtext)] do
    for w2 in [1..Length(t2.classtext)] do
      x:=Copy(t1.classtext[w1]);
      for s in t2.classtext[w2] do
        Add(x,s+Length(t1.cartan));
      od;
      Add(d.classtext,x);
      if IsBound(t1.irredinfo) and IsBound(t2.irredinfo) then
        t:=rec();
        if IsBound(t1.irredinfo[w1].charparam) and 
                    IsBound(t2.irredinfo[w2].charparam) then
          t.charparam:=[t1.irredinfo[w1].charparam,t2.irredinfo[w2].charparam];
        elif IsBound(t1.irredinfo[w1].charname) and 
                    IsBound(t2.irredinfo[w2].charname) then
          t.charname:=[t1.irredinfo[w1].charname,t2.irredinfo[w2].charname];
        fi;
        Add(d.irredinfo,t);
      fi;
    od;
  od;
  Unbind(d.fusionsource);
  Unbind(d.fusions);
  return d;
end;

############################################################################
##
#F  HeckeInductionPolynomials( <tr>, <ti> )   . . . . . . . . . . . . . . .
#F  . . . . . . . . . . .  induction matrix for characters of Hecke algebras
##
##  'HeckeInductionPolynomials' returns a matrix which, multiplied  from the
##  right with a list of characters of the table <tr>,  gives  the  list  of
##  induced characters for the table <ti>. The table  <tr>  must  contain  a
##  component in 'fusions' as returned, e.g., by the function 'HeckeFusion'.
##
HeckeInductionPolynomials:=function(arg)
  local wjdj,W,U,D,J,e,i,d,w,ux,x,y,mat,p;

  W:=Weyl(arg[2].cartan);
  Hecke(W,arg[2].parameter);
  U:=Weyl(arg[1].cartan);
  Hecke(U,arg[1].parameter);
  for i in arg[1].fusions do
    if i.name=arg[2].name then
      J:=i.J;
    fi;
  od;
  if not IsBound(J) then
    Error("no fusion");
  fi;
  D:=WeylRightCosetRepresentatives(W,[1..W.dim],J);

  wjdj:=function(w)
    local ww,r,c;
    ww:=PermWeylWord(W,w);
    if ww=( ) then
      return [[ ],[ ]];
    else
      if w in D then
        return [[ ],w];
      fi;
      for r in J do
        if r^ww>W.N then
          c:=wjdj(WeylWordPerm(W,W.permgens[r]*ww)); 
          return [ReducedWeylWord(W,Concatenation([r],c[1])),c[2]];
        fi;
      od;
    fi;
  end;

  mat:=[];
  Print("#I \c");
  for w in arg[2].classtext do
    Print(Position(arg[2].classtext,w)," \c");
    p:=List(arg[1].classtext,i->0*arg[2].parameter[1]);
    for d in D do
      y:=W.T(d)*W.T(w);
      for i in y.basrep do 
        x:=wjdj(i[2]);
        if x[2]=d then
          ux:=List(x[1],j->Position(J,j));
          if IsBound(arg[3]) then
            e:=1;
            while arg[3][e][1]<>ux do
              e:=e+1;
            od;
            p:=p+i[1]*arg[3][e][2];
          else
            p:=p+i[1]*WeylClassPolynomials(U,arg[1].classtext,ux);
          fi;
        fi;
      od;
    od;
    Add(mat,p);
  od;
  Print("\n");
  return TransposedMat(mat);
end;

############################################################################
##
#F  HeckeScalarProducts( <ti>, <char1>, <char2> )  . . . . . . . . . . . . .
#F   . . . . . . . . . . . . . . . . . .  scalar products between characters
##
##  'HeckeScalarProducts' specializes the parameters to  1  and  returns the
##  matrix of ordinary scalar products between the  specialized  characters.
##
HeckeScalarProducts:=function(ti,char1,char2)
  local val,i,j,i1,l,mat;
  if ForAny(ti.parameter,i->i<>1 and IsPolynomial(i)=false) then
    Print("W parameter must be indeterminates \n");
    return false;
  fi;
  val:=function(p)
    if not IsPolynomial(p) then
      return p;
    else
      return Value(p,1);
    fi;
  end;
  mat:=[];
  for i in char1 do
    i1:=List(i,val);
    l:=[];
    for j in char2 do
      Add(l,ScalarProduct(ti,i1,List(j,val)));
    od;
    Add(mat,l);
  od;
  return mat;
end;


############################################################################
##
#F  HeckeReducedChar( <ti>, <irrs>, <reds> )   . . . . . . . . . . . . . . .
#F   . . . . . reduce reducible characters with a given list of irreducibles
##
##  'HeckeReducedChar'    is  the  equivalent  of  the  usual  GAP  function
##  'ReducedOrdinary'  for  characters of Hecke algebras.  So,  <ti>  is the
##  character table  of  a  Hecke  algebra,  <irrs> is a list of irreducible
##  characters and <reds> a list of reducible characters.
##
HeckeReducedChar:=function(ti,irrs,reds)
  local val,red,irr,j,i1,r1,s;
  val:=function(p)
    if not IsPolynomial(p) then
      return p;
    else
      return Value(p,1);
    fi;
  end;
  red:=Copy(reds);
  for j in red do
    i1:=List(j,val);
    if ScalarProduct(ti,i1,i1)=1 then
      if not j in irrs then
        Add(irrs,j);
      fi;
      j:=0;
    fi;
  od;
  for irr in irrs do
    i1:=List(irr,val);
    for j in [1..Length(red)] do
      if red[j]<>0*red[j] then
        r1:=List(red[j],val);
        s:=ScalarProduct(ti,i1,r1);
        if s<>0 then
          red[j]:=red[j]-s*irr;
          if ScalarProduct(ti,r1,r1)-s=1 then
            if not red[j] in irrs then
              Print("#I new irreducible of degree ",val(red[j][1]),"\n");
              Add(irrs,red[j]);
            fi;
            red[j]:=0*red[j];
          fi;
        fi;
      fi;
    od; 
  od;
  return rec(irreducibles:=irrs,
             reducibles:=Set(Filtered(red,i->i<>0*i)));
end;      
