// ************************************************************************

#include <RpSGIWriter.h>
#include <assert.h>
#include <string.h> // memset()

RpSGIWriter::RpSGIWriter( const char *cpOutFilename, RpSampledImage *pInput )
: _source( pInput, pInput->getType() )
{
   if ( cpOutFilename && pInput )
      _iOK = writeSGIFile( cpOutFilename );
   else
      _iOK = 0;
   return;
}



RpSGIWriter::RpSGIWriter( const char *cpOutFilename, RpSampledImage *pInput, RpType setType )
: _source( pInput, setType )
{
   if ( cpOutFilename && pInput )
      _iOK = writeSGIFile( cpOutFilename );
   else
      _iOK = 0;
   return;
}


RpSGIWriter::RpSGIWriter( const char *cpOutFilename, RpInputImage *pInput, const RpImageArea& setArea, RpType setType )
: _source( pInput, setArea, setType )
{ 
   _iOK = writeSGIFile( cpOutFilename );
}


int
RpSGIWriter::writeSGIFile( const char *cpFile )
{
   int		iOK = -1;
   RpType	fileType;
   ofstream	out( cpFile );
   
   if ( cpFile && ( !out.fail() ) ) {
      
      //
      // set data type:
      //
      fileType = RpUShort;
      if ( _source.getType() == RpUChar )
         fileType = RpUChar;
         
      //
      // open & fill header:
      //
      iOK &= validateFile( out, _source.getArea(), fileType );
      
      //
      // fill rows:
      //
      if ( iOK )
         iOK &= writeRows( out, fileType );
 
   } else
      iOK = 0;
   
   return ( iOK );
}


int
RpSGIWriter::validateFile( ofstream& out, const RpImageArea& area, const RpType fileType )
{
   int 			iOK = -1;
   unsigned char	cp[512];
   
   assert(( fileType == RpUChar )||( fileType == RpUShort));

   //
   // init to zeros
   //    RLE == off ( 0 )
   //
   memset( cp, 0, 512 );
   
   //  magic cookie:
   castFromShort( cp, 474 );
   
   // bytes per pix
   if ( fileType == RpUChar )
      cp[3] = 1;
   else
      cp[3] = 2;
      
   // x, y + c number of channels:
   castFromShort( cp + 4, 3 );
   
   // x, y, c
   castFromShort( cp + 6, area.width );
   castFromShort( cp + 8, area.height );
   castFromShort( cp + 10, area.c );
   
   // min & max (unused)
   castFromLong( cp + 12, 0 );
   if ( fileType == RpUChar )
      castFromLong( cp + 16, 0x00FF );
   else
      castFromLong( cp + 16, 0x0FFFF );
   
   //
   // commit:
   //
   out.write( cp, 512 );

   if ( out.fail() )
      iOK = 0;
   
   return ( iOK );
}


int
RpSGIWriter::writeRows( ofstream& outfile, const RpType setType )
{
   int			iOK = -1;
   int			iChannels;
   signed		iLoop;
   RpImageArea		reqArea; 
   RpImageTile 		*pTile; 
   
   assert( (setType == RpUShort )|| (setType == RpUChar) );
   
   reqArea = _source.getArea();
   reqArea.height = 1;
   reqArea.z = 0;
   for( iChannels=0; iChannels<_source.getArea().c; iChannels++ )
   {
      reqArea.c = iChannels;
      
      //
      // compiler requires explicit typecast to
      // (signed) in the following for-compare:
      //
      for( iLoop=_source.getArea().y; iLoop<(signed)(_source.getArea().y + _source.getArea().height); iLoop++ )
      {
         reqArea.y = iLoop;
            
	 pTile = _source.newTile( reqArea );
	 if ( pTile )
	 {
	    pTile->typecast( setType );
	    _commit( outfile, pTile->getBuffer(), reqArea.width, setType );
            pTile->deleteTile(); 
         }
	 else
	 {
            iOK = 0;
            break;
	 }
      }
   }
   return( iOK );
}


void
RpSGIWriter::_commit( ofstream& outfile, void *vpBuffer, unsigned uPix, RpType setType )
{
   unsigned char	*cpBuffer;
   unsigned short	*spBuffer;   
   unsigned		uLoop;
   
   cpBuffer = (unsigned char *) vpBuffer;
   spBuffer = (unsigned short *) vpBuffer;
   
   if ( setType == RpUChar )
      outfile.write( cpBuffer, uPix );
   else
   {
      assert( setType == RpUShort );
      for( uLoop=0; uLoop<uPix; uLoop++ )
      {
         //
         // endian safe:
         //
         castFromShort( cpBuffer, *spBuffer );
         cpBuffer += sizeof( unsigned short );
         spBuffer++;
      }
      outfile.write( (unsigned char *) vpBuffer, uPix * sizeof(unsigned short));
   }
   return;
}
