#ifndef RpSoftPullMan_CLASS
#define RpSoftPullMan_CLASS

#include <RpReferenced.h>

#include <RpImageMan.h>
#include <RpSampledImage.h>

#define			MIN_TILEBOUND	( 64 )

#define			TE_MEMMGR	( 512 )
#define			CHE_MEMMGR	( 64 )
#define			CAE_MEMMGR	( 16 )


// ************************************************************************
// CLASS : RpSoftPullMan
// DESC:
//    This class implements a non-aggressive pull-through image
//    processing engine.  You shouldn't ever need to mess with this.
// NOTE:
//    Use RpManReference objects to interface with image manager
//    instances!
//
class RpSoftPullMan : public RpImageMan {
   public:
      ~RpSoftPullMan( void );   
         
      RpImageTile *
      newTile( RpSampledImage *pCaller, const RpImageArea& area );
      
      void
      forceCache( RpSampledImage * );

      void
      flushCache( RpSampledImage * )
      { flushCache(); }
   
      void
      flushCache( void );
      
      unsigned long
      getRAMSize( void ) const;
      
      void
      setRAMBound( unsigned long ulBytes );
      
      void
      setTileBound( unsigned long ulPixels )
      {
         if ( ulPixels < MIN_TILEBOUND )
            ulPixels = MIN_TILEBOUND + 1;
            
         RpImageMan::setTileBound( ulPixels );
         return;
      }
      
   protected:

      struct requestKey {
	 public:
	    RpSampledImage 	*p;
	    signed		c;
	    signed		z;

	    requestKey( void )
	    { p=0; c=z=0; }   

	    requestKey( RpSampledImage *p1, const RpImageArea& area )
	    { 
	       z = area.z;
	       c = area.c;
	       p = p1; 
	       if ( p )
	          p->registerReference();
	    }

	    requestKey( const requestKey& src )
	    {
	       z=src.z;
	       c=src.c;
	       p=src.p;
	       if ( p )
	          p->registerReference();
	    }

	    ~requestKey( void )
	    {
	       if ( p )
	          p->unregisterReference();
	       return;
	    }


	    requestKey&
	    operator = ( const requestKey& src )
	    {
	       z=src.z;
	       c=src.c;
	       if ( src.p != p )
	       {
		  if ( p )
	             p->unregisterReference();
                  p = src.p;                  
		  if ( p  )
	             p->registerReference();
	       }
	       return( *this );
	    }
	    
	    int
	    operator > ( const requestKey& op2 ) const
	    { return 
               (
        	  ( p > op2.p ) ||
        	  ( ( p == op2.p ) && ( c > op2.c ) ) ||
        	  ( ( p == op2.p ) && ( c == op2.c ) && ( z > op2.z ) )
               );
	    }
	    

	    int
	    operator < ( const requestKey& op2 ) const
	    { return 
               (
        	  ( p < op2.p ) ||
        	  ( ( p == op2.p ) && ( c < op2.c ) ) ||
        	  ( ( p == op2.p ) && ( c == op2.c ) && ( z < op2.z ) )
               );
	    }

	    int
	    operator == ( const requestKey& op2 ) const
	    {
	       return (
		(p==op2.p) && (c==op2.c) && (z==op2.z)
	       );
	    }

	    int
	    operator != ( const requestKey& op ) const
	    { return (!( (*this) == op )); }
      };

      
      class MapNode {
         public:
            MapNode( const requestKey& setKey );            
            
            virtual
            ~MapNode( void );
            
            virtual void
            deleteInstance( void ){ delete this; }
            
            void
            reset( const requestKey& setKey );
            
            requestKey		_key;         
            MapNode 		*_pLeft;
            MapNode 		*_pRight;
    
      };
      
      
      class Map {
         public:
            Map( void );
            ~Map( void );
            
            void
            insert( MapNode *pInsertMe );
            
            void
            erase( const requestKey& key );
            
            MapNode *
            find( const requestKey& key );
            
            void
            swapContents( Map& src );
            
            static MapNode *
            min( MapNode *pRoot, int(*cmp)(MapNode*,MapNode*) );
            
            MapNode *
            root( void )
            { return( _pRoot ); }
            
            const MapNode *
            root( void ) const
            { return (_pRoot ); }
            
            unsigned
            size( void ) const
            { return ( _uSize ); }
            
         protected:
            MapNode	*_pRoot;
            unsigned	_uSize;
            
            static MapNode *
            splay( const requestKey& key, MapNode *pTree );
      };


      typedef requestKey callerID;

      class cacheElement;
      class candidateElement : public MapNode {
         public:
	    signed	priority;
	    callerID	caller;
	    float	cost;	    
	    
            candidateElement( const requestKey& setKey ) 
               : MapNode( setKey ) { }
            
            ~candidateElement( void ) { }
               
            static candidateElement *
            newInstance( const requestKey& setKey );
                        
            void
            deleteInstance( void );
                        
            static candidateElement *
            newInstance( const cacheElement& src );
            
            static candidateElement *
            newInstance( const candidateElement& src );

         protected:
         
            static unsigned ca_memmgr_count;
            static candidateElement* ca_memmgr_ppStore[ CAE_MEMMGR ];
      };


      class cacheElement : public MapNode {
	 public:
	    signed	priority;
	    callerID	caller;
	    float	cost;
	    RpImageTile *pTile;
	    
	    cacheElement( const requestKey& setKey )
	       : MapNode( setKey ) { }
	    
	    static cacheElement *
	    newInstance( const requestKey& setKey );
	    
	    void
	    deleteInstance( void );
	    
            static cacheElement *
	    newInstance( const cacheElement& src ); // shallow copy
	    
	    static cacheElement *
	    newInstance( const candidateElement& src );
	    
	    ~cacheElement( void )
	    {
	       if ( pTile )
	          pTile->deleteTile();
	    }
	    
	 protected:
	    static unsigned
	    ch_memmgr_count;
	    
	    static cacheElement*
	    ch_memmgr_ppStore[ CHE_MEMMGR ];	    
      };
      
      
      class topologyElement : public MapNode {
         public:
            topologyElement( const requestKey& setKey )
               : MapNode( setKey ) { }
               
            ~topologyElement( void ) { }
            
            static topologyElement *
            newInstance( const requestKey& setKey );
            
            void
            deleteInstance( void );
            
         public:
            
            float		cost;
            
            static unsigned	te_memmgr_count;            
            static topologyElement
            			*te_memmgr_ppStore[ TE_MEMMGR ];
      };
    
      
   protected:
   
      RpSoftPullMan( unsigned long ulRAM, unsigned long ulTileBound )
         : RpImageMan( ulRAM, ulTileBound ) 
         { 
            _iPriorityCounter = 0;
            _uRecurseCounter = 0;
            _pTopology = 0;
         }
         
      friend class RpManReference;
      
   private:
      signed		_iPriorityCounter;
      callerID		_currentCaller;

      // ------------------------------------------------------------------  
      unsigned		_uRecurseCounter;
      
      void
      incrRecurseCounter( RpSampledImage *pCaller, const RpImageArea& area );
      
      void
      decrRecurseCounter( void );      
       // ------------------------------------------------------------------

      Map		*_pTopology;
      
      void
      registerTopology( RpSampledImage *pCaller, const RpImageArea& area );
      
      static int
      worstTopology( MapNode *p1, MapNode *p2 );      
      
      void
      submitCandidate( void );
      
      void
      clearTopology( void );
      // ------------------------------------------------------------------
      
      Map		_candidates;
      
      int
      candidateServiced( RpImageTile *pWriteHere, RpSampledImage *pCaller );
      
      void
      promoteCandidate( candidateElement *pCandidate );
      
      static int
      minCandidateCost( MapNode *p1, MapNode *p2 );
      
      int
      gotEnoughRAM( candidateElement *pCandidateElement, int iDoFree );
      
      int
      priorityLessThan( const RpSoftPullMan::cacheElement *c1, const RpSoftPullMan::candidateElement *c2 ) const;
      
      // ------------------------------------------------------------------
      
      cacheElement *
      minCacheElement( cacheElement * ) const;
      
      int
      cacheServiced( RpImageTile *pWriteHere, RpSampledImage *pCaller );
      
      void
      demoteCacheElement( cacheElement *pCached );
      
      Map		_cache;
      
      // ------------------------------------------------------------------
      void
      calcDecomposition( const RpImageArea& area, unsigned& uXDiv, unsigned& uYDiv );

      void
      decomposeRequest( RpImageTile *pRet, RpSampledImage *pCaller ); 
      // ------------------------------------------------------------------

      void
      flushCacheMap( void );
      
      void
      flushCandidateMap( void );
      
      void
      trimCandidates( void );
      
      static unsigned long
      _getRAMSize( cacheElement* );
      
      // ------------------------------------------------------------------

};

#endif
