// ************************************************************************
// FILE : RpBinaryBuffer.h
// DESCRIPTION :
//     Container for binary data; dynamically resizable.

#ifndef	RpBinaryBuffer_CLASS
#define	RpBinaryBuffer_CLASS
#include <stdlib.h> // free()

// ************************************************************************
// CLASS : RpBinaryBuffer
// NOTE:
//    internal RAM allocation is aligned to hold any built-in data type.
//
class RpBinaryBuffer {

   public:
      RpBinaryBuffer( void ) { _vpBuffer=0; _uSize=_uUsed=0; }
      RpBinaryBuffer( unsigned uUseBytes ) { _vpBuffer=0; _uSize=_uUsed=0; useBytes(uUseBytes,0); }
      ~RpBinaryBuffer( void ) { if (_vpBuffer) free( _vpBuffer); }
      
      // ------------------------------------------------------------------
      // description:
      //    sets the number of bytes needed. Will attempt to resize if
      //    necessary.
      // post:
      //    returns a non-zero if buffer can hold requested number of bytes
      // post:
      //    if the buffer currently contains data (via a previous call
      //    to useBytes()) AND (iKeepContents) then that data will be 
      //    preserved; UNLESS the number of bytes in this call is less
      //    than the number of bytes previously being used.  In that
      //    case, the data will be truncated.
      //
      int		useBytes( unsigned uNeededSize, int iKeepContents=-1 );
      
      // ------------------------------------------------------------------
      // description:
      //    query to get the current number of bytes that this buffer
      //    has been told to hold.
      //
      unsigned		usingBytes( void ) const
      				{ return (_uUsed); }
      				
      // ------------------------------------------------------------------
      // description:
      //   Access to the buffer. user is responsible for bounds-checking.
      //
      void *		getBuffer(void)
      				{ return (_vpBuffer); }
      const void *	getBuffer(void) const
      				{ return ((const void*)_vpBuffer); }
      
      // ------------------------------------------------------------------
      // description:
      //    Internally, the size of the buffer can grow, but does not
      //    automatically shrink.
      //    Use this method to query the actual RAM consumption in bytes;
      //    NOT the size of memory that the user has requested for use.
      //
      unsigned		getRAMSize( void ) const
      				{ return (_uSize); }      
      
      // ------------------------------------------------------------------
      // description:
      //    Internally, the size of the buffer can grow, but does not
      //    automatically shrink.
      //    This is to minimize counter-acting new/delete operations.
      //    Use this method to force a shrinkage in the size of the memory
      //    allocation, thereby freeing extra RAM.
      //
      void		compact( void )
			      { resetSize( _uUsed, -1 ); }
      
   protected:
      void *		_vpBuffer;
      unsigned 		_uSize;
      unsigned		_uUsed;
      
      int resetSize( unsigned uSetBytes, int iKeepContents );      
};

#endif
