/* Copyright (c) 1994 Gregory P. Ward.  All rights reserved.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided that the above copyright
 * notice and the following two paragraphs appear in all copies of
 * this software.  
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : gl_mpeg.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: 
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/20?, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/time.h>
#include <signal.h>
#include <gl.h>
#include "mpeg.h"
#include "gl_mpeg.h"


/* Global variables set by ParseCmdLine() and used by main() */


char       *DisplayName = NULL;
BufferEnum  BufferType = NO_BUFFER;
BufferEnum  BufferFallback = NO_BUFFER;
int         MaxMemory = 0;
Boolean     KeepTempFlag = FALSE;
Boolean     QuietFlag = TRUE;


/* Some enum->string mappings (for debugging and/or command-line options) */

char *StyleNames [] = { "forward", "backward", "rock" };
char *BufferNames [] = { "none", "memory", "disk" };




/*
 *--------------------------------------------------------------
 *
 * int_handler --
 *
 *	Handles Cntl-C interupts..
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void
int_handler()
{
   if (!QuietFlag) {
     fprintf(stderr, "Interrupted!\n");
   }
/*
   if (curVidStream != NULL)
      DestroyVidStream(curVidStream);
*/
   exit(1);
}



double ReadSysClock(void)
{
  struct timeval tv;
  (void) gettimeofday(&tv, (struct timezone *)NULL);
  return (tv.tv_sec + tv.tv_usec / 1000000.0);
}


int qprintf (char *format, ...)
{
   va_list  arglist;
   int      n;
   
   if (!QuietFlag)
   {
      va_start (arglist, format);
      n = vprintf (format, arglist);
      fflush (stdout);
      va_end (arglist);
      return (n);
   }
   return (0);
}





/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitializeMovie
@INPUT      : pargc - pointer to main()'s argc
              argv  - same as main()'s argv
@OUTPUT     : Directly set by InitializeMovie and possibly by ParseCmdLine:
                 Movie->Filename
		 Movie->InputStream
		 Movie->Window
		 Movie->{TotFrames,CurFrame}
	      Set only by ParseCmdLine (or left with default values):
		 Movie->Options.*
@RETURNS    : (void)
              Note that ParseCmdLine() calls usage(), which exit()'s, in
	      the case of an error.
@DESCRIPTION: Sets up global variables and most of the MovieState structure,
              and calls ParseCmdLine() to possibly modify them.
@METHOD     : 
@GLOBALS    : Set only by ParseCmdLine (or left with default values):
	         BufferType
	         MaxMemory
@CALLS      : ParseCmdLine()
@CREATED    : 94/6/22, Greg Ward (from code in main()) 
@MODIFIED   : 94/6/27, GW: added argument Movie - cuts down on globals,
                           increases modularity, etc. etc.
              94/6/28, GW: renamed from InitializeGlobals, improved comments
              94/7/3,  GW: moved inconsistency checking to ParseCmdLine()
---------------------------------------------------------------------------- */
void InitializeMovie (int *pargc, char *argv[], MovieState *Movie)
{
   /* Set things so that the default behaviour is to take the window
    * size from the zoom factors multiplied by the image size, and the
    * default zoom is 1.0.  To override this, we'll need a way (on the
    * command line) to explicitly supply window width/height.
    */

   Movie->Filename = NULL;
   Movie->InputStream = stdin;	/* default if no filename given */
   
   Movie->Window.ZoomX = Movie->Window.ZoomY = 1.0;
   Movie->Window.Width = Movie->Window.Height = 0;

   Movie->TotFrames = Movie->CurFrame = 0;
   Movie->CurrentImage = NULL;
   
   /* 
    * Parse the command line.  Note that any of the global variables 
    * at the top of this file may be changed by ParseCmdLine() (which,
    * after all, is why they're global variables).  It may also call
    * SetMPEGOption() to set options internal to the MPEG decoder
    * (in particular, dithering options).  Of course, various fields
    * of the Movie struct (nameley the flags and window size/zoom 
    * options) may also be changed by ParseCmdLine().
    */
   
   ParseCmdLine (pargc, argv, Movie);
   
}     /* InitializeMovie () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : FlipImage
@INPUT      : Image - describes image size (height, width, bits/pixel)
              Data - image data, presumably in format described by Image
@OUTPUT     : Data - same, but flipped vertically
@RETURNS    : (void)
@DESCRIPTION: Vertically flips an image.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/28, Greg Ward
@MODIFIED   : 
@COMMENTS   : Untested on anything other than 24-bit (32 bits per pixel)
              lrectwrite()-style data.
	      Wouldn't be necessary if pixmode() worked as advertised 
	      on the 4D35's.
---------------------------------------------------------------------------- */
void FlipImage (ImageDesc *Image, char *Data)
{
   int   RowSize;
   char *CurRow, *StopRow, *OtherRow, *TempRow;

   
   RowSize = Image->Width * Image->PixelSize / 8;
   TempRow = (char *) malloc (RowSize);
   StopRow = Data + RowSize * (Image->Height/2 - 1);
   OtherRow = Data + RowSize * (Image->Height-1);

   for (CurRow = Data; CurRow <= StopRow; CurRow += RowSize)
   {
      memcpy (TempRow, CurRow, RowSize);
      memcpy (CurRow, OtherRow, RowSize);
      memcpy (OtherRow, TempRow, RowSize);
      OtherRow -= RowSize;
   }

   free (TempRow);

}     /* FlipImage () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : DecodeMovie
@INPUT      : Movie:
                 all-purpose structure containing handy data such as the
		 window size and image size/zoom/location, movie flags and
		 status
@OUTPUT     : sets Movie->TotFrames to the total number of frames received
              from the MPEG decoding engine
@RETURNS    : -1 if OpenMPEG, InitColorDisplay, or SetupBuffer fail
              0 if we finish because the user aborted (ie. in HandleEvents())
              1 if we finish because we've displayed the last frame
@DESCRIPTION: Prepares to decode an MPEG movie.  Initializes the
              display and/or the frame buffer, depending on the value
              of the global variable BufferType and the movie option
              flag Preview.  Calls MPEG library to decode frames; as
              each frame is receieved, it is either buffered or
              displayed.
@METHOD     : 
@GLOBALS    : BufferType, MaxMemory
@CALLS      : OpenMPEG
              InitColorDisplay
	      SetupBuffer
              GetMPEGFrame
	      FlipImage
	      DisplayFrame
	      SaveFrame
	      HandleEvents
@CREATED    : 94/6/22, Greg Ward (from code in main())
@MODIFIED   : 94/6/27, GW: replaced Image and Window arguments with Movie
              94/7/07, GW: little changes to reflect the new structure
                           of movie options and status flags
---------------------------------------------------------------------------- */
int DecodeMovie (MovieState *Movie)
{
   Boolean Done = FALSE;
   Boolean MoreFrames = TRUE;
   Boolean FirstPlay = TRUE;
   int     NumFrames;
   int     RetValue;

   /* Prepare to read/decode the MPEG */
   
   if (!OpenMPEG (Movie->InputStream, &Movie->Image))
   {
      return (-1);
   }
   qprintf ("Decoding MPEG");
   
   /* Initialize the display and/or the buffer */
   
   if (Movie->Options.Preview)
   {
      if (!InitColorDisplay (Movie))
      {
	 return (-1);
      }
   }

   if (BufferType != NO_BUFFER)
   {
      if (!SetupBuffer (BufferType, &Movie->Image, MaxMemory, -1))
      {
	 return (-1);
      }
   }
   
   Movie->TotFrames = 0;
   Movie->CurFrame = -1;
   Movie->CurrentImage = (char *) malloc (Movie->Image.Size);
   Movie->Status.Decoding = TRUE;
   Movie->Status.Paused = FALSE;
   Movie->Status.Forward = TRUE;
   
   /* 
    * Loop control: Done means the user aborted (this is handled and
    * reported by HandleEvents()); MoreFrames means we haven't yet
    * decoded or displayed the last frame.
    */

   while (!Done && MoreFrames)
   {
      MoreFrames = GetMPEGFrame (Movie->CurrentImage);
      FlipImage (&Movie->Image, Movie->CurrentImage);
      Movie->CurFrame++;
      if (FirstPlay)
      {	
	 Movie->TotFrames++;
      }
      qprintf ("..%d", Movie->CurFrame+1);

      /* Display and/or save the frame */
      
      if (Movie->Options.Preview)
      {
/*	 sginap (Movie->FrameDelay);         */
	 DisplayFrame (Movie);
	 HandleEvents (&Done, Movie);

	 if (!MoreFrames && BufferType == NO_BUFFER)
	 {
	    if (!Movie->Options.Continuous)
	    {
	       Movie->Status.Paused = TRUE;
	       HandleEvents (&Done, Movie);
	    }	       

	    RewindMPEG (Movie->InputStream, &Movie->Image);
	    Movie->CurFrame = -1;
	    MoreFrames = TRUE;
	    FirstPlay = FALSE;
	 }

      }

      if (BufferType != NO_BUFFER)
      {
	 SaveFrame (Movie->CurFrame, &Movie->Image, Movie->CurrentImage);
      }
   }     /* while */

   /* 
    * Now put the movie into pause mode and take user input.  The
    * user's options at this point are basically: continue to buffered
    * display (assuming buffering is on); single-step frames (again,
    * assuming buffered), or abort.  If frames were not buffered, then
    * continuing (unpausing) and aborting will look the same to the
    * user -- ie. the program will quit.
    */

   dprintf ("Done decoding: Done=%d, Continuous=%d, Preview=%d\n",
	    Done, Movie->Options.Continuous, Movie->Options.Preview);
   Movie->Status.Decoding = FALSE;
   if (!Done && !Movie->Options.Continuous && Movie->Options.Preview && Movie->PlayStyle != ROCK)
   {
      Movie->Status.Paused = TRUE;
      HandleEvents (&Done, Movie);
   }

   /* If the user aborted (either in main loop, or just above) flag it */

   RetValue = (Done) ? 0 : 1;
   
   qprintf ("\n");
   free (Movie->CurrentImage);
   Movie->CurrentImage = NULL;
   return (RetValue);

}     /* DecodeMovie () */



#undef DEBUG
/* ----------------------------- MNI Header -----------------------------------
@NAME       : AdvanceFrame
@INPUT      : Movie - same as usual
@OUTPUT     : (none)
@RETURNS    : TRUE if the "new" frame (ie. the frame advanced to) is
              the last of the current sequence.  (If the play style is
              FORWARD, then the "last frame of the sequence" is the
              last frame of the movie; if the play style is BACKWARD
              or ROCK, then it is the first frame of the movie.)

@DESCRIPTION: Advances the frame number depending on a fairly complicated
              interaction between CurDirection, CurFrame, Options, and Status.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/27, Greg Ward
@MODIFIED   : 94/7/07, GW: adapted to the new structure of movie 
                           option and status flags
                           made done an argument rather than returning it
---------------------------------------------------------------------------- */
Boolean AdvanceFrame (MovieState *Movie)
{
   Boolean LastFrame = FALSE;

   /* Are we playing forwards right now? */

   if (Movie->Status.Forward)
   {
      Movie->CurFrame++;

      /* Have we moved *past* the last frame? */

      if (Movie->CurFrame == Movie->TotFrames)
      {

	 /* Are we now supposed to play the movie backwards? */
	 
	 if (Movie->PlayStyle == ROCK || Movie->PlayStyle == BACKWARD)
	 {
	    Movie->Status.Forward = FALSE;
	    Movie->CurFrame -= 2;
	 }

	 /* Else, play style is forwards only so go back to frame 0 */

	 else
	 {
	    Movie->CurFrame = 0;
	 }     /* if/else: BACKWARDs or ROCK? */

      }     /* if reached last frame */
   }     /* if current direction is forwards */
   
   
   /* Now similar logic for going backwards ... */

   else
   {
      if (Movie->CurFrame == 0)
      {
	 /* Are we supposed to play the movie forwards now? */
	 
	 if (Movie->PlayStyle == FORWARD || Movie->PlayStyle == ROCK)
	 {
	    Movie->Status.Forward = TRUE;
	    Movie->CurFrame++;
	 }
	 
	 /* Else, just play it backwards again */
	 
	 else
	 {
	    Movie->CurFrame = Movie->TotFrames - 1;
	 }
      }

      /* Haven't reached frame 0, so just decrement */

      else
      {
	 Movie->CurFrame--;
      }
   }


   /* Have we moved to the last frame of the sequence? */

   if (Movie->PlayStyle == FORWARD)
   {
      if (Movie->CurFrame == Movie->TotFrames-1)
      {
	 LastFrame = TRUE;
      }
   }
   else
   {
      if (Movie->CurFrame == 0)
      {
	 LastFrame = TRUE;
      }
   }

   dprintf ("AdvanceFrame: just advanced to frame %d; play style is %s; LastFrame=%d\n",
	    Movie->CurFrame, StyleNames[Movie->PlayStyle], LastFrame);
   return (LastFrame);

}     /* AdvanceFrame () */



#define DEBUG



/* ----------------------------- MNI Header -----------------------------------
@NAME       : DisplayMovie
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: 
@METHOD     : 
@GLOBALS    : BufferType
@CALLS      : InitColorDisplay, DisplayFrame, CloseBuffer
@CREATED    : 94/6/22, Greg Ward, from code in main()
@MODIFIED   : 94/6/27, GW: Replaced old arguments (Image, Window, NumFrames)
                           with the single argument Movie; replaced the
			   two for-loops with a call to AdvanceFrame()
---------------------------------------------------------------------------- */
void DisplayMovie (MovieState *Movie)
{
   Boolean Done = FALSE;
   Boolean LastFrame = FALSE;

   if (!Movie->Options.Preview)	/* if we haven't ALREADY opened window... */
   {
      InitColorDisplay (Movie);
   }
   Movie->Status.Paused = FALSE;

   /* For playing FORWARD or ROCK'ing, start at frame 0 and go forward */
   
   if (Movie->PlayStyle == FORWARD || Movie->PlayStyle == ROCK)
   {
      Movie->CurFrame = 0;
      Movie->Status.Forward = TRUE;
   }
   
   /* Exclusively backwards play: start at last frame, go backwards */

   else
   {
      Movie->CurFrame = Movie->TotFrames-1;
      Movie->Status.Forward = FALSE;
   }

   /* 
    * Done means the user aborted (set by HandleEvents()); LastFrame
    * means we have displayed the last frame of the current sequence.
    * This is defined as the last frame of the movie (ie. frame N-1)
    * for FORWARD or ROCK play style, and frame 0 for BACKWARD display.
    * LastFrame is set by AdvanceFrame() by looking at the play
    * style and frame number; it doesn't give a hoot whether we're in
    * continuous or single play mode.
    */

   LastFrame = FALSE;
   while (!Done)
   {
      
      /* Load frame from buffer, display frame, and check event queue */

      Movie->CurrentImage = 
	 LoadFrame (Movie->CurFrame, &Movie->Image);
      sginap (Movie->FrameDelay);
      DisplayFrame (Movie);
      HandleEvents (&Done, Movie);

      /* 
       * If the last call to AdvanceFrame() (ie. the call that set the
       * current value of Movie->CurFrame) set LastFrame, and the user
       * selected single-play mode, then pause between plays.
       */

      dprintf ("Just displayed frame %d; LastFrame=%d, Continuous=%d\n",
	       Movie->CurFrame, LastFrame, Movie->Options.Continuous);
      if (LastFrame && !Movie->Options.Continuous && Movie->PlayStyle != ROCK)
      {
	 SetPause (Movie, TRUE);
	 HandleEvents (&Done, Movie);
      }

      /* 
       * Now advance to the next frame, following a rather complicated
       * formula that depends on myriad movie option and status flags.
       */

      LastFrame = AdvanceFrame (Movie);

   }

   CloseBuffer (BufferType, Movie->TotFrames);
}     /* DisplayFrame () */




void main(int argc, char *argv[])
{
   MovieState   Movie /*= 
   { 
      NULL, NULL, 
      { 0, 0, 0, 0.0, 0.0, 0, 0 },
      { 0, 0, 0, 0, 0, 0 },
      PLAY, FORWARD,
      0, 0, 0,
      { 0, 1, 0, 1, 0, 0, 1 },
      NULL
   }*/;

   mallopt (M_DEBUG, 1);
   InitializeMovie (&argc, argv, &Movie);

   /*
    * Now decode the movie; if DecodeMovie() returns anything other
    * than 1, we are done (either an error or a user abort).  Otherwise
    * check if buffering and continuous play were both enabled, and
    * display the movie if so.
    */
   if (DecodeMovie (&Movie) == 1 && 
       BufferType != NO_BUFFER)
   {
      DisplayMovie (&Movie);
   }
}
