/*
 * PSutils.c -  Routines to interface to display postscript.
 *
 * Written by:  Wesley C. Barris
 *              AHPCRC
 *              Minnesota Supercomputer Center, Inc.
 * Date:        May 12, 1992
 * Copyright @ 1992, Minnesota Supercomputer Center, Inc.
 * RESTRICTED RIGHTS LEGEND
 *
 * Use, duplication, or disclosure of this software and its documentation
 * by the Government is subject to restrictions as set forth in subdivision
 * { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 * clause at 52.227-7013.
 *
 */
/*
 * Here are the routines contained in this file:
 *	psinit()
 *	psclose()
 *	psfindfont(font, size)
 *	pssetrotation(ang)
 *  int	psgetstrwidth(text)
 *	psgetbbox(text, llx, lly, urx, ury)
 *	psgetblocksize(text, totwidth, totheight)
 *	pstexttoctxt(text, justify)
 *	pscopypixmap(topixmap)
 *	psorpixmap(topixmap)
 *   GC	psgetbitmapgc(topixmap)
 *XImage* psctxttobitmap()
 *	psbitmaptorle(bitmap, cv)
 *	psbitmaptogl(cx, cy, bitmap)
 *	psbitmaptolrect(cx, cy, bitmap)
 *	psbitmaptopixmap(cx, cy, bitmap, dest, cv)
 *	psprstr(text)
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <sys/utsname.h>
#include <DPS/dpsXclient.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <gl.h>
#include "PSutilsP.h"

/*
 * DPS Prototypes
 */
/*
DPSContext	XDPSCreateSimpleContext(Display*, Drawable, GC, int, int, DPSTextProc, DPSErrorProc, DPSSpace);
void		DPSPrintf(DPSContext, char*, ...);
void		DPSSetContext(DPSContext);
void		DPSDestroyContext(DPSContext);
void		DPSWaitContext(DPSContext);
*/


#define grayMax 1	/* white */
#define grayMult 1	/* distance between black and white */
#define firstGray 0	/* black */

static DPSContext	ctxt;		/* Display PostScript context */
static Display	*display;
static GC	dpsgc;
static GC	bitmapgc = NULL;
static Pixmap	dpspixmap;
static float	xdpi, ydpi;
static int	pixwidth, pixheight;
static char	extName[] = "DPSExtension";
static int	totstrwidth;
static int	fontheight, ymax;
static int	xoffset, yoffset;
static int	angle = 0;
static double	rangle = 0.0;
static unsigned int	totstrheight;

/*****************************************************************************/
/*
 * Determine the maximum of two numbers.
 */
static int
max(int i1, int i2)
{
   if (i1 > i2)
      return i1;
   else
      return i2;
}
/*****************************************************************************/
/*
 * Convert point size to X dots per inch.
 */
static float
pttoxdpi(float ptsize)
{
   return ptsize*xdpi/72.0;
}

/*****************************************************************************/
/*
 * Convert point size to Y dots per inch.
 */
static float
pttoydpi(float ptsize)
{
   return ptsize*ydpi/72.0;
}

/*****************************************************************************/
/*
 * Convert X dots per inch to point size.
 */
static int
xtopt(int x)
{
   return (int)(x*72.0/xdpi + 0.5);
}

/*****************************************************************************/
/*
 * Convert Y dots per inch to point size.
 */
static int
ytopt(int y)
{
   return (int)(y*72.0/ydpi + 0.5);
}

/*****************************************************************************/
/*
 * This routine will initialize a DPS pixmap for PS rendering.
 */
void
psinit(Display *dpy, int w, int h)
{
   int mOptCode, fEvent, fError;   /* major opcode, first event, first error */
   int screen;
   int	ramp[3];
   struct utsname name;
/*
 * Save these for later use.
 */
   display = dpy;
   pixwidth = w;
   pixheight = h;
/*
 * If the Display PostScript extension is not available, exit.
 */
   if (!XQueryExtension(display, extName, &mOptCode, &fEvent, &fError)) {
      printf("This program requires Display PostScript.  The Display\n");
      printf("PostScript extension of the X server is not available.\n");
      exit(1);
      }
   uname(&name);
/*
 * I compute the screen dots per inch here.  This is used when converting
 * between points and screen coords.  I know that adding 1.0 is not really
 * correct here, but the resulting text widths "look" more correct when I
 * do this.
 */
   xdpi = (float)getgdesc(GD_XPMAX)/getgdesc(GD_XMMAX)*25.4 + 1.0;
   ydpi = (float)getgdesc(GD_YPMAX)/getgdesc(GD_YMMAX)*25.4 + 1.0;
   screen = DefaultScreen(display);
/*
 * IRIX 5.x.x.x requires that the dps pixmap has 8 bit planes.
 * Others releases do not.
 */
   if (name.release[0] == '4')
      dpspixmap = XCreatePixmap(display, RootWindow(display, screen),
                                pixwidth, pixheight, 1);
   else
      dpspixmap = XCreatePixmap(display, RootWindow(display, screen),
                                pixwidth, pixheight, 8);
   dpsgc = XCreateGC(display, dpspixmap, 0, NULL);
/*
 * create a simple DPS context
 * NOTE: This statement causes the X server to crash if you are running
 * with a default 24 bit visual.
 */
   ctxt = XDPSCreateSimpleContext(display, dpspixmap, dpsgc, 0, pixheight,
                          DPSDefaultTextBackstop, DPSDefaultErrorProc, NULL);
   if (ctxt == NULL) {
      fprintf(stderr, "Unable to create a Display PostScript context.\n");
      exit(1);
      }
/*
 * Create a gray ramp appropriate for 1-bit deep pixmap, attach
 * pixmap and GC to context.  White pixels will have the value 1,
 * while black pixels will have the value 0.
 */
   ramp[0] = grayMax;
   ramp[1] = grayMult;
   ramp[2] = firstGray;
   DPSPrintf(ctxt, "%d %d %d %d [%d %d %d] setXgcdrawablecolor\n",
             XGContextFromGC(dpsgc), dpspixmap, 0, pixheight,
             ramp[0], ramp[1], ramp[2]);
             /*dpsgc->gid, dpspixmap, 0, pixheight, ramp[0], ramp[1], ramp[2]);*/
   DPSSetContext(ctxt);
}

/*****************************************************************************/
/*
 * This routine is used to free pixmap memory and terminate DPS functions.
 */
void
psclose(void)
{
   DPSDestroyContext(ctxt);
   XFreeGC(display, dpsgc);
   XFreePixmap(display, dpspixmap);
}

/*****************************************************************************/
/*
 * This routine is used to select a certain font for printing text.
 */
void
psfindfont(char *font, float ptsize)
{
   int i;
/*
 * Put the selected font on the stack.
 */
   /*PSWSelectFont(ctxt, font, ptsize);*/
   DPSselectfont(ctxt, font, ptsize);
   fontheight = pttoydpi(ptsize) + 1.0;	/* height based on point size */
/*
 * Compute the X and Y offsets for this font.
 */
   for (i=0; i<N_EXT_FONTS; i++) {
      if (!(strncmp(font, font_info[i].name, strlen(font)))) {
         xoffset = font_info[i].xmult*pttoxdpi(ptsize)+1.0;
         yoffset = font_info[i].lomult*pttoydpi(ptsize)+1.0;
         ymax = font_info[i].himult*pttoydpi(ptsize)+1.0;
         }
      }
}

/*****************************************************************************/
/*
 * This routine is used to set a rotation for text.
 */
void
pssetrotation(int ang)
{
/*
 * Send the rotation angle.
 */
   PSWSetRotation(ctxt, ang);
   angle = ang;
   rangle = ang*3.141592654/180.0;
}

/*****************************************************************************/
/*
 * This routine will return the width in pixels of the simple text string.
 */
int
psgetstrwidth(char *text)
{
   float	width, height;
   int		iwidth;

   /*PSWStringWidth(ctxt, text, &width);*/
   DPSstringwidth(ctxt, text, &width, &height);
   iwidth = (int)(pttoxdpi(width) + 0.5) + xoffset;/* round off to nearest int */
/*
 * To compute width of rotated text I would have to:
 *	- determine the number of lines
 *	- determine the font height
 * 	- compute iwidth = (int)(width*cos(rangle) + height*sin(rangle));
 */
   return iwidth;
}

/*****************************************************************************/
/*
 * This routine will return the bounding box of one string of text.
 */
void
psgetbbox(char *text, int *llx, int *lly, int *urx, int *ury)
{
/*
 * This doesn't work after several calls???
 */
   DPSmoveto(ctxt, 0.0, 0.0);
   DPScharpath(ctxt, text, 0);
   DPSflattenpath(ctxt);
   DPSpathbbox(ctxt, llx, lly, urx, ury);
}

/*****************************************************************************/
/*
 * This routine will return both the width and height of the text block
 * which contains carriage return characters.
 */
void
psgetblocksize(char *text, int *totwidth, int *totheight)
{
   char		**strarr, *startP, *endP, *sP;
   float	strwidth, hgt;
   int		i, nstr, nchar, nlines, spec_char;
   float	box[4];

   sP = text;
   nlines = 1;
   while(*sP != '\0') {
      if (*sP++ == '\n')
         nlines++;
      }
   strarr = (char **)malloc(nlines*sizeof(char *));
   startP = text;				/* point to beginning of text */
   endP = strchr(text, '\n');		/* point to first <CR> */
   nstr = 0;
   if (text[0] == '.') {
      while (endP) {
         nchar = (int)(endP-startP + 1);	/* # of chars in first str */
         strarr[nstr] = (char *)malloc(nchar);
         strncpy(strarr[nstr], startP, nchar-1);
         strarr[nstr][nchar-1] = '\0';		/* terminate string */
         spec_char = atoi(&strarr[nstr][1]);
         strarr[nstr][0] = (char)spec_char;
         strarr[nstr++][1] = '\0';
         startP = endP + 1;			/* point to next str */
         endP = strchr(startP, '\n');		/* find next <CR> */
         }
      strarr[nstr] = (char *)malloc(strlen(startP)+1);
      strcpy(strarr[nstr], startP);
      spec_char = atoi(&strarr[nstr][1]);
      strarr[nstr][0] = (char)spec_char;
      strarr[nstr++][1] = '\0';
      }
   else {
      while (endP) {
         nchar = (int)(endP-startP + 1);	/* # of chars in first str */
         strarr[nstr] = (char *)malloc(nchar);
         strncpy(strarr[nstr], startP, nchar-1);
         strarr[nstr++][nchar-1] = '\0';	/* terminate string */
         startP = endP + 1;			/* point to next str */
         endP = strchr(startP, '\n');		/* find next <CR> */
         }
      strarr[nstr] = (char *)malloc(strlen(startP)+1);
      strcpy(strarr[nstr++], startP);
      }
   *totheight = yoffset + (nlines-1)*fontheight + ymax;
   /*fprintf(stderr, "%d %d %d\n", yoffset, fontheight, ymax);*/
   if (*totheight > pixheight)
      *totheight = pixheight;
   *totwidth = 0;
      for (i=0; i<nlines; i++) {
         /*PSWStringWidth(ctxt, strarr[i], &strwidth);*/
         DPSstringwidth(ctxt, strarr[i], &strwidth, &hgt);
         *totwidth = max(*totwidth, (int)(pttoxdpi(strwidth)+1.0));
         }
   /*(*totwidth)++;*/
    *totwidth += xoffset;
/*
 * To handle rotated text I would have to recompute:
 *	newwidth  = oldwidth*cos(rangle) + oldheight*sin(rangle);
 *	newheight = oldheight*cos(rangle) + oldwidth*sin(rangle);
 */
}

/*****************************************************************************/
/*
 * This routine will print a block of text that contains carriage return
 * characters.
 */
void
pstexttoctxt(char *text, int justify)
{
   char		**strarr, *startP, *endP, *sP;
   float	strwidth, hgt;
   int		i, nchar;
   int		nstr, nlines;
   int		xorig, yorig;
   int		spec_char;
   short	r, g, b;

   sP = text;
   nlines = 1;
   while(*sP != '\0') {
      if (*sP++ == '\n')
         nlines++;
      }
   strarr = (char **)malloc(nlines*sizeof(char *));
   startP = text;				/* point to beginning of text */
   endP = strchr(text, '\n');		/* point to first <CR> */
   nstr = 0;
   if (text[0] == '.') {
      while (endP) {
         nchar = (int)(endP-startP + 1);	/* # of chars in first str */
         strarr[nstr] = (char *)malloc(nchar);
         strncpy(strarr[nstr], startP, nchar-1);
         strarr[nstr][nchar-1] = '\0';		/* terminate string */
         spec_char = atoi(&strarr[nstr][1]);
         strarr[nstr][0] = (char)spec_char;
         strarr[nstr++][1] = '\0';
         startP = endP + 1;			/* point to next str */
         endP = strchr(startP, '\n');		/* find next <CR> */
         }
      strarr[nstr] = (char *)malloc(strlen(startP)+1);
      strcpy(strarr[nstr], startP);
      spec_char = atoi(&strarr[nstr][1]);
      strarr[nstr][0] = (char)spec_char;
      strarr[nstr++][1] = '\0';
      }
   else {
      while (endP) {
         nchar = (int)(endP-startP + 1);	/* # of chars in first str */
         strarr[nstr] = (char *)malloc(nchar);
         strncpy(strarr[nstr], startP, nchar-1);
         strarr[nstr++][nchar-1] = '\0';	/* terminate string */
         startP = endP + 1;			/* point to next str */
         endP = strchr(startP, '\n');		/* find next <CR> */
         }
      strarr[nstr] = (char *)malloc(strlen(startP)+1);
      strcpy(strarr[nstr++], startP);
      }
/*
 * Figure out the height and width.
 */
   totstrheight = yoffset + (nlines-1)*fontheight + ymax;
   if (totstrheight > pixheight)
      totstrheight = pixheight;
   totstrwidth = 0;
      for (i=0; i<nlines; i++) {
         /*PSWStringWidth(ctxt, strarr[i], &strwidth);*/
         DPSstringwidth(ctxt, strarr[i], &strwidth, &hgt);
         totstrwidth = max(totstrwidth, (int)(pttoxdpi(strwidth)+1.0));
         }
   /*totstrwidth++;*/
   totstrwidth += xoffset;
/*
 * The pixmap is not initialized.  Paint it black.
 */
   DPSPrintf(ctxt, "erasepage\n"); /* or: PSerasepage(); */
/*
 *  Print all the strings.
 */
   /*if (spec_char)
      PSWDisplayChar(ctxt, 0, ytopt(fontheight*0.25), spec_char);
   else*/
      for (i=0; i<nlines; i++) {
         if (justify == JUSTIFY_CENTER) {
            /*PSWStringWidth(ctxt, strarr[i], &strwidth);*/
            DPSstringwidth(ctxt, strarr[i], &strwidth, &hgt);
            xorig = (totstrwidth-pttoxdpi(strwidth)-xoffset)/2 + xoffset;
            }
         else if (justify == JUSTIFY_RIGHT) {
            /*PSWStringWidth(ctxt, strarr[i], &strwidth);*/
            DPSstringwidth(ctxt, strarr[i], &strwidth, &hgt);
            xorig = totstrwidth-pttoxdpi(strwidth)-xoffset + xoffset;
            }
         else
            xorig = xoffset;
         yorig = fontheight*(nlines-i-1) + yoffset;
         if (yorig < pixheight)
            PSWDisplayText(ctxt, xtopt(xorig), ytopt(yorig), strarr[i]);
         }
   DPSWaitContext(ctxt);	/* wait for postscript to finish */
}

/*****************************************************************************/
/*
 * This routine will copy the DPS pixmap to a pixmap of your choice.
 */
void
pscopypixmap(Pixmap topixmap)
{
   if (!bitmapgc)
      bitmapgc = XCreateGC(display, topixmap, 0, NULL);
   XCopyPlane(display, dpspixmap, topixmap, bitmapgc, 0, pixheight-totstrheight,
              totstrwidth, totstrheight, 0, 0, (unsigned long)0x01);
}

/*****************************************************************************/
/*
 * This routine will return the gc used for bitmap operations.
 */
GC
psgetbitmapgc(Pixmap topixmap)
{
   if (!bitmapgc)
      bitmapgc = XCreateGC(display, topixmap, 0, NULL);
   return bitmapgc;
}

/*****************************************************************************/
/*
 * This routine will or the DPS pixmap to a pixmap of your choice.
 */
void
psorpixmap(Pixmap topixmap, int offset)
{
   if (!bitmapgc)
      bitmapgc = XCreateGC(display, topixmap, 0, NULL);
   XSetFunction(display, bitmapgc, GXor);
   XCopyPlane(display, dpspixmap, topixmap, bitmapgc, 0, pixheight-totstrheight,
              totstrwidth, totstrheight, offset, offset, (unsigned long)0x01);
   XSetFunction(display, bitmapgc, GXcopy);
}

/*****************************************************************************/
/*
 * This routine will copy the DPS pixmap to a bitmap.
 */
XImage*
psctxttobitmap(void)
{
   XImage       *bitmap;
/*
 * Transfer the text image from the server to the client.
 */
   if (totstrwidth > pixwidth) {
      fprintf(stderr, "WARNING!  Text too wide for default pixmap -- cropping.\n");
      totstrwidth = pixwidth;
      }
   bitmap = XGetImage(display, dpspixmap, 0, pixheight-totstrheight,
                      totstrwidth, totstrheight, (unsigned long)0x01, XYPixmap);
   if (bitmap == NULL) {
      fprintf(stderr, "Unable to get the contents of a pixmap.\n");
      exit(1);
      }
   return bitmap;
}

/*****************************************************************************/
/*
 * This routine will copy the content of a bitmap to a 24 bit rle file.
 */
void
psbitmaptorle(XImage *bitmap, short cv[4])
{
   FILE                 *pixout;
   char                 cmd[80];
   unsigned long        pixelvalue;
   unsigned long        *wbuf = NULL;
   int                  bufsiz;
   int                  i, x, y;
   int                  black_text;
/*
 * Grab some memory for the image.
 */
   bufsiz = bitmap->width*bitmap->height*4;
   if ((wbuf = (unsigned long *) malloc(bufsiz)) == NULL)
      fprintf(stderr, "Cannot malloc %d bytes.\n", bufsiz);
   if (cv[0] == 0 && cv[1] == 0 && cv[2] == 0) {
      black_text = 1;
      cv[0] = 1;
      }
   else
      black_text = 0;
   for(y=bitmap->height-1, i=0; y>=0; y--) {
      for(x=0; x<bitmap->width; x++, i++) {
         if ((pixelvalue = XGetPixel(bitmap, x, y)) == 0) {
             wbuf[i] = (black_text ? (cv[3] << 24) : cv[0]
                                                  | (cv[1] << 8)
                                                  | (cv[2] << 16)
                                                  | (cv[3] << 24) );
             }
         else
             wbuf[i] = 0;
         }
      }
/*
 *  Make an RLE file.
 */
   sprintf(cmd,"rawtorle -w %d -h %d -n 4 -r",bitmap->width,bitmap->height);
   if ((pixout = popen(cmd, "w")) == NULL)
      fprintf(stderr, "Cannot open pipe.\n");
   if ((fwrite(wbuf, bufsiz, 1, pixout)) != 1)
      fprintf(stderr, "Cannot write command to pipe.\n");
   pclose(pixout);
   free(wbuf);
}


/*****************************************************************************/
/*
 * This routine will copy the content of a bitmap to a 24 bit gl window.
 */
void
psbitmaptogl(short cx, short cy, XImage *bitmap)
{
   short		vector[2];
   int			x, y;
   unsigned long	pixelvalue;
/*
 * Copy text from bitmap to glx window.
 */
   bgnpoint();
   for(y=0; y<bitmap->height; y++) {
      for(x=0; x<bitmap->width; x++) {
         if ((pixelvalue = XGetPixel(bitmap, x, y)) == 0) {
            vector[0] = cx + x;
            vector[1] = cy + bitmap->height-y-1;
            v2s(vector);
            }
         }
      }
   endpoint();
}


/*****************************************************************************/
/*
 * This routine will copy the content of a bitmap to a 24 bit gl window
 * via lrectwrite.
 */
void
psbitmaptolrect(short cx, short cy, XImage *bitmap)
{
   int			x, y;
   unsigned long	pixelvalue, *pbuf;
   short		r, g, b;
/*
 * Copy bitmap containing text into an unsigned long array using current
 * color.  Then lrectwrite the buffer to the screen.
 */
   /*r = 240;g = 240;b = 40;*/
   gRGBcolor(&r, &g, &b);
   pbuf = (unsigned long *)malloc(bitmap->width*bitmap->height*sizeof(long));
   for(y=0; y<bitmap->height; y++)
      for(x=0; x<bitmap->width; x++)
         if ((pixelvalue = XGetPixel(bitmap, x, y)) == 0)
            pbuf[x + (bitmap->height-y-1)*bitmap->width] = 255 << 24 | b << 16 | g << 8 | r;
         else
            pbuf[x + (bitmap->height-y-1)*bitmap->width] = 0;
   blendfunction(BF_SA, BF_MSA);
   lrectwrite(0, 0, bitmap->width-1, bitmap->height-1, pbuf);
   blendfunction(BF_ONE, BF_ZERO);
   free(pbuf);
}

/*****************************************************************************/
/*
 * This routine will copy the contents of a bitmap to a drawable.
 */
void
psbitmaptopixmap(short cx, short cy, XImage *bitmap, Window dest, short cv[4])
{
   Colormap	colormap;
   XColor	screen_in_out;
   int		x, y;
   unsigned long pixelvalue;

/*
 * Get the index of the colorcell that matches cv most closely.
 */
   colormap = DefaultColormap(display, DefaultScreen(display));
   screen_in_out.red   = (cv[0]/255.0)*65535;
   screen_in_out.green = (cv[1]/255.0)*65535;
   screen_in_out.blue  = (cv[2]/255.0)*65535;
   XAllocColor(display, colormap, &screen_in_out);
   fprintf(stderr, "Pixel value: %d\n", screen_in_out.pixel);
   XSetForeground(display, DefaultGC(display, 0), screen_in_out.pixel);
/*
 * Copy from bitmap to dest window.
 */
   for(y=0; y<bitmap->height; y++)
      for(x=0; x<bitmap->width; x++)
         if ((pixelvalue = XGetPixel(bitmap, x, y)) == 0)
            XDrawPoint(display, dest, DefaultGC(display, 0), x, y);
}

/*****************************************************************************/
/*
 * This routine will print a simple text string.
 */
void
psprstr(char *text)
{

   XImage	*bitMap;
   char		*strarr[20], *startP, *endP;
   float	strwidth, hgt;
   int		i, x, y, nchar;
   int		nstr = 0;
   int		xorig, yorig;
   int		spec_char = 0;
   short	vector[2];
   unsigned long pixelvalue;

   startP = text;				/* point to beginning of text */
   endP = strchr(text, '\n');			/* point to first <CR> */
   while (endP) {
      nchar = (int)(endP-startP + 1);	/* # of chars in first str */
      strarr[nstr] = (char *)malloc(nchar);
      strncpy(strarr[nstr], startP, nchar-1);
      strarr[nstr++][nchar-1] = '\0';	/* terminate string */
      startP = endP + 1;			/* point to next str */
      endP = strchr(startP, '\n');		/* find next <CR> */
      }
   strarr[nstr] = (char *)malloc(strlen(startP)+1);
   strcpy(strarr[nstr++], startP);
   totstrheight = yoffset + (nstr-1)*fontheight + ymax;
   totstrwidth = 0;
      for (i=0; i<nstr; i++) {
         /*PSWStringWidth(ctxt, strarr[i], &strwidth);*/
         DPSstringwidth(ctxt, strarr[i], &strwidth, &hgt);
         totstrwidth = max(totstrwidth, (int)(pttoxdpi(strwidth)+1.0));
         }
   totstrwidth += xoffset;
/*
 * The pixmap is not initialized.  Paint it white.
 */
   DPSPrintf(ctxt, "erasepage\n"); /* or: PSerasepage(); */
/*
 *  Print all the strings.
 */
   if (spec_char)
      /*PSWDisplayChar(ctxt, 0, (int)(fontheight*0.25), spec_char);*/
      PSWDisplayChar(ctxt, 0, ytopt(fontheight*0.25), spec_char);
   else
      for (i=0; i<nstr; i++) {
         xorig = 0;
         yorig = fontheight*(nstr-i-1) + fontheight*0.25;
         PSWDisplayText(ctxt, xtopt(xorig), ytopt(yorig), strarr[i]);
         }
   DPSWaitContext(ctxt);	/* wait for postscript to finish */
/*
 * Transfer the text image from the server to the client.
 */
   if (totstrwidth > pixwidth) {
      fprintf(stderr,
              "WARNING!  Text too wide for default pixmap -- cropping to %d.\n",
              pixwidth);
      totstrwidth = pixwidth;
      }
   bitMap = XGetImage(display, dpspixmap, 0, pixheight-totstrheight, totstrwidth, totstrheight,
                      (unsigned long)0x01, XYPixmap);
   if (bitMap == NULL) {
      fprintf(stderr, "Unable to get the contents of a pixmap.\n");
      exit(1);
      }
   /*XFreePixmap(display, dpspixmap);*/
   /*CreateURTImage(bitMap, totstrwidth, totstrheight);*/
/*
 * Copy text from bitmap to glx window.
 */
   bgnpoint();
   for(y=0; y<bitMap->height; y++) {
      for(x=0; x<bitMap->width; x++) {
         if ((pixelvalue = XGetPixel(bitMap, x, y)) == 0) {
            vector[0] = x;
            vector[1] = bitMap->height-y;
            v2s(vector);
            }
         }
      }
   endpoint();
}
