/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: gbcio.c,v 1.12 1997/08/19 01:34:12 timj Exp $")


#define		__gbcio_c__

#include	"gbcio.h"
#include	"gsi.h"
#include	"widdata.h"
#include	"structures.h"
#include	"misc.h"
#include	"defines.h"
#include	<errno.h>
#include	<stdarg.h>
#include	<string.h>



/* --- defines --- */


/* the STRUCT_HAS_FIELD_HASH_TABLE() macro is used to determine wether
 * the struct_type has a hash table for it's field names or not.
*/
#define	STRUCT_HAS_FIELD_HASH_TABLE(struct_type)	\
	( gsi_struct_info(struct_type)!=NULL )


/* the HANDLE_DUPS() macro registers the occourance of a duplicate
 * symbol name in the gbc_htab_dups_ref and renames the widget.
 * it also issues a warning.
*/
#define	HANDLE_DUPS(org_name, widget_data)	{			\
	if (strcmp((org_name),						\
	           widget_data_get_symbol_name((widget_data)))!=0) {	\
		g_hash_table_insert(gbc_htab_dups_ref,		\
				    g_strdup((org_name)),		\
				    (widget_data));			\
		gbc_warn("duplicate widget name `%s': renamed to: `%s'",	\
			(org_name),					\
			widget_data_get_symbol_name((widget_data)));	\
	}								\
}


#define	gbc_should_abort()		(!yyin || gbc_pa_errors>=gbc_pa_errors_max)
#define	gbc_unexp_widget(widget_name,reason)		\
	gbc_error("unexpected widget `%s'%s%s%s",	\
		 (widget_name),				\
		 (reason) ? " (" : "",			\
		 (reason) ? (reason) : "",		\
		 (reason) ? ")" : "")



/* --- variables --- */
extern	FILE		*yyin;		/* from gbclex.l */
	guint		yyline=0;	/* referenced in gbclex.l */
	yytoken_U	yylval;		/* referenced in gbclex.l */
static	gchar		*gbc_name=NULL;
static	guint		gbc_indent_level=0;
static	guint		gbc_pa_errors=0;
static	guint		gbc_pa_errors_max=0;
static	guint		gbc_signal_handler_field_hash_offset=42;
static	GHashTable	*gbc_htab_dups_ref=NULL;
static	GHashTable	*gbc_htab_struct_type=NULL;
static	GHashTable	*gbc_htab_signal_handler_fields=NULL;
static	GHashTable	**gbc_htab_array_struct_field=NULL;


/* --- static prototypes --- */


/* the yyfetch() function scans expected_token from yyin. if something
 * else is found, TOKEN_ERROR will be returned.
 * if expected_token is TOKEN_ANY any tokens but TOKEN_EOF will be accepted.
 * if expected_token is TOKEN_EOF any tokens including TOKEN_EOF will be
 * accepted.
 * if TOKEN_ERROR is returned and EOF is not reached,
 * a warning will be issued.
*/
static	gint	yyfetch		(gint		expected_token);


static	void	gbc_unexp_token	(gint		token,
				 gint		expected_token);
static	void	gbc_error	(const gchar	*format,
				 ...)
				 #ifdef         __GNUC__
				 	__attribute__
				 	((format (printf, 1, 2)))
				 #endif         /*__GNUC__*/
				 ;
/**/	void	gbc_warn		(const gchar	*format,
				 ...)		 	/* used in gbclex.l */
				 #ifdef         __GNUC__
				 	__attribute__
				 	((format (printf, 1, 2)))
				 #endif         /*__GNUC__*/
				 ;


static	void	gbc_writef	(FILE		*file,
				 const gchar	*format,
				 ...)
				 #ifdef         __GNUC__
				 	__attribute__
				 	((format (printf, 2, 3)))
				 #endif         /*__GNUC__*/
				 ;


/* widget data io functions
*/
static	void	gbc_hash_table_init	(void);
static	void	gbc_htab_destroy	(void);
static	void	gbc_htab_free_key	(gpointer		key,
					 gpointer		value,
					 gpointer		user_data);
static	void	gbc_write_generic_fields(FILE			*f_out,
					 gb_any_S		*Struct,
					 gboolean		skip_defaults);
static	void	gbc_write_linkage	(FILE			*f_out,
					 gb_wdat_base_S		*widget_data,
					 gboolean		skip_defaults);
static	void	gbc_write_signal_handler_stack (FILE		*f_out,
					 gb_wdat_base_S	*widget_data);
static	void	gbc_p_structure_R	(tree_S			*tree,
					 gb_any_S		*Struct);
static	void	gbc_p_generic_field	(gb_any_S		*Struct,
					 gchar			*field_name);
static	void	gbc_p_signal_handler	(gb_wdat_base_S		*widget_data);



/* --- functions --- */
void
gbc_error	(const char	*format,
		 ...)
{
		 va_list	args;
	
	g_assert(format);
	
	gbc_pa_errors++;
	fprintf(GBC_STREAM_ERROR,
		"%s:%d: error: ",
		gbc_name,
		yyline);
	va_start(args, format);
	vfprintf(GBC_STREAM_ERROR,
		format,
		args);
	va_end(args);
	fprintf(GBC_STREAM_ERROR,
		"\n");
	fflush(GBC_STREAM_ERROR);
}


void
gbc_warn		(const char	*format,
		 ...)
{
		 va_list	args;
	
	g_assert(format);
	
	fprintf(GBC_STREAM_WARN,
		"%s:%d: warning: ",
		gbc_name,
		yyline);
	va_start(args, format);
	vfprintf(GBC_STREAM_WARN,
		format,
		args);
	va_end(args);
	fprintf(GBC_STREAM_WARN,
		"\n");
	fflush(GBC_STREAM_WARN);
}


void
gbc_unexp_token	(gint		token,
		 gint		expected_token)
{
	register gchar	*token_string;
	register guint	token_string_len;
	register gchar	*expected_string;
	register guint	expected_string_len;
	
	token_string_len=64;
	token_string=g_new(gchar, token_string_len+1);
	expected_string_len=64;
	expected_string=g_new(gchar, expected_string_len+1);
	
	switch (token) {
	
	case	' ' ... '~':
		snprintf(token_string, token_string_len, "character `%c'", token);
		break;
	
	case	TOKEN_NULL:
		snprintf(token_string, token_string_len, "identifier `NULL'");
		break;
	
	case	TOKEN_LONG:
		snprintf(token_string, token_string_len, "number `%ld'", yylval.Long);
		break;
	
	case	TOKEN_DOUBLE:
		snprintf(token_string, token_string_len, "number `%.1f'", yylval.Double);
		break;
	
	case	TOKEN_STRING:
		snprintf(token_string, token_string_len, "%sstring \"%s\"",
			strlen(yylval.String)>0 ? "" : "empty ",
			yylval.String);
		break;
	
	case	TOKEN_TREE:
		snprintf(token_string, token_string_len, "tree identifier `%s'", yylval.String);
		break;
	
	case	TOKEN_STRUCT:
		snprintf(token_string, token_string_len, "structure identifier `GB_%s'", yylval.String);
		break;
	
	case	TOKEN_WIDGET:
		snprintf(token_string, token_string_len, "widget identifier `GB_WIDGET_%s'", yylval.String);
		break;
	
	
	case	TOKEN_FIELD:
		snprintf(token_string, token_string_len, "field identifier `%s'", yylval.String);
		break;
	
	case	TOKEN_ERROR:
		g_assert_not_reached();
		break;
	
	default:
		snprintf(token_string, token_string_len, "token <%d>", token);
		break;
	}

	switch (expected_token) {
	
	case	TOKEN_ANY:
	case	TOKEN_EOF:
		g_free(expected_string);
		expected_string=NULL;
		break;
	
	case	' ' ... '~':
		snprintf(expected_string, expected_string_len, "character `%c'", expected_token);
		break;
	
	case	TOKEN_NULL:
		snprintf(expected_string, expected_string_len, "`NULL'");
		break;
	
	case	TOKEN_LONG:
		snprintf(expected_string, expected_string_len, "number (integer)");
		break;
	
	case	TOKEN_DOUBLE:
		snprintf(expected_string, expected_string_len, "number (float)");
		break;
	
	case	TOKEN_STRING:
		snprintf(expected_string, expected_string_len, "string constant");
		break;
	
	case	TOKEN_TREE:
		snprintf(expected_string, expected_string_len, "tree identifier");
		break;
	
	case	TOKEN_STRUCT:
	case	TOKEN_WIDGET:
		snprintf(expected_string, expected_string_len, "structure identifier");
		break;
	
	case	TOKEN_FIELD:
		snprintf(expected_string, expected_string_len, "field identifier");
		break;
	
	default:
		snprintf(expected_string, expected_string_len, "token <%d>", expected_token);
		break;
	}


	if (token==TOKEN_EOF) {
		g_assert(!yyin);
		gbc_error("unexpected end of file");
	} else if (expected_string) {
		gbc_error("unexpected %s, expected %s", token_string, expected_string);
	} else
		gbc_error("unexpected %s", token_string);
	
	g_free(token_string);
	g_free(expected_string);
}


void
gbc_hash_table_init	(void)
{
	register guint	i;
	
	if (!gbc_htab_struct_type) {
		g_assert(!gbc_htab_array_struct_field);
	
		gbc_htab_struct_type=g_hash_table_new((GHashFunc)g_string_hash,
							(GCompareFunc)g_string_equal);
		gbc_htab_array_struct_field=g_new0(GHashTable*, GB_STRUCT_LAST);
	
		for (i=0; i<GB_STRUCT_LAST; i++) {
			if (STRUCT_HAS_FIELD_HASH_TABLE(i))
			    	g_hash_table_insert(gbc_htab_struct_type,
					    (gpointer)gsi_struct_info(i)->struct_name,
					    (gpointer)i);
		}
	
		for (i=0; i<GB_STRUCT_LAST; i++) {
			register GList			 *list, *free_list;
		
			if (!STRUCT_HAS_FIELD_HASH_TABLE(i)) {
				gbc_htab_array_struct_field[i]=NULL;
				continue;
			}
			gbc_htab_array_struct_field[i]=g_hash_table_new((GHashFunc)g_string_hash,
									  (GCompareFunc)g_string_equal);
			g_assert(gbc_htab_array_struct_field[i]);
			
			free_list=list=gsi_struct_field_list(gsi_struct_info(i));
			while (list) {
				register const gsi_field_info_S	*field_info;
				
				field_info=list->data;
				
#ifdef	PARANOID
				g_assert(strcmp(to_downcase(field_info->field_name), field_info->field_name)==0);
#endif	PARANOID
				g_hash_table_insert(gbc_htab_array_struct_field[i],
						    (gpointer)field_info->field_name,
						    (gpointer)field_info);
				list=list->next;
			}
			g_list_free(free_list);
		}
		
		
		/* initialize signal handler parser
		*/
		g_assert(!gbc_htab_signal_handler_fields);
		gbc_htab_signal_handler_fields=g_hash_table_new((GHashFunc)g_string_hash,
								(GCompareFunc)g_string_equal);
		for (i=0; i<GB_SIGH_N_FIELDS; i++) {
			g_hash_table_insert(gbc_htab_signal_handler_fields,
					    gb_signal_handler_field_info[i].field_name,
					    (gpointer) (i + gbc_signal_handler_field_hash_offset));
		}
	}
	
	g_assert(!gbc_htab_dups_ref);
	gbc_htab_dups_ref=g_hash_table_new((GHashFunc)g_string_hash,
					     (GCompareFunc)g_string_equal);
}


void
gbc_htab_free_key	(gpointer	key,
			 gpointer	value,
			 gpointer	user_data)
{
	g_free(key);
}


void
gbc_htab_destroy	(void)
{
	g_hash_table_foreach(gbc_htab_dups_ref, gbc_htab_free_key, NULL);
	g_hash_table_destroy(gbc_htab_dups_ref);
	gbc_htab_dups_ref=NULL;
}


gint
yyfetch	(gint	expected_token)
{
	register int	yytoken;
	
	g_assert(expected_token!=TOKEN_ERROR);
	
	if (!yyin)
		if (expected_token==TOKEN_EOF)
			return TOKEN_EOF;
		else
			return TOKEN_ERROR;
	
	yytoken=yylex();
	
	if (yytoken<=0) {
		yyin=NULL;
		if (expected_token!=TOKEN_EOF) {
			gbc_unexp_token(TOKEN_EOF, expected_token);
			return TOKEN_ERROR;
		} else
			return TOKEN_EOF;
	}
	
	if (yytoken!=expected_token && expected_token!=TOKEN_ANY && expected_token!=TOKEN_EOF) {
		gbc_unexp_token(yytoken, expected_token);
		if (yytoken==TOKEN_STRING)
			g_free(yylval.String);
		return TOKEN_ERROR;
	}
	
	return yytoken;
}


void
gbc_writef	(FILE		*file,
		 const gchar	*format,
		 ...)
{
	static	 gchar		*buffer=NULL;
	static	 guint		buffer_len=DFL_BUFFER_SIZE;
		 va_list	args;
	register gchar		*buf2;
	register guint		i;
	
	g_assert(format);
	
	
	va_start(args, format);
	do {
		register guint		len;
		
		if (buffer==NULL) {
			buffer=g_new(gchar, buffer_len+1);
		}
		len=vsnprintf(buffer, buffer_len, format, args);
		if (len==buffer_len) {
			buffer_len+=DFL_BUFFER_SIZE;
			g_free(buffer);
			buffer=NULL;
		}
	} while (!buffer);
	va_end(args);
	
	buf2=strchr(buffer, '\177');
	if (buf2)
		*(buf2++)=0;
	
	i=0;
	while (i<gbc_indent_level*strlen(GBC_INDENT_STRING))
		i+=fprintf(file, "%s", GBC_INDENT_STRING);
	i=0;
	i+=fprintf(file, "%s", buffer);
	
	if (buf2) {
		do {
			i+=fprintf(file, " ");
		} while (i<GBC_FILLUP_LENGTH);
	}
	i+=fprintf(file, "%s\n", buf2 ? buf2 : "");
	
	fflush(file);
}


void
gbc_write_widget_data	(FILE		*f_out,
			 gb_wdat_base_S	*widget_data,
			 gboolean	skip_defaults,
			 gboolean	recurse)
{
	register GList	*list;
	
	g_assert(GB_IS_WIDDAT(widget_data));
	if (GUBI_DATA(widget_data)->category & WID_CAT_AUXILLARY) {
		g_assert(widget_data->parent=NULL);
		g_assert(widget_data->next=NULL);
	}
		
	
	list=GUBI_DATA(widget_data)->children;
	
	
	/* write widget header
	*/
	if ((!GUBI_DATA(widget_data)->category&WID_CAT_GENERIC) ||
	    GUBI_DATA(widget_data)->link_refs)
		gbc_writef(f_out, "GB_WIDGET_%s \"%s\" {",
			to_UPCASE(GUBI_DATA(widget_data)->struct_info->struct_name),
			widget_data_get_symbol_name(widget_data));
	else
		gbc_writef(f_out, "GB_WIDGET_%s \"\" {",
			to_UPCASE(GUBI_DATA(widget_data)->struct_info->struct_name));
	gbc_indent_level++;
	
	
	/* save fields
	*/
	gbc_write_generic_fields(f_out, GB_CAST(any, widget_data), skip_defaults);
	
	
	/* save linkage information
	*/
	gbc_write_linkage(f_out, widget_data, skip_defaults);
	
	
	/* save handler stack
	*/
	if (widget_data->signal_handler_stack)
		gbc_write_signal_handler_stack(f_out, widget_data);
	
	
	/* save children
	*/
	if (recurse) while (list) {
		register gb_wdat_base_S	*child_data=list->data;
		
		g_assert(GB_IS_WIDDAT(child_data));
		
		gbc_write_widget_data(f_out, child_data, skip_defaults, recurse);
		
		list=list->next;
	}
	
	
	/* write widget footer
	*/
	gbc_indent_level--;
	gbc_writef(f_out, "}");
}


void
gbc_write_linkage	(FILE		*f_out,
			 gb_wdat_base_S	*widget_data,
			 gboolean	skip_defaults)
{
	g_assert(GB_IS_WIDDAT(widget_data));
	
	if (widget_data_check_linkage(widget_data))
		return;
	
	
	/* write linkage header
	*/
	gbc_writef(f_out, "GB_%s {",
		to_UPCASE(gsi_struct_info(widget_data->linkage->type)->struct_name));
	gbc_indent_level++;
	
	
	/* save linkage fields
	*/
	gbc_write_generic_fields(f_out, widget_data->linkage, skip_defaults);
	
	
	/* write linkage footer
	*/
	gbc_indent_level--;
	gbc_writef(f_out, "}");
}


void
gbc_write_signal_handler_stack	(FILE		*f_out,
				 gb_wdat_base_S	*widget_data)
{
	register gb_signal_handler_S	*handler;
	register guint			count;
	
	g_assert(GB_IS_WIDDAT(widget_data));
	
	handler=widget_data->signal_handler_stack;
	count=0;
	
	if (handler) while (handler->signal_name) {
		gbc_writef(f_out, "%s {", gb_signal_handler_struct_name);
		gbc_indent_level++;
		gbc_writef(f_out, "%s\177%d",
			gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].field_name,
			handler->connect_options);
		gbc_writef(f_out, "%s\177\"%s\"",
			gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].field_name,
			handler->signal_name);
		gbc_writef(f_out, "%s\177\"%s\"",
			gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].field_name,
			(char*)handler->handler_func);
		gbc_writef(f_out, "%s\177\"%s\"",
			gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].field_name,
			(char*)handler->data.func_data);
		gbc_indent_level--;
		gbc_writef(f_out, "}");
		count++;
		handler++;
	}
}


void
gbc_write_generic_fields(FILE			*f_out,
			 gb_any_S		*Struct,
			 gboolean		skip_defaults)
{
	register GList				*list, *free_list;
	register const gsi_struct_info_S	*struct_info;
	
	g_assert(Struct);
	struct_info=gsi_struct_info(Struct->type);
	g_assert(struct_info);
	
	free_list=list=gsi_struct_field_list(struct_info);
	while (list) {
		register const	gsi_field_info_S	*field_info;
		
		field_info=list->data;
		
		if (!(gsi_field_has_default(Struct, field_info)&&skip_defaults)) {
			register const	gchar		*string;
			register gsi_base_S		*link_p;
			
			switch (field_info->field_type) {
			case	GSI_FIELD_BOOLEAN:
				gbc_writef(f_out, "%s\177%u",
					field_info->field_name,
					gsi_field_get_value(Struct, field_info).v_boolean);
				break;
			
			case	GSI_FIELD_BITS:
			case	GSI_FIELD_ENUM:
				gbc_writef(f_out, "%s\177%u",
					field_info->field_name,
					gsi_field_get_value(Struct, field_info).v_bits);
				break;
			
			case	GSI_FIELD_STRING:
				string=gsi_field_get_value_as_string(Struct, field_info);
				gbc_writef(f_out, "%s\177%s",
					field_info->field_name,
					string[0]==0 ? "\"\"" : string);
				break;
			
			case	GSI_FIELD_STRUCT_LINK:
				link_p=gsi_field_get_value(Struct, field_info).v_struct_link;
				if (link_p) {
					g_assert(GB_IS_WIDDAT(link_p));
					string=widget_data_get_symbol_name(GB_wCAST(base, link_p));
				} else
					string="";
				gbc_writef(f_out, "%s\177\"%s\"",
					field_info->field_name,
					string);
				break;
			
			case	GSI_FIELD_POINTER:
				break;
			
			default:
				string=gsi_field_get_value_as_string(Struct, field_info);
				gbc_writef(f_out, "%s\177%s",
					field_info->field_name,
					string);
				break;
			}
		}
		
		list=list->next;
	}
	g_list_free(free_list);
}


void
gbc_write_window(FILE			*f_out,
		 gchar			*filename,
		 gb_wdat_window_S	*window_data,
		 gboolean		skip_defaults)
{
	g_assert(f_out);
	g_assert(filename);
	g_assert(GB_IS_WIDDAT_WINDOW(window_data));
	
	
	
	/* prepare global vars
	*/
	gbc_name=filename;
	gbc_indent_level=0;
	
	
	/* save window's widget tree
	*/
	gbc_writef(f_out, "\n\n/* widget tree for %s\n*/", to_DownCase(PRGNAME_SHORT));
	gbc_write_widget_data(f_out, GB_wCAST(base, window_data), skip_defaults, TRUE);
}


guint
gbc_read_trees	(FILE			*f_in,
		 gchar			*filename,
		 guint			max_errors)
{
	register int		yytoken;
	register tree_S		*tree;
	register guint		i;
	
	
	/* prepare global variables
	*/
	for (i=strlen(filename); i>0; i--)
		if (filename[i-1]=='/')
			break;
	if (i>0 && i<strlen(filename))
		gbc_name=&filename[i];
	else
		gbc_name=filename;
	yyin=NULL;
	yyrestart(f_in);
	yyline=1;
	gbc_pa_errors=0;
	gbc_pa_errors_max=max_errors;
	
	gbc_hash_table_init();
	
	/* scan for new trees
	*/
	yytoken=TOKEN_EOF;
	while (!gbc_should_abort()) {
		
		yytoken=yyfetch(TOKEN_EOF);

		switch (yytoken) {
		register char			*string;
		register gb_struct_type_E	struct_type;
		register gb_wdat_window_S	*window_data;

		case	TOKEN_WIDGET:
			struct_type=(gb_struct_type_E)g_hash_table_lookup(gbc_htab_struct_type,
							(gpointer)to_downcase(yylval.String));
			if (!GB_TYPE_IS_WIDDAT_WINDOW(struct_type)) {
				gbc_unexp_widget(yylval.String, NULL);
				g_free(yylval.String);
				break;
			}
			g_free(yylval.String);
			
			if (yyfetch(TOKEN_STRING)==TOKEN_ERROR)
				break;
			string=symbol_name_make_valid(yylval.String);
			
			if (yyfetch('{')==TOKEN_ERROR) {
				g_free(string);
				break;
			}
			tree=tree_new(TRUE, FALSE);
			
			window_data=GB_wCAST(window, widget_data_new(struct_type, string, FALSE));
			
			
			/* we need to check for duplicate identifiers
			*/
			if (string && string[0]!=0)
				HANDLE_DUPS(string, GB_wCAST(base, window_data));
			
			g_free(string);
			tree_insert_widget_data_R(tree, NULL, GB_wCAST(base, window_data), 0);
			tree_set_current_widget_data(tree, GB_wCAST(base, window_data));
			
			gbc_p_structure_R(tree, GB_CAST(any, window_data));
			
			tree_show(tree);
			break;
			
		default:
			gbc_unexp_token(yytoken, TOKEN_TREE);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	gbc_htab_destroy();
	
	return gbc_pa_errors;
}


void
gbc_p_structure_R	(tree_S		*tree,
			 gb_any_S	*Struct)
{
	register int		yytoken;
	register gboolean	scan_children_only;
	
	scan_children_only=FALSE;
	while ((yytoken=yyfetch(TOKEN_ANY))!='}') {
		
		if (gbc_should_abort())
			break;
		
		
		/* if we have reached the end of a widget
		 * definition, we only accept new widgets
		 * (yytoken==TOKEN_WIDGET) or
		 * the footer (yytoken=='}')
		*/
		if (scan_children_only &&
		    yytoken!='}' &&
		    yytoken!=TOKEN_WIDGET &&
		    yytoken!=TOKEN_ERROR) {
			gbc_unexp_token(yytoken, TOKEN_ANY);
			continue;
		}
		
		switch (yytoken) {
		register gb_struct_type_E	struct_type;
		register gchar			*string;
		register gb_wdat_base_S		*widget_data;
		register gb_wdat_base_S		*child_data;
		
		case	TOKEN_FIELD:
			string=yylval.String;
			gbc_p_generic_field(Struct, string);
			g_free(string);
			break;

		case	TOKEN_STRUCT:
			if (!GB_TYPE_IS_WIDDAT(Struct->type)) {
				gbc_unexp_token(TOKEN_STRUCT, TOKEN_FIELD);
				break;
			}
			
			widget_data=GB_wCAST(base, Struct);
				
			struct_type=(gb_struct_type_E)
				g_hash_table_lookup(gbc_htab_struct_type,
						(gpointer)to_downcase(yylval.String));
			g_free(yylval.String);
			
			switch (struct_type) {
			
			case	GB_LINKAGE_PACK:
			case	GB_LINKAGE_BOX:
			case	GB_LINKAGE_DIALOG:
			case	GB_LINKAGE_TABLE:
			case	GB_LINKAGE_NOTEBOOK:
				if (widget_data->linkage) {
					gbc_unexp_token(TOKEN_STRUCT, TOKEN_FIELD);
					break;
				}
				widget_data_new_linkage(widget_data);
				if (!widget_data->linkage) {
					gbc_unexp_token(TOKEN_STRUCT, TOKEN_FIELD);
					break;
				}
				
				if (yyfetch('{')==TOKEN_ERROR) {
					g_free(widget_data->linkage);
					widget_data->linkage=NULL;
					break;
				}
				gbc_p_structure_R(tree, widget_data->linkage);
				break;
			
			default:
				gbc_unexp_token(yytoken, TOKEN_FIELD);
			case	TOKEN_ERROR:
			case	TOKEN_EOF:
				break;
			}
			break;

		case	TOKEN_WIDGET:
			if (!GB_TYPE_IS_WIDDAT(Struct->type)) {
				gbc_unexp_token(TOKEN_WIDGET, TOKEN_FIELD);
				break;
			}
			
			struct_type=(gb_struct_type_E)g_hash_table_lookup(gbc_htab_struct_type,
							(gpointer)to_downcase(yylval.String));
			
			if (!GB_TYPE_IS_WIDDAT(struct_type)) {
				gbc_error("structure is no widget: <%s>", yylval.String);
				g_free(yylval.String);
				break;
			}
			
			
			/* we don't accept windows as children
			*/
			if (GB_TYPE_IS_WIDDAT_WINDOW(struct_type)) {
				gbc_unexp_widget(yylval.String, "window widget");
				g_free(yylval.String);
				break;
			}
			
			
			/* we need to be a container if we get a widget
			*/
			if (!GB_TYPE_IS_WIDDAT_CONTAINER(Struct->type)) {
				gbc_unexp_widget(yylval.String, "parent is no container");
				g_free(yylval.String);
				break;
			}
			
			
			/* we need to have space for our widget
			*/
			if (HAS_MAX_CHILD_COUNT(GB_wCAST(base, Struct))) {
				gbc_unexp_widget(yylval.String, "maximum child count reached in parent");
				g_free(yylval.String);
				break;
			}
			
			g_free(yylval.String);
			
			
			if (yyfetch(TOKEN_STRING)==TOKEN_ERROR)
				break;
			string=symbol_name_make_valid(yylval.String);
			if (yyfetch('{')==TOKEN_ERROR) {
				g_free(string);
				break;
			}
			child_data=widget_data_new(struct_type, string, FALSE);
			
			
			/* we need to check for duplicate identifiers
			*/
			if (string && string[0]!=0)
				HANDLE_DUPS(string, child_data);
			
			g_free(string);
			
			tree_insert_widget_data_R(tree, GB_wCAST(base, Struct), child_data, ~0);
			
			gbc_p_structure_R(tree, GB_CAST(any, child_data));
			
			scan_children_only=TRUE;
			break;

		default:
			gbc_unexp_token(yytoken, TOKEN_FIELD);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	
	/* check for linkage specific stuff
	*/
	if (GB_TYPE_IS_WIDDAT(Struct->type)) {
		register gb_wdat_base_S	*widget_data=GB_wCAST(base, Struct);
		
		if (widget_data_check_linkage(widget_data)) {
			g_free(widget_data->linkage);
			widget_data->linkage=NULL;
		}
		
		widget_data_children_check_linkage(widget_data);
 	}
}


void
gbc_p_generic_field	(gb_any_S	*Struct,
			 gchar		*field_name)
{
	register const gsi_field_info_S	*field_info;
	register gsi_value_U		value;
	register gboolean		set_value;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(field_name);
	
	field_info=NULL;
	
	
	/* lookup the field type
	*/
	field_info=g_hash_table_lookup(gbc_htab_array_struct_field[Struct->type],
				       to_downcase(field_name));
	
	
	/* check for extensions on WIDDAT,
	 * warn if we don't know the field type
	*/
	if (!field_info) {
		if (GB_TYPE_IS_WIDDAT(Struct->type) &&
		    strcmp(gb_signal_handler_struct_name, to_downcase(field_name))==0) {
				gbc_p_signal_handler(GB_wCAST(base, Struct));
				return;
		} else {
			gbc_warn("invalid field identifier (foreign?): `%s'", field_name);
			yyfetch(TOKEN_EOF);
			return;
		}
	}
	
	
	/* scan the field's value
	*/
	set_value=FALSE;
	yylval.String=NULL;
	switch (field_info->field_type) {
	register guint	yytoken;
	register gb_wdat_base_S	*widget_data;
	register gb_wdat_base_S	*link_data;
	
	case	GSI_FIELD_FLOAT:
		yytoken=yyfetch(TOKEN_ANY);
		switch (yytoken) {
		
		case	TOKEN_LONG:
			yylval.Double=yylval.Long;
		case	TOKEN_DOUBLE:
			value.v_float=yylval.Double;
			set_value=TRUE;
			break;
		
		case	TOKEN_ERROR:
			break;
			
		default:
			gbc_unexp_token(yytoken, TOKEN_DOUBLE);
			break;
		}
		yylval.String=NULL;
		break;
	
	case	GSI_FIELD_BITS:
		if (yyfetch(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.v_bits=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
		
	case	GSI_FIELD_INT:
		if (yyfetch(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.v_int=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
		
	case	GSI_FIELD_ENUM:
		if (yyfetch(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.v_enum=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		if (yyfetch(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.v_boolean=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
	
	case	GSI_FIELD_STRING:
		if (yyfetch(TOKEN_STRING)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.v_string=yylval.String;
		set_value=TRUE;
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		if (yyfetch(TOKEN_STRING)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		symbol_name_make_valid(yylval.String);
		
		value=gsi_field_get_value(Struct, field_info);
		
		widget_data=GB_wCAST(base, Struct);
		
		g_assert(GB_IS_WIDDAT(widget_data));
		
		link_data=g_hash_table_lookup(gbc_htab_dups_ref, yylval.String);
		if (!link_data)
			link_data=widget_data_lookup(yylval.String);
		
		if (link_data) {
			if (GUBI_DATA(widget_data)->tree!=GUBI_DATA(link_data)->tree) {
				gbc_warn("out of tree reference `%s' for `%s'",
					yylval.String,
					widget_data_get_symbol_name(widget_data));
			} else if (link_data->type!=widget_data->type) {
				gbc_warn("invalid type in reference `%s' for `%s'",
					yylval.String,
					widget_data_get_symbol_name(widget_data));
			} else {
				value.v_struct_link=GB_CAST(any, link_data);
				set_value=TRUE;
			}
		} else {
			if (yylval.String[0]!=0) {
				gbc_warn("unknown reference `%s' for `%s'",
					yylval.String,
					widget_data_get_symbol_name(widget_data));
			}
		}
		break;
		
	case	GSI_FIELD_POINTER:
	default:
		yylval.String=NULL;
		g_assert_not_reached();
		break;
	}
	
	
	/* assign the field's value
	*/
	if (set_value)
		gsi_field_set_value(Struct, field_info, value);
	
	g_free(yylval.String);
}


void
gbc_p_signal_handler	(gb_wdat_base_S	*widget_data)
{
	register int		yytoken;
	register int		connect_options;
	register gboolean	connect_options_valid;
	register gchar		*func_name;
	register gchar		*signal_name;
	register gchar		*data_name;
	register gchar		*warn_string;
	
	
	g_assert(GB_IS_WIDDAT(widget_data));
	
	
	if (yyfetch('{')==TOKEN_ERROR)
		return;
	
	connect_options=0;
	connect_options_valid=FALSE;
	func_name=NULL;
	signal_name=NULL;
	data_name=NULL;
	
	while ((yytoken=yyfetch(TOKEN_ANY))!='}') {
		
		if (gbc_should_abort())
			break;
		
		switch (yytoken) {
			register gchar	*string;
			register gint	h;
		
		case	TOKEN_FIELD:
			string=yylval.String;
			h=(gint) g_hash_table_lookup(gbc_htab_signal_handler_fields,
								to_downcase(string));
				
			switch (h - gbc_signal_handler_field_hash_offset) {
				
			case	GB_SIGH_SIGNAL_NAME_INDX:
				if (yyfetch(TOKEN_STRING)==TOKEN_ERROR)
					break;
				g_free(signal_name);
				signal_name=yylval.String;
				break;
				
			case	GB_SIGH_FUNC_NAME_INDX:
				if (yyfetch(TOKEN_STRING)==TOKEN_ERROR)
					break;
				g_free(func_name);
				func_name=symbol_name_make_valid(yylval.String);
				break;
			
			case	GB_SIGH_DATA_NAME_INDX:
				if (yyfetch(TOKEN_STRING)==TOKEN_ERROR)
					break;
				g_free(data_name);
				data_name=yylval.String;
				break;
			
			case	GB_SIGH_CONNECT_OPTIONS_INDX:
				if (yyfetch(TOKEN_LONG)==TOKEN_ERROR)
					break;
				connect_options= yylval.Long &
							( GB_CONNECT |
							  GB_CONNECT_OBJECT |
							  GB_CONNECT_AFTER |
							  GB_CONNECT_DATA_POINTER );
				connect_options_valid= connect_options == yylval.Long;
				break;
			
			default:
				gbc_warn("invalid field identifier: `%s'", string);
				yyfetch(TOKEN_EOF);
				break;
			}
			g_free(string);
			break;
		
		default:
			gbc_unexp_token(yytoken, TOKEN_FIELD);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	warn_string=NULL;
	if (!signal_name)
		warn_string=gb_signal_handler_field_info[GB_SIGH_SIGNAL_NAME_INDX].field_name;
	else if (!func_name)
		warn_string=gb_signal_handler_field_info[GB_SIGH_FUNC_NAME_INDX].field_name;
	else if (!data_name)
		warn_string=gb_signal_handler_field_info[GB_SIGH_DATA_NAME_INDX].field_name;
	else if (!connect_options_valid)
		warn_string=gb_signal_handler_field_info[GB_SIGH_CONNECT_OPTIONS_INDX].field_name;
	
	if (warn_string)
		gbc_warn("%s: missing %s specification", gb_signal_handler_struct_name, warn_string);
	else
		widget_data_add_handler(widget_data, signal_name, func_name, data_name, connect_options);
		
	g_free(func_name);
	g_free(signal_name);
	g_free(data_name);
}
