/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* structures.h: functions for handling of gb_*_S structures
 * RCS_ID("$Id: structures.h,v 1.5 1997/05/09 16:29:43 tim Exp $")
*/


#ifndef		__structures_h__
#define		__structures_h__

#ifdef		__cplusplus
  extern	"C"	{
#endif

#include	"config.h"
#include	"gbconf.h"



/* --- typedefs --- */
typedef	void		(*fpointer)		(void);
typedef	void		(*ipointer_field_func)	(GtkWidget*, gpointer);
typedef	const char	*(*enum_field_func)	(const guint32, gboolean);
typedef	struct		struct_info_S		struct_info_S;
typedef	union		field_value_U		field_value_U;
typedef	struct		field_info_S		field_info_S;

typedef	enum {
	FIELD_FLOAT,
	FIELD_INT,
	FIELD_BITS,
	FIELD_ENUM,
	FIELD_BOOLEAN,
	FIELD_STRING,
	FIELD_POINTER,
	FIELD_WD_LINK,
	FIELD_AUX_LINK
} field_type_E;



/* --- structures --- */
union	field_value_U {
	gfloat			float_val;
	gint			int_val;
	guint32			bit_val;
	guint			enum_val;
	gboolean		bool_val;
	gchar			*string;
	gpointer		link;
};

struct	struct_info_S {
	const	guint32			struct_type;
	const	gchar	*const		struct_name;
	const	guint			field_count;
	const	field_info_S		*fields;
	const	guint			widget_child_count
					/* extend general structure
					 * description to support widgets
					*/;
};

struct	field_info_S {
	const	field_type_E		field_type;
	const	gchar	*const		field_name;
	const	float			minimum
					/* minimum for float_val,
					 * int_val, bit_val or enum_val
					 * fields.
					*/;
	const	float			dfl_val
					/* default for float_val,
					 * int_val, bit_val or enum_val
					 * fields.
					 * contains field_options_E for
					 * FIELD_STRING or FIELD_POINTER.
					*/;
	const	float			maximum
					/* maximum for float_val, int_val,
					 * enum_val or bit_val fields.
					*/;
	const	fpointer		field_func
					/* function to return a string for
					 * enum values or single bits.
					*/;
	const	gint			field_offset
					/* field offset within gb_wdat_*_S
					*/;
};



/* --- prototypes --- */


/* the struct_info() function returns a pointer to the struct_info_S
 * for a specific structure type.
 * returns:
 * a pointer to a constant struct_info_S or NULL.
*/
const	struct_info_S*
	structure_info			(const gb_struct_type_E	struct_type);


/* the field_info() function returns a pointer to the field_info_S for a
 * specific field of a given widget specified by StrInf.
 * the existence of StrInf as well as a field of #field_num within
 * the existence of *StrInf is asserted!
 * returns:
 * a pointer to a constant field_info_S.
*/
const	field_info_S*
	field_info			(const struct_info_S	*StrInf
					 /* *valid* pointer to a
					  * struct_info_S.
					 */,
					 const guint		field_num
					 /* *valid field index for the
					  * corresponding struct_info_S.
					 */);


/* the field_info_by_widget_type() is similar to field_info() except that the
 * StrInf structure is fetched from widget_struct_info(widget_type).
 * returns:
 * a pointer to a constant field_info_S.
*/
const	field_info_S*
	field_info_by_struct_type	(const gb_struct_type_E	struct_type,
					 const guint		field_num);


/* the structure_field_set_default() function resets the specified field
 * in Struct to it's default value.
 * the existence of *Struct and *FldInf is asserted!
*/
void	structure_field_set_default	(gb_any_S		*Struct
					 /* *valid* pointer to a structure
					 */,
					 const field_info_S	*FldInf
					 /* *valid* pointer to a
					  * field_info_S
					 */);


/* the structure_field_has_default() function returns wether the
 * specified field has been changed since the last call to
 * structure_field_set_default().
 * the existence of *Struct and *FldInf is asserted!
*/
gboolean
	structure_field_has_default	(gb_any_S		*Struct,
					 const field_info_S	*FldInf);


/* the structure_field_set_value() function sets the specified field
 * in Struct to value, taking care for it's validity.
 * the existence of *Struct and *FldInf is asserted!
*/
void	structure_field_set_value	(gb_any_S		*Struct,
					 field_value_U		value,
					 const field_info_S	*FldInf);


/* the structure_field_get_value() function returns the value of
 * a specified field in Struct.
 * on empty strings this returns NULL.
 * the existence of *Struct and *FldInf is asserted!
*/
field_value_U
	structure_field_get_value	(gb_any_S		*Struct,
					 const	field_info_S	*FldInf);


/* the structure_field_get_value_str() determines the field value
 * from structure_field_get_value() and converts it into a string.
 * a NULL string is returned as an empty string (that is "").
 * enumaration values get returned as their corresponding constant,
 * as well as boolean values. (a boolean value of 42 will cause this
 * function to return "TRUE").
 * FIXME: unfinished.
 * the existence of *Struct and *FldInf is asserted!
 * returns:
 * a pointer to a constant string.
 * the returned string can only be used until the next call to this function.
*/
const gchar*
	structure_field_get_value_str	(gb_any_S		*Struct,
					 const	field_info_S	*FldInf);


/* the structure_field_bits_count() function just computes the amount
 * of usable bits in a structure field of type FIELD_BITS.
 * the existence of *FldInf and FldInf->type==FIELD_BITS is asserted!
*/
guint	structure_field_bits_count	(const	field_info_S	*FldInf);


/* the structure_field_bit_value() function returns the value
 * representing bit_num. bit_num is the index into usable bits of the
 * specified field (see structure_field_bits_count()).
 * the existence of *FldInf and bit_num is asserted!
*/
guint32	structure_field_bit_value	(const	field_info_S	*FldInf,
					 guint			bit_num);


/* the structure_field_bit_set() function sets the corresponding bit
 * in the specified structure field to TRUE or FALSE.
 * (see structure_field_bits_count()).
 * the existence of *Struct, *FldInf and bit_num is asserted!
*/
void	structure_field_bit_set		(gb_any_S		*Struct,
					 const	field_info_S	*FldInf,
					 guint			bit_num,
					 gboolean		value);


/* the structure_fields_initialize() function sets all fields of Struct
 * to their default values.
 * in contrast to structure_field_set_default() this function doesn't
 * care about any existing values.
 * (e.g. a string pointer pointing to an allocated buffer that needs to be
 * free()d).
 * the existence of *Struct is asserted!
*/
void	structure_fields_initialize	(gb_any_S		*Struct,
					 const struct_info_S	*StrInf);


/* the structure_fields_free() function frees any data that might be
 * allocated by some widget fields and needs to be free()d.
 * the existence of *Struct is asserted!
*/
void	structure_fields_free		(gb_any_S		*Struct,
					 const struct_info_S	*StrInf);


/* the widget_data_base_child_count() function returns the child count of
 * WidDat, that is caused by it's field values. on most widget types this
 * is simply 0, but on e.g. buttons a non NULL string pointer for the button
 * label causes at least a child count of 1.
 * the existence of *WidDat is asserted!
*/
guint	widget_data_base_child_count	(gb_wdat_base_S		*WidDat);


/* the widget_data_link_remove() function removes any occourances of
 * WidDat in LnkDat's link reference list. it also reset's all fields
 * of WidDat that refere to LnkDat to the default value.
 * the existence of *WidDat and *LnkDat is asserted!
*/
void	widget_data_link_remove		(gb_wdat_base_S		*WidDat,
					 gb_wdat_base_S		*LnkDat);






#ifdef		__cplusplus
  }
#endif
#endif		/*__structures_h__*/
