/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: c_source.c,v 1.6 1997/05/09 16:17:57 tim Exp $")


#define		__c_source_c__

#include	"builder.h"
#include	"widdata.h"
#include	"structures.h"
#include	"misc.h"
#include	"defines.h"
#include	<errno.h>


/* --- global variables --- */
const	builder_templ_S	C_builder_Files[]={
/*
{ base		s-postf	tpl-postfix	dynamic	trunc	source_type		}
*/
{ "files",	"doc",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN_DESC	},
{ "gbconf",	"h",	"h.tpl",	FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "gbconf", 	"c",	"c.tpl",	FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "widgets",	"h",	"h.tpl",	TRUE,	TRUE,	GB_SOURCE_MAIN		},
{ "widgets", 	"c",	"c.tpl",	TRUE,	TRUE,	GB_SOURCE_MAIN		},
{ "ufunc", 	"h",	"h.tpl",	TRUE,	FALSE,	GB_SOURCE_MAIN		},
{ "ufunc",	"c",	"c.tpl",	TRUE,	FALSE,	GB_SOURCE_MAIN		},
{ "main", 	"c",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "RCS",	"h",	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
{ "Makefile",	NULL,	".tpl",		FALSE,	FALSE,	GB_SOURCE_MAIN		},
};
const	guint		C_builder_Files_count=sizeof(C_builder_Files)/sizeof(builder_templ_S);



/* --- variable --- */
static	gboolean	Skip_Handler=TRUE;


/* --- static prototypes --- */
static	gboolean	ExpandMacro	(const	gchar	*macro);


/* write @c_widget_data_protos@ or @c_widget_data_decls@
 * as described in templates/macros.doc for a whole widget tree.
*/
static	void	Write_widget_tree	(gboolean	protos
					 /* write prototypes?
					  * contents otherwise.
					 */,
					 gboolean	internal
					 /* write internal prototypes?
					 */);


/* write @c_widget_data_protos@ or @c_widget_data_decls@
 * as described in templates/macros.doc.
*/
static	void	Write_widget_data_R	(gb_wdat_base_S	*WidDat,
					 gboolean	protos,
					 gboolean	internal);


/* write @c_window_data_pointers@
 * as described in templates/macros.doc.
*/
static	void	Write_window_data_pointers	(void);


/* write the handler array of a gb_wdat_base_S
*/
static	void	Write_handler_stack	(gb_wdat_base_S		*WidDat);


/* write the linkage information of a gb_wdat_base_S
*/
static	void	Write_linkage		(gb_wdat_base_S		*WidDat);


/* write the fields of gb_wdat_base_S only
*/
static	void	Write_widget_fields	(gb_wdat_base_S		*WidDat,
					 gboolean		protos,
					 gboolean		internal);


/* write the fields of a gb_wdat_*_S excluding
 * the ones from gb_wdat_base_S
*/
static	void	Write_generic_fields	(gb_any_S		*Struct,
					 const struct_info_S	*StrInf);



/* --- functions --- */
gboolean
ExpandMacro	(const	char	*macro)
{
	Char_Buffer[0]=0;
	
	if (strcmp(macro,				"c_window_data_pointers")==0)
		Write_window_data_pointers();
	else if (strcmp(macro,				"c_widget_data_pointer_protos")==0)
		Write_widget_tree(TRUE , FALSE);
	else if (strcmp(macro,				"c_widget_data_iprotos")==0)
		Write_widget_tree(TRUE , TRUE);
	else if (strcmp(macro,				"c_widget_data_decls")==0)
		Write_widget_tree(FALSE, FALSE);
	else
		return FALSE;

	FLUSH_TARGET();
	
	return TRUE;
}


gint
Write_c_source	(const	gchar			*target_dir,
		 const	gboolean		dynamic_only,
		 const	gboolean		skip_handler,
		 const	gboolean		trunc_override,
		 const	guint16			source_mask)
{
	register const builder_templ_S	*source_set;
	register guint			i;
	register gint			error;
	
	g_assert(target_dir);
	
	source_set=C_builder_Files;
	
	error=FALSE;
	
	Skip_Handler=skip_handler;
	
	for (i=0; i<C_builder_Files_count; i++) {
		register gchar		*err_name;
		
		g_assert(source_set[i].base);
		g_assert(source_set[i].template_postfix);
		g_assert(source_set[i].dynamic || !source_set[i].trunc);
		
		
		if ( source_set[i].source_type&source_mask &&
		     (source_set[i].dynamic || !dynamic_only) ) {
			sprintf(Char_Buffer, "%s%s", source_set[i].base, source_set[i].template_postfix);
			err_name=NULL;
		
			err_name=builder_init(DIR_TEMPLATE,
						Char_Buffer,
						target_dir,
						source_set[i].base,
						source_set[i].postfix,
						dynamic_only,
						source_set[i].dynamic,
						source_set[i].trunc | trunc_override);
			if (err_name) {
				error=TRUE;
				g_warning("%s\n%s:\n%s:\n%s",
					FUNCNAME,
					err_name,
					g_strerror(EIO),
					g_strerror(errno));
				g_free(err_name);
			} else
				builder_expand_file(ExpandMacro);
			builder_reset();
		}
	}
	
	return error ? EIO : FALSE;
}


void
Write_window_data_pointers	(void)
{
	register GList	*list;
	register guint	i;
	
	list=UD_tree_list;
	
	i=0;
	while (list) {
		register tree_S			*tree;
		register gb_wdat_window_S	*WinDat;
		
		g_assert((tree=list->data));
		g_assert(tree->widget_data_list);
		WinDat=tree->widget_data_list->data;
		g_assert(GB_IS_WINDOW_WIDGET(WinDat));
		
		sprintf(Char_Buffer, "\t%s&%s%s%s%s,\t/* \"%s\" */\n",
			WinDat->type==GB_WIDGET_WINDOW ? "" : "GB_wCAST(window,\n\t  ",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(GB_wCAST(base, WinDat)),
			WIDGET_DATA_NAME_POSTFIX,
			WinDat->type==GB_WIDGET_WINDOW ? "" : ")",
			WinDat->title);
		FLUSH_TARGET();
		
		i++;
		list=list->next;
	}
	
	if (i==0) {
		sprintf(Char_Buffer, "\tNULL, /* ANSI C forbids empty initializer braces */\n");
		FLUSH_TARGET();
	}
	
	Char_Buffer[0]=0;
}


void
Write_widget_tree	(gboolean	protos,
			 gboolean	internal)
{
	register GList		*tree_list;
	
	
	/* for each tree write it's widgets
	*/
	tree_list=UD_tree_list;
	while (tree_list) {
		register gb_wdat_base_S	*WinDat;
		register tree_S		*tree;
		
		g_assert((tree=tree_list->data));
		g_assert(tree->widget_data_list);
		WinDat=tree->widget_data_list->data;
		g_assert(GB_IS_WINDOW_WIDGET(WinDat));
		
		Write_widget_data_R(WinDat, protos, internal);
		
		tree_list=tree_list->next;
	}
}


void
Write_widget_data_R	(gb_wdat_base_S	*WidDat,
			 gboolean	protos,
			 gboolean	internal)
{
	register GList	*list;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	
	/* write prototype or structure
	*/
	Write_widget_fields(WidDat, protos, internal);
	
	
	/* write children
	*/
	list=GUBI_DATA(WidDat)->children;
	while (list) {
		register gb_wdat_base_S	*ChildDat;
		
		ChildDat=list->data;
		g_assert(GB_IS_WIDGET(ChildDat));
		
		Write_widget_data_R(ChildDat, protos, internal);
		
		list=list->next;
	}
}


void
Write_widget_fields	(gb_wdat_base_S	*WidDat,
			 gboolean	protos,
			 gboolean	internal)
{
	g_assert(GB_IS_WIDGET(WidDat));
	
	
	/* if we should write the gb_wdat_*_S contents
	 * let's first write auxillary structures
	*/
	if (!protos) {
		Write_handler_stack(WidDat);
		Write_linkage(WidDat);
	}
	
	
	/* write prototype part of a c structure
	 * or it's begining, if the fields follow
	*/
	if (protos) {
		if (internal)
			sprintf(Char_Buffer, "static\tgb_wdat_%s_S\t%s%s%s;\n",
				structure_info(WidDat->type)->struct_name,
				WIDGET_DATA_NAME_PREFIX,
				widget_data_symbol_name_get(WidDat),
				WIDGET_DATA_NAME_POSTFIX);
		else
			sprintf(Char_Buffer, "extern\tgb_wdat_%s_S\t*const\t%s;\n",
				structure_info(WidDat->type)->struct_name,
				widget_data_symbol_name_get(WidDat));
	} else
		sprintf(Char_Buffer, "static\tgb_wdat_%s_S\t%s%s%s = {\n",
			structure_info(WidDat->type)->struct_name,
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			WIDGET_DATA_NAME_POSTFIX);
	FLUSH_TARGET();
	Char_Buffer[0]=0;

	if (protos)
		return;
	
	
	/* write the non-generic base field values
	*/
	sprintf(Char_Buffer, "\tGB_WIDGET_%s,\n\t\t\t/* widget_type */\n",
		to_UPCASE(structure_info(WidDat->type)->struct_name));
	FLUSH_TARGET();
	if (WidDat->parent)
		sprintf(Char_Buffer, "\tGB_CAST(wdat_base, &%s%s%s),\n\t\t\t/* parent */\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat->parent),
			WIDGET_DATA_NAME_POSTFIX);
	else
		sprintf(Char_Buffer, "\tNULL,\t\t/* parent */\n");
	FLUSH_TARGET();
	
	sprintf(Char_Buffer, "\tFALSE,\t\t/* free_clone */\n");
	FLUSH_TARGET();
	sprintf(Char_Buffer, "\tNULL,\t\t/* clone */\n");
	FLUSH_TARGET();
	
	if (WidDat->next)
		sprintf(Char_Buffer, "\tGB_CAST(wdat_base, &%s%s%s),\n\t\t\t/* next */\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat->next),
			WIDGET_DATA_NAME_POSTFIX);
	else
		sprintf(Char_Buffer, "\tNULL,\t\t/* next */\n");
	FLUSH_TARGET();
	
	sprintf(Char_Buffer, "\tNULL,\t\t/* widget */\n");
	FLUSH_TARGET();
	
	if (WidDat->handler_stack && !Skip_Handler)
		sprintf(Char_Buffer, "\t%s%s%s,\t\t/* handler_stack */\n",
			HANDLER_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			HANDLER_NAME_POSTFIX);
	else
		sprintf(Char_Buffer, "\tNULL,\t\t/* handler_stack */\n");
	FLUSH_TARGET();
	
	if (!widget_data_linkage_check(WidDat))
		sprintf(Char_Buffer, "\tGB_CAST(any, &%s%s%s),\n\t\t\t/* linkage */\n",
			LINKAGE_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			LINKAGE_NAME_POSTFIX);
	else
		sprintf(Char_Buffer, "\tNULL,\t\t/* linkage */\n");
	FLUSH_TARGET();
	
	
	/* generic fields
	*/
	Write_generic_fields(GB_CAST(any, WidDat), structure_info(GB_WIDGET_BASE));
	
	
	if (GB_TYPE_IS_CONTAINER_WIDGET(WidDat->type))
		Write_generic_fields(GB_CAST(any, WidDat), structure_info(GB_WIDGET_CONTAINER));

    	Write_generic_fields(GB_CAST(any, WidDat), structure_info(WidDat->type));
	
	sprintf(Char_Buffer, "};\n");
	FLUSH_TARGET();
	sprintf(Char_Buffer, "gb_wdat_%s_S\t*const %s\t= ",
		structure_info(WidDat->type)->struct_name,
		widget_data_symbol_name_get(WidDat));
	FLUSH_TARGET();
	sprintf(Char_Buffer, "&%s%s%s;\n",
			WIDGET_DATA_NAME_PREFIX,
			widget_data_symbol_name_get(WidDat),
			WIDGET_DATA_NAME_POSTFIX);
	FLUSH_TARGET();

	Char_Buffer[0]=0;
}


void
Write_generic_fields	(gb_any_S		*Struct,
			 const struct_info_S	*StrInf)
{
	register guint	i;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(StrInf);
	
	for (i=0; i<StrInf->field_count; i++) {
		register const	field_info_S	*FldInf;
		register const	gchar		*string;
		
		FldInf=field_info(StrInf, i);
		
		string=structure_field_get_value_str(Struct, FldInf);
		
		switch (FldInf->field_type) {
		
		case	FIELD_STRING:
			sprintf(Char_Buffer, "\t%s%s%s,%s\t/* %s */\n",
				string ? "\""	: "",
				string ? string	: "NULL",
				string ? "\""	: "",
				(string && strlen(string)>7) ? "\n\t\t" : "\t",
				FldInf->field_name);
			break;
		
		case	FIELD_WD_LINK:
			sprintf(Char_Buffer, "\t%s%s%s%s,%s\t/* %s */\n",
				string[0] ? "&" : "",
				string[0] ? WIDGET_DATA_NAME_PREFIX : "",
				string[0] ? string : "NULL",
				string[0] ? WIDGET_DATA_NAME_POSTFIX : "",
				strlen(string)>7 ? "\n\t\t" : "\t",
				FldInf->field_name);
			break;
		
		default:
			sprintf(Char_Buffer, "\t%s,%s\t/* %s */\n",
				string,
				strlen(string)>7 ? "\n\t\t" : "\t",
				FldInf->field_name);
			break;
		}
		
		FLUSH_TARGET();
	}
}


void
Write_handler_stack	(gb_wdat_base_S	*WidDat)
{
	register gb_handler_S	*h_next;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	if (Skip_Handler)
		return;
	
	
	/* there is nothing to write,
	 * if the list is empty
	*/
	if (!(h_next=WidDat->handler_stack))
		return;
	
	
	/* write prototype part of an array
	 * or it's begining, if the elements follow
	*/
	sprintf(Char_Buffer, "gb_%s_S\t%s%s%s[] = {\n",
		"handler",
		HANDLER_NAME_PREFIX,
		widget_data_symbol_name_get(WidDat),
		HANDLER_NAME_POSTFIX);
	FLUSH_TARGET();
	Char_Buffer[0]=0;

	
	/* write the handler values
	*/
	while (h_next->handler_func) {
		sprintf(Char_Buffer, "\t{\t%u,\t/* connect_options */\n",
			h_next->connect_options);
		FLUSH_TARGET();
		sprintf(Char_Buffer, "\t\t0,\t/* handler_id */\n");
		FLUSH_TARGET();
		sprintf(Char_Buffer, "\t\tGTK_SIGNAL_FUNC(%s),\t/* signal handler */\n",
			(char*)h_next->handler_func);
		FLUSH_TARGET();
		sprintf(Char_Buffer, "\t\t\"%s\",\t/* signal_name */\n",
			h_next->signal_name);
		FLUSH_TARGET();
		sprintf(Char_Buffer, "\t\t{ %s },\t/* function/object data */\n",
			(char*)h_next->data.func_data);
		FLUSH_TARGET();
		sprintf(Char_Buffer, "\t},\n");
		FLUSH_TARGET();
		h_next++;
	}
	sprintf(Char_Buffer, "\t{\t0, 0, NULL, NULL, { NULL }\t}\n");
	FLUSH_TARGET();
	sprintf(Char_Buffer, "};\n");
	FLUSH_TARGET();

	Char_Buffer[0]=0;
}


void
Write_linkage		(gb_wdat_base_S		*WidDat)
{
	register const struct_info_S	*LnkInf;
	
	g_assert(GB_IS_WIDGET(WidDat));
	
	if (widget_data_linkage_check(WidDat))
		return;
	
	g_assert((LnkInf=structure_info(WidDat->linkage->type)));
	
	
	/* write linkage structure begining
	*/
	sprintf(Char_Buffer, "gb_%s_S\t%s%s%s = {\n",
		LnkInf->struct_name,
		LINKAGE_NAME_PREFIX,
		widget_data_symbol_name_get(WidDat),
		LINKAGE_NAME_POSTFIX);
	FLUSH_TARGET();
	
	sprintf(Char_Buffer, "\tGB_%s,\n\t\t\t/* type */\n",
		to_UPCASE(LnkInf->struct_name));
	FLUSH_TARGET();
	
	/* write the linkage values
	*/
	Char_Buffer[0]=0;
	Write_generic_fields(WidDat->linkage, LnkInf);
	
	sprintf(Char_Buffer, "};\n");
	FLUSH_TARGET();

	Char_Buffer[0]=0;
}
