/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"gleeditor.h"
#include	"gleprivate.h"
#include	"glercargs.h"
#include	"glehandler.h"
#include	"gtkcluehunter.h"
#include	<string.h>
#include	<stdio.h>
#include	<stdlib.h>



/* --- macros --- */
#define	GOBJECT_SET_EDITOR(go, ed)	gle_gobject_set_qdata ((go), quark_editor, (ed))


/* --- signal list entries --- */
enum {
  SFIELD_OBJECT,
  SFIELD_SIGNAL,
  SFIELD_FUNC,
  SFIELD_DATA,
  SFIELD_CTYPE,
  SFIELD_N_COLUMNS
};
static gchar *sfield_names[SFIELD_N_COLUMNS] = {
  "Object",
  "Signal",
  "Function",
  "Data",
  "Type",
};
static gchar *sfield_tips[SFIELD_N_COLUMNS] = {
  NULL,
  ("Signal name entry, hit TAB multiple times "
   "to get a list of possible completions."),
  ("Function name entry, hit TAB multiple times "
   "to get a list of possible completions."),
  NULL,
  NULL,
};


/* --- signals --- */
enum {
  SIGNAL_REFRESH_VALUES,
  SIGNAL_RELOAD_VALUES,
  SIGNAL_APPLY_VALUES,
  SIGNAL_RESET_VALUES,
  SIGNAL_RESTORE_VALUES,
  SIGNAL_REFRESH_GARG,
  SIGNAL_LAST
};
typedef gboolean	(*SignalRefreshGArg)	(GtkObject *object,
						 GleGArg   *garg,
						 gpointer   func_data);


/* --- prototypes --- */
static void	gle_editor_destroy		(GtkObject		*object);
static void	gle_editor_class_init		(GleEditorClass		*class);
static void	gle_editor_init			(GleEditor		*editor);
static void	gle_editor_construct		(GleEditor		*editor,
						 GleGObject		*gobject);
static void	gle_editor_real_refresh_values	(GleEditor		*editor);
static void	gle_editor_real_reload_values	(GleEditor		*editor);
static void	gle_editor_real_apply_values	(GleEditor		*editor);
static void	gle_editor_real_reset_values	(GleEditor		*editor);
static void	gle_editor_real_restore_values	(GleEditor		*editor);
static void	gle_editor_real_refresh_garg	(GleEditor		*editor,
						 GleGArg		*garg);
static void	gle_editor_tree_select_row	(GleEditor		*editor,
						 GtkCTreeNode		*node,
						 gint			 column,
						 gpointer		 user_data);
static void	gle_editor_switch_page		(GleEditor		*editor,
						 GtkNotebookPage	*page,
						 guint			 page_num,
						 gpointer		 user_data);
static void	gle_editor_garg_widget_cb	(GleGArg		*garg,
						 GleGArgCbType		 cb_type);
static void	gle_editor_create_signal	(GleEditor		*editor);
static void	gle_editor_set_signal		(GleEditor		*editor);
static void	gle_editor_delete_signal	(GleEditor		*editor);
static void	gle_editor_signal_selected	(GleEditor		*editor,
						 gint       		 row);
static void	gle_editor_apply_signals	(GleEditor		*editor);


/* -- variables --- */
static GtkWindowClass	*parent_class = NULL;
static guint		 editor_signals[SIGNAL_LAST] = { 0 };
static GleEditorClass	*gle_editor_class = NULL;
static GQuark		 quark_editor = 0;


/* --- functions --- */
GtkType
gle_editor_get_type (void)
{
  static GtkType editor_type = 0;
  
  if (!editor_type)
    {
      GtkTypeInfo editor_info =
      {
	"GleEditor",
	sizeof (GleEditor),
	sizeof (GleEditorClass),
	(GtkClassInitFunc) gle_editor_class_init,
	(GtkObjectInitFunc) gle_editor_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      editor_type = gtk_type_unique (gtk_window_get_type (), &editor_info);
    }
  
  return editor_type;
}

static void
gle_editor_marshal_refresh_garg (GtkObject	*object,
				 GtkSignalFunc	func,
				 gpointer	func_data,
				 GtkArg		*args)
{
  SignalRefreshGArg sfunc = (SignalRefreshGArg) func;
  
  sfunc (object, GTK_VALUE_POINTER (args[0]), func_data);
}

static void
gle_editor_class_init (GleEditorClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;
  
  gle_editor_class = class;
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;
  
  parent_class = gtk_type_class (gtk_window_get_type ());
  
  quark_editor = g_quark_from_static_string (GLE_PRIVATE_KEY (editor));
  
  editor_signals[SIGNAL_REFRESH_VALUES] =
    gtk_signal_new ("refresh_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, refresh_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_RELOAD_VALUES] =
    gtk_signal_new ("reload_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, reload_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_APPLY_VALUES] =
    gtk_signal_new ("apply_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, apply_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_RESET_VALUES] =
    gtk_signal_new ("reset_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, reset_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_RESTORE_VALUES] =
    gtk_signal_new ("restore_values",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, restore_values),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  editor_signals[SIGNAL_REFRESH_GARG] =
    gtk_signal_new ("refresh_garg",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleEditorClass, refresh_garg),
		    gle_editor_marshal_refresh_garg,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_POINTER);
  gtk_object_class_add_signals (object_class, editor_signals, SIGNAL_LAST);
  
  object_class->destroy = gle_editor_destroy;
  
  class->refresh_values = gle_editor_real_refresh_values;
  class->reload_values = gle_editor_real_reload_values;
  class->apply_values = gle_editor_real_apply_values;
  class->reset_values = gle_editor_real_reset_values;
  class->restore_values = gle_editor_real_restore_values;
  class->refresh_garg = gle_editor_real_refresh_garg;
}

static void
gle_editor_init (GleEditor *editor)
{
  GtkWidget *main_vbox;
  GtkWidget *hbox;
  GtkWidget *any;
  GtkWidget *button;
  static gchar *ctree_titles[] = { "", "Category" };
  
  GLE_TAG (editor, "GLE-Editor");
  
  editor->gobject = NULL;
  editor->args_dirty = FALSE;
  editor->is_widget = FALSE;
  editor->reload_queued = FALSE;
  editor->garg_list = NULL;
  editor->child_args = NULL;
  editor->sfield_list = NULL;
  editor->sfield_signal = NULL;
  editor->sfield_func = NULL;
  editor->sfield_data = NULL;
  editor->sfield_ctype = NULL;
  
  /* setup the main window and its container and tooltips
   */
  gtk_widget_set (GTK_WIDGET (editor),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "GLE-Editor",
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  main_vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", editor,
		    "GtkWidget::visible", TRUE,
		    NULL);
  if (!gle_editor_class->tooltips)
    {
      gle_editor_class->tooltips = gtk_tooltips_new ();
      gtk_object_ref (GTK_OBJECT (gle_editor_class->tooltips));
      gtk_object_sink (GTK_OBJECT (gle_editor_class->tooltips));
      gtk_tooltips_set_delay (gle_editor_class->tooltips, GLE_RCVAL_TOOLTIPS_DELAY);
      gtk_tooltips_force_window (gle_editor_class->tooltips);
      gtk_widget_set_name (gle_editor_class->tooltips->tip_window, "GLE-TipWindow");
      gtk_signal_connect (GTK_OBJECT (gle_editor_class->tooltips),
			  "destroy",
			  gtk_widget_destroyed,
			  &gle_editor_class->tooltips);
    }
  else
    gtk_object_ref (GTK_OBJECT (gle_editor_class->tooltips));
  
  /* gle name
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  editor->gname_label =
    gtk_widget_new (GTK_TYPE_LABEL,
		    "GtkLabel::label", "<gname_label>",
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), editor->gname_label, FALSE, TRUE, 0);
  editor->gname_entry =
    gtk_widget_new (GTK_TYPE_ENTRY,
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::activate", gtk_window_activate_default, editor,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->gname_entry,
		    NULL);
  any =
    gtk_widget_new (GTK_TYPE_HSEPARATOR,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);
  
  /* object name label
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  editor->object_name_label =
    gtk_widget_new (GTK_TYPE_LABEL,
		    "GtkLabel::label", "<object_name_label>",
		    "GtkWidget::visible", TRUE,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->object_name_label,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), editor->object_name_label, FALSE, TRUE, 0);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);
  
  /* widget argument ctree/notebook
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, TRUE, TRUE, 0);
  any = gtk_widget_new (GTK_TYPE_SCROLLED_WINDOW,
			"visible", TRUE,
			"hscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"vscrollbar_policy", GTK_POLICY_AUTOMATIC,
			"parent", hbox,
			NULL);
  editor->ctree = (GtkCTree*) gtk_ctree_new_with_titles (2, 0, ctree_titles);
  gtk_container_add (GTK_CONTAINER (any), GTK_WIDGET (editor->ctree));
  editor->clist = GTK_CLIST (editor->ctree);
  gtk_widget_set (GTK_WIDGET (editor->ctree),
		  "GtkWidget::visible", TRUE,
		  "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->ctree,
		  "object_signal::tree_select_row", gle_editor_tree_select_row, editor,
		  "width", 120,
		  "selection_mode", GTK_SELECTION_BROWSE,
		  "line_style", GTK_CTREE_LINES_NONE,
		  NULL);
  gtk_clist_column_titles_passive (editor->clist);
  gtk_clist_set_column_width (editor->clist, 0, 12);
  editor->notebook =
    gtk_widget_new (GTK_TYPE_NOTEBOOK,
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &editor->notebook,
		    "object_signal_after::switch_page", gle_editor_switch_page, editor,
		    "GtkNotebook::scrollable", TRUE,
		    /* "GtkNotebook::tab_border", 0, */
		    "GtkNotebook::show_border", FALSE,
		    "GtkNotebook::enable_popup", TRUE,
		    "GtkNotebook::show_tabs", FALSE,
		    NULL);
  gtk_widget_set_name (GTK_NOTEBOOK (editor->notebook)->menu, "GLE-Notebook_Popup");
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);
  
  /* arg buttons, they map to the following actions:
   * "Reset"	- gle_editor_restore_values()
   * "Update"	- gle_editor_queue_reload_values()
   * %		- gle_editor_refresh_values()
   * %		- gle_editor_reset_values()
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "GtkButton::label", "Update",
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::clicked", gle_editor_queue_reload_values, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  gtk_tooltips_set_tip (gle_editor_class->tooltips, button,
			"Reload the editor contents from the GtkObject, "
			"use this if a GtkObject changed its arguments from outside "
			"of the editor's control.",
			NULL);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "GtkButton::label", "Reset",
		    "GtkWidget::visible", TRUE,
		    "GtkObject::object_signal::clicked", gle_editor_restore_values, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  gtk_tooltips_set_tip (gle_editor_class->tooltips, button,
			"Reset editor contents, "
			"use this after arguments have been changed accidentally.",
			NULL);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), any, FALSE, TRUE, 0);
  
  /* Apply/Close buttons
   */
  hbox =
    gtk_widget_new (GTK_TYPE_HBOX,
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_end (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "GtkButton::label", "Apply",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::has_default", TRUE,
		    "GtkObject::object_signal::clicked", gle_editor_apply_values, editor,
		    NULL);
  editor->default_default = button;
  gtk_tooltips_set_tip (gle_editor_class->tooltips, button,
			"Apply the editor contents to the GtkObject",
			NULL);
  button =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "GtkButton::label", "Close",
		    "GtkWidget::parent", hbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkObject::object_signal::clicked", gtk_widget_destroy, editor,
		    NULL);
  gtk_tooltips_set_tip (gle_editor_class->tooltips, button,
			"Close the editor without affecting the GtkObject",
			NULL);
}

GtkWidget*
gle_editor_new (GleGObject *gobject)
{
  GtkWidget *editor;
  
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  
  editor = gtk_type_new (gle_editor_get_type ());
  
  gle_editor_construct (GLE_EDITOR (editor), gobject);
  
  return editor;
}

void
gle_editor_destroy_args (GleEditor	*editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_container_foreach (GTK_CONTAINER (editor->notebook),
			 (GtkCallback) gtk_widget_destroy,
			 NULL);
  if (editor->ctree)
    gtk_ctree_remove_node (editor->ctree, NULL);
}

static void
gle_editor_tree_select_row (GleEditor	 *editor,
			    GtkCTreeNode *node,
			    gint	  column,
			    gpointer	  user_data)
{
  GtkWidget *vbox;
  GtkNotebook *notebook;
  
  notebook = GTK_NOTEBOOK (editor->notebook);
  
  vbox = gtk_ctree_node_get_row_data (editor->ctree, node);
  if (vbox)
    {
      gint num;
      
      num = gtk_notebook_page_num (notebook, vbox);
      if (gtk_notebook_get_current_page (notebook) != num)
	{
	  GtkWidget *focus;

	  focus = gtk_object_get_data (GTK_OBJECT (vbox), "focus");
	  gtk_notebook_set_page (notebook, num);
	  if (focus)
	    {
	      gtk_widget_grab_focus (focus);
	      gtk_widget_grab_default (focus);
	    }
	  else
	    {
	      gtk_widget_grab_default (editor->default_default);
	      gtk_widget_grab_focus (GTK_WIDGET (editor->ctree));
	    }
	}
    }
}

static void
gle_editor_switch_page (GleEditor	*editor,
			GtkNotebookPage *page,
			guint		 page_num,
			gpointer	 user_data)
{
  GtkCTreeNode *node;
  
  node = gtk_object_get_user_data (GTK_OBJECT (page->child));
  
  if (editor->ctree)
    gtk_ctree_select (editor->ctree, node);
}

static void
gle_editor_notebook_child_destroy (GtkWidget *child,
				   GleEditor *editor)
{
  GtkCTreeNode *node;
  
  if (editor->ctree)
    {
      node = gtk_object_get_user_data (GTK_OBJECT (child));
      gtk_ctree_remove_node (editor->ctree, node);
    }
}

static void
clue_hunter_add_signals (GtkClueHunter *clue_hunter,
			 GtkType	type)
{
  GtkObjectClass *class;
  GtkWidget *clist;

  clist = gtk_widget_new (GTK_TYPE_CLIST,
			  "n_columns", 2,
			  "titles_active", FALSE,
			  NULL);
  gtk_clist_set_auto_sort (GTK_CLIST (clist), FALSE);
  gtk_clist_column_titles_hide (GTK_CLIST (clist));

  class = gtk_type_class (type);
  while (class)
    {
      guint *object_signals;
      gint i;

      object_signals = class->signals;

      for (i = class->nsignals - 1; i >= 0; i--)
	{
	  GtkSignalQuery *query;

	  query = object_signals[i] ? gtk_signal_query (object_signals[i]) : NULL;
	  if (query)
	    {
	      gchar *text[2];

	      text[0] = gtk_type_name (query->object_type);
	      text[1] = (gchar*) query->signal_name;

	      gtk_clist_insert (GTK_CLIST (clist), -1, text);
	      g_free (query);
	    }
	}

      class = gtk_type_parent_class (class->type);
    }

  gtk_clue_hunter_set_clist (clue_hunter, clist, 1);
  gtk_clue_hunter_set_align_width (clue_hunter, FALSE);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (clue_hunter->scw),
				  GTK_POLICY_NEVER,
				  GTK_POLICY_AUTOMATIC);
}

static gboolean
clue_hunter_add_handler (GleHandler *handler,
			 gpointer    user_data)
{
  GtkClueHunter *clue_hunter = user_data;

  gtk_clue_hunter_add_string (clue_hunter, g_quark_to_string (handler->id));
  
  return TRUE;
}

static GtkWidget*
create_signal_field (GleEditor *editor,
		     guint	sfield,
		     GtkBox    *parent_box)
{
  GtkWidget *box;
  GtkWidget *label;
  GtkWidget *entry;

  box = gtk_widget_new (GTK_TYPE_HBOX,
			"visible", TRUE,
			"homogeneous", FALSE,
			"spacing", 10,
			"border_width", 0,
			NULL);
  gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
  label = gtk_widget_new (GTK_TYPE_LABEL,
			  "label", sfield_names[sfield],
			  "justify", GTK_JUSTIFY_LEFT,
			  "visible", TRUE,
			  NULL);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
  entry = gtk_widget_new (GTK_TYPE_ENTRY,
			  "width", 200,
			  "visible", TRUE,
			  "user_data", GUINT_TO_POINTER (sfield),
			  NULL);
  if (sfield_tips[sfield])
    gtk_tooltips_set_tip (gle_editor_class->tooltips, entry, sfield_tips[sfield], NULL);

  if (sfield == SFIELD_SIGNAL || sfield == SFIELD_FUNC)
    {
      GtkWidget *any;

      any = gtk_widget_new (GTK_TYPE_CLUE_HUNTER,
			    "keep_history", FALSE,
			    "entry", entry,
			    "user_data", editor->gobject,
			    NULL);
      GLE_TAG (any, "GLE-Editor-clue-hunter-popup");
      if (sfield == SFIELD_FUNC)
	gle_handler_foreach (clue_hunter_add_handler, any);
      else
	clue_hunter_add_signals (GTK_CLUE_HUNTER (any),
				 GLE_GOBJECT_OTYPE (editor->gobject));
    }
  gtk_signal_connect_object (GTK_OBJECT (entry),
			     "activate",
			     GTK_SIGNAL_FUNC (gtk_window_activate_default),
			     GTK_OBJECT (editor));
  gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);

  return entry;
}

static GtkWidget*
gle_editor_signal_page (GleEditor *editor)
{
  GtkCTreeNode *node;
  GtkWidget *vbox;
  GtkWidget *box;
  GleGObject *gobject;
  gchar *pattern;
  GtkWidget *title;
  gchar *page_name = "Signals";
  GtkWidget *any;

  g_return_val_if_fail (editor->sfield_list == NULL, NULL);

  gobject = editor->gobject;
  
  node = gtk_ctree_insert_node (editor->ctree, NULL, NULL, NULL, 0,
				NULL, NULL, NULL, NULL,
				TRUE, TRUE);
  gtk_ctree_node_set_text (editor->ctree, node, 1, page_name);
  
  pattern = g_strnfill (strlen (page_name), '_');
  title =
    gtk_widget_new (GTK_TYPE_ALIGNMENT,
		    "xalign", 0.5,
		    "yalign", 0.5,
		    "xscale", 1.0,
		    "yscale", 1.0,
		    "visible", TRUE,
		    "child", gtk_widget_new (GTK_TYPE_LABEL,
					     "label", page_name,
					     "pattern", pattern,
					     "visible", TRUE,
					     NULL),
		    NULL);
  g_free (pattern);
  vbox =
    gtk_widget_new (GTK_TYPE_VBOX,
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::visible", TRUE,
		    "user_data", node,
		    "signal::destroy", gle_editor_notebook_child_destroy, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (vbox), title, FALSE, FALSE, 5);

  /* Signal:
   */
  editor->sfield_signal = create_signal_field (editor, SFIELD_SIGNAL, GTK_BOX (vbox));
  gtk_widget_set (editor->sfield_signal,
		  "signal::destroy", gtk_widget_destroyed, &editor->sfield_signal,
		  NULL);
  /* Function:
   */
  editor->sfield_func =  create_signal_field (editor, SFIELD_FUNC, GTK_BOX (vbox));
  gtk_widget_set (editor->sfield_func,
		  "signal::destroy", gtk_widget_destroyed, &editor->sfield_func,
		  NULL);
  /* Data:
   */
  editor->sfield_data =  create_signal_field (editor, SFIELD_DATA, GTK_BOX (vbox));
  gtk_widget_set (editor->sfield_data,
		  "signal::destroy", gtk_widget_destroyed, &editor->sfield_data,
		  NULL);
  /* Signal Type
   */
  {
    struct {
      gchar *name;
      guint  ctype;
    } ctypes[] = {
      { "normal",	GLE_CONNECTION_NORMAL, },
      { "object",	GLE_CONNECTION_OBJECT, },
      { "after",	GLE_CONNECTION_AFTER, },
      { "object after",	GLE_CONNECTION_OBJECT_AFTER, },
      { NULL,		0, },
    }, *ct = ctypes;
    GtkWidget *menu;
    GtkWidget *label;

    box = gtk_widget_new (GTK_TYPE_HBOX,
			  "homogeneous", FALSE,
			  "spacing", 10,
			  "border_width", 0,
			  "visible", TRUE,
			  NULL);
    gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, TRUE, 0);
    label = gtk_widget_new (GTK_TYPE_LABEL,
			    "label", "Connect",
			    "justify", GTK_JUSTIFY_LEFT,
			    "visible", TRUE,
			    NULL);
    gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
    editor->sfield_ctype = gtk_option_menu_new ();
    gtk_widget_set (editor->sfield_ctype,
		    "signal::destroy", gtk_widget_destroyed, &editor->sfield_ctype,
		    NULL);
    gtk_widget_show (editor->sfield_ctype);
    gtk_box_pack_end (GTK_BOX (box), editor->sfield_ctype, FALSE, TRUE, 0);
    
    menu = gtk_menu_new ();
    GLE_TAG (menu, "GLE-Editor-Signal-Popup");
    
    while (ct->name)
      {
	GtkWidget *item;
	
	item = gtk_menu_item_new_with_label (ct->name);
	gtk_widget_lock_accelerators (item);
	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), GUINT_TO_POINTER (ct->ctype));
	gtk_container_add (GTK_CONTAINER (menu), item);

	ct++;
      }

    gtk_option_menu_set_menu (GTK_OPTION_MENU (editor->sfield_ctype), menu);
  }
  
  box = gtk_widget_new (GTK_TYPE_HBOX,
			"homogeneous", FALSE,
			"spacing", 5,
			"border_width", 0,
			"visible", TRUE,
			NULL);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, TRUE, 0);
  any =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "label", "Create",
		    "can_default", TRUE,
		    "object_signal::clicked", gle_editor_create_signal, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (box), any, FALSE, TRUE, 0);
  any =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "label", "Change",
		    "can_default", TRUE,
		    "object_signal::clicked", gle_editor_set_signal, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (box), any, FALSE, TRUE, 0);
  gtk_object_set_data (GTK_OBJECT (vbox), "focus", any);
  any =
    gtk_widget_new (GTK_TYPE_BUTTON,
		    "visible", TRUE,
		    "label", "Delete",
		    "can_default", TRUE,
		    "object_signal::clicked", gle_editor_delete_signal, editor,
		    NULL);
  gtk_box_pack_start (GTK_BOX (box), any, FALSE, TRUE, 0);
  any =
    gtk_widget_new (GTK_TYPE_SCROLLED_WINDOW,
		    "visible", TRUE,
		    "hscrollbar_policy", GTK_POLICY_AUTOMATIC,
		    "vscrollbar_policy", GTK_POLICY_AUTOMATIC,
		    "parent", vbox,
		    NULL);
  editor->sfield_list =
    gtk_widget_new (GTK_TYPE_CLIST,
		    "n_columns", SFIELD_N_COLUMNS,
		    "visible", TRUE,
		    "parent", any,
		    "selection_mode", GTK_SELECTION_BROWSE,
		    "signal::destroy", gtk_widget_destroyed, &editor->sfield_list,
		    "object_signal::select_row", gle_editor_signal_selected, editor,
		    NULL);
  gtk_clist_set_auto_sort (GTK_CLIST (editor->sfield_list), TRUE);
  gtk_clist_set_column_title (GTK_CLIST (editor->sfield_list), SFIELD_OBJECT, sfield_names[SFIELD_OBJECT]);
  gtk_clist_set_column_title (GTK_CLIST (editor->sfield_list), SFIELD_SIGNAL, sfield_names[SFIELD_SIGNAL]);
  gtk_clist_set_column_title (GTK_CLIST (editor->sfield_list), SFIELD_FUNC, sfield_names[SFIELD_FUNC]);
  gtk_clist_set_column_title (GTK_CLIST (editor->sfield_list), SFIELD_DATA, sfield_names[SFIELD_DATA]);
  gtk_clist_set_column_title (GTK_CLIST (editor->sfield_list), SFIELD_CTYPE, sfield_names[SFIELD_CTYPE]);
  gtk_clist_column_titles_show (GTK_CLIST (editor->sfield_list));
  gtk_clist_column_titles_passive (GTK_CLIST (editor->sfield_list));
  gtk_ctree_node_set_row_data (editor->ctree, node, vbox);
      
  /* we already rely on GtkNotebook::switch_page here */
  gtk_notebook_append_page (GTK_NOTEBOOK (editor->notebook),
			    vbox,
			    gtk_label_new (page_name));
  
  return vbox;
}

static GtkWidget*
gle_editor_new_page (GleEditor	 *editor,
		     const gchar *group,
		     gboolean	  with_children)
{
  GtkCTreeNode *node;
  GtkWidget *vbox;
  
  node = gtk_ctree_insert_node (editor->ctree, NULL, NULL, NULL, 0,
				NULL, NULL, NULL, NULL,
				TRUE, TRUE);
  gtk_ctree_node_set_text (editor->ctree, node, 1, group);
  
  if (!with_children)
    {
      /* we need the correct ctree style before referencing it's fields,
       * or we'll operate on colors from some gtk default style
       */
      gtk_widget_ensure_style (GTK_WIDGET (editor->ctree));
      
      gtk_ctree_node_set_foreground (editor->ctree,
				     node,
				     &GTK_WIDGET (editor->ctree)->style->fg[GTK_STATE_INSENSITIVE]);
      gtk_ctree_node_set_background (editor->ctree,
				     node,
				     &GTK_WIDGET (editor->ctree)->style->bg[GTK_STATE_INSENSITIVE]);
      gtk_ctree_node_set_selectable (editor->ctree, node, FALSE);
      vbox = NULL;
    }
  else
    {
      gchar *pattern;
      GtkWidget *title;
      
      pattern = g_strnfill (strlen (group), '_');
      title =
	gtk_widget_new (GTK_TYPE_ALIGNMENT,
			"xalign", 0.5,
			"yalign", 0.5,
			"xscale", 1.0,
			"yscale", 1.0,
			"visible", TRUE,
			"child", gtk_widget_new (GTK_TYPE_LABEL,
						 "label", group,
						 "pattern", pattern,
						 "visible", TRUE,
						 NULL),
			NULL);
      g_free (pattern);
      vbox =
	gtk_widget_new (GTK_TYPE_VBOX,
			"GtkBox::homogeneous", FALSE,
			"GtkBox::spacing", 0,
			"GtkContainer::border_width", 5,
			"GtkWidget::visible", TRUE,
			"user_data", node,
			"signal::destroy", gle_editor_notebook_child_destroy, editor,
			NULL);
      gtk_box_pack_start (GTK_BOX (vbox), title, FALSE, FALSE, 5);
      
      gtk_ctree_node_set_row_data (editor->ctree, node, vbox);
      
      /* we already rely on GtkNotebook::switch_page here */
      gtk_notebook_append_page (GTK_NOTEBOOK (editor->notebook),
				vbox,
				gtk_label_new (group));
    }
  
  return vbox;
}

static void
gle_editor_garg_widget_cb (GleGArg		  *garg,
			   GleGArgCbType	   cb_type)
{
  GleEditor *editor;
  
  editor = gtk_object_get_data_by_id (GTK_OBJECT (garg->widget), quark_editor);
  switch (cb_type)
    {
    case GLE_GARG_CB_ARG_RESET:
    case GLE_GARG_CB_ARG_RESTORE:
    case GLE_GARG_CB_ARG_GET:
      gle_editor_refresh_garg (editor, garg);
      break;
    case GLE_GARG_CB_ARG_SET:
    case GLE_GARG_CB_ARG_SAVE:
    default:
      break;
    }
}

void
gle_editor_rebuild (GleEditor  *editor)
{
  GleGArgGroup *group;
  GleGObject *gobject;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  g_return_if_fail (editor->gobject != NULL);
  g_return_if_fail (editor->is_widget != FALSE);
  
  gobject = editor->gobject;
  
  gtk_widget_hide (editor->notebook->parent);
  
  gle_editor_destroy_args (editor);
  
  /* create object argument fields in notebook
   */
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      GtkWidget *vbox;
      
      vbox = NULL;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg	*garg;
	  
	  garg = slist->data;
	  if (!GLE_GARG_IS_READWRITE (garg))
	    continue;
	  
	  if (garg->widget)
	    {
	      g_warning ("GLE: garg->widget already exists?");
	      continue;
	    }
	  
	  if (!vbox)
	    vbox = gle_editor_new_page (editor, g_quark_to_string (group->garg_group), TRUE);
	  
	  gle_arg_create_field (&garg->object_arg,
				gtk_arg_name_strip_type (garg->arg_name),
				&garg->widget,
				GTK_BOX (vbox),
				GTK_WINDOW (editor));
	  gtk_object_set_data_by_id (GTK_OBJECT (garg->widget), quark_editor, editor);
	  gle_garg_widget_set_cb (garg->widget, gle_editor_garg_widget_cb);
	  if (GLE_GARG_IS_CONSTRUCT_ONLY (garg))
	    gtk_widget_set_sensitive (garg->widget, FALSE);
	}
      
      if (!vbox)
	vbox = gle_editor_new_page (editor, g_quark_to_string (group->garg_group), FALSE);
    }

  gle_editor_signal_page (editor);

  gtk_notebook_set_page (GTK_NOTEBOOK (editor->notebook), 0);

  gle_editor_queue_reload_values (editor);
  
  gtk_widget_show (editor->notebook->parent);
}

static void
gle_editor_construct (GleEditor	 *editor,
		      GleGObject *gobject)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  g_return_if_fail (gobject != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (gobject));
  g_return_if_fail (editor->gobject == NULL);
  g_return_if_fail (gle_editor_from_gobject (gobject) == NULL);
  
  editor->gobject = gobject;
  editor->is_widget = gtk_type_is_a (GLE_GOBJECT_OTYPE (gobject), GTK_TYPE_WIDGET);
  
  GOBJECT_SET_EDITOR (gobject, GTK_WIDGET (editor));
  GLE_NOTIFIER_INSTALL_DATA (gobject, "pre_destroy", gtk_widget_destroy, editor);
  GLE_NOTIFIER_INSTALL_DATA (gobject, "rebuilt_args", gle_editor_rebuild, editor);

#if 0
  if (editor->is_widget)
    {
      gtk_signal_connect_object_after (editor->gobject->object,
				       "hide",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
      gtk_signal_connect_object_after (editor->gobject->object,
				       "show",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
      gtk_signal_connect_object_after (editor->gobject->object,
				       "style_set",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
      gtk_signal_connect_object_after (editor->gobject->object,
				       "state_changed",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
      /* we connect to size_request and draw because a *lot* of the
       * random widget arguments cause either a redraw or a resize. if
       * we trigger that, we are mostly set wrt updating our arguments.
       */
      gtk_signal_connect_object_after (editor->gobject->object,
				       "size_request",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
      gtk_signal_connect_object_after (editor->gobject->object,
				       "draw",
				       gle_editor_queue_reload_values,
				       GTK_OBJECT (editor));
    }
#endif
  gle_editor_rebuild (editor);
}

static void
gle_editor_destroy (GtkObject *object)
{
  GleEditor *editor;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_EDITOR (object));
  
  editor = GLE_EDITOR (object);
  
  gtk_object_unref (GTK_OBJECT (gle_editor_class->tooltips));
  
  GLE_NOTIFIER_REMOVE_MFD (editor->gobject, "rebuilt_args", gle_editor_rebuild, editor);
  GLE_NOTIFIER_REMOVE_MFD (editor->gobject, "pre_destroy", gtk_widget_destroy, editor);
  
  GOBJECT_SET_EDITOR (editor->gobject, NULL);
  editor->gobject = NULL;
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
gle_editor_signal_selected (GleEditor *editor,
			    gint       row)
{
  GtkCListRow *clist_row;
  GtkCList *clist;
  gchar *text;
  GtkWidget *menu;
  GleConnectionType ctype;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  if (row < 0 || row >= clist->rows)
    return ;

  clist_row = g_list_nth (clist->row_list, row)->data;

  text = clist_row->cell[SFIELD_SIGNAL].u.text;
  gtk_entry_set_text (GTK_ENTRY (editor->sfield_signal), text ? text : "");
  text = clist_row->cell[SFIELD_FUNC].u.text;
  gtk_entry_set_text (GTK_ENTRY (editor->sfield_func), text ? text : "");
  text = clist_row->cell[SFIELD_DATA].u.text;
  gtk_entry_set_text (GTK_ENTRY (editor->sfield_data), text ? text : "");
  text = clist_row->cell[SFIELD_CTYPE].u.text;
  ctype = GLE_CONNECTION_NORMAL;
  if (text && (text[0] == 'O' || text[1] == 'O'))
    ctype |= GLE_CONNECTION_OBJECT;
  if (text && (text[0] == 'A' || text[1] == 'A'))
    ctype |= GLE_CONNECTION_AFTER;

  menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (editor->sfield_ctype));
  if (menu)
    {
      guint n;
      GList *list;

      n = 0;
      for (list = GTK_MENU_SHELL (menu)->children; list; list = list->next)
	{
	  GtkWidget *item;
	  gpointer data;

	  item = list->data;
	  data = gtk_object_get_user_data (GTK_OBJECT (item));
	  if (GPOINTER_TO_UINT (data) == ctype)
	    {
	      gtk_option_menu_set_history (GTK_OPTION_MENU (editor->sfield_ctype), n);
	      break;
	    }
	  n++;
	}
    }
  
}

void
gle_editor_update_signals (GleEditor *editor)
{
  GtkCList *clist;
  GleConnection *connection;
  gboolean need_resize;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  gtk_clist_freeze (clist);

  need_resize = clist->rows < 1;

  gtk_clist_clear (clist);

  for (connection = editor->gobject->connections; connection; connection = connection->next)
    {
      GtkSignalQuery *query;
      gchar buffer[32], *text[SFIELD_N_COLUMNS] = { NULL, };
      gchar ctype_string[3] = { 0, 0, 0, }, *ctype_p = ctype_string;
      gint row;
      
      query = gtk_signal_query (connection->signal_id);

      text[SFIELD_OBJECT] = gtk_type_name (query->object_type);
      text[SFIELD_SIGNAL] = (gchar*) query->signal_name;
      text[SFIELD_FUNC] = connection->handler;
      sprintf (buffer, "%p", connection->data);
      text[SFIELD_DATA] = buffer;
      if (connection->ctype & GLE_CONNECTION_OBJECT)
	*(ctype_p++) = 'O';
      if (connection->ctype & GLE_CONNECTION_AFTER)
	*(ctype_p++) = 'A';
      text[SFIELD_CTYPE] = ctype_string;

      row = gtk_clist_insert (clist, -1, text);
      gtk_clist_set_row_data (clist, row, connection);
      
      g_free (query);
    }

  if (need_resize)
    gtk_clist_columns_autosize (clist);

  gtk_clist_thaw (clist);
}

static void
gle_editor_change_signal (GleEditor        *editor,
			  gint              row,
			  gchar            *signal,
			  gchar            *func,
			  gchar            *data,
			  GleConnectionType ctype)
{
  GtkCList *clist;
  GtkSignalQuery *query;
  gchar buffer[32];
  gchar ctype_string[3] = { 0, 0, 0, }, *ctype_p = ctype_string;
  guint signal_id;
  guint base;

  clist = GTK_CLIST (editor->sfield_list);

  gtk_clist_freeze (clist);
  
  signal_id = gtk_signal_lookup (signal, GLE_GOBJECT_OTYPE (editor->gobject));
  if (!signal_id)
    signal_id = 1;
  query = gtk_signal_query (signal_id);
  if (!query)
    gtk_signal_query (1);

  gtk_clist_set_text (clist, row, SFIELD_OBJECT, gtk_type_name (query->object_type));
  gtk_clist_set_text (clist, row, SFIELD_SIGNAL, query->signal_name);
  gtk_clist_set_text (clist, row, SFIELD_FUNC, func);
  if (data && data[0] == '0')
    {
      base = 8;
      data++;
      if (data[0] == 'x' || data[0] == 'X')
	{
	  base = 16;
	  data++;
	}
    }
  else
    base = 10;
  sprintf (buffer, "%p", GINT_TO_POINTER (strtol (data, NULL, base)));
  gtk_clist_set_text (clist, row, SFIELD_DATA, buffer);
  if (ctype & GLE_CONNECTION_OBJECT)
    *(ctype_p++) = 'O';
  if (ctype & GLE_CONNECTION_AFTER)
    *(ctype_p++) = 'A';
  gtk_clist_set_text (clist, row, SFIELD_CTYPE, ctype_string);

  g_free (query);

  gtk_clist_thaw (clist);

  gle_editor_signal_selected (editor, row);
}

static void
gle_editor_create_signal (GleEditor *editor)
{
  static gchar *text[SFIELD_N_COLUMNS] = { NULL, };
  GtkCList *clist;
  gboolean need_resize;
  gint row;
  gchar *signal;
  gchar *func;
  gchar *data;
  gpointer ctype;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  signal = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_signal)));
  func = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_func)));
  data = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_data)));
  ctype = gtk_object_get_user_data (GTK_OBJECT (GTK_OPTION_MENU (editor->sfield_ctype)->menu_item));

  gtk_clist_freeze (clist);
  
  need_resize = clist->rows < 1;
      
  row = gtk_clist_insert (GTK_CLIST (clist), 0, text);

  gle_editor_change_signal (editor, row, signal, func, data, GPOINTER_TO_UINT (ctype));

  g_free (signal);
  g_free (func);
  g_free (data);

  if (need_resize)
    gtk_clist_columns_autosize (clist);

  gtk_clist_thaw (clist);
}

static void
gle_editor_set_signal (GleEditor *editor)
{
  GtkCList *clist;
  gint row;
  gchar *signal;
  gchar *func;
  gchar *data;
  gpointer ctype;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  row = clist->selection ? GPOINTER_TO_INT (clist->selection->data) : -1;

  if (row < 0 || row >= clist->rows)
    return;
  
  signal = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_signal)));
  func = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_func)));
  data = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->sfield_data)));
  ctype = gtk_object_get_user_data (GTK_OBJECT (GTK_OPTION_MENU (editor->sfield_ctype)->menu_item));

  gle_editor_change_signal (editor, row, signal, func, data, GPOINTER_TO_UINT (ctype));

  g_free (signal);
  g_free (func);
  g_free (data);

  gle_editor_signal_selected (editor, row);
}

static void
gle_editor_apply_signals (GleEditor *editor)
{
  GtkCList *clist;
  GSList *delete_connections = NULL;
  GleConnection *cwalk;
  GSList *slist;
  GList *list;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  for (cwalk = editor->gobject->connections; cwalk; cwalk = cwalk->next)
    delete_connections = g_slist_prepend (delete_connections, cwalk);

  for (list = clist->row_list; list; list = list->next)
    {
      GtkCListRow *clist_row = list->data;
      gchar *text;
      guint signal_id;
      gchar *handler;
      gpointer data;
      GleConnectionType ctype;
      guint base;

      if (clist_row->data && !g_slist_find (delete_connections, clist_row->data))
	{
	  clist_row->data = NULL;
	  continue;
	}

      text = clist_row->cell[SFIELD_SIGNAL].u.text;
      signal_id = gtk_signal_lookup (text ? text : "", GLE_GOBJECT_OTYPE (editor->gobject));
      if (!signal_id)
	signal_id = 1;
      
      handler = clist_row->cell[SFIELD_FUNC].u.text;
      if (!handler)
	handler = "gtk_false";
      
      text = clist_row->cell[SFIELD_DATA].u.text;
      if (text && text[0] == '0')
	{
	  base = 8;
	  text++;
	  if (text[0] == 'x' || text[0] == 'X')
	    {
	      base = 16;
	      text++;
	    }
	}
      else
	base = 10;
      data = (gpointer) strtol (text ? text : "", NULL, base);

      text = clist_row->cell[SFIELD_CTYPE].u.text;
      ctype = GLE_CONNECTION_NORMAL;
      if (text && (text[0] == 'O' || text[1] == 'O'))
	ctype |= GLE_CONNECTION_OBJECT;
      if (text && (text[0] == 'A' || text[1] == 'A'))
	ctype |= GLE_CONNECTION_AFTER;

      if (!clist_row->data)
	clist_row->data = gle_connection_new (editor->gobject,
					      signal_id,
					      handler,
					      data,
					      ctype);
      else
	{
	  delete_connections = g_slist_remove (delete_connections, clist_row->data);
	  gle_connection_change (clist_row->data,
				 signal_id,
				 handler,
				 data,
				 ctype);
	}
    }
  
  for (slist = delete_connections; slist; slist = slist->next)
    gle_connection_destroy (slist->data);

  g_slist_free (delete_connections);

  gle_gobject_connect (editor->gobject);
}

static void
gle_editor_delete_signal (GleEditor *editor)
{
  GtkCList *clist;
  gint row;

  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));

  clist = GTK_CLIST (editor->sfield_list);

  row = GPOINTER_TO_INT (clist->selection->data);

  if (row < 0 || row >= clist->rows)
    return;
  
  gtk_clist_remove (GTK_CLIST (clist), row);
}

void
gle_editor_update_names (GleEditor *editor)
{
  static gchar *generic_prompt = "Generic GLE Name:";
  static const guint prompt_offs = 8;
  gchar *string;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  if (editor->is_widget)
    {
      GtkWidget *widget;
      
      widget = GLE_GWIDGET_WIDGET (editor->gobject);
      string = g_strconcat (gtk_type_name (GLE_GOBJECT_OTYPE (editor->gobject)),
			    "::",
			    "\"",
			    widget && widget->name ? widget->name : "",
			    "\"",
			    NULL);
    }
  else
    string = g_strconcat (gtk_type_name (GLE_GOBJECT_OTYPE (editor->gobject)), NULL);
  
  if (!g_str_equal (GTK_LABEL (editor->object_name_label)->label, string))
    gtk_label_set_text (GTK_LABEL (editor->object_name_label), string);
  g_free (string);
  
  if (GLE_GOBJECT_GENERIC_GNAME (editor->gobject))
    {
      if (!g_str_equal (GTK_LABEL (editor->gname_label)->label, generic_prompt))
	gtk_label_set_text (GTK_LABEL (editor->gname_label), generic_prompt);
    }
  else
    {
      if (!g_str_equal (GTK_LABEL (editor->gname_label)->label, generic_prompt + prompt_offs))
	gtk_label_set_text (GTK_LABEL (editor->gname_label), generic_prompt + prompt_offs);
    }
  
  if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (editor->gname_entry)), editor->gobject->gname))
    {
      gtk_entry_set_text (GTK_ENTRY (editor->gname_entry),
			  editor->gobject->gname ? editor->gobject->gname : "");
      gtk_entry_set_position (GTK_ENTRY (editor->gname_entry), 0);
    }
}

void
gle_editor_refresh_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_REFRESH_VALUES]);
}

static void
gle_editor_real_refresh_values (GleEditor *editor)
{
  GleGArgGroup *group;
  GleGObject *gobject;
  
  gobject = editor->gobject;
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  
	  if (garg->widget)
	    gle_editor_refresh_garg (editor, garg);
	}
    }
}

static gint
gle_editor_reload_handler (GleEditor *editor)
{
  g_return_val_if_fail (editor != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_EDITOR (editor), FALSE);
  
  editor->reload_queued = FALSE;
  
  if (!GTK_OBJECT_DESTROYED (editor))
    gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_RELOAD_VALUES]);
  
  gtk_object_unref (GTK_OBJECT (editor));
  
  return FALSE;
}

void
gle_editor_queue_reload_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  if (!editor->reload_queued)
    {
      editor->reload_queued = TRUE;
      
      gtk_object_ref (GTK_OBJECT (editor));
      
      gtk_idle_add ((GtkFunction) gle_editor_reload_handler, editor);
    }
}

static void
gle_editor_real_reload_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gle_gobject_update_all_args (editor->gobject);
  gle_editor_update_names (editor);
  gle_editor_update_signals (editor);
}

void
gle_editor_apply_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_APPLY_VALUES]);
}

static void
gle_editor_real_apply_values (GleEditor *editor)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  gchar *gname;
  
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gobject = editor->gobject;
  
  gname = g_strdup (gtk_entry_get_text (GTK_ENTRY (editor->gname_entry)));
  gle_gname_canonicalize (gname);
  if (gname[0] &&
      !g_str_equal ((gchar*) editor->gobject->gname, gname) &&
      !gle_gname_lookup (gname))
    gle_gobject_set_gname (editor->gobject, gname);
  g_free (gname);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  
	  if (GLE_GARG_IS_READWRITE (garg) && garg->widget)
	    garg->needs_set |= gle_arg_set_from_field (&garg->object_arg, garg->widget);
	}
    }
  if (GLE_GOBJECT_IS_INSTANTIATED (editor->gobject))
    gle_gobject_apply_all_args (editor->gobject);
  else
    gle_gobject_save_all_args (editor->gobject);
  
  gle_editor_apply_signals (editor);

  gle_editor_queue_reload_values (editor);
}

void
gle_editor_reset_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_RESET_VALUES]);
}

static void
gle_editor_real_reset_values (GleEditor *editor)
{
  gle_gobject_reset_all_args (editor->gobject);
  /* if (editor->is_widget)
   *   gle_gwidget_reset_child_args (GLE_GWIDGET (editor->gobject));
   */
}

void
gle_editor_restore_values (GleEditor *editor)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_RESTORE_VALUES]);
}

static void
gle_editor_real_restore_values (GleEditor *editor)
{
  gle_gobject_restore_all_args (editor->gobject);
  gle_editor_refresh_values (editor);
  /* if (editor->is_widget)
   *   gle_gwidget_restore_child_args (GLE_GWIDGET (editor->gobject));
   */
}

void
gle_editor_refresh_garg (GleEditor	*editor,
			 GleGArg	*garg)
{
  g_return_if_fail (editor != NULL);
  g_return_if_fail (GLE_IS_EDITOR (editor));
  g_return_if_fail (garg != NULL);
  if (!garg->widget)
    return;
  g_return_if_fail (gtk_widget_get_toplevel (garg->widget) == GTK_WIDGET (editor));
  
  gtk_signal_emit (GTK_OBJECT (editor), editor_signals[SIGNAL_REFRESH_GARG], garg);
}

static void
gle_editor_real_refresh_garg (GleEditor	     *editor,
			      GleGArg	     *garg)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (garg->widget != NULL);
  
  gle_arg_refresh_field (&garg->object_arg, garg->widget);
}

GleEditor*
gle_editor_from_gobject (GleGObject	*gobject)
{
  g_return_val_if_fail (gobject != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GOBJECT (gobject), NULL);
  
  return gle_gobject_get_qdata (gobject, quark_editor);
}

static void
gle_tb_sensitize (GtkToggleButton *tb,
		  GtkWidget	  *widget)
{
  gtk_widget_set_sensitive (widget, tb->active);
}

static const gchar *key_arg_ev = "gle-arg-item-enum-value";

void
gle_arg_create_field (GtkArg	  *arg,
		      const gchar *name,
		      GtkWidget	 **widget_p,
		      GtkBox	  *parent_box,
		      GtkWindow	  *window)
{
  register GtkWidget *widget;
  GtkType arg_type;
  
  if (widget_p)
    *widget_p = NULL;
  else
    g_return_if_fail (widget_p != NULL);
  
  g_return_if_fail (arg != NULL);
  if (!name)
    {
      g_return_if_fail (arg->name != NULL);
      name = arg->name;
    }
  g_return_if_fail (parent_box != NULL);
  g_return_if_fail (GTK_IS_BOX (parent_box));
  if (window)
    g_return_if_fail (GTK_IS_WINDOW (window));
  
  widget = NULL;
  arg_type = arg->type;
  switch (GTK_FUNDAMENTAL_TYPE (arg_type))
    {
      GtkWidget *box;
      GtkWidget *bar;
      GtkWidget *label;
      GtkWidget *entry;
      GtkWidget *button;
      GtkEnumValue *ev;
      
    case GTK_TYPE_BOOL:
      widget = gtk_widget_new (GTK_TYPE_CHECK_BUTTON,
			       "GtkButton::label", name,
			       "GtkWidget::visible", TRUE,
			       NULL);
      gtk_misc_set_alignment (GTK_MISC (GTK_BIN (widget)->child), 0, 0.5);
      gtk_box_pack_start (parent_box, widget, FALSE, FALSE, 0);
      break;
      
    case GTK_TYPE_INT:
    case GTK_TYPE_UINT:
    case GTK_TYPE_LONG:
    case GTK_TYPE_ULONG:
      box = gtk_widget_new (GTK_TYPE_HBOX,
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (GTK_TYPE_LABEL,
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      entry = gtk_widget_new (GTK_TYPE_ENTRY,
			      "GtkWidget::width", 50,
			      "GtkWidget::visible", TRUE,
			      NULL);
      if (window)
	gtk_signal_connect_object (GTK_OBJECT (entry),
				   "activate",
				   GTK_SIGNAL_FUNC (gtk_window_activate_default),
				   GTK_OBJECT (window));
      gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);
      widget = entry;
      break;
      
    case GTK_TYPE_FLOAT:
    case GTK_TYPE_DOUBLE:
      box = gtk_widget_new (GTK_TYPE_HBOX,
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (GTK_TYPE_LABEL,
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      entry = gtk_widget_new (GTK_TYPE_ENTRY,
			      "GtkWidget::width", 80,
			      "GtkWidget::visible", TRUE,
			      NULL);
      if (window)
	gtk_signal_connect_object (GTK_OBJECT (entry),
				   "activate",
				   GTK_SIGNAL_FUNC (gtk_window_activate_default),
				   GTK_OBJECT (window));
      gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);
      widget = entry;
      break;
      
    case GTK_TYPE_STRING:
      box = gtk_widget_new (GTK_TYPE_HBOX,
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (GTK_TYPE_LABEL,
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      button = gtk_widget_new (GTK_TYPE_CHECK_BUTTON,
			       "GtkWidget::visible", TRUE,
			       NULL);
      entry = gtk_widget_new (GTK_TYPE_ENTRY,
			      "GtkWidget::width", 140,
			      "GtkWidget::visible", TRUE,
			      "GtkObject::user_data", button,
			      NULL);
      if (window)
	gtk_signal_connect_object (GTK_OBJECT (entry),
				   "activate",
				   GTK_SIGNAL_FUNC (gtk_window_activate_default),
				   GTK_OBJECT (window));
      gtk_signal_connect (GTK_OBJECT (button),
			  "clicked",
			  GTK_SIGNAL_FUNC (gle_tb_sensitize),
			  entry);
      gle_tb_sensitize (GTK_TOGGLE_BUTTON (button), entry);
      gtk_box_pack_end (GTK_BOX (box), entry, FALSE, TRUE, 0);
      gtk_box_pack_end (GTK_BOX (box), button, FALSE, FALSE, 0);
      widget = entry;
      break;
      
    case GTK_TYPE_ENUM:
      box = gtk_widget_new (GTK_TYPE_HBOX,
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (GTK_TYPE_LABEL,
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      widget = gtk_option_menu_new ();
      gtk_widget_show (widget);
      gtk_box_pack_end (GTK_BOX (box), widget, FALSE, TRUE, 0);
      
      ev = gtk_type_enum_get_values (arg_type);
      if (ev)
	{
	  GtkWidget *menu;
	  
	  menu = gtk_menu_new ();
	  GLE_TAG (menu, "GLE-Editor-Enum-Popup");
	  
	  while (ev->value_nick)
	    {
	      GtkWidget *item;
	      
	      item = gtk_menu_item_new_with_label (ev->value_nick);
	      gtk_widget_lock_accelerators (item);
	      gtk_widget_show (item);
	      gtk_object_set_data (GTK_OBJECT (item), key_arg_ev, ev);
	      gtk_container_add (GTK_CONTAINER (menu), item);
	      
	      ev++;
	    }
	  
	  gtk_option_menu_set_menu (GTK_OPTION_MENU (widget), menu);
	}
      else
	gtk_widget_set_sensitive (GTK_WIDGET (box), FALSE);
      break;
      
    case GTK_TYPE_FLAGS:
      box = gtk_widget_new (GTK_TYPE_HBOX,
			    "GtkBox::homogeneous", FALSE,
			    "GtkBox::spacing", 10,
			    "GtkContainer::border_width", 0,
			    "GtkWidget::visible", TRUE,
			    NULL);
      gtk_box_pack_start (parent_box, box, FALSE, TRUE, 0);
      label = gtk_widget_new (GTK_TYPE_LABEL,
			      "GtkLabel::label", name,
			      "GtkLabel::justify", GTK_JUSTIFY_LEFT,
			      "GtkWidget::visible", TRUE,
			      NULL);
      gtk_box_pack_start (GTK_BOX (box), label, FALSE, TRUE, 0);
      bar = gtk_menu_bar_new ();
      gtk_widget_show (bar);
      gtk_box_pack_end (GTK_BOX (box), bar, FALSE, TRUE, 0);
      widget = gtk_menu_item_new_with_label ("Flags");
      gtk_widget_lock_accelerators (widget);
      gtk_widget_show (widget);
      gtk_menu_bar_append (GTK_MENU_BAR (bar), widget);
      
      ev = gtk_type_flags_get_values (arg_type);
      if (ev)
	{
	  GtkWidget *menu;
	  
	  menu = gtk_menu_new ();
	  GLE_TAG (menu, "GLE-Editor-Flags-Popup");
	  
	  while (ev->value_nick)
	    {
	      GtkWidget *item;
	      
	      item = gtk_check_menu_item_new_with_label (ev->value_nick);
	      gtk_widget_lock_accelerators (item);
	      gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (item), TRUE);
	      gtk_widget_show (item);
	      gtk_object_set_data (GTK_OBJECT (item), key_arg_ev, ev);
	      gtk_container_add (GTK_CONTAINER (menu), item);
	      
	      ev++;
	    }
	  
	  gtk_menu_item_set_submenu (GTK_MENU_ITEM (widget), menu);
	}
      else
	gtk_widget_set_sensitive (GTK_WIDGET (box), FALSE);
      break;
      
    default:
      widget = gtk_widget_new (GTK_TYPE_LABEL,
			       "GtkLabel::label", name,
			       "GtkWidget::sensitive", FALSE,
			       "GtkWidget::visible", TRUE,
			       NULL);
      gtk_misc_set_alignment (GTK_MISC (widget), 0, 0.5);
      gtk_box_pack_start (parent_box, widget, FALSE, FALSE, 0);
      break;
    }
  
  if (widget)
    {
      *widget_p = widget;
      gtk_signal_connect (GTK_OBJECT (widget),
			  "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  widget_p);
      gle_arg_refresh_field (arg, widget);
    }
}

void
gle_arg_refresh_field (GtkArg	 *arg,
		       GtkWidget *widget)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (arg != NULL);
  
  switch (GTK_FUNDAMENTAL_TYPE (arg->type))
    {
      gchar *string;
      gchar buffer[64];
      GtkWidget *menu;
      GtkWidget *button;
      
    case GTK_TYPE_BOOL:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget),
				    GTK_VALUE_BOOL (*arg));
      break;
      
    case GTK_TYPE_INT:
      sprintf (buffer, "%d", GTK_VALUE_INT (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_LONG:
      sprintf (buffer, "%ld", GTK_VALUE_LONG (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_UINT:
      sprintf (buffer, "%u", GTK_VALUE_UINT (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_ULONG:
      sprintf (buffer, "%lu", GTK_VALUE_ULONG (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_FLOAT:
      sprintf (buffer, "%f", GTK_VALUE_FLOAT (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_DOUBLE:
      sprintf (buffer, "%f", GTK_VALUE_DOUBLE (*arg));
      if (!g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), buffer))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), buffer);
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      break;
      
    case GTK_TYPE_STRING:
      string = GTK_VALUE_STRING (*arg);
      if (!string || !g_str_equal (gtk_entry_get_text (GTK_ENTRY (widget)), string))
	{
	  gtk_entry_set_text (GTK_ENTRY (widget), string ? string : "");
	  gtk_entry_set_position (GTK_ENTRY (widget), 0);
	}
      button = gtk_object_get_user_data (GTK_OBJECT (widget));
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), string != NULL);
      break;
      
    case GTK_TYPE_ENUM:
      menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (widget));
      if (menu)
	{
	  guint n;
	  GList *list;
	  
	  n = 0;
	  for (list = GTK_MENU_SHELL (menu)->children; list; list = list->next)
	    {
	      GtkWidget *item;
	      GtkEnumValue *ev;
	      
	      item = list->data;
	      ev = gtk_object_get_data (GTK_OBJECT (item), key_arg_ev);
	      if (ev->value == GTK_VALUE_ENUM (*arg))
		{
		  gtk_option_menu_set_history (GTK_OPTION_MENU (widget), n);
		  break;
		}
	      n++;
	    }
	}
      break;
      
    case GTK_TYPE_FLAGS:
      menu = GTK_MENU_ITEM (widget)->submenu;
      if (menu)
	{
	  GList *list;
	  
	  for (list = GTK_MENU_SHELL (menu)->children; list; list = list->next)
	    {
	      GtkWidget *item;
	      GtkEnumValue *ev;
	      
	      item = list->data;
	      ev = gtk_object_get_data (GTK_OBJECT (item), key_arg_ev);
	      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), GTK_VALUE_FLAGS (*arg) & ev->value);
	    }
	}
      break;
      
    default:
      break;
    }
}

gboolean
gle_arg_set_from_field (GtkArg	  *arg,
			GtkWidget *widget)
{
  gboolean dirty;
  gchar *dummy = NULL;
  
  g_return_val_if_fail (arg != NULL, FALSE);
  g_return_val_if_fail (widget != NULL, FALSE);
  
  dirty = FALSE;
  switch (GTK_FUNDAMENTAL_TYPE (arg->type))
    {
      gboolean bool_data;
      gchar *string;
      gulong ulong_data;
      glong long_data;
      gdouble double_data;
      GtkWidget *item;
      GtkWidget *button;
      GtkWidget *menu;
      gchar *string2;
      guint base;
      
    case GTK_TYPE_BOOL:
      bool_data = GTK_TOGGLE_BUTTON (widget)->active;
      if (bool_data != GTK_VALUE_BOOL (*arg))
	{
	  GTK_VALUE_BOOL (*arg) = bool_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_INT:
      string = gtk_entry_get_text (GTK_ENTRY (widget));
      if (string && string[0] == '0')
	{
	  base = 8;
	  string++;
	  if (string[0] == 'x' || string[0] == 'X')
	    {
	      base = 16;
	      string++;
	    }
	}
      else
	base = 10;
      long_data = strtol (string, &dummy, base);
      if (long_data != GTK_VALUE_INT (*arg))
	{
	  GTK_VALUE_INT (*arg) = long_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_LONG:
      string = gtk_entry_get_text (GTK_ENTRY (widget));
      if (string && string[0] == '0')
	{
	  base = 8;
	  string++;
	  if (string[0] == 'x' || string[0] == 'X')
	    {
	      base = 16;
	      string++;
	    }
	}
      else
	base = 10;
      long_data = strtol (string, &dummy, base);
      if (long_data != GTK_VALUE_LONG (*arg))
	{
	  GTK_VALUE_LONG (*arg) = long_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_UINT:
      string = gtk_entry_get_text (GTK_ENTRY (widget));
      if (string && string[0] == '0')
	{
	  base = 8;
	  string++;
	  if (string[0] == 'x' || string[0] == 'X')
	    {
	      base = 16;
	      string++;
	    }
	}
      else
	base = 10;
      ulong_data = strtol (string, &dummy, base);
      if (ulong_data != GTK_VALUE_UINT (*arg))
	{
	  GTK_VALUE_UINT (*arg) = ulong_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_ULONG:
      string = gtk_entry_get_text (GTK_ENTRY (widget));
      if (string && string[0] == '0')
	{
	  base = 8;
	  string++;
	  if (string[0] == 'x' || string[0] == 'X')
	    {
	      base = 16;
	      string++;
	    }
	}
      else
	base = 10;
      ulong_data = strtol (string, &dummy, base);
      if (ulong_data != GTK_VALUE_ULONG (*arg))
	{
	  GTK_VALUE_ULONG (*arg) = ulong_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_FLOAT:
      double_data = g_strtod (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy);
      if (double_data != GTK_VALUE_FLOAT (*arg))
	{
	  GTK_VALUE_FLOAT (*arg) = double_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_DOUBLE:
      double_data = g_strtod (gtk_entry_get_text (GTK_ENTRY (widget)), &dummy);
      if (double_data != GTK_VALUE_DOUBLE (*arg))
	{
	  GTK_VALUE_DOUBLE (*arg) = double_data;
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_STRING:
      button = gtk_object_get_user_data (GTK_OBJECT (widget));
      if (GTK_TOGGLE_BUTTON (button)->active)
	string = gtk_entry_get_text (GTK_ENTRY (widget));
      else
	string = NULL;
      string2 = GTK_VALUE_STRING (*arg);
      if (string && string2 && strcmp (string, string2) == 0)
	;
      else if (string != string2)
	{
	  g_free (GTK_VALUE_STRING (*arg));
	  GTK_VALUE_STRING (*arg) = g_strdup (string);
	  dirty = TRUE;
	}
      break;
      
    case GTK_TYPE_ENUM:
      item = GTK_OPTION_MENU (widget)->menu_item;
      if (item)
	{
	  GtkEnumValue *ev;
	  
	  ev = gtk_object_get_data (GTK_OBJECT (item), key_arg_ev);
	  
	  if (GTK_VALUE_ENUM (*arg) != ev->value)
	    {
	      GTK_VALUE_ENUM (*arg) = ev->value;
	      dirty = TRUE;
	    }
	}
      break;
      
    case GTK_TYPE_FLAGS:
      menu = GTK_MENU_ITEM (widget)->submenu;
      if (menu)
	{
	  GList *list;
	  guint v;
	  
	  v = 0;
	  for (list = GTK_MENU_SHELL (menu)->children; list; list = list->next)
	    {
	      GtkEnumValue *ev;
	      
	      item = list->data;
	      ev = gtk_object_get_data (GTK_OBJECT (item), key_arg_ev);
	      if (GTK_CHECK_MENU_ITEM (item)->active)
		v |= ev->value;
	    }
	  
	  if (v != GTK_VALUE_FLAGS (*arg))
	    {
	      GTK_VALUE_FLAGS (*arg) = v;
	      dirty = TRUE;
	    }
	}
      break;
      
    default:
      break;
    }
  
  return dirty;
}
