/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleshell.h"
#include	"glepublic.h"
#include	"gleprivate.h"
#include	"glegobject.h"
#include	"glegwidget.h"
#include	"gleselector.h"
#include	"glemenus.h"
#include	"gleeditor.h"
#include	<gdk/gdk.h>



/* --- signals --- */
enum
{
  SIGNAL_LAST
};


/* --- clist columns --- */
enum
{
  CLIST_COL_TYPE,
  CLIST_COL_STATE,
  CLIST_COL_REFCOUNT,
  CLIST_COL_GLENAME,
  CLIST_COL_NAME,
  CLIST_N_COLS
};
static guint	clist_init_col_width[CLIST_N_COLS] =
{
  50	/* CLIST_COL_TYPE */,
  50	/* CLIST_COL_STATE */,
  50	/* CLIST_COL_REFCOUNT */,
  50	/* CLIST_COL_GLENAME */,
  50	/* CLIST_COL_NAME */,
};
#define	CLIST_INITIAL_COL_WIDTH(x)	(clist_init_col_width[(x)])


/* --- clist row data --- */
#define	GLE_ROW_DATA_INIT_ALLOC		(64)
typedef	struct	_GleRowData	GleRowData;
struct _GleRowData
{
  GleShell *shell;
  GtkWidget *widget;
  gint row;
  gint destroy_handler;
};


/* --- prototypes --- */
static guint		gle_shell_get_type		(void);
static void		gle_shell_class_init		(GleShellClass	*class);
static void		gle_shell_init			(GleShell	*shell);
static void		gle_shell_set_customer_name	(GleShell	*shell);
static gint		gle_shell_deletion_request	(GleShell	*shell);
static void		gle_shell_destroy		(GtkObject	*object);
static void		gle_shell_finalize		(GtkObject	*object);
static void		gle_shell_row_selected		(GtkCList	*clist,
							 gint		 row,
							 gint		 column,
							 GdkEventButton	*event,
							 GleShell	*shell);
static void		gle_shell_customer_destroyed	(GtkWidget	*customer,
							 GleShell	*shell);
static void		gle_shell_clist_append_widget	(GtkWidget	*widget,
							 GleShell   *shell);


/* --- variables --- */
static GtkWindowClass	*parent_class = NULL;
static GleShell		*gle_shell = NULL;


/* --- functions --- */
static guint
gle_shell_get_type (void)
{
  static guint shell_type = 0;
  
  if (!shell_type)
    {
      GtkTypeInfo shell_info =
      {
	"GleShell",
	sizeof (GleShell),
	sizeof (GleShellClass),
	(GtkClassInitFunc) gle_shell_class_init,
	(GtkObjectInitFunc) gle_shell_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      shell_type = gtk_type_unique (gtk_window_get_type (), &shell_info);
    }
  
  return shell_type;
}

static void
gle_shell_class_init (GleShellClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  widget_class->delete_event = (gint(*)(GtkWidget*,GdkEventAny*)) gle_shell_deletion_request;

  object_class->destroy = gle_shell_destroy;
  object_class->finalize = gle_shell_finalize;
}

static void
gle_shell_candidate_check (GtkObject      *caller,
			   GtkWidget      *new_candidate,
			   gint           *candidate_ok)
{
  g_return_if_fail (caller != NULL);
  g_return_if_fail (GLE_IS_SHELL (caller));

  *candidate_ok = (*candidate_ok &&
		   new_candidate &&
		   new_candidate != GTK_WIDGET (caller) &&
		   !GLE_HAS_TAG (new_candidate));
}

static void
gle_shell_detach_list_popup (GtkWidget	   *widget,
			     GtkMenu	   *menu)
{
  GleShell *shell;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GLE_IS_SHELL (widget));

  shell = GLE_SHELL (widget);
  g_return_if_fail (shell->list_popup == (GtkWidget*) menu);

  shell->list_popup = NULL;
}

static void
gle_shell_init (GleShell *shell)
{
  GtkWidget *shell_vbox;
  GtkWidget *main_vbox;
  GtkWidget *status_hbox;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *any;
  gchar *string;
  guint i;
  gchar *clist_titles[CLIST_N_COLS] =
  {
    "Type"	/* CLIST_COL_TYPE */,
    "State"	/* CLIST_COL_STATE */,
    "RefCount"	/* CLIST_COL_REFCOUNT */,
    "GleName"	/* CLIST_COL_GLENAME */,
    "Name"	/* CLIST_COL_NAME */,
  };

  shell->customer = NULL;
  shell->customer_destroy_handler = 0;

  string = g_strconcat ("GLE Engine", "	 [", gle_prg_name (), "]", NULL);
  gtk_widget_set (GTK_WIDGET (shell),
		  "GtkWidget::name", "GLE-Shell",
		  "GtkWindow::title", string,
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::allow_shrink", TRUE,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  g_free (string);
  shell_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", shell,
		    "GtkWidget::visible", TRUE,
		    NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", shell_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (shell_vbox), main_vbox, TRUE, TRUE, 0, GTK_PACK_START);
  status_hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), status_hbox, FALSE, TRUE, 0, GTK_PACK_START);
  shell->customer_label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "<None>",
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_set_child_packing (GTK_BOX (main_vbox), shell->customer_label, FALSE, TRUE, 0, GTK_PACK_START);
  shell->clist = gtk_clist_new_with_titles (CLIST_N_COLS, clist_titles);
  gtk_widget_set (shell->clist,
		  "GtkWidget::width", 200,
		  "GtkWidget::height", 200,
		  "GtkObject::signal::select_row", gle_shell_row_selected, shell,
		  "GtkWidget::visible", TRUE,
		  NULL);
  gtk_clist_set_policy (GTK_CLIST (shell->clist), GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_selection_mode (GTK_CLIST (shell->clist), GTK_SELECTION_BROWSE);
  gtk_clist_column_titles_passive (GTK_CLIST (shell->clist));
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (shell->clist), i, CLIST_INITIAL_COL_WIDTH (i));
  gtk_box_pack_start (GTK_BOX (main_vbox), shell->clist, TRUE, TRUE, 0);
  shell->row_data_mem_chunk =
    g_mem_chunk_new ("GleRowData mem chunks",
		     sizeof (GleRowData),
		     sizeof (GleRowData) * GLE_ROW_DATA_INIT_ALLOC,
		     G_ALLOC_AND_FREE);
  shell->clist_indent = NULL;
  hbox =
    gtk_widget_new (gtk_hbox_get_type (),
		    "GtkBox::homogeneous", TRUE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), hbox, FALSE, TRUE, 0);
  shell->update_button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Update Tree",
		    "GtkObject::object_signal::clicked", gle_shell_update_tree, shell,
		    "GtkWidget::sensitive", FALSE,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), shell->update_button, FALSE, TRUE, 0);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Select Window",
		    "GtkObject::object_signal::clicked", gle_shell_popup_selector, shell,
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 0);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::visible", TRUE,
		    NULL);
  gtk_box_pack_start (GTK_BOX (shell_vbox), any, FALSE, TRUE, 0);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Close",
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::has_default", TRUE,
		    "GtkObject::object_signal::clicked", gle_shell_deletion_request, shell,
		    NULL);
  gtk_box_pack_start (GTK_BOX (shell_vbox), button, FALSE, TRUE, 0);

  shell->list_popup = gle_menu_widget_ops_new ();
  gtk_menu_attach_to_widget (GTK_MENU (shell->list_popup),
			     GTK_WIDGET (shell),
			     gle_shell_detach_list_popup);

  shell->selector =
    gtk_widget_new (gle_selector_get_type (),
		    "GtkObject::object_signal::candidate_check", gle_shell_candidate_check, shell,
		    "GtkObject::signal::destroy", gtk_widget_destroyed, &shell->selector,
		    "GtkWindow::title", "GLE - Selector",
		    NULL);
}

static void
gle_shell_destroy (GtkObject *object)
{
  GleShell *shell;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_SHELL (object));
  
  shell = GLE_SHELL (object);
  g_assert (shell == gle_shell);

  gle_set_flash_widget (NULL);

  gle_shell_set_customer (shell, NULL);

  if (shell->list_popup)
    gtk_widget_destroy (shell->list_popup);

  gtk_widget_destroy (shell->selector);
  
  gle_root_shutdown ();
  
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);

  gle_shell = NULL;
}

static void
gle_shell_finalize (GtkObject *object)
{
  GleShell *shell;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_WINDOW (object));
  
  shell = GLE_SHELL (object);
  g_mem_chunk_destroy (shell->row_data_mem_chunk);
  shell->row_data_mem_chunk = NULL;
  
  GTK_OBJECT_CLASS(parent_class)->finalize (object);
}

GleShell*
gle_shell_get (void)
{
  if (!gle_shell)
    {
      gle_shell = GLE_SHELL (gtk_widget_new (gle_shell_get_type (), NULL));
      GLE_TAG (gle_shell);
      GLE_TAG (gle_shell->list_popup);
      /* FIXME: GLE_TAG (gle_shell->selector); */
    }

  return gle_shell;
}

void
gle_shell_popup (GleShell *shell)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));
  
  if (!GTK_WIDGET_VISIBLE (shell))
    gtk_widget_show (GTK_WIDGET (shell));

  gdk_window_raise (GTK_WIDGET (shell)->window);
}

void
gle_shell_popup_selector (GleShell	 *shell)
{
  GtkWidget *customer;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));

  gle_shell_set_customer (shell, NULL);

  customer = gle_selector_make_selection (GLE_SELECTOR (shell->selector));

  if (customer && GTK_OBJECT (customer)->ref_count > 1)
    {
      gle_selector_reset (GLE_SELECTOR (shell->selector));
      gle_shell_set_customer (shell, customer);
    }
  else
    gle_selector_reset (GLE_SELECTOR (shell->selector));
}

static void
gle_shell_customer_destroyed (GtkWidget	     *customer,
			      GleShell	     *shell)
{
  if (shell->customer == customer)
    {
      shell->customer = NULL;
      gtk_signal_disconnect (GTK_OBJECT (customer), shell->customer_destroy_handler);
      shell->customer_destroy_handler = 0;
      gtk_widget_unref (customer);
      gle_shell_set_customer_name (shell);
      gle_shell_update_tree (shell);
    }
}

void
gle_shell_set_customer (GleShell	*shell,
			GtkWidget	*customer)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));

  if (customer)
    {
      g_return_if_fail (GTK_IS_WIDGET (customer));
      g_return_if_fail (customer->parent == NULL);
      g_return_if_fail (!GLE_HAS_TAG (customer));
    }

  if (shell->customer)
    gle_shell_customer_destroyed (shell->customer, shell);
  
  if (customer)
    {
      shell->customer = customer;
      gtk_widget_ref (customer);

      shell->customer_destroy_handler =
	gtk_signal_connect (GTK_OBJECT (shell->customer),
			    "destroy",
			    GTK_SIGNAL_FUNC (gle_shell_customer_destroyed),
			    shell);
      gle_shell_set_customer_name (shell);
      gle_shell_update_tree (shell);
    }
}

GtkWidget*
gle_shell_get_customer (GleShell       *shell)
{
  g_return_val_if_fail (shell != NULL, NULL);
  g_return_val_if_fail (GLE_IS_SHELL (shell), NULL);

  return shell->customer;
}

static void
gle_shell_set_customer_name (GleShell *shell)
{
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));

  if (shell->customer)
    {
      gchar *string;
      gchar *name;

      if (shell->customer->name)
	name = shell->customer->name;
      else if (GTK_IS_WINDOW (shell->customer) &&
	       GTK_WINDOW (shell->customer)->title)
	name = GTK_WINDOW (shell->customer)->title;
      else if (GTK_IS_MENU (shell->customer))
	name = "<Unnamed>";
      else
	name = "<Unknown>";
      
      string = g_strconcat (gtk_type_name (GTK_OBJECT_TYPE (shell->customer)),
			    "::",
			    name,
			    NULL);
      gtk_label_set (GTK_LABEL (shell->customer_label), string);
      gtk_widget_set_sensitive (shell->customer_label, TRUE);
      gtk_widget_set_sensitive (shell->update_button, TRUE);
      g_free (string);
    }
  else
    {
      gtk_label_set (GTK_LABEL (shell->customer_label), "<None>");
      gtk_widget_set_sensitive (shell->customer_label, FALSE);
      gtk_widget_set_sensitive (shell->update_button, FALSE);
    }
}

static void
gle_row_data_chunk_free (GtkCList	*clist,
			 guint		 row,
			 gpointer	*row_data_p,
			 gpointer	 func_data)
{
  if (*row_data_p)
    {
      GMemChunk *mem_chunk;
      GleRowData *row_data;

      mem_chunk = (GMemChunk*) func_data;
      row_data = *row_data_p;

      /* printf ("clear: row_data(%p) widget(%p) row(%d) hid(%d)\n", row_data, row_data->widget, row_data->row, row_data->destroy_handler); */

      g_return_if_fail (row_data->row >= 0);

      *row_data_p = NULL;
      gtk_signal_disconnect (GTK_OBJECT (row_data->widget), row_data->destroy_handler);
      g_mem_chunk_free (mem_chunk, row_data);
    }
}

void
gle_shell_update_tree (GleShell	      *shell)
{
  guint i;

  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));

  gtk_clist_freeze (GTK_CLIST (shell->clist));
  gtk_clist_row_foreach (GTK_CLIST (shell->clist),
			 gle_row_data_chunk_free,
			 shell->row_data_mem_chunk);
  gtk_clist_clear (GTK_CLIST (shell->clist));
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (shell->clist), i, CLIST_INITIAL_COL_WIDTH (i));
  if (shell->customer)
    {
      shell->clist_indent = g_strdup ("");
      gle_shell_clist_append_widget (shell->customer, (gpointer) shell);
      g_free (shell->clist_indent);
      shell->clist_indent = NULL;
    }

  gtk_clist_thaw (GTK_CLIST (shell->clist));
}

static void
gle_row_data_destroyed (GtkWidget *widget,
			GleRowData *row_data)
{
  GtkCList *clist;
  GleShell *shell;
  guint row;

  /* printf ("destroy: x-widget(%p) row_data(%p) widget(%p) row(%d) hid(%d)\n", widget, row_data, row_data->widget, row_data->row, row_data->destroy_handler); */

  shell = row_data->shell;
  clist = GTK_CLIST (shell->clist);
  row = row_data->row;
  row_data->row = -44;
  g_return_if_fail (row >= 0);
  gtk_clist_set_row_data (clist, row, NULL);
  gtk_signal_disconnect (GTK_OBJECT (row_data->widget), row_data->destroy_handler);
  gtk_clist_set_text (clist, row, CLIST_COL_REFCOUNT, "????");
  gtk_clist_set_text (clist, row, CLIST_COL_STATE, "Destroyed");
  gtk_clist_set_foreground (clist, row, & GTK_WIDGET (clist)->style->fg[GTK_STATE_INSENSITIVE]);
  gtk_clist_set_background (clist, row, & GTK_WIDGET (clist)->style->bg[GTK_STATE_INSENSITIVE]);
  g_mem_chunk_free (shell->row_data_mem_chunk, row_data);
}

static void
gle_shell_clist_append_widget (GtkWidget  *widget,
			       GleShell	  *shell)
{
  GtkCList *clist;
  GleRowData *row_data;
  gchar *text[CLIST_N_COLS];
  const gchar *s;
  guint i;

  clist = GTK_CLIST (shell->clist);

  text[CLIST_COL_TYPE] = g_strconcat (shell->clist_indent,
				      gtk_type_name (GTK_OBJECT_TYPE (widget)),
				      NULL);
  text[CLIST_COL_REFCOUNT] = g_new (gchar, 32);
  sprintf (text[CLIST_COL_REFCOUNT], "%04d", GTK_OBJECT (widget)->ref_count);
  s = gle_widget_get_glename (widget);
  text[CLIST_COL_GLENAME] = g_strconcat ("\"",
					 s ? s : "",
					 "\"",
					 NULL);
  text[CLIST_COL_NAME] = g_strconcat ("\"",
				      widget->name ? widget->name : "",
				      "\"",
				      NULL);
  switch (widget->state)
    {
    case  GTK_STATE_ACTIVE:		text[CLIST_COL_STATE] = g_strdup ("(x) Active"); break;
    case  GTK_STATE_PRELIGHT:		text[CLIST_COL_STATE] = g_strdup ("(x) Prelight"); break;
    case  GTK_STATE_SELECTED:		text[CLIST_COL_STATE] = g_strdup ("(x) Selected"); break;
    case  GTK_STATE_INSENSITIVE:	text[CLIST_COL_STATE] = g_strdup ("(x) Insensitive"); break;
    default:				text[CLIST_COL_STATE] = g_strdup ("(x) Normal"); break;
    }
  text[CLIST_COL_STATE][1] = GTK_WIDGET_VISIBLE (widget) ? 'V' : 'H';
  for (i = 0; i < CLIST_N_COLS; i++)
    {
      gint new_width;

      new_width = gdk_text_width (GTK_WIDGET (clist)->style->font, text[i], strlen (text[i]));
      if (clist->column[i].width < new_width)
	gtk_clist_set_column_width (clist, i, new_width);
    }

  row_data = g_mem_chunk_alloc (shell->row_data_mem_chunk);
  row_data->shell = shell;
  row_data->widget = widget;
  row_data->row = gtk_clist_append (clist, text);
  gtk_clist_set_row_data (clist, row_data->row, row_data);
  row_data->destroy_handler =
    gtk_signal_connect (GTK_OBJECT (widget),
			"destroy",
			GTK_SIGNAL_FUNC (gle_row_data_destroyed),
			row_data);
  for (i = 0; i < CLIST_N_COLS; i++)
    g_free (text[i]);

  if (GTK_IS_CONTAINER (widget))
    {
      gchar *old_indent;

      old_indent = shell->clist_indent;
      shell->clist_indent = g_strconcat (old_indent, "	 ", NULL);
      gtk_container_foreach (GTK_CONTAINER (widget),
			     (GtkCallback) gle_shell_clist_append_widget,
			     shell);
      g_free (shell->clist_indent);
      shell->clist_indent = old_indent;
    }
}

static gint
gle_shell_deletion_request (GleShell	*shell)
{
  gle_set_flash_widget (NULL);

  gtk_widget_destroy (GTK_WIDGET (shell));

  return FALSE;
}

static void
gle_shell_row_selected (GtkCList       *clist,
			gint		row,
			gint		column,
			GdkEventButton *event,
			GleShell       *shell)
{
  g_return_if_fail (clist != NULL);
  g_return_if_fail (GTK_IS_CLIST (clist));
  g_return_if_fail (shell != NULL);
  g_return_if_fail (GLE_IS_SHELL (shell));

  if (row >= 0)
    {
      gle_set_flash_widget (NULL);

      if (event)
	{
	  if (event->type == GDK_BUTTON_PRESS)
	    {
	      if (event->button == 2)
		{
		  GleRowData *row_data;

		  row_data = gtk_clist_get_row_data (clist, row);
		  if (row_data)
		    {
		      GtkWidget *dialog;

		      dialog = gle_editor_popup (row_data->widget);
		      gle_root_connect_object_life (GTK_OBJECT (dialog));
		    }
		}
	      else if (event->button == 3)
		{
		  GleRowData *row_data;

		  row_data = gtk_clist_get_row_data (clist, row);
		  if (row_data)
		    {
		      gle_menu_widget_ops_popup (GTK_MENU (shell->list_popup),
						 row_data->widget,
						 event->x_root,
						 event->y_root,
						 event->button,
						 event->time);
		    }
		}
	    }
	  else if (event->type == GDK_2BUTTON_PRESS)
	    {
	      if (column == CLIST_COL_REFCOUNT &&
		  event->button == 1 &&
		  (event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_CONTROL_MASK))
		{
		  GleRowData *row_data;
		  
		  row_data = gtk_clist_get_row_data (clist, row);
		  if (row_data)
		    {
		      gchar string[32];
		      
		      if (event->state & GDK_SHIFT_MASK)
			gtk_object_ref (GTK_OBJECT (row_data->widget));
		      else
			gtk_object_unref (GTK_OBJECT (row_data->widget));
		      sprintf (string, "%04d", GTK_OBJECT (row_data->widget)->ref_count);
		      gtk_clist_set_text (clist, row, CLIST_COL_REFCOUNT, string);
		    }
		}
	    }
	}
    }
}
