/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glemisc.h"
#include	"glepopups.h"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<ctype.h>
#include	<gtk/gtkprivate.h>




/* --- variables --- */
static GtkWidget *gle_flash_widget = NULL;


/* --- functions --- */
void
gle_widget_make_sensitive (GtkWidget	*widget)
{
  if (!GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, TRUE);
}

void
gle_widget_make_insensitive (GtkWidget	*widget)
{
  if (GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, FALSE);
}


static gboolean	gtk_types_hard_initialized = FALSE;

static void
gle_hard_initialize_gtk_types (void)
{
  gtk_types_hard_initialized = TRUE;
#define GLE_TYPE_CALL(type_func)	type_func ();
#  include "gletypecalls.c"
#undef	GLE_TYPE_CALL
}

GtkType
gle_type_from_name (const gchar      *arg_name,
		    const gchar	**class_name_p)
{
  static gchar	*buffer = NULL;
  static guint	buffer_len = 0;
  GtkType	widget_type;
  const gchar	*p;
  guint		class_length;

  g_return_val_if_fail (arg_name != NULL, 0);
  
  p = strchr (arg_name, ':');
  if (!p)
    p = arg_name + strlen (arg_name);
  
  class_length = p - arg_name;

  g_return_val_if_fail (class_length > 0, 0);
  
  if (class_length < buffer_len)
    {
      g_free (buffer);
      buffer_len = class_length + 1;
      buffer = g_new (gchar, buffer_len);
    }
  g_memmove (buffer, arg_name, class_length);
  buffer[class_length + 1] = 0;
  
  widget_type = gtk_type_from_name (buffer);
  if (!widget_type && !gtk_types_hard_initialized)
    {
      gle_hard_initialize_gtk_types ();
      widget_type = gtk_type_from_name (buffer);
    }

  if (widget_type && class_name_p)
    *class_name_p = buffer;

  return widget_type;
}

static gint
gle_flash_timer (gpointer data)
{
  static gboolean toggle = FALSE;
  GtkWidget *widget;
  GdkGC *gc;

  toggle = !toggle;

  widget = gle_flash_widget;
  if (!widget || !widget->window)
    return TRUE;

  if (toggle)
    gc = gtk_widget_get_default_style ()->white_gc;
  else
    gc = gtk_widget_get_default_style ()->black_gc;

  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width-1,
			   widget->allocation.height-1);
  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width,
			   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x,
		   widget->allocation.y,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y +
		   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y,
		   widget->allocation.x,
		   widget->allocation.y +
		   widget->allocation.height);
  

  return TRUE;
}

static void
gle_expose_widget (GtkWidget *widget)
{
  if (widget->window)
    gdk_window_clear_area_e (widget->window, 0, 0, 0, 0);
}

void
gle_set_flash_widget (GtkWidget *widget)
{
  static gint flash_timer = 0;
  static gint flash_widget_handler1;
  static gint flash_widget_handler2;

  if (widget)
    g_return_if_fail (GTK_IS_WIDGET (widget));

  if (!flash_timer && widget)
    flash_timer = gtk_timeout_add (GLE_FLASH_MSEC, gle_flash_timer, NULL);

  if (flash_timer && !widget)
    {
      gtk_timeout_remove (flash_timer);
      flash_timer = 0;
    }

  if (gle_flash_widget)
    {
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler1);
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler2);
      gle_expose_widget (gle_flash_widget);
    }

  gle_flash_widget = widget;
  if (gle_flash_widget)
    {
      flash_widget_handler1 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "destroy",
							 GTK_SIGNAL_FUNC (gle_set_flash_widget),
							 NULL);
      flash_widget_handler2 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "size_allocate",
							 GTK_SIGNAL_FUNC (gle_expose_widget),
							 GTK_OBJECT (gle_flash_widget));
    }
}

GtkWidget*
gle_get_flash_widget (void)
{
  return gle_flash_widget;
}

gint
gtk_clist_find_row (GtkCList *clist,
		    gpointer  data)
{
  GList *list;
  gint n;

  g_return_val_if_fail (clist != NULL, -1);
  g_return_val_if_fail (GTK_IS_CLIST (clist), -1);

  if (clist->rows < 1)
    return -1;

  n = 0;
  list = clist->row_list;
  while (list)
    {
      GtkCListRow *clist_row;

      clist_row = list->data;
      if (clist_row->data == data)
	break;
      n++;
      list = list->next;
    }

  if (list)
    return n;

  return -1;
}

static	gint	destroy_handler = 0;
static GList	*destroy_queue = NULL;

static gint
gtk_delayed_destroy_handler (gpointer data)
{
  GList *list;

  list = destroy_queue;
  while (list)
    {
      GtkWidget *widget;

      widget = list->data;
      list = list->next;

      gtk_widget_destroy (widget);
      destroy_queue = g_list_remove (destroy_queue, widget);
    }

  destroy_handler = 0;
  return FALSE;
}

void
gtk_clist_row_foreach (GtkCList               *clist,
		       GtkCListRowFunc        func,
		       gpointer               func_data)
{
  GList *list;
  guint n;

  g_return_if_fail (clist != NULL);
  g_return_if_fail (GTK_IS_CLIST (clist));
  g_return_if_fail (func != NULL);

  n = 0;
  list = clist->row_list;
  while (list)
    {
      GtkCListRow *clist_row;

      clist_row = list->data;
      list = list->next;
      (*func) (clist, n++, &clist_row->data, func_data);
    }
}

void
gtk_widget_queue_destroy (GtkWidget      *widget)
{
  if (!g_list_find (destroy_queue, widget))
    {
      if (!destroy_handler)
	destroy_handler = gtk_idle_add (gtk_delayed_destroy_handler, NULL);
      destroy_queue = g_list_prepend (destroy_queue, widget);
    }
}

GList*
gtk_widget_query_arg_list (GtkWidget *widget,
			   GtkType    class_type)
{
  GtkArg* args;
  gint nargs;
  GList *list;
  guint i;

  nargs = 0;
  args = gtk_object_query_args	 (class_type, &nargs);
  gtk_widget_set (widget, "GtkObject::signal::destroy", g_free, args, NULL);

  list = NULL;
  for (i = nargs; i > 0; i--)
    {
      list = g_list_prepend (list, &args[i - 1]);
    }

  return list;
}

gchar*
g_strdown (gchar  *string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = tolower (*s);
      s++;
    }

  return string;
}

gchar*
g_strup (gchar	*string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = toupper (*s);
      s++;
    }

  return string;
}

gchar*
g_basename (const gchar	   *file_name)
{
  register gchar *base;

  g_return_val_if_fail (file_name != NULL, NULL);

  base = strrchr (file_name, '/');
  if (base)
    base++;
  else
    base = (gchar*) file_name;

  return base;
}

gchar*
g_strconvert (gchar	     *mod_string,
	      gchar	     new_delim,
	      gboolean	     downcase,
	      gboolean	     upcase)
{
  register gchar *delimiters = "_-|> <.";
  register gchar *string;

  g_return_val_if_fail (mod_string != NULL, NULL);

  string = mod_string;
  while (*string)
    {
      if (new_delim && strchr (delimiters, *string))
	*string = new_delim;
      else if (upcase)
	*string = toupper (*string);
      else if (downcase)
	*string = tolower (*string);

      string++;
    }

  return mod_string;
}
