/*	ScianFileSystem.c
	manages file system window for scian

	Jim Lyons
	created 4/17/91

	4/28/91 version 2 of files window
	5/1/91  version 3: made file objects
	5/3/91	added format dialog routines
	5/29/91 EMP removed lib headers
	6/25/91 removed icon definitions
	7/22/91 added Read Data Files menu item
	8/8/91	guarded all strcpy and strcat calls
	9/9/91	added help strings
	9/16/91	redesigned file window
	9/19/91 added file info window
	10/12/91 removed Set File Type stuff
	10/28/91 changed interface to File Readers (with EMP)
*/

#include "Scian.h"
#include "ScianStyle.h"
#include "ScianTypes.h"
#include "ScianMethods.h"
#include "ScianArrays.h"
#include "ScianWindows.h"
#include "ScianTextBoxes.h"
#include "ScianSliders.h"
#include "ScianButtons.h"
#include "ScianTitleBoxes.h"
#include "ScianObjWindows.h"
#include "ScianColors.h"
#include "ScianControls.h"
#include "ScianTimers.h"
#include "ScianLists.h"
#include "ScianIDs.h"
#include "ScianEvents.h"
#include "ScianErrors.h"
#include "ScianDialogs.h"
#include "ScianIcons.h"
#include "ScianFiles.h"
#include "ScianDatasets.h"
#include "ScianScripts.h"
#include "ScianPreferences.h"
#include "ScianFileSystem.h"

#include <pwd.h>
#include <time.h>

/* defs for file window parameters */

#undef TITLEBOXTOP
#define TITLEBOXTOP	14

#define PARENTBTNWID	84
#define PARENTBTNHT	70
#define MINBTNWID	120
#define FILTERBTNWID	208

#define FILESWIDTH	(2*MINORBORDER + 2*SMALLGAP + 3*MINBTNWID)
#define FILESHEIGHT	330


/* defs for file format dialog box */
#define RADIOWID	84
#define BOTPART (BOTMARGIN + OKBTNHT + INTERSPACE)

#define STICKYALL (STICKYRIGHT+STICKYLEFT+STICKYBOTTOM+STICKYTOP)

/* defs for file info window */
#define NINFOITEMS	5
#define INFOITEMHT	40
#define INFOITEMWID	110
#define INFONAMEWIDTH	70
#define INFOFIELDWIDTH	200
#define INFOFIELDHEIGHT	(NINFOITEMS*INFOITEMHT)
#define INFOWIDTH	(3*MINORBORDER + INFONAMEWIDTH + INFOFIELDWIDTH)
#define INFOHEIGHT	(MINORBORDER + MAJORBORDER + INTERSPACE + INFOFONTSIZE + BARWIDTH + INFOFIELDHEIGHT)
#define INFOFONT	"Helvetica-Bold"
#define INFOFONTSIZE	12


/* file selector bit mask */
#define MASK(f)		(1<<f)

#define DEFAULTFILTER	(MASK(DIRFILE) | MASK(DATAFILE) | MASK(LOGFILE))
#define ALLFILES	(MASK(NFILETYPES) - 1)

/*** for ScianIDs.h ***/
#ifndef FIELD
#define FIELD 737
#define OPEN  747
#define READER 757
#endif

/*** for ScianIcons.h ***/
#ifndef ICONUNKNOWN
#define ICONUNKNOWN 61
#endif

/* static function prototypes */
#ifdef PROTO

static ObjPtr ReadDataFile(ObjPtr);
static ObjPtr SetFormat(ObjPtr);
static ObjPtr ShowInfo(ObjPtr);
static char *GetFileInfo(char *);
static void FileFormatDialog(FuncTyp);
static ObjPtr CancelDialog(ObjPtr);
static ObjPtr OKDialog(ObjPtr);
static ObjPtr OKFormat(int, ObjPtr);
static void ChangeFileType(ObjPtr, int);
static ObjPtr SelFileList(ObjPtr);
static ObjPtr SelDirList(ObjPtr);
static ObjPtr NewFileWindow(WinInfoPtr, char *);
static Bool ReadDirectory(ObjPtr);
static ObjPtr ShowFiles(ObjPtr);
static ObjPtr DoParent(ObjPtr);
static ObjPtr DoOpen(ObjPtr);
static ObjPtr OpenFolder(ObjPtr);
static ObjPtr NewFilter(ObjPtr);
static int FileType(char *);

#else

static ObjPtr ReadDataFile();
static ObjPtr SetFormat();
static ObjPtr ShowInfo();
static char *GetFileInfo();
static void FileFormatDialog();
static ObjPtr CancelDialog();
static ObjPtr OKDialog();
static ObjPtr OKFormat();
static void ChangeFileType();
static ObjPtr SelFileList();
static ObjPtr SelDirList();
static ObjPtr NewFileWindow();
static Bool ReadDirectory();
static ObjPtr ShowFiles();
static ObjPtr DoParent();
static ObjPtr DoOpen();
static ObjPtr OpenFolder();
static ObjPtr NewFilter();
static int FileType();

#endif

/********************************************************* GLOBALS */

int fileIcon[NFILETYPES] = {ICONFOLDER, ICONBINFILE, ICONOBJFILE, ICONTEXTFILE, ICONDATAFILE, 
			ICONSCIANDOC, ICONUNKNOWN};
char *fileTypeName[NFILETYPES] = {"Dir", "Bin", "Obj", "Text", "Data", 
			"SciAn Log", "Other"};
ObjPtr stash;

 /* pointers for deferred task to activate buttons in file window */
static ObjPtr gOpenBtn, gSetFormatBtn, gShowInfoBtn, gIconList = NULLOBJ;

/********************************************************** INIT FILE SYSTEM */
void InitFileSystem(void)
{
	stash = NewObject(NULLOBJ, 0);
	AddToReferenceList(stash);
}

void KillFileSystem(void)
{
	DeleteThing(stash);
}

/******************************************************* DO NEW FILE WINDOW */
void DoNewFileWindow(void)
{
	ObjPtr theStr;
	WinInfoPtr askWindow;

	if (logging)
	{
		Log("Show FilesWindow\n");
		InhibitLogging(TRUE);
	}
	if (theStr = GetVar(stash, DATA))
	{
		strncpy(tempStr, GetString(theStr), TEMPSTRSIZE);
		tempStr[TEMPSTRSIZE] = '\0';
	}
	else if (PrefExists(PREF_DEFDIR))
	{
		strncpy(tempStr, GetPrefString(PREF_DEFDIR), TEMPSTRSIZE);
		tempStr[TEMPSTRSIZE] = '\0';
	}
	else getwd(tempStr);
	if (selWindow) askWindow = AskUser(selWinInfo, "Enter directory name:",
		 (FuncTyp) NewFileWindow, tempStr);
	if (askWindow) SetVar((ObjPtr) askWindow, HELPSTRING, NewString("This dialog window \
allows you to enter the name of the directory for which you want to open a new File Window. The \
default directory name, showing when the window first comes up, may be set in the Preferences window."));
	if (logging) InhibitLogging(FALSE);
}

static ObjPtr NewFileAgain(WinInfoPtr win, int n)
{
	if (n == 1) DoNewFileWindow();
	else SetVar(stash, DATA, NULLOBJ); /* clear error string */
	return NULLOBJ;
}

#ifdef PROTO
static ObjPtr NewFileWindow(WinInfoPtr owner, char *replyText)
#else
static ObjPtr NewFileWindow(owner, replyText)
WinInfoPtr owner;
char *replyText;
#endif
{
	char *s, *t = NULL;
	WinInfoPtr alert;

	if (*replyText == '~' && (s = getenv("HOME"))) /* handle squiggle */
	{
		t = malloc(strlen(replyText) + strlen(s) + 1);
		/***/
		strcpy(t, s);
		strcat(t, "/");
		strcat(t, replyText+1);
	}

	if (chdir(t ? t : ( *replyText ? replyText : "/") ) == 0)
	{
		GetFileWindow();
		SetVar(stash, DATA, NULLOBJ); /* clear error string */
	}
	else /* error */
	{
#define PROMPT1 "Could not open directory "
#define PROMPT2 ". Do you want to enter another directory or cancel the request?"

		/* save text for user reediting */
		SetVar(stash, DATA, NewString(t ? t : replyText));

		/* set up prompt in tempStr */
		strcpy(tempStr, PROMPT1);
		strncat(tempStr, (t ? t : replyText), 
				TEMPSTRSIZE - strlen(PROMPT1) - strlen(PROMPT2) - 1);
		strcat(tempStr, PROMPT2);
		alert = AlertUser(/***/ UIYELLOW, owner, tempStr, NewFileAgain,
				2, "Cancel", "Re-enter");
		if (alert) SetVar((ObjPtr) alert, HELPSTRING, NewString("This dialog window \
came up because the directory entered in the New File Window dialog could not be opened. Press \
the Re-enter button to correct that entry, or press the Cancel key to cancel the request."));
	}
	if (t) free(t);
}

/******************************************************* SET FORMAT */
#ifdef PROTO
static ObjPtr SetFormat(ObjPtr obj)
#else
static ObjPtr SetFormat(obj)
ObjPtr obj;
#endif
{
	DoTask(DoSetFileFormat);
	return NULLOBJ;
}

void DoSetFileFormat(void)
{
	ObjPtr corral, theList;
	int fmt, i,n;

	if (!selWindow || !(corral = GetVar((ObjPtr) selWinInfo, CORRAL))
		|| ListCount(theList = SelFileList(corral)) == 0) return;

	if (logging)
	{
		Log("Set FileFormat\n");
		InhibitLogging(TRUE);
	}

	/* now make dialog */
	FileFormatDialog(OKFormat);

	/*** assuming selWinInfo is now the new dialog ... */
	SetVar((ObjPtr) selWinInfo, FILELIST, theList); /* stash file list with dialog */
	SetVar(theList, CORRAL, corral); /* ref by OKFormat */

	if (logging) InhibitLogging(FALSE);
	return;
}

#ifdef PROTO
static ObjPtr OKFormat(int fmt, ObjPtr list)
#else
static ObjPtr OKFormat(fmt, list)
int fmt;
ObjPtr list;
#endif
{
	ObjPtr theFile, theRdr, corral, theIcon;
	int i, n = ListCount(list);

	/* fmt is the number of the button that was selected in the format group */
	if (fmt < 0) return ObjFalse; /* no choice made */
	else if (fmt >= ListCount(allFileReaders)) /* choice was "none" */
	{
		for (i=0; i<n; ++i) /* clear FORMAT and READER, leave filetype same */
		{
			theFile = GetListElem(list, i);
			theIcon = GetVar(theFile, REPICON);
			SetVar(theFile, READER, NULLOBJ);
			SetVar(theIcon, FORMAT, NULLOBJ);
		}
	}
	else
	{
		for (i=0; i<n; ++i)
		{
			theFile = GetListElem(list, i);
			theRdr = GetListElem(allFileReaders, fmt);
			SetVar(theFile, READER, theRdr);
			ChangeFileType(theFile, DATAFILE);
			SetVar(GetVar(theFile, REPICON), FORMAT, 
			       GetVar(theRdr, NAME));
		}
	}
	if (!(corral = GetVar(list, CORRAL))) return ObjFalse;
	/* reshow files if necessary */
	if (GetInt(GetVar(corral, SELECTOR)) & MASK(DATAFILE)) return ObjTrue;
	WakeMe(corral, SHOWFILES, Clock());
	return ObjTrue;
}

/* ChangeFileType routine is somewhat obsolete since we eliminated the Set File Type command.
	It is used in one place for Set File Format (in OKFormat above) just to change
	a file's type to DATA. I kept the general form in case needed later. JL */
#ifdef PROTO
static void ChangeFileType(ObjPtr theFile, int newType)
#else
static void ChangeFileType(theFile, newType)
ObjPtr theFile;
int newType;
#endif
{
	ObjPtr theIcon, theCorral, fileLists;
	int oldType;

	oldType = GetInt(GetVar(theFile, FILETYPE));
	if (oldType == newType) return;

	theIcon = GetVar(theFile, REPICON);
	theCorral = GetVar(theIcon, PARENT);
	fileLists = GetVar(theCorral, FILELIST);

	/* take file off old type list, add to new type list */
	DeleteFromList(GetListElem(fileLists, oldType), theFile);
	PrefixList(GetListElem(fileLists, newType), theFile);

	/* change the file type in file and icon */
	SetVar(theFile, FILETYPE, NewInt(newType));
	SetVar(theIcon, FILETYPE, NewInt(newType));

	/* change the icon */
	SetVar(theIcon, WHICHICON, NewInt(fileIcon[newType]));

	if (newType == DATAFILE) /* just in case a file was changed from data, then back */
	{
		char *ext, *name;
		ObjPtr theRdr;

		name = GetString(GetVar(theFile, NAME));
		ext = strrchr(name, '.');
		if (ext++ && (theRdr = FindExtensionReader(ext) ))
		{
			/* known data file format */
			SetVar(theFile, READER, theRdr);
			SetVar(theIcon, FORMAT, GetVar(theRdr, NAME));
		}
	}
	else /* clear format in icon and file */
	{
		SetVar(theFile, READER, NULLOBJ);
		SetVar(theIcon, FORMAT, NULLOBJ);
	}
	return;
}

/**************************************************************** READ DATA FILE */
#ifdef PROTO
static ObjPtr ReadDataFile(ObjPtr obj)
#else
static ObjPtr ReadDataFile(obj)
ObjPtr obj;
#endif
{
	DoTask(DoReadDataFile);
	return NULLOBJ;
}

void DoReadDataFile()
{
	ObjPtr corral, theList, theFile, theFormat, thePath;
	ObjPtr theIcon;
	char *pathName;
	DIR *dirp;
	ThingListPtr p;
	Bool formatError = FALSE;

	if (logging)
	{
		InhibitLogging(TRUE);
	}
	if (selWindow)
	{
		if (!(corral = GetVar((ObjPtr) selWinInfo, CORRAL)) ) return;

		/* get the selected files */
		theList = SelFileList(corral);
		if ( p = LISTOF(theList) )
		{
			/* get directory name */
			thePath = GetVar(corral, PATH);
			if (!thePath || chdir(pathName = GetString(thePath)) != 0)
			{
				ReportError("DoReadDataFile", "Could not change directory.");
				return; /* this can't be happening! */
			}

			/* open directory */
			dirp = opendir(pathName);
			if (dirp == NULL)
			{
				ReportError("DoReadDataFile","Could not open directory.");
				return; /* this can't be happening! */
			}

			/* now check and open the files */
			while (p)
			{
				theFile = p->thing;
				if (GetInt(GetVar(theFile, FILETYPE)) != DATAFILE
					|| !( theFormat = GetVar(theFile, READER) ))
				{
	    				formatError = TRUE;
				}
				else
				{
					LongOperation();
					ReadFormattedFile(theFormat,
							GetString(GetVar(theFile, NAME)));
					/* deselect after reading */
					SelectIcon(GetVar(theFile, REPICON), FALSE);
				}
				p = p->next;
			}
			if (formatError) DoTask(ReportFileFormatError);
		}

		/* now get the selected directory icons */
		theList = SelDirList(corral);
		if ( p = LISTOF(theList) )
		{
			while (p)
			{
				theIcon = p->thing;
				DeferMessage(theIcon, OPEN);
				SelectIcon(theIcon, FALSE); /* deselect after opening */
				p = p->next;
			}
		}
	}
	if (logging) InhibitLogging(FALSE);
	return;
}

/******************************************************************** SHOW INFO */
#ifdef PROTO
static ObjPtr ShowInfo(ObjPtr obj)
#else
static ObjPtr ShowInfo(obj)
ObjPtr obj;
#endif
{
	DoTask(DoShowInfo);
	return NULLOBJ;
}

void DoShowInfo(void)
{
	WinInfoPtr infoWin;
	ObjPtr corral, textbox, thePath, theList, theString;
	int i;

	if (!selWindow || !(corral = GetVar((ObjPtr) selWinInfo, CORRAL))) return;
	theList = SelFileList(corral);
	if (!LISTOF(theList)) return; /* no files selected */
	if (logging)
	{
		Log("Show Controls\n");
		InhibitLogging(TRUE);
	}
	thePath = GetVar(corral, PATH);
	if (chdir(GetString(thePath)) != 0) /* error, bad, nono, crapola */
	{
		ReportError("DoShowInfo", "cd error");
		if (logging) InhibitLogging(FALSE);
		return;
	}

	/* get the info window */
	theString = NewString("File Info");
	infoWin = GetDialog((WinInfoPtr) NULLOBJ, theString, "File Info",
		INFOWIDTH, INFOHEIGHT, INFOWIDTH, INFOHEIGHT, WINRGB + WINDBUF + WINFIXEDSIZE);
	if (textbox = GetVar((ObjPtr) infoWin, REPOBJ)) /* dialog exists */
	{
		ThingListPtr p;
		ObjPtr theFile, field, contents;
		char *fileName;
		int x, itemWid;
	
		/* pop info window */
		PopWindow(infoWin);
		SetVar(textbox, VALUE, thePath);

		/* put in file info textbox for each file on the list */
		field = GetVar((ObjPtr) infoWin, FIELD);
		SetVar(field, CONTENTS, contents = NewList());
		p = LISTOF(theList);
		x = MINORBORDER;
		SetupFont(INFOFONT, INFOFONTSIZE); /* so we can check filename length */
		while (p)
		{
			theFile = p -> thing;
			fileName = GetString(GetVar(theFile, NAME));
			itemWid = StrWidth(fileName);
			if (itemWid < INFOITEMWID) itemWid = INFOITEMWID;
			textbox = NewTextBox(x, x + itemWid, 
				 -INFOFIELDHEIGHT + MINORBORDER, -MINORBORDER,
				PLAIN, "File Info", GetFileInfo(fileName));
			PrefixList(contents, textbox);
			SetVar(textbox, PARENT, field);
			SetTextFont(textbox, INFOFONT);
			SetTextSize(textbox, INFOFONTSIZE);
			SetTextLineSpace(textbox, INFOITEMHT - INFOFONTSIZE);
			p = p -> next;
			x = x + itemWid + 2*MINORBORDER;
		}
		ScrollHome(field);
		RecalcScroll(field);
	}
	else /* create info window */
	{
		ObjPtr field, panel, contents, button;
		ThingListPtr p;
		ObjPtr theFile;
		char *fileName;
		int x, itemWid;

		contents = GetVar((ObjPtr) infoWin, CONTENTS); /* contents of window */

		/* put in help string */
		SetVar((ObjPtr) infoWin, HELPSTRING, NewString("This window shows directory \
information for the selected file(s) in the file window. If more than one file was selected, \
you can use the scrollbar at the bottom of the window to view information for each of them."));

		/* put in panel */
		panel = NewPanel(greyPanelClass, 0, INFOWIDTH, 0, INFOHEIGHT);
		PrefixList(contents, panel);
		SetVar(panel, PARENT, (ObjPtr) infoWin);

		contents = GetVar(panel, CONTENTS); /* contents of panel */

		/* put path textbox at top */
		textbox = NewTextBox(MINORBORDER, INFOWIDTH - MINORBORDER,
				INFOHEIGHT - MAJORBORDER - INFOFONTSIZE,
				INFOHEIGHT - MAJORBORDER, PLAIN, "Pathname", GetString(thePath));
		SetVar(textbox, PARENT, panel);
		PrefixList(contents, textbox);
		SetTextFont(textbox, INFOFONT);
		SetTextSize(textbox, INFOFONTSIZE);
		SetTextAlign(textbox, RIGHTALIGN);
		SetVar((ObjPtr) infoWin, REPOBJ, textbox); /* ref to by this routine */

		/* put in "fieldnames" textbox */
		textbox = NewTextBox(MINORBORDER, MINORBORDER + INFONAMEWIDTH, 
			MINORBORDER,
			INFOHEIGHT - MAJORBORDER - INTERSPACE - INFOFONTSIZE - MINORBORDER - 1,
			PLAIN, "Fieldnames", 
		  	"Name\nSize\nOwner\nMode\nModified");
		SetVar(textbox, PARENT, panel);
		PrefixList(contents, textbox);
		SetTextFont(textbox, INFOFONT);
		SetTextSize(textbox, INFOFONTSIZE);
		SetTextAlign(textbox, RIGHTALIGN);
		SetTextLineSpace(textbox, INFOITEMHT-INFOFONTSIZE);

		/* put in field for info text boxes */
		field = NewControlField(2*MINORBORDER + INFONAMEWIDTH, INFOWIDTH - MINORBORDER,
			MINORBORDER, MINORBORDER + INFOFIELDHEIGHT + BARWIDTH, 
			"Info Field", OBJECTSFROMTOP + BARBOTTOM);
		SetVar(field, BACKGROUND, NewInt(UIGRAY75));
		SetVar(field, BORDERTYPE, NewInt(1));
		SetVar(field, PARENT, panel);
		PrefixList(contents, field);
		SetVar(field, CONTENTS, contents = NewList());
		SetVar((ObjPtr) infoWin, FIELD, field);

		/* put in file info textbox for each file on the list */
		p = LISTOF(theList);
		x = MINORBORDER;
		SetupFont(INFOFONT, INFOFONTSIZE); /* so we can check filename length */
		while (p)
		{
			theFile = p -> thing;
			fileName = GetString(GetVar(theFile, NAME));
			itemWid = StrWidth(fileName);
			if (itemWid < INFOITEMWID) itemWid = INFOITEMWID;
			textbox = NewTextBox(x, x + itemWid, 
				 -INFOFIELDHEIGHT + MINORBORDER, -MINORBORDER,
				PLAIN, "File Info", GetFileInfo(fileName));
			PrefixList(contents, textbox);
			SetVar(textbox, PARENT, field);
			SetTextFont(textbox, INFOFONT);
			SetTextSize(textbox, INFOFONTSIZE);
			SetTextLineSpace(textbox, INFOITEMHT - INFOFONTSIZE);
			p = p -> next;
			x = x + itemWid + 2*MINORBORDER;
		}
		RecalcScroll(field);
	}
	if (logging) InhibitLogging(FALSE);
	return;
}

#ifdef PROTO
static char *GetFileInfo(char *fileName)
#else
static char *GetFileInfo(fileName)
char *fileName;
#endif
{
	char *p = tempStr, *s, buf[20];
	struct stat stbuf;
	struct passwd *pwbuf;
	struct tm *tmbuf;
	short mode;

	s = fileName;
	while (*p++ = *s++) ;
	*(p - 1) = '\n';

	/* get file info */
	if (stat(fileName, &stbuf) == -1) /* This happens! (eg, link to a bin, dead link) */
	{
		*p++ = '?';
		*p = '\0';
	}
	else
	{
		/* get size in K */
		sprintf(buf,"%dK",(stbuf.st_size + 1023)/1024);
		s = buf;
		while (*p++ = *s++) ;
		*(p - 1) = '\n';

		/* get owner name */
		pwbuf = getpwuid(stbuf.st_uid);
		s = pwbuf -> pw_name;
		while (*p++ = *s++) ;
		*(p - 1) = '\n';

		/* get mode bits */
		mode = stbuf.st_mode;
		*p++ = mode & S_IRUSR ? 'r' : '-';
		*p++ = mode & S_IWUSR ? 'w' : '-';
		*p++ = mode & S_IXUSR ? 'x' : '-';
		*p++ = ' ';
		*p++ = mode & S_IRGRP ? 'r' : '-';
		*p++ = mode & S_IWGRP ? 'w' : '-';
		*p++ = mode & S_IXGRP ? 'x' : '-';
		*p++ = ' ';
		*p++ = mode & S_IROTH ? 'r' : '-';
		*p++ = mode & S_IWOTH ? 'w' : '-';
		*p++ = mode & S_IXOTH ? 'x' : '-';
		*p++ = '\n';

		/* get modified date */
		tmbuf = localtime(&stbuf.st_mtime);
		sprintf(buf,"%02d/%02d/%02d %02d:%02d\n", tmbuf->tm_mon + 1, tmbuf->tm_mday,
			tmbuf->tm_year, tmbuf->tm_hour, tmbuf->tm_min);
		s = buf;
		while (*p = *s++) ++p;
	}
	return tempStr;
}

/********************************************************************* FILE FORMAT DIALOG
*	Puts up a dialog for selecting data file format.
*	The OKRoutine is called with the selected format and a list of files:
*
*		void OKRoutine(int fmt, ObjPtr theList)
*/
#ifdef PROTO
static void FileFormatDialog(FuncTyp OKRoutine)
#else
static void FileFormatDialog(OKRoutine)
FuncTyp OKRoutine;
#endif
{
	WinInfoPtr theDialog;
	ObjPtr panel, alertPanel, contents, promptBox, radioBtn, cancelBtn, okBtn;
	ObjPtr thePrompt, radioGroup;
	int row, col, nRows, nCols, totWid, totHt;
	int n, nBtns;

	nBtns = ListCount(allFileReaders) + 1;

	/* first compute sizes from the number of buttons */
	/* number of buttons is one more than number of formats to allow for "none" */
	if (nBtns < 11) nCols = 2, nRows = (nBtns + 1)/2;
	else if (nBtns < 25) nCols = 3, nRows = (nBtns + 2)/3;
	else nCols = 4, nRows = (nBtns + 3)/4;
	totWid = 2*SIDEMARGIN + nCols*RADIOWID + (nCols - 1)*GAP;
	totHt = BOTMARGIN + OKBTNHT + 2*INTERSPACE + nRows*CHECKBOXHEIGHT + (nRows - 1)*GAP
		+ PROMPTTEXTSIZE + TOPMARGIN;

	/* see if dialog exists */
	if (!selWindow) return;
	thePrompt = NewString("Set data format to:");
	theDialog = GetDialog(selWinInfo, thePrompt, "Set Data Format",
		totWid, totHt, totWid, totHt, 
		WINRGB + WINDBUF + WINFIXEDSIZE + WINCENTERED + WINNOFRAME);
	if(!theDialog) return;

	if (GetVar((ObjPtr) theDialog, REPOBJ)) /* dialog exists */
	{
		/* pop the dialog */
		PopWindow(theDialog);
	}
	else /* make the dialog box */
	{
		ThingListPtr runner;
		contents = GetVar((ObjPtr) theDialog, CONTENTS); /* contents of window */
		SetVar((ObjPtr) theDialog, HELPSTRING, NewString("This window shows \
the data file formats recognized by this version of SciAn. Click on the proper format \
for the selected files and press the OK button. For more information, see the section \
on data file formats in the SciAn user manual."));

		/* put in panel */
		panel = NewPanel(greyPanelClass, 0, totWid, 0, totHt);
		PrefixList(contents, panel);
		SetVar(panel, PARENT, (ObjPtr) theDialog);

		contents = GetVar(panel, CONTENTS); /* contents of panel */

		/* put in pretty panel */
		alertPanel = NewAlertPanel(0, totWid, 0, totHt, "Fmt Panel");
		PrefixList(contents, alertPanel);
		SetVar(alertPanel, PARENT, panel);

		/* put in prompt */
		promptBox = NewTextBox(SIDEMARGIN, totWid - SIDEMARGIN,
				totHt - TOPMARGIN - PROMPTTEXTSIZE, totHt - TOPMARGIN,
				PLAIN, "Prompt Box", GetString(thePrompt));
		PrefixList(contents, promptBox);
		SetVar(promptBox, PARENT, panel);
		SetTextFont(promptBox, PROMPTTEXTFONT);
		SetTextSize(promptBox, PROMPTTEXTSIZE);
		SetTextColor(promptBox, PROMPTTEXTCOLOR);

		/* put in the bottom buttons */
		cancelBtn = NewButton(totWid - SIDEMARGIN - 2*OKBTNWID - GAP,
				totWid - SIDEMARGIN - OKBTNWID - GAP,
				BOTMARGIN, BOTMARGIN + OKBTNHT, "Cancel");
		PrefixList(contents, cancelBtn);
		SetVar(cancelBtn, PARENT, panel);
		SetMethod(cancelBtn, CHANGEDVALUE, CancelDialog);
		SetVar(cancelBtn, HELPSTRING, NewString("Pressing this button \
will cancel this dialog window without changing the format of the selected file(s)."));

		okBtn = NewButton(totWid - SIDEMARGIN - OKBTNWID,
				totWid - SIDEMARGIN,
				BOTMARGIN, BOTMARGIN + OKBTNHT, "OK");
		PrefixList(contents, okBtn);
		SetVar(okBtn, PARENT, panel);
		SetMethod(okBtn, OKMETHOD, OKRoutine); /* for ref by OKDialog */
		SetMethod(okBtn, CHANGEDVALUE, OKDialog);
		SetVar(okBtn, HELPSTRING, NewString("Pressing this button \
will dismiss this dialog window and set the format of the selected file(s)."));
		SetVar(okBtn, REPOBJ, (ObjPtr) theDialog);

		/* oh, yeah -- guess we need a few radio buttons */
		radioGroup = NewRadioButtonGroup("Data Format");
		PrefixList(contents, radioGroup);
		SetVar(radioGroup, PARENT, panel);
		runner = LISTOF(allFileReaders);
		for (n=0, row=nRows-1; row>=0 && runner; --row)
		{
			for (col = 0; col<nCols && runner; ++col)
			{
				ObjPtr theName;
				theName = GetVar(runner -> thing, NAME);
				radioBtn = NewRadioButton(
					SIDEMARGIN + col*(RADIOWID + GAP),
					SIDEMARGIN + (col+1)*RADIOWID + col*GAP,
					BOTPART + row*(CHECKBOXHEIGHT + GAP),
					BOTPART + (row+1)*CHECKBOXHEIGHT + row*GAP,
					theName ? GetString(theName) : "");
				AddRadioButton(radioGroup, radioBtn);
				runner = runner -> next; 
			}
		}
		/* now add "none" button */
		row = 0; col = nCols - 1;
		radioBtn = NewRadioButton(
			SIDEMARGIN + col*(RADIOWID + GAP),
			SIDEMARGIN + (col+1)*RADIOWID + col*GAP,
			BOTPART + row*(CHECKBOXHEIGHT + GAP),
			BOTPART + (row+1)*CHECKBOXHEIGHT + row*GAP,
			"NONE");
		AddRadioButton(radioGroup, radioBtn);
		
		SetVar(okBtn, DATA, radioGroup); /* ref by OKDialog */
#if 0
		SetValue(radioGroup, NewInt(nFormats)); /* make "none" the default */
#endif
	}
	return;
}

#ifdef PROTO
static ObjPtr OKDialog(ObjPtr btn)
#else
static ObjPtr OKDialog(btn)
ObjPtr btn;
#endif
{
	FuncTyp OKRtn;
	ObjPtr list;
	int val;

	list = GetVar(GetVar(btn, REPOBJ), FILELIST);
	val = GetInt(GetVar(GetVar(btn, DATA), VALUE));
	OKRtn = GetMethod(btn, OKMETHOD);

	OKRtn(val, list);
	DoTask(DoClose);
	return ObjTrue;
}

#ifdef PROTO
static ObjPtr CancelDialog(ObjPtr btn)
#else
static ObjPtr CancelDialog(btn)
ObjPtr btn;
#endif
{
	DoTask(DoClose);
	return ObjFalse;
}

/********************************************************************* SELECTED FILE LIST
*	Returns a list of files selected in corral of a files window.
*	Excludes directories. If no files are selected, returns an empty list.
*/
#ifdef PROTO
static ObjPtr SelFileList(ObjPtr corral)
#else
static ObjPtr SelFileList(corral)
ObjPtr corral;
#endif
{
	ObjPtr theFile, theIcon, theList;
	ThingListPtr p;

	theList = NewList();
	if(!IsObject(corral)) return theList;
	p = LISTOF(GetVar(corral, CONTENTS));
	while (p)
	{
		theIcon = p->thing;
		if ( IsObject(theIcon)
		&& InClass(theIcon, iconClass)
		&& GetPredicate(theIcon, SELECTED)
		&& IsObject(theFile = GetVar(theIcon, REPOBJ))
		&& InClass(theFile, fileClass)
		&& GetInt(GetVar(theFile, FILETYPE)) != DIRFILE )
			PrefixList(theList, theFile);
		p = p->next;
	}
	return theList;
}

/********************************************************************* SELECTED DIR LIST
*	Returns a list of directory icons selected in corral of a files window.
*	Excludes files. If none are selected, returns an empty list.
*/
#ifdef PROTO
static ObjPtr SelDirList(ObjPtr corral)
#else
static ObjPtr SelDirList(corral)
ObjPtr corral;
#endif
{
	ObjPtr theIcon, theList;
	ThingListPtr p;

	theList = NewList();
	if(!IsObject(corral)) return theList;
	p = LISTOF(GetVar(corral, CONTENTS));
	while (p)
	{
		theIcon = p->thing;
		if ( IsObject(theIcon)
		&& InClass(theIcon, iconClass)
		&& GetPredicate(theIcon, SELECTED)
		&& GetInt(GetVar(theIcon, FILETYPE)) == DIRFILE )
			PrefixList(theList, theIcon);
		p = p->next;
	}
	return theList;
}

/******************************************************************* GET FILE WINDOW */
static void GetFileWindow(void)
/* Create/pop Files dialog */
{
	WinInfoPtr fileWin;
	ObjPtr panel, contents, corral, thePath;
	ObjPtr btn, btnGroup;
	int btnWid;
	Screencoord left, right, bottom, top;

	/* get the dialog window */
	getwd(tempStr); /* should already be switched to new directory */
	thePath = NewString(tempStr);
	fileWin = GetDialog((WinInfoPtr) NULLOBJ, thePath, tempStr,
		FILESWIDTH, FILESHEIGHT, SCRWIDTH, SCRHEIGHT, WINRGB + WINDBUF);
	if(!fileWin) return;

	getviewport(&left, &right, &bottom, &top);

	if (GetVar((ObjPtr) fileWin, PATH)) /* dialog exists */
	{
		/* pop files dialog */
		PopWindow(fileWin);
	}
	else /* create file window */
	{
		InhibitLogging(TRUE);
		SetVar((ObjPtr) fileWin, PATH, thePath); /* ref by this routine */
		contents = GetVar((ObjPtr) fileWin, CONTENTS); /* contents of window */

		DefineMenuItem((ObjPtr) fileWin, FILEMENU, "Open", DoReadDataFile);
		DefineMenuItem((ObjPtr) fileWin, FILEMENU, "Set Format", DoSetFileFormat);
		DefineMenuItem((ObjPtr) fileWin, FILEMENU, "Show Info", DoShowInfo);
		DefineMenuItem((ObjPtr) fileWin, OBJECTMENU, "Select All", DoSelectAllIcons);
		DefineMenuItem((ObjPtr) fileWin, OBJECTMENU, "Deselect All", DoDeselectAllIcons);

		/* put in help string */
		SetVar((ObjPtr) fileWin, HELPSTRING, NewString("This window shows the files \
and sub-directories located in the directory given by the window's title. The files appear as \
icons corresponding to file type; sub-directories appear as folders. To open a file or \
folder, select its icon and press the Open button, or simply double click on it."));

		/* put in panel */
		panel = NewPanel(greyPanelClass, 0, right, 0, top);
		PrefixList(contents, panel);
		SetVar(panel, PARENT, (ObjPtr) fileWin);
		SetVar(panel, STICKINESS, NewInt(STICKYALL));

		contents = GetVar(panel, CONTENTS); /* contents of panel */

		/* put in icon corral */
		corral = NewIconCorral(NULLOBJ, MINORBORDER, 
				right - MINORBORDER, TITLEBOXTOP + MINORBORDER + OKBTNHT,
				top - (TITLEBOXTOP + MINORBORDER + PARENTBTNHT),
				OBJECTSFROMTOP + BARRIGHT + BARBOTTOM);
		SetVar(corral, STICKINESS, NewInt(STICKYALL));
		PrefixList(contents, corral);
		SetVar(corral, PARENT, panel);
		SetVar(corral, PATH, thePath);
		SetVar(corral, SELECTOR, NewInt(DEFAULTFILTER)); /* file selector bits */
		SetMethod(corral, SHOWFILES, ShowFiles);

		/* put in title box for filter buttons */
		btn = NewTitleBox(MINORBORDER, MINORBORDER + FILTERBTNWID + 2*SMALLGAP,
			top - TITLEBOXTOP - PARENTBTNHT, top - 4, "Files");
		PrefixList(contents, btn);
		SetVar(btn, PARENT, panel);
		SetVar(btn, STICKINESS, NewInt(STICKYTOP + STICKYLEFT));

		/* put in file filter radio group */
		btnGroup = NewRadioButtonGroup("File Filter");
		SetVar(btnGroup, PARENT, panel);
		PrefixList(contents, btnGroup);
		SetVar(btnGroup, CORRAL, corral);
		SetVar(btnGroup, STICKINESS, NewInt(STICKYTOP + STICKYLEFT));
		SetVar(btnGroup, HELPSTRING, NewString("These buttons determine which file types \
to show in the File Window. (Folders are always shown.) If the top button is checked, only SciAn \
data files and log files are shown; the files must have proper filename \
extensions to be recognized by SciAn. If the bottom button is checked, all files are shown."));

		btn = NewRadioButton(left + MINORBORDER + SMALLGAP, 
			left + MINORBORDER + SMALLGAP + FILTERBTNWID,
			top - TITLEBOXTOP - PARENTBTNHT + SMALLGAP + CHECKBOXHEIGHT + CHECKBOXSPACING,
			top - TITLEBOXTOP - PARENTBTNHT + SMALLGAP + 2*CHECKBOXHEIGHT + CHECKBOXSPACING,
			"Show SciAn files only");
		AddRadioButton(btnGroup, btn);
		SetVar(btn, STICKINESS, NewInt(STICKYTOP + STICKYLEFT));
		SetVar(btn, SELECTOR, NewInt(DEFAULTFILTER));

		btn = NewRadioButton(left + MINORBORDER + SMALLGAP, 
			left + MINORBORDER + SMALLGAP + FILTERBTNWID,
			top - TITLEBOXTOP - PARENTBTNHT + SMALLGAP,
			top - TITLEBOXTOP - PARENTBTNHT + SMALLGAP + CHECKBOXHEIGHT,
			"Show all files");
		AddRadioButton(btnGroup, btn);
		SetVar(btn, STICKINESS, NewInt(STICKYTOP + STICKYLEFT));
		SetVar(btn, SELECTOR, NewInt(ALLFILES));
		SetValue(btnGroup, NewInt(0)); /* make "Show SciAn files only" the default */
		SetMethod(btnGroup, CHANGEDVALUE, NewFilter);

		/* put in parent button */
		btn = NewIconLabeledButton(right - MINORBORDER - PARENTBTNWID, 
			right - MINORBORDER, top - TITLEBOXTOP - PARENTBTNHT,
			top - TITLEBOXTOP, ICONPARENT, UIWHITE, "Open Parent");
		SetVar(btn, STICKINESS, NewInt(STICKYTOP + STICKYRIGHT));
		SetVar(btn, PARENT, panel);
		SetVar(btn, LABEL, NewString("Open Parent"));
		SetVar(btn, HELPSTRING, NewString("Pressing this button will \
open a new file window at the next higher directory level. The directory of the current window \
will appear as a folder in the new window."));
		PrefixList(contents, btn);
		MakeButtonToggled(btn, TRUE); /* make parent button icon be lit up always */
		SetValue(btn, NewInt(TRUE));
		MakeButtonToggled(btn, FALSE);
		SetVar(btn, PATH, thePath); /* ref by DoParent */
		SetMethod(btn, CHANGEDVALUE, DoParent);

		/* put in action buttons */
		btnWid = (right - left - 2*MINORBORDER - 2*SMALLGAP)/3;
		btn = NewButton(left + MINORBORDER, left + MINORBORDER + btnWid,
				bottom + MINORBORDER, bottom + MINORBORDER + OKBTNHT,
				"Open");
		SetVar(btn, PARENT, panel);
		PrefixList(contents, btn);
		ActivateButton(btn, FALSE);
		SetVar(btn, STICKINESS, NewInt(STICKYLEFT + FLOATINGRIGHT));
		SetMethod(btn, CHANGEDVALUE, ReadDataFile);
		SetVar(btn, HELPSTRING, NewString("Pressing this button, or choosing Open \
from the File menu, will read the selected files into the Datasets window and open new file windows \
for the selected folders. The button is not active if no files or folders are selected."));
		SetVar(corral, OPENBTN, btn); /* ref by SetFileButtons */

		btn = NewButton(left + MINORBORDER + btnWid + SMALLGAP, 
		left + MINORBORDER + SMALLGAP + 2*btnWid,
				bottom + MINORBORDER, bottom + MINORBORDER + OKBTNHT,
				"Set Format");
		SetVar(btn, PARENT, panel);
		PrefixList(contents, btn);
		ActivateButton(btn, FALSE);
		SetVar(btn, STICKINESS, NewInt(FLOATINGRIGHT + FLOATINGLEFT));
		SetMethod(btn, CHANGEDVALUE, SetFormat);
		SetVar(btn, HELPSTRING, NewString("Pressing this button, or choosing Set Format \
from the File menu, will open a dialog window allowing you to choose the data format for the selected \
files. The button is not active if no files are selected."));
		SetVar(corral, SETFORMATBTN, btn); /* ref by SetUpButtons */

		btn = NewButton(right - MINORBORDER - btnWid, right - MINORBORDER,
				bottom + MINORBORDER, bottom + MINORBORDER + OKBTNHT,
				"Show Info");
		SetVar(btn, PARENT, panel);
		PrefixList(contents, btn);
		ActivateButton(btn, FALSE);
		SetVar(btn, STICKINESS, NewInt(FLOATINGLEFT + STICKYRIGHT));
		SetMethod(btn, CHANGEDVALUE, ShowInfo);
		SetVar(btn, HELPSTRING, NewString("Pressing this button, or choosing Show Info \
from the File menu, will open a window showing directory information for the selected files. \
The button is not active if no files are selected."));
		SetVar(corral, SHOWINFOBTN, btn); /* ref by SetUpButtons */

		/* set up reference */
		SetVar((ObjPtr) fileWin, CORRAL, corral); /* ref by this routine */

		/* now put in the file icons */
		ReadDirectory(corral);
		ShowFiles(corral);
		InhibitLogging(FALSE);
	}
}

/*********************************************************************** SET UP BUTTONS */
void DoSetFileButtons(void)
/* deferred task to set up buttons based on icons selected in file window */
{
    ThingListPtr runner;
    int dirCount=0, fileCount=0;

    if (!gIconList) return;
    runner = LISTOF(gIconList);
    while (runner)
    {
	if (GetPredicate(runner -> thing, SELECTED))
	{
		if (GetInt(GetVar(runner -> thing, FILETYPE)) == DIRFILE) ++dirCount;
		else ++fileCount;
	}
	runner = runner -> next;
    }

    ActivateButton(gOpenBtn, fileCount + dirCount > 0);
    ActivateButton(gSetFormatBtn, fileCount > 0);
    ActivateButton(gShowInfoBtn, fileCount > 0);

    gIconList = NULLOBJ;
    return;
}

#ifdef PROTO
static ObjPtr SetFileButtons(ObjPtr icon)
#else
static ObjPtr SetFileButtons(icon)
ObjPtr icon;
#endif
/* CHANGEDVALUE method for icons used to set up buttons in file window */
{
	ObjPtr corral;

	if (!(corral = GetVar(icon, PARENT))) return ObjFalse;

	gOpenBtn = GetVar(corral, OPENBTN);
	gSetFormatBtn = GetVar(corral, SETFORMATBTN);
	gShowInfoBtn = GetVar(corral, SHOWINFOBTN);
	gIconList = GetVar(corral, CONTENTS);

	if (!gIconList) return ObjFalse;

	DoUniqueTask(DoSetFileButtons);
	return ObjTrue;
}

#ifdef PROTO
ObjPtr IconHelp(ObjPtr icon, ObjPtr class)
#else
ObjPtr IconHelp(icon, class)
ObjPtr icon, class;
#endif
{
	int type = GetInt(GetVar(icon, FILETYPE));

	switch (type)
	{
		case DIRFILE:
			SetVar(class, HELPSTRING, NewString("This icon represents a sub-directory \
within the directory of this file window. Double clicking on the folder will open a file window \
for the sub-directory."));
			break;

		case DATAFILE:
			if (GetVar(icon, FORMAT)) SetVar(class, HELPSTRING, NewString("\
This icon represents a data file with a format recognized by this version of SciAn. The \
name of the format is shown under the file name. \
Double clicking on this icon will read the file into the Datasets window."));

		/* without Set File Type, this can't happen; all data files have known formats */
		/***	else SetVar(class, HELPSTRING, NewString("This icon represents a data \
file with an unknown file format. Before SciAn can read the file, you must identify the data \
format. Select the file, press the Set Format button, and then select the proper format. \
For more information, see the chapter on data file formats in the SciAn user manual.")); ***/
			break;

		case OTHFILE:
			SetVar(class, HELPSTRING, NewString("This icon represents a file whose type \
SciAn could not determine from the filename extension. If the file is a data file, you must \
identify the data format before SciAn can read the file. Select the file, press the Set \
Format button, and then select the proper format."));
			break;

		default:
			SetVar(class, HELPSTRING, NewString("This icon represents a file in the \
directory of this file window."));
			break;
	}
	return NULLOBJ;
}

/************************************************************************ READ DIRECTORY 
*	Reads directory of 'path' associated with given corral.
*	Creates file objects and sets parameters for them.
*/
#ifdef PROTO
static Bool ReadDirectory(ObjPtr corral)
#else
static Bool ReadDirectory(corral)
ObjPtr corral;
#endif
{
	struct stat stbuf;
	int fd;
#ifdef DIRENT
	struct dirent *dp;
#else
	struct direct *dp;
#endif
	DIR *dirp;
	ObjPtr thePath, theFile;
	ObjPtr fileLists;
	int i, fileType;
	ObjPtr theRdr;
	char *pathName, *ext;

	/* this may take a while */
	MySetCursor(WATCHCURSOR);

	thePath = GetVar(corral, PATH);
	if (thePath) pathName = GetString(thePath);
	else return FALSE;

	/* open directory */
	dirp = opendir(pathName);
	if (dirp == NULL) return FALSE; /* error opening directory */

	/* set up master list and file type lists */
	fileLists = NewList();
	for (i=0; i<NFILETYPES; ++i) PrefixList(fileLists, NewList());

	/* read directory */
	while ((dp = readdir(dirp)) != NULL) /***/
	{

		if (strcmp(dp->d_name, ".") == 0 || strcmp(dp->d_name, "..") == 0) continue;
		stat(dp->d_name, &stbuf);
		if((stbuf.st_mode & S_IFMT) == S_IFLNK) continue; /*** must be a dead link? */
		if ((stbuf.st_mode & S_IFMT) == S_IFSOCK) continue; /* socket file */

		if (*dp->d_name == '.') continue; /* ignore dot files */
		ext = strrchr(dp->d_name, '.'); /* point to filename extension */

		/* make a file object */
		theFile = NewObject(fileClass, 0);
		SetVar(theFile, NAME, NewString(dp->d_name));
		SetVar(theFile, PATH, thePath);

		/* figure out file type */
		if((stbuf.st_mode & S_IFMT) == S_IFDIR)
		{
			/* entry is a directory */
			SetVar(theFile, FILETYPE, NewInt(DIRFILE));
			PrefixList(GetListElem(fileLists, DIRFILE), theFile);
		}
		else if (stbuf.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH))
		{
			/* entry is (probably) executable file */
			SetVar(theFile, FILETYPE, NewInt(BINFILE));
			PrefixList(GetListElem(fileLists, BINFILE), theFile);
		}
		else
		{
			/* check file extension to identify file type */
			if (ext++) /* skip dot */
			{
				if (strcmp(ext, "o") == 0)
				{
					/* object file */
					SetVar(theFile, FILETYPE, NewInt(OBJFILE));
					PrefixList(GetListElem(fileLists, OBJFILE), theFile);
				}
				else if (strcmp(ext, "txt") == 0
					|| strcmp(ext, "c") == 0
					|| strcmp(ext, "h") == 0
					|| strcmp(ext, "f") == 0 )
				{
					/* text file */
					SetVar(theFile, FILETYPE, NewInt(TEXTFILE));
					PrefixList(GetListElem(fileLists, TEXTFILE), theFile);
				}
				else if ( theRdr = FindExtensionReader(ext) )
				{
					/* known data file extension */
					SetVar(theFile, FILETYPE, NewInt(DATAFILE));
					SetVar(theFile, READER, theRdr);
					PrefixList(GetListElem(fileLists, DATAFILE), theFile);
				}
				else if (strcmp(ext, "log") == 0) /*** what should the ext be? */
				{
					/* script or log file */
					SetVar(theFile, FILETYPE, NewInt(LOGFILE));
					PrefixList(GetListElem(fileLists, LOGFILE), theFile);
				}
				else /* other file type */
				{
					SetVar(theFile, FILETYPE, NewInt(OTHFILE));
					PrefixList(GetListElem(fileLists, OTHFILE), theFile);
				}
			}
			else /* no dot: other file type */
			{
				SetVar(theFile, FILETYPE, NewInt(OTHFILE));
				PrefixList(GetListElem(fileLists, OTHFILE), theFile);
			}
		}
	}
	closedir(dirp);
	SetVar(corral, FILELIST, fileLists);
}

/*********************************************************************** SHOW FILES
*	Puts the icons for the selected files into the given corral.
*	Calls ShowFileList for each of the selected file types.
*/
#ifdef PROTO
static ObjPtr ShowFiles(ObjPtr corral)
#else
static ObjPtr ShowFiles(corral)
ObjPtr corral;
#endif
{
	ObjPtr theFile, theIcon, theRdr, iconList;
	ObjPtr fileLists, theType, thePath;
	ObjPtr contents;
	int i, x, y, fileSelector, left, right, bottom, top, width;
	Bool stagger, stagCol = TRUE;
	ThingListPtr p;
	FuncTyp ReScroll;

	/* this may take a while */
	MySetCursor(WATCHCURSOR);

	fileSelector = GetInt(GetVar(corral, SELECTOR));
	fileLists = GetVar(corral, FILELIST);
	thePath = GetVar(corral, PATH);

	Get2DIntBounds(corral, &left, &right, &bottom, &top);
	width = right - left;

	SetVar(corral, CONTENTS, contents = NewList());

 	/* set up for loop */
	x = MINORBORDER + ICONLEFTBORDER;
	y = -MINORBORDER - ICONTOPBORDER;
	stagger = GetPrefTruth(PREF_STAGGERICONS); 

	/* now drop the selected filetype icons in the corral */
	for (i=0; i<NFILETYPES; ++i)
	{
		if (!(fileSelector & MASK(i))) continue;
		theType = NewInt(i);
		p = LISTOF(GetListElem(fileLists, i));
		while(p)
		{
			theFile = p -> thing;
			if (!(theIcon = GetVar(theFile, REPICON)))
			{
				/* make icon for this file */
				theIcon = NewIcon(x, y - (stagger && (stagCol = !stagCol) ? ICONYSPACE/2 : 0),
						fileIcon[i], GetString(GetVar(theFile, NAME)));
				SetVar(theFile, REPICON, theIcon);
				SetVar(theIcon, REPOBJ, theFile);
				SetVar(theIcon, PARENT, corral);
				SetVar(theIcon, FILETYPE, theType);
				SetVar(theIcon, PATH, thePath);
				SetMethod(theIcon, MAKE1HELPSTRING, IconHelp);
				SetMethod(theIcon, CHANGEDVALUE, SetFileButtons);

				if (i == DIRFILE)
				{
					SetMethod(theIcon, DOUBLECLICK, DoOpen);
					SetMethod(theIcon, OPEN, OpenFolder);
				}
				else SetMethod(theIcon, DOUBLECLICK, DoOpen);

				if (theRdr = GetVar(theFile, READER)) /* put format name on icon */
					SetVar(theIcon, FORMAT, GetVar(theRdr, NAME));
			}
			else
			{
				ObjPtr locArray = NewRealArray(1, 2L);
				real loc[2];

				loc[0] = x;
				loc[1] = y - (stagger && (stagCol = !stagCol) ? ICONYSPACE/2 : 0);
				CArray2Array(locArray, loc);
				SetVar(theIcon, ICONLOC, locArray);
			}
			SelectIcon(theIcon, FALSE); /* ensure not selected */
			PrefixList(contents, theIcon);

			/* increment location */
			if ((x += ICONXSPACE) > right - ICONRIGHTBORDER)
			{
				/* new row */
				stagCol = TRUE;
				x = MINORBORDER + ICONLEFTBORDER;
				y -= ICONYSPACE;
			}
			p = p -> next;
		}
	}
	ScrollHome(corral);
	RecalcScroll(corral);
	return ObjTrue; /* function is used as method */
}

#ifdef PROTO
static ObjPtr DoParent(ObjPtr button)
#else
static ObjPtr DoParent(button)
ObjPtr button;
#endif
{
	strncpy(tempStr, GetString(GetVar(button, PATH)), TEMPSTRSIZE);
	tempStr[TEMPSTRSIZE] = '\0';
	*(strrchr(tempStr, '/')) = '\0';
	if (chdir(*tempStr ? tempStr : "/") == 0)
	{
		DoTask(GetFileWindow);
		return ObjTrue;
	}
	else
	{
		/* my God! what do we do now? */
		fputc (7, stderr); /* just beep and do nothing */
		return ObjFalse;
	}
}

/*********************************************************************** NEW FILTER */
#ifdef PROTO
static ObjPtr NewFilter(ObjPtr b)
#else
static ObjPtr NewFilter(b)
ObjPtr b;
#endif
{
	ObjPtr corral;
	int selector;

	corral = GetObjectVar("NewFilter", b, CORRAL);
	selector = GetInt(GetVar(b, VALUE)) ? ALLFILES : DEFAULTFILTER;
	SetVar(corral, SELECTOR, NewInt(selector));
	ShowFiles(corral);
	return ObjTrue;
}

/********************************************************************* DO OPEN */
/* This routine is the DOUBLECLICK method for icons */
#ifdef PROTO
static ObjPtr DoOpen(ObjPtr icon)
#else
static ObjPtr DoOpen(icon)
ObjPtr icon;
#endif
{
	ObjPtr btn, corral;

	corral = GetVar(icon, PARENT);
	btn = GetVar(corral, OPENBTN);
	SetValue(btn, NULLOBJ);
	return ObjTrue;
}

/* This routine is called only as a result of a deferred OPEN message */
#ifdef PROTO
static ObjPtr OpenFolder(ObjPtr icon)
#else
static ObjPtr OpenFolder(icon)
ObjPtr icon;
#endif
{
	char *path, *name;

	path = GetString(GetVar(icon, PATH));
	name = GetString(GetVar(icon, NAME));
	if (strlen(path) + 1 + strlen(name) > TEMPSTRSIZE)
	{
		/* the dang pathname is too long! */
		ReportError("OpenFolder","The dang pathname is too long!");
		return ObjFalse;
	}
	strcpy(tempStr, path);
	strcat(tempStr, "/");
	strcat(tempStr, name);
	tempStr[TEMPSTRSIZE] = '\0';

	if(chdir(tempStr) != 0)
	{
		/* Hey! There shouldn't be an error here! */
		ReportError("OpenFolder","Could not open the dang folder!");
		return ObjFalse;
	}

	/* now open new files window */
	GetFileWindow();

	return ObjTrue;
}
