/*
  File: wff2vcr.c
  Author: K.R. Sloan
  Last Modified: 17 February 1992
  Purpose: convert a .wff file to VICAR

 NOTE: this program is based on bare minimum specifications for VICAR
       images.  It works on the images that I have seen, and on the images
       produced by wff2vcr (of course?).  It is not to be taken as a 
       specification of VICAR images.  If you spot an obvious error, or
       know enough to provide guidance on further extensions, please
       contact <sloan@cis.uab.edu>

  NOTE: the VICAR files have 8 bits per sample

        the .wff files must be I.  If the BitsPerSample is less than
        8, then samples are extended to 8 bits (correctly!).  If the 
        BitsPerSample is greater than 8, the samples are truncated.   
 */

#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <sys/types.h>
#include "wff.h"

unsigned char *malloc();

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is:\n\t%s [-h][-v]\n",RoutineName);
 } 

static void FatalError(s)
 char *s;
{
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s); exit(-1);
}


/* VICAR stuff */
 /* tags, and guesses as to meaning... */
static int  LBLSIZE;       /* size of header, must be int mult of NS */
static char FORMAT[80];    /* 'BYTE' is OK */
static char TYPE[80];      /* 'IMAGE' is OK */
static int  BUFSIZe;       /* integer multiple of NS ? */
static int  DIM;           /* == 3? */
static int  EOL;           /* == 0? */
static int  RECSIZE;       /* == LBLSIZE? */
static char ORG[80];       /* `BSQ` is OK */
static int  NL;            /* height */
static int  NS;            /* width */ 
static int  NB;            /* samples per pixel? */
static int  N1;            /* == NL? */ 
static int  N2;            /* == NS? */
static int  N3;            /* == NB? */
static int  N4;            /* 0 is OK */
static int  NBB;           /* 0 is OK */ 
static int  NLB;           /* 0 is OK */
static char HOST[80];      /* machine type? */
static char INTFMT[80];    /* integer format? */
static char REALFMT[80];   /* read format? */
static char TASK[80];      /* processing applied? */
static char USER[80];      /* who was responsible? */
static char DAT_TIM[80];   /* when? */
static char COMMENT[80];   /* comment! */
    
WriteVICARHeader(fd, width, height, BandsPerPixel)
 FILE *fd;
 int width, height, BandsPerPixel;
 {
  int pad;

  if (1 != BandsPerPixel) FatalError("Sorry, we can only handle 1 band");

  /*
     LBLSIZE must be an integer multiple of width.
     It also needs to be large enough for everything below to fit.
     We use 1024 as a reasonable minimum size, and pick the first integer
     multiple of 'width' to be the LBLSIZE.

     Look - I don't really understand VICAR format.  We're just hacking...

   */
  LBLSIZE = width;  while(LBLSIZE < 1024) LBLSIZE += width;

  fprintf(fd,"LBLSIZE=%-d",LBLSIZE);   /* see above */
  fprintf(fd," FORMAT='BYTE'");        
  fprintf(fd," TYPE='IMAGE'");
  fprintf(fd," BUFSIZ=%-d",20*LBLSIZE);
  fprintf(fd," DIM=3");  
  fprintf(fd," EOL=0");             
  fprintf(fd," RECSIZE=%-d",LBLSIZE); 
  fprintf(fd," ORG='BSQ'");         
  fprintf(fd," NL=%-d",height);     
  fprintf(fd," NS=%-d",width);      
  fprintf(fd," NB=1");              
  fprintf(fd," N1=%-d",height);     
  fprintf(fd," N2=%-d",width);      
  fprintf(fd," N3=1");              
  fprintf(fd," N4=0");              
  fprintf(fd," NBB=0");             
  fprintf(fd," NLB=0");             
  fprintf(fd," COMMENT='created by wff2vcr complain to Ken Sloan'");

  for(pad = LBLSIZE - ftell(fd); pad > 0; pad--) fputc(0,fd);
 }

static void WriteVICARScanLine(fd, VICARScanLine, VICARScanLineLength)
 FILE *fd;
 unsigned char *VICARScanLine;
 int VICARScanLineLength;
 {
  (void)fwrite(VICARScanLine, 1, VICARScanLineLength, fd);
 }

static void Pass(inFD, outFD)
 FILE *inFD;
 FILE *outFD;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand;
  int KnownColorSystem, BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  int x,y,b,width,height,i,c;
  unsigned short *WFFScanLine;
  int WFFScanLineLength;
  unsigned char *VICARScanLine;
  int VICARScanLineLength, depth;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer)) 
   FatalError("OpenFB failed");
  
  if (FAILURE == ReadImage(inFD, FrameBuffer)) 
   FatalError("ReadImage failed");

  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right)) 
   FatalError("GetBounds failed");

  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   FatalError("GetColorsystem failed");

  if (VERBOSE)
   fprintf(stderr,"%s: WhatBands = %s; BitsPerBand = %d\n",
                   RoutineName, WhatBands, BitsPerBand);

  if (0 == strcmp(WhatBands,"I")) BandsPerPixel = 1;
  else                            FatalError("Sorry, we only do I\n");

  width = Right - Left + 1;
  height = Top - Bottom + 1;

  WFFScanLineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanLine = (unsigned short *) malloc(WFFScanLineLength);
  if ((unsigned short *)0 == WFFScanLine)
   FatalError("no memory for WFFScanLine");

  VICARScanLineLength = width*BandsPerPixel*(sizeof (unsigned char));
  VICARScanLineLength = VICARScanLineLength + (VICARScanLineLength % 2);
  VICARScanLine = (unsigned char *) malloc(VICARScanLineLength);
  if ((unsigned char *)0 == VICARScanLine)
   FatalError("no memory for VICARScanLine");

  if (VERBOSE) fprintf(stderr,"%s: Writing VICARHeader\n", RoutineName);

  WriteVICARHeader(outFD, width, height, BandsPerPixel); 

  if (VERBOSE) fprintf(stderr,"%s: Writing VICAR image", RoutineName);

  for (y=Top;y>=Bottom;y--)
   {
    if (FAILURE == GetBlock(FrameBuffer,y,Left,y,Right, WFFScanLine))
     FatalError("GetBlock Failed");

    i = 0;  c = 0;

    for(x=Left;x<=Right;x++)
     {  /* convert a sample from BitsPerBand to 8 bits */
      register unsigned int j, sample, all;
      sample = ((int)WFFScanLine[c++]) << (16-BitsPerBand) >> 8;
      for(all=0, j=0; j<8; j += BitsPerBand)
       all = (all >> BitsPerBand) | sample;
      VICARScanLine[i++] = (unsigned char) (all & 0x000000ff);
     }
    WriteVICARScanLine(outFD, VICARScanLine, VICARScanLineLength);
    if (VERBOSE) fprintf(stderr,".");
   }
  
   if (VERBOSE) fprintf(stderr,"\n");

   if (ferror(outFD))
    fprintf(stderr,"%s: Error writing image\n", RoutineName);

   if (VERBOSE)
    fprintf(stderr,"%s: finished writing the image\n", RoutineName);

   fflush(outFD); (void)CloseFB(&FrameBuffer);
   free(VICARScanLine); free(WFFScanLine);
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;

  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': { usage(); exit(-1); }
     }
   }
 
  Pass(stdin, stdout);

  exit(0); 
 }
