/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Program:  IMAGE.C                                                         */
/*                                                                           */
/* Purpose:  This is version 2 of the image processing library.  It is       */
/*           intended to provide a simple interface for reading and          */
/*           writing images created by version 1 of the library.  This       */
/*           collection of routines differs from the original routines       */
/*           in the following ways:                                          */
/*                                                                           */
/*           1)  New types of images are handled.  The set now includes      */
/*               GREY, COLOR, COLORPACKED, BYTE, SHORT, LONG, REAL,          */
/*               COMPLEX and USERPACKED.                                     */
/*                                                                           */
/*           2)  Faster access to pixels is provided.  Special routines      */
/*               are included which are optimal for reading/writing 1D,      */
/*               2D and 3D images.  The N dimensional pixel access routine   */
/*               has also been improved.                                     */
/*                                                                           */
/*           3)  Three new routines imread, imwrite and imheader.            */
/*               See the iman entries for a full description of these        */
/*               new routines.                                               */
/*                                                                           */
/* Contains: imcreat            - Image initialization routines              */
/*           imopen                                                          */
/*           imclose                                                         */
/*                                                                           */
/*           imread             - Pixel access routines                      */
/*           imwrite                                                         */
/*           imgetpix                                                        */
/*           imputpix                                                        */
/*           GetPut2D                                                        */
/*           GetPut3D                                                        */
/*           GetPutND                                                        */
/*                                                                           */
/*           imheader           - Information access routines                */
/*           imdim                                                           */
/*           imbounds                                                        */
/*           imgetdesc                                                       */
/*           imtest                                                         */
/*           imgettitle                                                      */
/*           imputtitle                                                      */
/*           imgetinfo                                                       */
/*           imputinfo                                                       */
/*           imcopyinfo                                                      */
/*           iminfoids                                                       */
/*           imerror                                                         */
/*           im_snap                                                         */
/*                                                                           */
/* Author:   John Gauch - Version 2                                          */
/*           Zimmerman, Entenman, Fitzpatrick, Whang - Version 1             */
/*                                                                           */
/* Date:     February 23, 1987                                               */
/*                                                                           */
/* Revised:  October 29, 1987 - Fixed one problem with the way histograms    */
/*           are calculated in imgetdesc.  Changes marked with JG1.          */
/*                                                                           */
/* Revised:  June 2, 1989 - Added imtest.  A. G. Gash.			     */
/*	     Sep 20, 189 - Added im_snap.  A. G. Gash.			     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
#include "image.h"
char *malloc();
long lseek();

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  These declarations are private to this library.                 */
/*           These MACROs copy an error message into the error string.       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Constants for lseek calls */
#define FROMBEG		0
#define FROMHERE	1
#define FROMEND		2

/* Address index constants */
#define aMAXMIN		0
#define aHISTO		1
#define aTITLE		2
#define aPIXFORM	3
#define aDIMC		4
#define aDIMV		5
#define aPIXELS		6
#define aINFO		7
#define aVERNO		8

/* Modes for GetPut routines */
#define READMODE	0
#define WRITEMODE	1

/* Blocking factor for imgetdesc */
#define MAXGET 4096

#define Error(Mesg)\
   {\
   strcpy(_imerrbuf, Mesg);\
   return(INVALID);\
   }

#define ErrorNull(Mesg)\
   {\
   strcpy(_imerrbuf, Mesg);\
   return(NULL);\
   }

#define Warn(Mesg)\
   {\
   strcpy(_imerrbuf, Mesg);\
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine creates an image.  The user specified image        */
/*           parameters are stored in the new image record.  For historical  */
/*           reasons, 4092 bytes are unused between the end of the           */
/*           histogram and the start of the pixel format field - gauchj.     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
#define UNUSED 4092
IMAGE *imcreat(Name, Protection, PixForm, Dimc, Dimv)
   char *Name;
   int Protection;
   int PixForm;
   int Dimc;
   int *Dimv;
   {
   IMAGE *Image;
   int Cnt;
   int Fd;
   int i;
   char Null = '\0';
   
   /* Check parameters */
   if (Name == NULL) ErrorNull("Null image name");
   if ((PixForm != GREY) && (PixForm != COLOR) && (PixForm != COLORPACKED) && 
       (PixForm != BYTE) && (PixForm != SHORT) && (PixForm != LONG) && 
       (PixForm != REAL) && (PixForm != COMPLEX) && (PixForm != USERPACKED)) 
      ErrorNull("Invalid pixel format");
   if ((Dimc < 1) || (Dimc > nDIMV)) ErrorNull("Illegal number of dimensions");

   /* Create image file */
   Fd = open(Name,CREATE,Protection);
   if (Fd == EOF) ErrorNull("Image already exists");

   /* Allocate image record */
   Image = (IMAGE *)malloc((unsigned)sizeof(IMAGE));
   if (Image == NULL) ErrorNull("Allocation error");

   /* Initialize image record */   
   Image->Title[0] = Null;
   Image->ValidMaxMin = FALSE;
   Image->MaxMin[0] = MAXVAL;
   Image->MaxMin[1] = MINVAL;
   Image->ValidHistogram = FALSE;
   for (i=0; i<nHISTOGRAM; i++)
      Image->Histogram[i] = 0;
   Image->PixelFormat = PixForm;
   Image->Dimc = Dimc;

   /* Determine number of pixels in image */
   Image->PixelCnt = 1;
   for (i=0; i<Image->Dimc; i++) {
      Image->Dimv[i] = Dimv[i];
      Image->PixelCnt = Image->PixelCnt * Dimv[i];
   }

   /* Determine size of each pixel */
   switch (Image->PixelFormat) {
      case GREY		: Image->PixelSize = 2; break;
      case COLOR	: Image->PixelSize = 4; break;
      case COLORPACKED	: Image->PixelSize = 4; break;
      case BYTE		: Image->PixelSize = 1; break;
      case SHORT	: Image->PixelSize = 2; break;
      case LONG		: Image->PixelSize = 4; break;
      case REAL		: Image->PixelSize = 4; break;
      case COMPLEX	: Image->PixelSize = 8; break;
      case USERPACKED	: Image->PixelSize = 4; break;
   }

   /* Initialize image addresses (do NOT change this) */
   Image->Address[aTITLE] = sizeof( Image->Address );
   Image->Address[aMAXMIN] = Image->Address[aTITLE] 
      + sizeof( Image->Title );
   Image->Address[aHISTO] = Image->Address[aMAXMIN] 
      + sizeof( Image->ValidMaxMin ) + sizeof( Image->MaxMin );
   Image->Address[aPIXFORM] = Image->Address[aHISTO] + UNUSED
      + sizeof( Image->ValidHistogram ) + sizeof( Image->Histogram );
   Image->Address[aDIMC] = Image->Address[aPIXFORM] 
      + sizeof( Image->PixelFormat );
   Image->Address[aDIMV] = Image->Address[aDIMC] 
      + sizeof( Image->Dimc );
   Image->Address[aPIXELS] = Image->Address[aDIMV] 
      + sizeof( Image->Dimv );
   Image->Address[aINFO] = Image->Address[aPIXELS] 
      + Image->PixelCnt * Image->PixelSize;
   Image->Address[aVERNO] = 1;

   /* Write null information field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aINFO], FROMBEG);
   Cnt = write(Fd, (char *)&Null, sizeof(Null));
   if (Cnt != sizeof(Null)) ErrorNull("Image write failed");
   Image->InfoCnt = 0;

   /* Save file pointer */
   Image->Fd = Fd;
   return(Image);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine opens an image.  The image parameters are          */
/*           read from the file and stored in the image record.              */
/*                                                                           */
/*---------------------------------------------------------------------------*/
IMAGE *imopen(ImName, Mode)
   char *ImName;
   int Mode;
   {
   IMAGE *Image;
   int Cnt;
   int Fd;
   int i;
   int Length;
   char *Buffer;
   char *StrPtr;
   char *Name;
   char *Data;

   /* Check parameters */
   if (ImName == NULL) ErrorNull("Null image name");
   if ((Mode != READ) && (Mode != UPDATE)) ErrorNull("Invalid open mode");

   /* Open image file */
   Fd = open(ImName,Mode);
   if (Fd == EOF) ErrorNull("Image file not found");

   /* Allocate image record */
   Image = (IMAGE *)malloc((unsigned)sizeof(IMAGE));
   if (Image == NULL) ErrorNull("Allocation error");

   /* Read addresses of image header fields */
   Cnt = read(Fd, (char *)&Image->Address[0], sizeof(Image->Address));
   if (Cnt != sizeof(Image->Address)) ErrorNull("Image read failed");
   
   /* Read Title field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aTITLE], FROMBEG);
   Cnt = read(Fd, (char *)&Image->Title[0], sizeof(Image->Title));
   if (Cnt != sizeof(Image->Title)) ErrorNull("Image read failed");

   /* Read MaxMin field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aMAXMIN], FROMBEG);
   Cnt = read(Fd, (char *)&Image->ValidMaxMin, sizeof(Image->ValidMaxMin));
   if (Cnt != sizeof(Image->ValidMaxMin)) ErrorNull("Image read failed");
   Cnt = read(Fd, (char *)&Image->MaxMin[0], sizeof(Image->MaxMin));
   if (Cnt != sizeof(Image->MaxMin)) ErrorNull("Image read failed");

   /* Read Histogram field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aHISTO], FROMBEG);
   Cnt = read(Fd, (char *)&Image->ValidHistogram,sizeof(Image->ValidHistogram));
   if (Cnt != sizeof(Image->ValidHistogram)) ErrorNull("Image read failed");
   Cnt = read(Fd, (char *)&Image->Histogram[0], sizeof(Image->Histogram));
   if (Cnt != sizeof(Image->Histogram)) ErrorNull("Image read failed");

   /* Read PixelFormat field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aPIXFORM], FROMBEG);
   Cnt = read(Fd, (char *)&Image->PixelFormat, sizeof(Image->PixelFormat));
   if (Cnt != sizeof(Image->PixelFormat)) ErrorNull("Image read failed");

   /* Determine size of each pixel */
   switch (Image->PixelFormat) {
      case GREY		: Image->PixelSize = 2; break;
      case COLOR	: Image->PixelSize = 4; break;
      case COLORPACKED	: Image->PixelSize = 4; break;
      case BYTE		: Image->PixelSize = 1; break;
      case SHORT	: Image->PixelSize = 2; break;
      case LONG		: Image->PixelSize = 4; break;
      case REAL		: Image->PixelSize = 4; break;
      case COMPLEX	: Image->PixelSize = 8; break;
      case USERPACKED	: Image->PixelSize = 4; break;
      }

   /* Read DimC field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aDIMC], FROMBEG);
   Cnt = read(Fd, (char *)&Image->Dimc, sizeof(Image->Dimc));
   if (Cnt != sizeof(Image->Dimc)) ErrorNull("Image read failed");

   /* Read DimV field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aDIMV], FROMBEG);
   Cnt = read(Fd, (char *)&Image->Dimv[0], sizeof(Image->Dimv));
   if (Cnt != sizeof(Image->Dimv)) ErrorNull("Image read failed");
    
   /* Determine number of pixels in image */
   Image->PixelCnt = 1;
   for (i=0; i<Image->Dimc; i++)
      Image->PixelCnt = Image->PixelCnt * Image->Dimv[i];

   /* Determine length of information field */
   Cnt = (int)lseek(Fd, (long)0, FROMEND);
   if (Cnt == -1) ErrorNull("Seek EOF failed");
   Length = Cnt - Image->Address[aINFO];

   /* Allocate buffer for information field */
   if (Length < 1) ErrorNull("Invalid information field");
   Buffer = (char *)malloc((unsigned)Length);
   if (Buffer == NULL) ErrorNull("Allocation error");

   /* Read whole information field into a buffer */
   Cnt = (int)lseek(Fd, (long)Image->Address[aINFO], FROMBEG);
   Cnt = read(Fd, (char *)Buffer, Length);
   if (Cnt != Length) ErrorNull("Image read failed");

   /* Prepare to loop through all fields */
   StrPtr = Buffer;
   Length = strlen(StrPtr) + 1;
   Image->InfoCnt = 0;

   /* Loop through all fields */
   while ((Length > 1) && (Image->InfoCnt < nINFO))
      {
      /* Reading field name from buffer */
      Name = (char *)malloc((unsigned)Length);
      if (Name == NULL) ErrorNull("Allocation error");
      strcpy(Name, StrPtr);
      StrPtr = StrPtr + Length;
      Length = strlen(StrPtr) + 1;

      /* Reading field data from buffer */
      Data = (char *)malloc((unsigned)Length);
      if (Data == NULL) ErrorNull("Allocation error");
      strcpy(Data, StrPtr);
      StrPtr = StrPtr + Length;
      Length = strlen(StrPtr) + 1;

      /* Saving name and data in image header */
      Image->InfoName[Image->InfoCnt] = Name;
      Image->InfoData[Image->InfoCnt] = Data;
      Image->InfoCnt ++;
      }

   /* Free buffer used for information string */
   free(Buffer);

   /* Save file pointer */
   Image->Fd = Fd;
   return(Image);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine closes an image.  The image parameters are         */
/*           written to the file from the image record.                      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imclose(Image)
   IMAGE *Image;
   {
   int Fd;
   int Cnt;
   int Length;
   char Null = '\0';
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");

   /* Check that file is open */
   Fd = Image->Fd;
   if (Fd == EOF) Error("Image not open");

   /* Write addresses of image header fields */
   Cnt = (int)lseek(Fd, (long)0, FROMBEG);
   Cnt = write(Fd, (char *)&Image->Address[0], sizeof(Image->Address));
   if (Cnt != sizeof(Image->Address)) Warn("Image write failed");

   /* Write Title field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aTITLE], FROMBEG);
   Cnt = write(Fd, (char *)&Image->Title[0], sizeof(Image->Title));
   if (Cnt != sizeof(Image->Title)) Warn("Image write failed");

   /* Write MaxMin field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aMAXMIN], FROMBEG);
   Cnt = write(Fd, (char *)&Image->ValidMaxMin, sizeof(Image->ValidMaxMin));
   if (Cnt != sizeof(Image->ValidMaxMin)) Warn("Image write failed");
   Cnt = write(Fd, (char *)&Image->MaxMin[0], sizeof(Image->MaxMin));
   if (Cnt != sizeof(Image->MaxMin)) Warn("Image write failed");

   /* Write Histogram field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aHISTO], FROMBEG);
   Cnt = write(Fd,(char *)&Image->ValidHistogram,sizeof(Image->ValidHistogram));
   if (Cnt != sizeof(Image->ValidHistogram)) Warn("Image write failed");
   Cnt = write(Fd, (char *)&Image->Histogram[0], sizeof(Image->Histogram));
   if (Cnt != sizeof(Image->Histogram)) Warn("Image write failed");

   /* Write PixelFormat field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aPIXFORM], FROMBEG);
   Cnt = write(Fd, (char *)&Image->PixelFormat, sizeof(Image->PixelFormat));
   if (Cnt != sizeof(Image->PixelFormat)) Warn("Image write failed");

   /* Write DimC field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aDIMC], FROMBEG);
   Cnt = write(Fd, (char *)&Image->Dimc, sizeof(Image->Dimc));
   if (Cnt != sizeof(Image->Dimc)) Warn("Image write failed");

   /* Write DimV field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aDIMV], FROMBEG);
   Cnt = write(Fd, (char *)&Image->Dimv[0], sizeof(Image->Dimv));
   if (Cnt != sizeof(Image->Dimv)) Warn("Image write failed");

   /* Write Info field */
   Cnt = (int)lseek(Fd, (long)Image->Address[aINFO], FROMBEG);
   for (i=0; i<Image->InfoCnt; i++)
      {
      /* Write name of field and free string */
      Length = strlen(Image->InfoName[i]) + 1;
      Cnt = write(Fd, (char *)Image->InfoName[i], Length);
      if (Cnt != Length) Warn("Image write failed");
      free(Image->InfoName[i]);

      /* Write field data and free string */
      Length = strlen(Image->InfoData[i]) + 1;
      Cnt = write(Fd, (char *)Image->InfoData[i], Length);
      if (Cnt != Length) Warn("Image write failed");
      free(Image->InfoData[i]);
      }
   Cnt = write(Fd, (char *)&Null, sizeof(Null));
   
   /* Close file and free image record */
   free((char *)Image);
   close(Fd);
   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads pixel data from an image.                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imread(Image, LoIndex, HiIndex, Buffer)
   IMAGE *Image;
   int LoIndex;
   int HiIndex;
   GREYTYPE *Buffer;
   {
   int Cnt;
   int Length;
   int Offset;   

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Buffer == NULL) Error("Null read buffer");
   if ((LoIndex < 0) || (HiIndex > Image->PixelCnt) ||
      (LoIndex > HiIndex)) Error("Invalid pixel index");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Determine number of bytes to read and the lseek offset */
   Length = (HiIndex - LoIndex +1) * Image->PixelSize;
   Offset = Image->Address[aPIXELS] + LoIndex * Image->PixelSize;

   /* Read pixels into buffer */
   Cnt = (int)lseek(Image->Fd, (long)Offset, FROMBEG);
   Cnt = read(Image->Fd, (char *)Buffer, Length);
   if (Cnt != Length) Error("Image pixel read failed");

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes pixel data to an image.  For GREY images,   */
/*           the maximum and minimum values are also updated, but the        */
/*           histogram is NOT updated.  The algorithm used to update the     */
/*           maximum and minimum values generates a bound on the image       */
/*           intensities, but not always the tightest bound.  This is the    */
/*           best we can do by looking at pixels as they are written.  To    */
/*           compute the tightest upper bound requires that we look at all   */
/*           pixels in the image.  This is what imgetdesc does.              */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imwrite(Image, LoIndex, HiIndex, Buffer)
   IMAGE *Image;
   int LoIndex;
   int HiIndex;
   GREYTYPE *Buffer;
   {
   int Cnt;
   int Length;
   int Offset;   
   int TempMin;
   int TempMax;
   int PixelCnt;
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Buffer == NULL) Error("Null read buffer");
   if ((LoIndex < 0) || (HiIndex > Image->PixelCnt) ||
      (LoIndex > HiIndex)) Error("Invalid pixel index");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Determine number of bytes to write and the lseek offset */
   Length = (HiIndex - LoIndex +1) * Image->PixelSize;
   Offset = Image->Address[aPIXELS] + LoIndex * Image->PixelSize;

   /* Write pixels into buffer */
   Cnt = (int)lseek(Image->Fd, (long)Offset, FROMBEG);
   Cnt = write(Image->Fd, (char *)Buffer, Length);
   if (Cnt != Length) Error("Image pixel write failed");

   /* Invalidate the MaxMin and Histogram fields */
   Image->ValidMaxMin = FALSE;
   Image->ValidHistogram = FALSE;

   /* Compute new Maximum and Minimum for GREY images */
   if (Image->PixelFormat == GREY)
      {
      TempMax = Image->MaxMin[1];
      TempMin = Image->MaxMin[0];
      PixelCnt = HiIndex - LoIndex + 1;
      for (i=0; i<PixelCnt; i++)
         {
         if (Buffer[i] > TempMax) TempMax = Buffer[i];
         if (Buffer[i] < TempMin) TempMin = Buffer[i];
         }
      Image->MaxMin[1] = TempMax;
      Image->MaxMin[0] = TempMin;
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads an arbitrary subwindow of an image.  It      */
/*           does optimal I/O for 1D, 2D and 3D images.  More general I/O    */
/*           is used for higher dimension images.                            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imgetpix(Image, Endpts, Coarseness, Pixels)
   IMAGE *Image;
   int Endpts[][2];
   int Coarseness[];
   GREYTYPE *Pixels;
   {
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Pixels == NULL) Error("Null pixel buffer");
   
   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Check endpoints */
   for (i=0; i<Image->Dimc; i++)
      {
      if (Coarseness[i] != 1) Error("Coarseness not implemented");
      if (Endpts[i][0] < 0) Error("Bad endpoints range");
      if (Endpts[i][1] >= Image->Dimv[i]) Error("Bad endpoints range");
      if (Endpts[i][1] < Endpts[i][0]) Error("Bad endpoints order");
      }

   /* Check for image dimensions */
   switch (Image->Dimc) {

      /* Handle 1D images */
      case 1: 
         return(imread(Image, Endpts[0][0], Endpts[0][1], Pixels));
         break;

      /* Handle 2D images */
      case 2:
         return(GetPut2D(Image, Endpts, Pixels, READMODE));
         break;

      /* Handle 3D images */
      case 3:
         return(GetPut3D(Image, Endpts, Pixels, READMODE));
         break;

      /* Handle higher dimension images */
      default:
         return(GetPutND(Image, Endpts, Pixels, READMODE));
         break;
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes an arbitrary subwindow of an image.  It     */
/*           does optimal I/O for 1D, 2D and 3D images.  More general I/O    */
/*           is used for higher dimension images.                            */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imputpix(Image, Endpts, Coarseness, Pixels)
   IMAGE *Image;
   int Endpts[][2];
   int Coarseness[];
   GREYTYPE *Pixels;
   {
   int i;
   int PixelCnt;
   int TempMax;
   int TempMin;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Pixels == NULL) Error("Null pixel buffer");
   
   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Check endpoints */
   PixelCnt = 1;
   for (i=0; i<Image->Dimc; i++)
      {
      if (Coarseness[i] != 1) Error("Coarseness not implemented");
      if (Endpts[i][0] < 0) Error("Bad endpoints range");
      if (Endpts[i][1] >= Image->Dimv[i]) Error("Bad endpoints range");
      if (Endpts[i][1] < Endpts[i][0]) Error("Bad endpoints order");
      PixelCnt = PixelCnt * (Endpts[i][1] - Endpts[i][0] + 1);
      }

   /* Invalidate the MaxMin and Histogram fields */
   Image->ValidMaxMin = FALSE;
   Image->ValidHistogram = FALSE;

   /* Compute new Maximum and Minimum for GREY images */
   if (Image->PixelFormat == GREY)
      {
      TempMax = Image->MaxMin[1];
      TempMin = Image->MaxMin[0];
      for (i=0; i<PixelCnt; i++)
         {
         if (Pixels[i] > TempMax) TempMax = Pixels[i];
         if (Pixels[i] < TempMin) TempMin = Pixels[i];
         }
      Image->MaxMin[1] = TempMax;
      Image->MaxMin[0] = TempMin;
      }

   /* Check for image dimensions */
   switch (Image->Dimc) {

      /* Handle 1D images */
      case 1: 
         return(imwrite(Image, Endpts[0][0], Endpts[0][1], Pixels));
         break;

      /* Handle 2D images */
      case 2:
         return(GetPut2D(Image, Endpts, Pixels, WRITEMODE));
         break;

      /* Handle 3D images */
      case 3:
         return(GetPut3D(Image, Endpts, Pixels, WRITEMODE));
         break;

      /* Handle higher dimension images */
      default:
         return(GetPutND(Image, Endpts, Pixels, WRITEMODE));
         break;
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads and writes pixels to 2D images.  The         */
/*           minimum number of I/O calls are used.                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetPut2D(Image, Endpts, Pixels, Mode)
   IMAGE *Image;
   int Endpts[][2];
   GREYTYPE *Pixels;
   int Mode;
   {
   int Xdim;
   int Ydim;
   int Xlength;
   int Ylength;
   int ReadBytes;
   int Yloop;
   int Yskipcnt;
   int FirstPixel;
   int Cnt;
   int i;
   char *PixelPtr;

   /* The 0th dimension is y; the 1st is x */
   Ydim = 0;
   Xdim = 1;
	    
   /* Determine how many pixels to READ/WRITE in each dimension */
   Ylength = (Endpts[Ydim][1] - Endpts[Ydim][0]) + 1;
   Xlength = (Endpts[Xdim][1] - Endpts[Xdim][0]) + 1;
	    
   /* Determine how many pixels to SKIP in each dimension */
   Yskipcnt = (Image->Dimv[Xdim]-Xlength) * Image->PixelSize;

   /* Determine the maximum number of consecutive pixels that can */
   /* be read in at one time and the number of non-contiguous     */
   /* sections that span the y dimension                          */
   if (Xlength != Image->Dimv[Xdim])
      {
      ReadBytes = Xlength * Image->PixelSize;
      Yloop = Ylength;
      }
   else 
      {
      ReadBytes = Ylength * Xlength * Image->PixelSize;
      Yloop = 1;
      }
 
   /* Calculate position of first pixel */
   FirstPixel = ((Endpts[Ydim][0] * Image->Dimv[Xdim]) 
              + Endpts[Xdim][0]) * Image->PixelSize
              + Image->Address[aPIXELS];

   /* Seek to first pixel in image */
   Cnt = (int)lseek(Image->Fd, (long)FirstPixel, FROMBEG);
   if (Cnt == -1) Error("Seek first pixel failed");
	    
   /* Loop reading/writing pixel data in sections */
   PixelPtr = (char *) Pixels;
   for (i=0; i<Yloop; i++)
      {
      /* Read/write data into buffer */
      if (Mode == READMODE)
         Cnt = read(Image->Fd, (char *)PixelPtr, ReadBytes);
      else
         Cnt = write(Image->Fd, (char *)PixelPtr, ReadBytes);
      if (Cnt != ReadBytes) Error("Pixel read/write failed");

      /* Advance buffer pointer */
      PixelPtr += ReadBytes;

      /* Seek to next line of pixels to read/write */
      Cnt = (int)lseek(Image->Fd, (long)Yskipcnt, FROMHERE);
      if (Cnt == -1) Error("Seek next pixel failed");
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads and writes pixels to 3D images.  The         */
/*           minimum number of I/O calls are used.                           */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetPut3D(Image, Endpts, Pixels, Mode)
   IMAGE *Image;
   int Endpts[][2];
   GREYTYPE *Pixels;
   int Mode;
   {
   int Xdim;
   int Ydim;
   int Zdim;
   int Xlength;
   int Ylength;
   int Zlength;
   int ReadBytes;
   int Yloop;
   int Zloop;
   int Yskipcnt;
   int Zskipcnt;
   int FirstPixel;
   int Cnt;
   int i;
   int j;
   char *PixelPtr;

   /* 0th dimension is z; 1st is y; 2nd is x */
   Zdim = 0;
   Ydim = 1;
   Xdim = 2;

   /* Determine how many pixels to READ/WRITE in each dimension*/
   Zlength = (Endpts[Zdim][1] - Endpts[Zdim][0]) + 1;
   Ylength = (Endpts[Ydim][1] - Endpts[Ydim][0]) + 1;
   Xlength = (Endpts[Xdim][1] - Endpts[Xdim][0]) + 1;

   /* Determine how many pixels to SKIP in each dimension*/
   Yskipcnt = (Image->Dimv[Xdim]-Xlength) * Image->PixelSize;
   Zskipcnt = (Image->Dimv[Ydim]-Ylength) 
            * Image->Dimv[Xdim] * Image->PixelSize;

   /* Determine the maximum number of consecutive pixels that can */
   /* be read in at one time and the number of non-contiguous     */
   /* sections that span the y and z dimensions                   */
   if (Xlength != Image->Dimv[Xdim])
      {
      ReadBytes = Xlength * Image->PixelSize;
      Yloop = Ylength;
      Zloop = Zlength;
      }
   else if (Ylength != Image->Dimv[Ydim]) 
      {
      ReadBytes = Ylength * Xlength * Image->PixelSize;
      Yloop = 1;
      Zloop = Zlength;
      }
   else 
      {
      ReadBytes = Zlength * Ylength * Xlength * Image->PixelSize;
      Yloop = 1;
      Zloop = 1;
      }
	    
   /* Calculate position of first pixel in image */
   FirstPixel = ((Endpts[Zdim][0] * Image->Dimv[Xdim] * Image->Dimv[Ydim])
              + (Endpts[Ydim][0] * Image->Dimv[Xdim]) 
              +  Endpts[Xdim][0]) * Image->PixelSize
              + Image->Address[aPIXELS];

   /* Seek to first pixel in image */
   Cnt = (int)lseek(Image->Fd, (long)FirstPixel, FROMBEG);
   if (Cnt == -1) Error("Seek first pixel failed");

   /* Loop reading/writing pixel data in sections */
   PixelPtr = (char *) Pixels;
   for (i=0; i<Zloop; i++)
      {
      for (j=0; j<Yloop; j++)
         {
         /* Read/write data from file into buffer */
         if (Mode == READMODE)
            Cnt = read(Image->Fd, (char *)PixelPtr, ReadBytes);
         else
            Cnt = write(Image->Fd, (char *)PixelPtr, ReadBytes);
         if (Cnt != ReadBytes) Error("Pixel read/write failed");
   
         /* Advance buffer pointer */
         PixelPtr += ReadBytes;
   
         /* Seek to next line of pixels to read/write */
         Cnt = (int)lseek(Image->Fd, (long)Yskipcnt, FROMHERE);
         if (Cnt == -1) Error("Seek next pixel failed");
         }

      /* Seek to next slice of pixels to read/write */
      Cnt = (int)lseek(Image->Fd, (long)Zskipcnt, FROMHERE);
      if (Cnt == -1) Error("Seek next pixel failed");
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads and writes pixels to N dimensional images.   */
/*           The minimum number of I/O calls are used.                       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
GetPutND(Image, Endpts, Pixels, Mode)
   IMAGE *Image;
   int Endpts[][2];
   GREYTYPE *Pixels;
   int Mode;
   {
   int SliceSize[nDIMV];
   int ReadCnt[nDIMV];
   int SkipCnt[nDIMV];
   int Index[nDIMV];
   int Dimc;
   int ReadBytes;
   int NextPixel;
   int i;
   int Cnt;
   char *PixelPtr;

   /* Determine size of one "slice" in each dimension */
   Dimc = Image->Dimc;
   SliceSize[Dimc-1] = Image->PixelSize;
   for (i=Dimc-2; i>=0; i--)
      SliceSize[i] = SliceSize[i+1] * Image->Dimv[i+1];

   /* Determine number of "slices" to read and skip in each dimension */
   for (i=0; i<Dimc; i++)
      {
      ReadCnt[i] = Endpts[i][1] - Endpts[i][0] + 1;
      SkipCnt[i] = Image->Dimv[i] - ReadCnt[i];
      }
   ReadBytes = ReadCnt[Dimc-1] * SliceSize[Dimc-1];
    
   /* Seek to beginning of pixels */
   Cnt = (int)lseek(Image->Fd, (long)Image->Address[aPIXELS], FROMBEG);

   /* Find offset to first pixel */
   NextPixel = 0;
   for (i=0; i<Dimc; i++)
      {
      NextPixel = NextPixel + SliceSize[i]*Endpts[i][0];
      Index[i] = Endpts[i][0];
      }

   /* Loop reading and skipping pixels */
   PixelPtr = (char *) Pixels;
   while (Index[0] <= Endpts[0][1])
      {
      /* Seek to next line of pixels to read/write */
      Cnt = (int)lseek(Image->Fd, (long)NextPixel, FROMHERE);

      /* Read/write data from file into buffer */
      if (Mode == READMODE)
         Cnt = read(Image->Fd, (char *)PixelPtr, ReadBytes);
      else
         Cnt = write(Image->Fd, (char *)PixelPtr, ReadBytes);
      if (Cnt != ReadBytes) Error("Pixel read/write failed");
  
      /* Advance buffer pointer */
      PixelPtr += ReadBytes;
      Index[Dimc-1] = Endpts[Dimc-1][1] + 1;
 
      /* Find offset to next pixel */
      NextPixel = 0;
      for (i=Dimc-1; i>=0; i--)
         {
         if (Index[i] > Endpts[i][1])
            {
            NextPixel = NextPixel + SkipCnt[i] * SliceSize[i];
            if (i > 0) 
               {
               Index[i] = Endpts[i][0];
               Index[i-1]++;
               }
            }
         }
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads all useful image information from the        */
/*           image image header.  This routine can be used in place of       */
/*           imdim, imbounds, and imgetdesc.                                 */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imheader(Image, PixFormat, PixSize, PixCnt, Dimc, Dimv, MaxMin)
   IMAGE *Image;
   int *PixFormat;
   int *PixSize;
   int *PixCnt;
   int *Dimc;
   int *Dimv;
   int *MaxMin;
   {
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (PixFormat == NULL) Error("Null pixformat pointer");
   if (PixSize == NULL) Error("Null pixsize pointer");
   if (PixCnt == NULL) Error("Null pixcnt pointer");
   if (Dimc == NULL) Error("Null dimc pointer");
   if (Dimv == NULL) Error("Null dimv pointer");
   if (MaxMin == NULL) Error("Null maxmin pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Copy data */
   *PixFormat = Image->PixelFormat;
   *PixSize = Image->PixelSize;
   *PixCnt = Image->PixelCnt;
   *Dimc = Image->Dimc;
   for (i=0; i<*Dimc; i++)
      Dimv[i] = Image->Dimv[i];

   /* Get correct MINMAX field */
   imgetdesc(Image, MINMAX, MaxMin);
   
   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine returns the pixel format and dimension count.      */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imdim(Image, PixFormat, Dimc)
   IMAGE *Image;
   int *PixFormat;
   int *Dimc;
   {

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (PixFormat == NULL) Error("Null pixformat pointer");
   if (Dimc == NULL) Error("Null dimc pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Copy data */
   *PixFormat = Image->PixelFormat;
   *Dimc = Image->Dimc;

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine returns the image dimension vector.                */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imbounds(Image, Dimv)
   IMAGE *Image;
   int *Dimv;
   {
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Dimv == NULL) Error("Null dimv pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Copy data */
   for (i=0; i<Image->Dimc; i++)
      Dimv[i] = Image->Dimv[i];
   
   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads the MAXMIN or HISTO field from the image.    */
/*           The histogram field contains 4096 buckets which contain         */
/*           pixel counts for pixels in the range [MinPixel..MaxPixel].      */
/*           When there are less than 4096 pixel values this technique       */
/*           works perfectly.  When there are more than 4096 pixel values    */
/*           all pixels greater than MinPixel+4095 are included in the       */
/*           4096th bucket.  While this seems like a bit of a hack, it was   */
/*           the best solution given historical contraints.  Ideally, the    */
/*           size of the histogram array should be [MinPixel..MaxPixel].     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imgetdesc(Image, Type, Buffer)
   IMAGE *Image;
   int Type;
   int Buffer[];
   {
   GREYTYPE Pixels[MAXGET];
   int Low, High;
   int TempMax, TempMin;
   int PixelCnt;
   int Index;
   int i;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Image->PixelFormat != GREY) Error("Image type is not GREY");

   /* Handle request for MINMAX field */
   if (Type == MINMAX)
      {
      if (Image->ValidMaxMin == TRUE)
         {
         /* Copy current MaxMin field */
         Buffer[0] = Image->MaxMin[0];
         Buffer[1] = Image->MaxMin[1];
         }
      else
         {
         /* Compute new MaxMin field */
         TempMin = MAXVAL;
         TempMax = MINVAL;

         /* Loop reading pixels */
         for (Low=0; Low<Image->PixelCnt; Low=Low+MAXGET)
            {
            /* Compute read range */
            High = Low + MAXGET - 1;
            if (High >= Image->PixelCnt) High = Image->PixelCnt -1;
 
            /* Read pixels */
            if (imread(Image, Low, High, Pixels) == INVALID)
               Error("Could not read pixels");
            
            /* Search for new MaxMin Values */
            PixelCnt = High - Low + 1;
            for (i=0; i<PixelCnt; i++)
               {
               if (Pixels[i]<TempMin) TempMin = Pixels[i];
               if (Pixels[i]>TempMax) TempMax = Pixels[i];
               }
            }

         /* Save new MaxMin field */
         Image->ValidMaxMin = TRUE;
         Image->MaxMin[0] = Buffer[0] = TempMin;
         Image->MaxMin[1] = Buffer[1] = TempMax;
         }
      }

   /* Handle request for HISTOGRAM field */
   else if (Type == HISTO) 
      {
      if (Image->ValidHistogram == TRUE)
         {
         /* Copy current Histogram field */
         for (i=0; i<nHISTOGRAM; i++)
            Buffer[i] = (int) Image->Histogram[i];
         }
      else
         {
         /* Compute MAX and MIN pixel values JG1 */
         if (imgetdesc(Image, MINMAX, Buffer) == INVALID)
            Error("Could not obtain minmax values");
         TempMin = Buffer[0];
         TempMax = Buffer[1];
         
         /* Compute new Histogram field */
         for(i=0; i<nHISTOGRAM; i++)
	    Buffer[i] = 0;

         /* Loop reading pixels */
         for (Low=0; Low<Image->PixelCnt; Low=Low+MAXGET)
            {
            /* Compute read range */
            High = Low + MAXGET - 1;
            if (High >= Image->PixelCnt) High = Image->PixelCnt -1;
 
            /* Read pixels */
            if (imread(Image, Low, High, Pixels) == INVALID)
               Error("Could not read pixels");
            
            /* Update Histogram and handle any pixels out of range */
            for (i=0; i<=(High-Low); i++)
               {
               /* Determine index into histogram map JG1 */
               Index = Pixels[i] - TempMin;
               if (Index < 0)
                  Buffer[ 0 ]++;
               else if (Index >= nHISTOGRAM)
                  Buffer[ nHISTOGRAM-1 ]++;
               else
                  Buffer[ Index ]++;
               }
            }

         /* Save new Histogram field */
         Image->ValidHistogram = TRUE;
         for (i=0; i<nHISTOGRAM; i++)
            Image->Histogram[i] = Buffer[i];
         }
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine returns the validity of the HISTO or MAXMIN	     */
/*	     fields.							     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imtest(Image, Type)
   IMAGE *Image;
   int Type;
   {

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Handle request for MINMAX field */
   if (Type == MINMAX)
      return (Image->ValidMaxMin);

   /* Handle request for HISTOGRAM field */
   else if (Type == HISTO) 
      return (Image->ValidHistogram);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads the title field from the image header.       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imgettitle(Image, Title)
   IMAGE *Image;
   char *Title;
   {

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Title == NULL) Error("Null title string pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Copy title field */
   strcpy(Title, Image->Title);

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes the title field to the image header.        */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imputtitle(Image, Title)
   IMAGE *Image;
   char *Title;
   {
   int Length;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Title == NULL) Error("Null title string pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Copy title field */
   Length = strlen(Title) + 1;
   if (Length > nTITLE) Error("Title too long");
   strcpy(Image->Title, Title);

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine reads the specified information field.             */
/*                                                                           */
/*---------------------------------------------------------------------------*/
char *imgetinfo(Image, Name)
   IMAGE *Image;
   char *Name;
   {
   int i;
   int Length;
   char *Data;

   /* Check parameters */
   if (Image == NULL) ErrorNull("Null image pointer");
   if (Name == NULL) ErrorNull("Null field name pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) ErrorNull("Image not open");

   /* Initialize return pointer */
   Data = NULL;

   /* Search list of information fields */
   for (i=0; i<Image->InfoCnt; i++)
      {
      /* If names match, return the data */
      if (strcmp(Name, Image->InfoName[i]) == 0)
         {
         Length = strlen(Image->InfoData[i]) + 1;
         Data = (char *)malloc((unsigned)Length);
         if (Data == NULL) ErrorNull("Allocation error");
         strcpy(Data, Image->InfoData[i]);
         } 
      }

   return(Data);
   }
 
/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine writes the specified information field.  If the    */
/*           data is NULL, the field is undefined (ie removed from list).    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imputinfo(Image, Name, Data)
   IMAGE *Image;
   char *Name;
   char *Data;
   {
   int i;
   int Length;
   int Last;
   int Found;
   int Match;

   /* Check parameters */
   if (Image == NULL) Error("Null image pointer");
   if (Name == NULL) Error("Null field name pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) Error("Image not open");

   /* Search list of information fields */
   Found = FALSE;
   for (i=0; i<Image->InfoCnt; i++)
      {
      /* Determine if names match */
      Match = strcmp(Name, Image->InfoName[i]);

      /* Either: Replace the data */
      if ((Match == 0) && (Data != NULL))
         {
         free(Image->InfoData[i]);
         Length = strlen(Data) + 1;
         Image->InfoData[i] = (char *)malloc((unsigned)Length);
         if (Image->InfoData[i] == NULL) Error("Allocation error");
         strcpy(Image->InfoData[i], Data);
         Found = TRUE;
         break;
         }

      /* Or: Delete the data and field name */
      else if ((Match == 0) && (Data == NULL))
         {
         free(Image->InfoName[i]);
         free(Image->InfoData[i]);
         Last = Image->InfoCnt - 1;
         Image->InfoName[i] = Image->InfoName[Last];
         Image->InfoData[i] = Image->InfoData[Last];
         Image->InfoCnt--;
         Found = TRUE;
         break;
         } 
      }

   /* Append to end of information field if not found */
   if ((Found == FALSE) && (Image->InfoCnt < nINFO) && (Data != NULL))
      {
      /* Add name field */
      Length = strlen(Name) + 1;
      Image->InfoName[i] = (char *)malloc((unsigned)Length);
      if (Image->InfoName[i] == NULL) Error("Allocation error");
      strcpy(Image->InfoName[i], Name);

      /* Add data field */
      Length = strlen(Data) + 1;
      Image->InfoData[i] = (char *)malloc((unsigned)Length);
      if (Image->InfoData[i] == NULL) Error("Allocation error");
      strcpy(Image->InfoData[i], Data);

      /* Increment field counter */
      Image->InfoCnt ++;
      }

   return(VALID);
   }
 
/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine copies all information fields from Image1 to       */
/*           Image2.                                                         */
/*                                                                           */
/*---------------------------------------------------------------------------*/
imcopyinfo(Image1, Image2)
   IMAGE *Image1;
   IMAGE *Image2;
   {
   int Length;
   int i;

   /* Check parameters */
   if (Image1 == NULL) Error("Null image pointer");
   if (Image2 == NULL) Error("Null image pointer");

   /* Check that file is open */
   if (Image1->Fd == EOF) Error("Image not open");
   if (Image2->Fd == EOF) Error("Image not open");

   /* Loop through list of information fields */
   Image2->InfoCnt = 0;
   for (i=0; i<Image1->InfoCnt; i++)
      {
      /* Copy name field */
      Length = strlen(Image1->InfoName[i]) + 1;
      Image2->InfoName[i] = (char *)malloc((unsigned)Length);
      if (Image2->InfoName[i] == NULL) Error("Allocation error");
      strcpy(Image2->InfoName[i], Image1->InfoName[i]);

      /* Copy data field */
      Length = strlen(Image1->InfoData[i]) + 1;
      Image2->InfoData[i] = (char *)malloc((unsigned)Length);
      if (Image2->InfoData[i] == NULL) Error("Allocation error");
      strcpy(Image2->InfoData[i], Image1->InfoData[i]);

      /* Increment field counter */
      Image2->InfoCnt ++;
      }

   return(VALID);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine returns a pointer to a list of information         */
/*           field names.                                                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
char **iminfoids(Image)
   IMAGE *Image;
   {
   char **Name;
   int Length;
   int i;

   /* Check parameters */
   if (Image == NULL) ErrorNull("Null image pointer");

   /* Check that file is open */
   if (Image->Fd == EOF) ErrorNull("Image not open");

   /* Allocate array of pointers */
   Length = sizeof(char *) * (Image->InfoCnt + 1);
   Name = (char **)malloc((unsigned)Length);
   
   /* Loop through list of information fields */
   for (i=0; i<Image->InfoCnt; i++)
      {
      /* Copy name field */
      Length = strlen(Image->InfoName[i]) + 1;
      Name[i] = (char *)malloc((unsigned)Length);
      if (Name[i] == NULL) ErrorNull("Allocation error");
      strcpy(Name[i], Image->InfoName[i]);
      }

   /* Put a null pointer at the end of the list */
   Name[Image->InfoCnt] = 0;

   return(Name);
   }

/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Purpose:  This routine returns the error string from the error buffer.    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
char *imerror()
   {
   char *Message;
   int Length;

   /* Copy error string */
   Length = strlen(_imerrbuf) + 1;
   Message = (char *)malloc((unsigned)Length);
   if (Message == NULL) ErrorNull("Allocation error");
   strcpy(Message, _imerrbuf);

   return(Message);
   }



/*page*/
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* Function:  Im_snap							     */
/*                                                                           */
/* Purpose:  To privide a quick means to writing out a 2D image from a	     */
/*	     program undergoing development.  This is not intended to be     */
/*	     used in finished programs.					     */
/*                                                                           */
/* Return:  Returns INVALID if there is a failure, VALID if the image is     */
/*	    successfully dumped.					     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
im_snap(xdim, ydim, pixformat, name, newtitle, pixel)
int	xdim;
int	ydim;
int	pixformat;
char   *name;
char   *newtitle;
GREYTYPE       *pixel;
{

IMAGE  *image;
int	dimc;
int	dimv[nDIMV];
int	pixcnt;
int	maxmin[nMAXMIN];
int	histo[nHISTOGRAM];


	if (pixformat != GREY) {
		Error("Snapshot image must be GREY type");
	}

	pixcnt = xdim*ydim;
	dimc = 2;
	dimv[0] = ydim;
	dimv[1] = xdim;

	/* Create new image file */
	if ((image = imcreat(name, DEFAULT, GREY, dimc, dimv)) == INVALID) {
		Error("Can not create snapshot image");
	}

	/* Write out new image file */
	if (imwrite(image, 0, pixcnt - 1, pixel) == INVALID) {
		Error("Can not write pixels to snapshot image\n");
	}

	/* Update the values of minmax and histo for new image */
	if (imgetdesc(image, MINMAX, maxmin) == INVALID) {
		Error("Can not update maxmin field");
	}
	if (imgetdesc(image, HISTO, histo) == INVALID) {
		Error("Can not update histo field");
	}

	if (newtitle != NULL) 
		if (imputtitle(image, newtitle) == INVALID) {
			Error("Can not write title to new image");
		}

	/* Close image files */
	imclose(image);

	return (VALID);
}
