/*
  File:	wff3x3.c
  Authors: K.R. Sloan
  Last Modified: 11 March 1988
  Purpose: convolves a wff image with a 3x3 template,
           plus a constant (DC) value.
           only those pixels which fall under the center pixel of the 
           template are changed (i.e., a 1 pixel border of the window is 
           left alone.)

           Template (and DC) values are scaled integers
           (positive or negative.)  The scale factor is 2048.  So,
            2048 == 1.0, 1024 == 0.5, -512 == -0.25, etc.
           
           It is strongly recommended that the individual magnitudes not
           be too large, and that the sum of the values in the template
           (plus the DC value) lie in the range [0,2048] (i.e., [0.0,1.0])
           
 */

#include <stdio.h>
#include <wff.h>
#include <sys/file.h>
#include <sys/errno.h>

#define ShiftFactor 11
#define ScaleFactor 2048

static char RoutineName[] = "wff3x3";

int
wff3x3(fdIn, fdOut, Template, DC)
 FILE *fdIn, *fdOut;
 int Template[3][3]; 
 int DC;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10];
  int BitsPerBand;
  char Name[NameLength], Value[ValueLength];
  int x, y, n;
  int j, k, l, b;
  static unsigned short A[4096], B[4096], C[4096], NewLine[4096];
  unsigned short *LastLine, *ThisLine, *NextLine, *t, *p;
  int dc,t00,t01,t02,t10,t11,t12,t20,t21,t22,value;
  int MaxPixelValue;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   { (void)CloseFB(FBin); return 1; }
  BitsPerBand = atoi(Value);
  BandsPerPixel = strlen(WhatBands);

  if (16 < BitsPerBand )
   {
    fprintf(stderr,"wff3x3: can't handle %d BitsPerBand!",BitsPerBand);
    (void)CloseFB(&FBin);  
    return; 
   }
  if (FAILURE == OpenFB(&FBout)) { (void)CloseFB(&FBin);  return 1; }

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  passed = 0;
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */

  MaxPixelValue = (1 << BitsPerBand) - 1;
  dc  = DC*MaxPixelValue;
  t00 = Template[0][0];
  t01 = Template[0][1];
  t02 = Template[0][2];
  t10 = Template[1][0];
  t11 = Template[1][1];
  t12 = Template[1][2];
  t20 = Template[2][0];
  t21 = Template[2][1];
  t22 = Template[2][2];

  LastLine = &A[0]; ThisLine = &B[0]; NextLine = &C[0];
  /* load up two lines of buffer, and write out one unchanged line */
  p = ThisLine;
  for (x=Left;x<=Right;x++,p += BandsPerPixel) 
   {
    if (FAILURE == NextPixelIn(FBin,p)) return 1;
    if (FAILURE == NextPixelOut(FBout,p)) return 1;
   }

  p = NextLine;
  for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     if (FAILURE == NextPixelIn(FBin,p)) return 1;

  /* step through destination, one pixel at a time */
  for (y = Bottom+1; y < Top; y++)
   {
    /* move forward one line */
    t = LastLine; LastLine = ThisLine; ThisLine = NextLine; NextLine = t;

    p = NextLine;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     if (FAILURE == NextPixelIn(FBin,p)) return 1;


    /* copy the left and right hand pixels */

    j = 0;
    for (b=0;b<BandsPerPixel;b++,j++) NewLine[j] = ThisLine[j];
    j = Right*BandsPerPixel;
    for (b=0;b<BandsPerPixel;b++,j++) NewLine[j] = ThisLine[j];

    /* filter the interior of the line */
    j = 0; k = BandsPerPixel; l = k + BandsPerPixel;
    for (x = Left+1; x < Right; x++)
     for (b=0;b<BandsPerPixel; b++,j++,k++,l++)
      {
       value = dc + LastLine[j]*t00 + LastLine[k]*t01 + LastLine[l]*t02
                  + ThisLine[j]*t10 + ThisLine[k]*t11 + ThisLine[l]*t12
                  + NextLine[j]*t20 + NextLine[k]*t21 + NextLine[l]*t22;
       value =  value >> ShiftFactor;

      if      (value < 0)             value = 0;
      else if (value > MaxPixelValue) value = MaxPixelValue;
      NewLine[k] = (unsigned short) value;
     }

    p = NewLine;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
    if (FAILURE == NextPixelOut(FBout,p)) return 1;
   }

  p = NextLine;
  for (x=Left;x<=Right;x++,p += BandsPerPixel)
    if (FAILURE == NextPixelOut(FBout,p)) return 1;

  (void)CloseFB(&FBin);
  if (FAILURE == CloseFB(&FBout)) return 1;

  return 0;
 }
