/*
   File: mA2I.c 
   Author: Mike Schmidt 
   Last Modified: 11 September 1990
   Purpose: Convert a complex WFF image file to an intensity WFF image file.
            command line options select one of:
              -m - log Magnitude (the default)
              -M - Magnitude (usually useless - the DC value dominates)
              -A - Angle 
 */

#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

#define Magnitude 0
#define Angle     1
#define logMagnitude 2

static char *RoutineName;

static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-m][-M][-A][[-h]\n",
           RoutineName);
 }

static void
Pass(fdIn,fdOut,OutputBand)
 FILE *fdIn, *fdOut;
 int OutputBand;
 {
  FrameBufferType *FBin = NULL, *FBout = NULL;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength], WhatBands[NameLength];
  int x,y,n,BitsPerBand,MaxValue;
  short Pixel[2];
  int passed = 0;
  double MaxMag,LogMax;
  
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  return; }
 
  GetColorSystem(FBin,WhatBands,&BitsPerBand);
  if (0 != strcmp (WhatBands,"mA"))
   {
    fprintf(stderr,"Input file type is not mA!\n");
    (void)CloseFB (&FBin);
    (void)CloseFB (&FBout);
    return;
   }
   
  
  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }


  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Set up the output file and local variables. */
  SetColorSystem(FBout,"I",BitsPerBand);
  GetDescriptor(FBin, "X-MaxMag",Value);
  MaxMag = atof(Value); 
  LogMax = log (MaxMag + 1.0);
  MaxValue = (1 << BitsPerBand) - 1;

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
  for (y=Bottom;y<=Top;y++)
   {
    for (x=Left;x<=Right;x++)
     {
      if (FAILURE == NextPixelIn(FBin,Pixel))
       { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
      switch (OutputBand)
       {
        case Angle: 
         Pixel[0] = Pixel[1]; break;
        case Magnitude: 
         Pixel[0] = (exp(Pixel[0]*LogMax/MaxValue) - 1.0)/MaxMag*MaxValue;
         break;
        default:
        case logMagnitude: break;
       }
      if (FAILURE == NextPixelOut(FBout,Pixel))
       { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
     }
    wffFlush(FBout);
   }

  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);

 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int OutputBand = logMagnitude;
 
  RoutineName = argv[ArgsParsed++];
  for(; ArgsParsed < argc ;)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit (-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'm': OutputBand = logMagnitude; break;
      case 'M': OutputBand = Magnitude; break;
      case 'A': OutputBand = Angle; break;
      default:
      case 'h':  { usage(); exit (-1); }
     }
   }	

  Pass(stdin,stdout,OutputBand);

  fprintf(stderr,"\n                -30-\n");
  exit (0);
}

