/*
  File:	Rank3x3.c
  Authors: K.R. Sloan
  Last Modified: 13 April 1988
  Purpose: computes an order-based filter over 3x3 regions
           only those pixels which fall under the center pixel of the 
           template are changed (i.e., a 1 pixel border of the window is 
           left alone.)

           Ranks are in [0,10].  
           Rank =  0 suppresses non-maxima
           Rank =  1 generates a minimum filter 
           Rank =  5 generates a median filter.
           Rank =  9 generates a maximum filter.            
           Rank = 10 replaces a pixel by it's rank in the neighborhood
                     (scaled to the range)
 */

#include <stdio.h>
#include <wff.h>
#include <sys/file.h>
#include <sys/errno.h>

static char RoutineName[] = "Rank3x3";

int
Rank3x3(fdIn, fdOut, Rank)
 FILE *fdIn, *fdOut;
 int Rank;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10];
  int BitsPerBand;
  char Name[NameLength], Value[ValueLength];
  int x, y, n;
  int j, k, l, b;
  static unsigned short A[4096], B[4096], C[4096], NewLine[4096];
  unsigned short *LastLine, *ThisLine, *NextLine, *t, *p;
  unsigned int MaxValue, ScaleFactor, Gray;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   { (void)CloseFB(FBin); return 1; }
  BitsPerBand = atoi(Value);
  BandsPerPixel = strlen(WhatBands);

  if (16 < BitsPerBand )
   {
    fprintf(stderr,"%s: can't handle %d BitsPerBand!",
                   RoutineName, BitsPerBand);
    (void)CloseFB(&FBin);  
    return; 
   }
  if (FAILURE == OpenFB(&FBout)) { (void)CloseFB(&FBin);  return 1; }

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  passed = 0;
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */

  LastLine = &A[0]; ThisLine = &B[0]; NextLine = &C[0];

  MaxValue = (1 << BitsPerBand) - 1;
  Gray = MaxValue >> 1;
  ScaleFactor = (MaxValue+1) >> 3;

  /* load up ThisLine with Gray */
  p = ThisLine;
  for (x=Left-1; x<=Right+1;x++)
   for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;

  /* load up one line of buffer - Gray on each end */
  p = NextLine;
  for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;
  for (x=Left;x<=Right;x++,p += BandsPerPixel) 
   if (FAILURE == NextPixelIn(FBin,p)) return 1;
  for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;

  /* step through destination, one pixel at a time */
  for (y = Bottom; y <= Top; y++)
   {
    /* move forward one line */
    t = LastLine; LastLine = ThisLine; ThisLine = NextLine; NextLine = t;

    p = NextLine;
    for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;
    if (y == Top)
     for (x=Left;x<=Right;x++) 
      for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;
    else
     for (x=Left;x<=Right;x++,p += BandsPerPixel) 
      if (FAILURE == NextPixelIn(FBin,p)) return 1;
    for (b=0;b<BandsPerPixel;b++,p++) *p = Gray;

    /* filter the line */
    j = 0; k = BandsPerPixel; l = k + BandsPerPixel;
    for (x = Left; x <= Right; x++)
     for (b=0;b<BandsPerPixel; b++,j++,k++,l++)
      {
       unsigned short Sorted[11];

       Sorted[1] = LastLine[j];
       Sorted[2] = LastLine[k];
       Sorted[3] = LastLine[l];
       Sorted[4] = ThisLine[j];
       Sorted[5] = ThisLine[l];
       Sorted[6] = NextLine[j];
       Sorted[7] = NextLine[k];
       Sorted[8] = NextLine[l];

       Sorted[9] = ThisLine[k];  /* last, to set up special cases below */
       { 
        register int i,j,k;
        unsigned short u,v; 
        for(i=2;i<=9;i++)
         {
          u = Sorted[i]; Sorted[0] = u;
          for(j = i, k = i-1;;)
           {
            v = Sorted[k--]; 
            if (u<v) Sorted[j--] = v; else break;
           }
          Sorted[j] = u;
         }

        if (9 == j) 
         { /* this is a local max */
          Sorted[10] = MaxValue;
	 }
        else
         { 
          Sorted[ 0] = 0;
          Sorted[10] = (j-1)*ScaleFactor;
	 }
       }
       NewLine[k] = Sorted[Rank];  
      }

    p = NewLine+1;
    for (x=Left;x<=Right;x++,p += BandsPerPixel) 
     if (FAILURE == NextPixelOut(FBout,p)) return 1;
   }

  (void)CloseFB(&FBin);
  if (FAILURE == CloseFB(&FBout)) return 1;

  return 0;
 }
