/*
   File: Difference.c 
   Authors: Mike Schmidt,
            K.R. Sloan 
   Last Modified: 18 November 1991
   Purpose: Take two WFF input files and create an output file which
            contains the difference of both images to stdout.  

            The default behaviour is that the
            difference (File1 - File2) is rescaled to fit within the
            bounds provided by the number of bits used per band.

            Optionally, the absolute ('-a') difference is calculated, and
            clipped to [0, MaxValue].

 */
#include <stdio.h>
#include <strings.h>
#include <wff.h>

static char *RoutineName;
static int RESCALE = -1;
static int VERBOSE = 0;

static FrameBufferType *FB1in = NULL, *FB2in = NULL, *FBout = NULL;
 
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s InputFile1 InputFile2 [-h][-v][-r][-a]\n",
           RoutineName);
 }

static void ExitErr()
 {
  if ((FrameBufferType *)0 != FB1in)
   (void)CloseFB(&FB1in);
  if ((FrameBufferType *)0 != FB2in)
   (void)CloseFB(&FB2in);
  if ((FrameBufferType *)0 != FBout)
   (void)CloseFB(&FBout);
  exit(-1);
 }
 
static void
Pass(fn1In,fn2In,fdOut)
 char fn1In[],fn2In[];
 FILE *fdOut;
 {
  FILE *fd1In, *fd2In;
  int Bottom1, Left1, Top1, Right1, Bottom2, Left2, Top2, Right2;
  char Value1[ValueLength], Value2[ValueLength];
  int BitsPerBand1,BitsPerBand2,MaxValue,NumOfBands,x,y,n;
  short Pixel1[99],Pixel2[99];  /* that ought to be enough... */
 
  fd1In = fopen (fn1In,"r");
  if ((FILE *)0 == fd1In)
   {
    fprintf(stderr,"Can't open input file: %s\n",fn1In);
    exit(-1);
   }
  fd2In = fopen (fn2In,"r");
  if ((FILE *)0 == fd2In)
   {
    fprintf(stderr,"Can't open input file: %s\n",fn2In);
    exit(-1);
   }
  if (FAILURE == OpenFB(&FB1in)) ExitErr();
  if (FAILURE == OpenFB(&FB2in)) ExitErr();
  if (FAILURE == PassImageIn(fd1In, FB1in)) ExitErr();
  if (FAILURE == PassImageIn(fd2In, FB2in)) ExitErr();
  if (FAILURE == OpenFB(&FBout)) ExitErr();
 
  /* Check that files have same encoding, bits per band, and dimensions. */
  GetColorSystem (FB1in,Value1,&BitsPerBand1);
  GetColorSystem (FB2in,Value2,&BitsPerBand2);
  GetBounds(FB1in,&Bottom1,&Left1,&Top1,&Right1);
  GetBounds(FB2in,&Bottom2,&Left2,&Top2,&Right2);
  if ((0 != strcmp(Value1,Value2)) || (BitsPerBand1 != BitsPerBand2) ||
      (Bottom1 != Bottom2)         || (Left1 != Left2)               ||
      (Top1 != Top2)               || (Right1 != Right2))
   {
    fprintf (stderr,"Input files are not compatible!\n");
    ExitErr();
   }

  /* Make a new header for the output file. */
  SetColorSystem (FBout,Value1,BitsPerBand1);
  GetDescriptor(FB1in,"Encoding",Value1);
  SetDescriptor(FBout,"Encoding",Value1);
  SetBounds(FBout,Bottom1,Left1,Top1,Right1);

  /* Create an X-PassedBy name in the new header. */
  strcpy (Value1,fn1In);
  strcat (Value1," - ");
  strcat (Value1,fn2In);
  strcat (Value1,", ");
  strcat (Value1,RoutineName);
  SetDescriptor(FBout, "X-PassedBy", Value1);

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout)) ExitErr();
  MaxValue = (1 << BitsPerBand1) - 1;
  NumOfBands = strlen(Value2);

  /* Finally, pass the pixels */
  for (y=Bottom1;y<=Top1;y++)
   {
    for (x=Left1;x<=Right1;x++)
     {
       if (FAILURE == NextPixelIn (FB1in,Pixel1)) ExitErr();
       if (FAILURE == NextPixelIn (FB2in,Pixel2)) ExitErr();
       for (n = 0; n < NumOfBands; n++)
       if (RESCALE)
        {
         Pixel1[n] = (Pixel1[n] - Pixel2[n] + MaxValue) / 2;
        }
       else
        {
         int v;
         v = (int)Pixel1[n] - (int)Pixel2[n];
         if (v < 0) v = 0; else if (MaxValue < v) v = MaxValue;
         Pixel1[n] = v;
        }
       if (FAILURE == NextPixelOut(FBout,Pixel1)) ExitErr();
     }
    wffFlush(FBout);
   }

  (void)CloseFB(&FB1in);
  (void)CloseFB(&FB2in);
  (void)CloseFB(&FBout);

 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *File1, *File2;

  RoutineName = argv[ArgsParsed++];
  RESCALE = -1;

  if ((argc-ArgsParsed) < 2)  { usage(); exit(-1); }	
  File1 = argv[ArgsParsed++];
  File2 = argv[ArgsParsed++];

  while (argc > ArgsParsed)
   {
    if ('-' != argv[ArgsParsed][0]) {usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
      case 'r': RESCALE = -1; break;
      case 'a': RESCALE = 0; break;
     }
   }

  Pass(File1,File2,stdout);

  exit(0);
}

