/* file: tests.c - all routines to test */

#include "stdio.h"
#include "math.h"

#define X       0
#define Y       1
#define Z       2
#define W       3

/* return 'z' of left-hand or right-hand cross product */
#define GIScalerCrossZ( v0, v1 ) ( (v0)[X] * (v1)[Y] - (v0)[Y] * (v1)[X] )

/*****************************************************************************/
/*****************************************************************************/
/* Fred Fisher test */

#define	DEBUG	1
int	verbose;			/* for tracing level */

#if	DEBUG
#define GIAssure(cond)                                                  \
        if (!(cond)) {                                                  \
            printf( "\7\7\7??? 'cond' failed\n");                       \
            printf( "GIAssure: Line %d, File %s\n\n", __LINE__, __FILE__); \
        }

#define	GITrace(level, args)	if ( verbose >= level ) { args };

#else
#define GIAssure(cond)
#define	GITrace(level, args)
#endif

/*****************************************************************************/
#define	DirLeft		0
#define	DirRight	1

isConvexFred( nvert, pVert )
int	  nvert;
float	  pVert[][2];
/* determine if the polygon is:
 *	return 0 if degenerate (all points co-linear or co-incident
 *	return 1 if convex
 *	return 2 if non-convex
 */
{
    int		firsti1, i1, i2, dir, turn = 0;
    float	vec1[3], vec2[3], *pcur, *pnext, *ptemp, ccw = 0.0, 
		outside, 
		/* following keep track if all edges are degenerate */
		outsideCCW, totalOutsideCCW = 0.0;

    if ( nvert < 3  ) return(0);

    /***** find indices i1 and i2 to 3 non-coincident points */
    i1 = 1;
    pcur  = vec1;
    pnext = vec2;
    /* scan for first pair of non-coincident pts */
    do {
	/* this is faster than computing entire cross product and checking
	 * for zero. we'll use 'pcur' later to complete cross product.
	 */
	pcur[X] = pVert[i1][X] - pVert[0][X];
	pcur[Y] = pVert[i1][Y] - pVert[0][Y];
	if ( pcur[X] != 0.0  ||  pcur[Y] != 0.0 ) {
	    break;
	}
    } while ( ++i1 < nvert );
    if ( i1 >= nvert ) return(0);	/* all coincident */
    firsti1 = i1;			/* remember first non-coincident */

    /***** find the next point that's not coincident with coord 'i1' */
    i2 = i1;
    while ( ++i2 < nvert ) {
	pnext[X] = pVert[i2][X] - pVert[i1][X];
	pnext[Y] = pVert[i2][Y] - pVert[i1][Y];
	if ( pnext[X] == 0.0  &&  pnext[Y] == 0.0 )
	    continue;
	ccw = GIScalerCrossZ( pcur, pnext );
	/* must get out, consider [0,0] [1,0] [0,0] [0,1] */
	break;
    }
    if ( i2 >= nvert ) return(0);	/* all coincident */

#define	CheckNextVertexForDirection(from, to)				\
    if ( dir == DirLeft ) {						\
	if ( pVert[from][X] < pVert[to][X] ) {				\
	    turn++; 	dir = DirRight; 	/* changed direction */	\
	}								\
    } else {								\
	if ( pVert[from][X] > pVert[to][X] ) {				\
	    turn++; 	dir = DirLeft; 		/* changed direction */	\
	} else {							\
	    /* X is not changing, check last three Y's */               \
	    if ( !( (pcur[X] > 0.0  &&  pVert[to][X] > pVert[from][X]) || \
		    (pcur[X] < 0.0  &&  pVert[to][X] > pVert[from][X]) ) ) { \
		turn++;                         /* only y turned */     \
	    }								\
	}								\
    } 									\

    /***** we now have 3 non-colinear points, find first direction */
    if ( pVert[0][X] < pVert[i1][X] ) {
	/* first vector heads to right */
	dir = DirRight;
	CheckNextVertexForDirection( i1, i2 );	/* needn't check turn yet */
    } else {
	if ( pVert[0][X] > pVert[i1][X] ) {
  	    /* first vector heads to left */
    	    dir = DirLeft;
	    CheckNextVertexForDirection( i1, i2 );  /* needn't check turn yet */
	} else {
	    /* first vector is vertical, decide with second coord */
	    GIAssure( pVert[0][X] - pVert[i2][X] );
	    if ( pVert[0][X] < pVert[i2][X] ) {
	        dir = DirRight;
	    } else {
	        dir = DirLeft;
	    }
	}
    }

    if ( ccw < 0 ) { outside = -1.0; 	totalOutsideCCW = -outside; }
    else           { outside =  1.0;	totalOutsideCCW =  ccw; }

    GITrace( 5,
	printf(
	    "First three indices [%d, %d, %d], dir,outside,ccw = %d, %g, %g\n",
			0, i1, i2, dir, outside, ccw );
    )
    ptemp = pnext;
	    pnext = pcur;
		    pcur = ptemp;		/* swap vector ptrs */

    /***** go around polygon checking turn every 3 non-coincident vertices */
    for ( i1 = i2, ++i2;    i2 < nvert;    i2++ ) {
	pnext[X] = pVert[i2][X] - pVert[i1][X];
	pnext[Y] = pVert[i2][Y] - pVert[i1][Y];
	if ( pnext[X] == 0.0  &&  pnext[Y] == 0.0 )
	    continue;

	/* we have another non-coincident point */
	CheckNextVertexForDirection( i1, i2 );
        if ( turn > 2 ) return(2);

	ccw = GIScalerCrossZ( pcur, pnext );
	GITrace( 5,
	    printf(
	    "on %2d,%2d dir,turn,ccw=[%d,%d,%g], pcur=[%g,%g] pnext=[%g,%g]\n",
		 i1,i2,           dir, turn, ccw,
		 pcur[X], pcur[Y], 	pnext[X], pnext[Y] );
	)
	if ( (outsideCCW = ccw * outside) < 0.0 )
	    return(2);				/* found wrong turn */
	totalOutsideCCW += outsideCCW;
	i1 = i2;
	ptemp = pnext;
		pnext = pcur;
			pcur = ptemp;		/* swap vector ptrs */
    }

    /***** now check final direction */
    CheckNextVertexForDirection( nvert-1, 0 );
    if ( turn > 2 ) return(2);

    /***** and another cross product */
    i2 = 0;
    pnext[X] = pVert[i2][X] - pVert[i1][X];
    pnext[Y] = pVert[i2][Y] - pVert[i1][Y];
    ccw = GIScalerCrossZ( pcur, pnext );
    GITrace( 5,
	printf(
	    "on %2d,%2d dir,turn,ccw=[%d,%d,%g], pcur=[%g,%g] pnext=[%g,%g]\n",
		 i1,i2,           dir, turn, ccw,
		 pcur[X], pcur[Y], 	pnext[X], pnext[Y] );
    )
    if ( (outsideCCW = ccw * outside) < 0.0 )
	return(2);                          /* found wrong turn */
    totalOutsideCCW += outsideCCW;
    i1 = i2;
    ptemp = pnext;
	    pnext = pcur;
		    pcur = ptemp;           /* swap vector ptrs */

    /***** and one more cross product to first non-coincident */
    i2 = firsti1;
    pnext[X] = pVert[i2][X] - pVert[i1][X];
    pnext[Y] = pVert[i2][Y] - pVert[i1][Y];
    ccw = GIScalerCrossZ( pcur, pnext );
    GITrace( 5,
	printf(
	    "on %2d,%2d dir,turn,ccw=[%d,%d,%g], pcur=[%g,%g] pnext=[%g,%g]\n",
		 i1,i2,           dir, turn, ccw,
		 pcur[X], pcur[Y], 	pnext[X], pnext[Y] );
    )
    if ( (outsideCCW = ccw * outside) < 0.0 )
	return(2);                          /* found wrong turn */
    totalOutsideCCW += outsideCCW;

    if ( totalOutsideCCW == 0.0 ) return(0);
    return(1);
}

/*****************************************************************************/
/*****************************************************************************\
/* Steve Hollasch test */

#define EPSILON 1e-15

#define V2op2(X,op,Y) \
    do { X[0] op Y[0]; X[1] op Y[1]; } while (0)

#define V2op3(X,assign,Y,op,Z) \
    do { X[0] assign Y[0] op Z[0]; X[1] assign Y[1] op Z[1]; } while (0)

/*****************************************************************************
**  This routine reads in a stream of 2D polygon coordinates from the input
**  stream and returns 1 if the resulting closed polygon is convex, otherwise
**  it returns 0.  The parameter `stream' is a currently open input stream.
*****************************************************************************/

int 
isConvexSteve(nvert, pVert )
int       nvert;
float     pVert[][2];
{
    auto int    looking = 2;		/* Loop Termination Status */
    auto int    factor = 0;		/* Turn Direction Factor */
    auto float  A[2], B[2], C[2];	/* Current Vertices */
    auto float  oldA[2], oldB[2];	/* First Two Vertices */
    auto int    xdir, ydir;		/* Current Coordinate Direction */
    auto int    xturns=0, yturns=0;	/* Coordinate Direction Changes */
    auto float  dot;			/* Dot Product of Edge Norm and Edge */
    auto float  U[2], V[2];		/* Edge Vectors AB and BC */
    auto int	onvert;

    /* Read in the first two vertices. */

    if ( nvert < 3 ) return(0);
    onvert = 0;
    V2op2( B, =, pVert[0] );

    do {
	V2op2( C, =, pVert[onvert] );
	onvert++;
    } while ((B[0] == C[0]) && (B[1] == C[1]));

    V2op2 (oldA,=,B);		/* Save the first two vertices. */
    V2op2 (oldB,=,C);

    V2op3 (V,=,C,-,B);		/* Calculate the edge vector V0V1. */

    /* Set the X & Y coordinate directions if possible. */

    xdir = (B[0] < C[0]) ? 1 : (B[0] > C[0]) ? -1 : 0;
    ydir = (B[1] < C[1]) ? 1 : (B[1] > C[1]) ? -1 : 0;

    do	/* Shift the last two vertices and read in the next vertex. */
    {
	V2op2 (A,=,B);
	V2op2 (B,=,C);

	do
	{   switch (looking)
	    {   case 2:
		    V2op2( C, =, pVert[onvert] );
		    if ( onvert >= nvert ) {
		        looking = 1;
			V2op2 (C,=,oldA);
		    }
		    onvert++;
		    break;

		case 1:
		    looking = 0;
		    V2op2 (C,=,oldB);
		    break;
		
		case 0:
		    return 1;
	    }
	} while ((C[0] == B[0]) && (C[1] == B[1]));

	V2op2 (U,=,V);
	V2op3 (V,=,C,-,B);	/* Calculate the edge vector from B to C. */

	/* Get the dot product of the normal to edge AB and the vector BC.
	** Compare this result with the previous dot products.  As long as
	** the sign is the same as the previous ones (or zero), then
	** everything's cool, otherwise we found a dent in the polygon, so
	** return 0.  */

	dot = (U[0] * V[1]) - (U[1] * V[0]);
	if (fabs(dot) < EPSILON)
	    dot = 0.0;
	else
	{   if (!factor)
		factor = (dot < 0.0) ? -1 : 1;
	    else if ((factor < 0) ^ (dot < 0.0))
		return 0;
	}

	/* Check the X coordinate delta.  For a simple polygon, this can
	** change sign (direction) twice, but no more. */

#       define CHECK_DIRECTION(P,Q,dir,turns) \
	    do { \
		if (P < Q) \
		{   if (dir == -1) \
		    {   ++turns; \
			if (dot == 0.0) return 0; \
		    } \
		    dir =  1; \
		} \
		else if (P > Q) \
		{   if (dir ==  1) \
		    {   ++turns; \
			if (dot == 0.0) return 0; \
		    } \
		    dir = -1; \
		} \
	    } while (0)

	CHECK_DIRECTION (B[0],C[0], xdir, xturns);
	if (xturns > 2) return 0;

	CHECK_DIRECTION (B[1],C[1], ydir, yturns);
	if (yturns > 2) return 0;

    } while (looking);

    return 1;	/* All tests passed; polygon is simple and convex. */
}
