/*
  File: 4D.c
  Author: K.R. Sloan, Jr.,
          Tony DeRose,
          Jamie Painter
  Last Modified: 23 May 1986
  Purpose: 4D homogeneous coordinate manipulation package
 */


#include <stdio.h>
#include <math.h>
#include <4D.h>
static double epsilon = 0.00000000001;

void printT(T)
 TransformType4D T;
 {
  int i,j;
  for (j = 0; j<4; j++)
   {
    for (i = 0; i<4; i++)
     fprintf(stderr," %f",T.T[i][j]);
    fprintf(stderr,"\n");
   }
 }
void printP(P)
 PointType4D P;
 {
  int i;
  for (i = 0; i<4; i++)
   fprintf(stderr," %f",P.P[i]);
  fprintf(stderr,"\n");
 }

extern PointType4D PxT4D(P,T)
 PointType4D P;
 TransformType4D T;
 {
  PointType4D Q;
  int i,k;
  double Qi;

  for (i = 0; i < 4; i++)
   {
    Qi = 0.0;
    for (k = 0; k < 4; k++)
     Qi += P.P[k]*T.T[i][k];
    Q.P[i] = Qi;
   }
  return Q;
 }

extern TransformType4D TxT4D(A, B)
 TransformType4D A, B;
 {
  TransformType4D C;
  int i,j,k;
  double Cij;

  for (j = 0; j < 4; j++)
   for (i = 0; i < 4; i++)
    {
     Cij = 0.0;
     for (k = 0; k < 4; k++)
      Cij += A.T[k][j]*B.T[i][k];
     C.T[i][j] = Cij;
    }
  return C;
 }

extern TransformType4D Identity4D()
 {
  TransformType4D I;
  int i,j; 

  for (i = 0; i < 4; i++)
   for (j = 0; j < 4; j++)
    I.T[i][j] = (double)(i == j);
  return I; 
 }

extern TransformType4D Translate4D(P)
 PointType4D P;
 {
  TransformType4D T;
  double w;
  
  T = Identity4D();
  w = P.P[3];
  T.T[0][3] = P.P[0] / w;
  T.T[1][3] = P.P[1] / w;
  T.T[2][3] = P.P[2] / w;
  return T;
 }

extern TransformType4D Scale4D(P)
 PointType4D P;
 {
  TransformType4D T;

  T = Identity4D();
  T.T[0][0] = P.P[0];
  T.T[1][1] = P.P[1];
  T.T[2][2] = P.P[2];
  T.T[3][3] = P.P[3];
  return T;
 }

extern TransformType4D Rotate4D(P0, P1, Radians)
 PointType4D P0, P1;
 double Radians;
 {
  TransformType4D T, R1, R2, R3, Rotate;
  double w, a, b, c, l, v;
  
  w = P1.P[3]; a  = P1.P[0]/w; b  = P1.P[1]/w; c  = P1.P[2]/w;
  w = P0.P[3]; a -= P0.P[0]/w; b -= P0.P[1]/w; c -= P0.P[2]/w;	

  l = sqrt(a*a + b*b + c*c);

  if (l < epsilon)
   {
    fprintf(stderr,"Rotate4D: |P1-P0| is too small - can't rotate\n");
    return Identity4D();
   }
  else
   {
    a = a/l; b = b/l; c = c/l;
   }
  
  v = sqrt(b*b + c*c);
 
  P0.P[3] = -P0.P[3];  /* reverse direction by reversing w */
  T = Translate4D(P0); 

  R1 = Identity4D(); R2 = R1; R3 = R1;

  if (v > epsilon)
   {
                          R1.T[1][1] = c/v;         R1.T[2][1] = b/v;
                          R1.T[1][2] = -R1.T[2][1]; R1.T[2][2] = R1.T[1][1];
   }

  if ((v > epsilon) || (a > epsilon) || (a < (-epsilon)))
   {
    R2.T[0][0] = v;                                 R2.T[2][0] = a;

    R2.T[0][2] = -R2.T[2][0];                       R2.T[2][2] = v;
   }

  R3.T[0][0] = cos(Radians); R3.T[1][0] = -sin(Radians);
  R3.T[0][1] = -R3.T[1][0];  R3.T[1][1] = R3.T[0][0];

  Rotate = TxT4D(T,R1);
  Rotate = TxT4D(Rotate,R2);
  Rotate = TxT4D(Rotate,R3);

                                                    R2.T[2][0] = -R2.T[2][0];
  R2.T[0][2] = -R2.T[0][2];

                                                    R1.T[2][1] = -R1.T[2][1];
                           R1.T[1][2] = -R1.T[1][2];

  T.T[0][3] = -T.T[0][3];   T.T[1][3] = -T.T[1][3];   T.T[2][3] = -T.T[2][3]; 

  Rotate = TxT4D(Rotate,R2);
  Rotate = TxT4D(Rotate,R1);
  Rotate = TxT4D(Rotate,T);
  return Rotate;
 }

extern TransformType4D Perspective4D(f)
 double f;
 {
  TransformType4D P;
  
  P = Identity4D();
  if ((f < epsilon) && (f > (-epsilon)))
   {
    fprintf(stderr,"Perspective4D: f (%f) is too small.\n",f);
    if (f < 0.0) f = (-epsilon);
    else         f =   epsilon;
   }
  P.T[2][2] = 1.0/f;    P.T[3][2] = 1.0/f;
  P.T[2][3] = -1.0;     P.T[3][3] = 0.0;
  return P;
 }
