/*
  File: DrawContours.c
  Authors: K.R. Sloan, David Meyers   
  Last Modified: 8 November 1989
  Purpose: Read a section file and draw the contours in 3D.
           Accepts command line flags ala Render, where they make sense.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#include <3D.h>
#include <Lighting.h>
#include <Triangles.h>
#include <Render.h>
#include <Scanline.h>
#include <TypeDefinitions.h>
#include <Contours.h>
#include <InputStuff.h>

extern char *malloc();
extern double atof();
extern void exit();

int VERBOSE = 0;    /* exported for the real renderers to see */
static int DEBUG = 0;
static int Preview = 1;
static int TensorProduct = 0;
static sgpColorType InteriorColor = { 1.0, 1.0, 1.0 };
static sgpColorType OutlineColor  = { 0.0, 0.0, 0.0 };

static char *RoutineName;
static void usage()
 {
  (void) fprintf(stderr,
  "Usage is\n\t%s [-v][-D][-O x y z][-e x y z][-o dx dy dz][-u x y z][-f f]\n",
                 RoutineName);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  static int ArgsParsed = 0;
  static PointType3D  EyePoint    = { 10.0, 10.0, 10.0 };
  static VectorType3D OpticalAxis = { -1.0, -1.0, -1.0 };
  static PointType3D Origin       = {  0.0,  0.0,  0.0 };
  static VectorType3D Up          = {  0.0,  1.0,  0.0 };

  static double FocalLength = 0.0040;
  static sgpRectangleType Window   = {-0.0015, -0.0012, 0.0015, 0.0012};
  static sgpRectangleType Viewport = {-0.125,   0.00,   1.125,  1.00};  
  static double Exposure = 1.0;
  PointType3D LightPosition;
  VectorType3D Right;
  sgpColorType Color;
  sgpColorType Ambient;
  Section *TheSection;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'e':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         EyePoint.x = atof(argv[ArgsParsed++]);
         EyePoint.y = atof(argv[ArgsParsed++]);
         EyePoint.z = atof(argv[ArgsParsed++]);
         OpticalAxis.dx = Origin.x - EyePoint.x;
         OpticalAxis.dy = Origin.y - EyePoint.y;
         OpticalAxis.dz = Origin.z - EyePoint.z;
         break;
        case 'o':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         OpticalAxis.dx = atof(argv[ArgsParsed++]);
         OpticalAxis.dy = atof(argv[ArgsParsed++]);
         OpticalAxis.dz = atof(argv[ArgsParsed++]);
         break;
        case 'O':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Origin.x = atof(argv[ArgsParsed++]);
         Origin.y = atof(argv[ArgsParsed++]);
         Origin.z = atof(argv[ArgsParsed++]);
         OpticalAxis.dx = Origin.x - EyePoint.x;
         OpticalAxis.dy = Origin.y - EyePoint.y;
         OpticalAxis.dz = Origin.z - EyePoint.z;
         break;
        case 'u':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Up.dx = atof(argv[ArgsParsed++]);
         Up.dy = atof(argv[ArgsParsed++]);
         Up.dz = atof(argv[ArgsParsed++]);
         break;
        case 'f':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         FocalLength = atof(argv[ArgsParsed++]);
         break;
        case 'D':
         DEBUG = 1;
         break;
        case 'v':
         VERBOSE = 1;
         break;
        case 'h': 
        default:
         usage(); exit(-1);
       }
     }
    else { usage(); exit (-1); }
   }   

  if (DEBUG) sgpInit(DebugDisplay);
  else       sgpInit(ColorDisplay);

  sgpSetWindow(Window);
  sgpSetViewport(Viewport);

  Camera3D(EyePoint, OpticalAxis, Up);
  Lens3D(FocalLength);  SetHither(FocalLength);

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  if (VERBOSE)
   {
    (void) fprintf(stderr,"%s: EyePoint = { %f, %f, %f }\n", RoutineName,
                        EyePoint.x,
                        EyePoint.y,
                        EyePoint.z);
    (void) fprintf(stderr,"%s: OpticalAxis = { %f, %f, %f }\n", RoutineName,
                        OpticalAxis.dx,
                        OpticalAxis.dy,
                        OpticalAxis.dz);
    (void) fprintf(stderr,"%s: Up = { %f, %f, %f }\n", RoutineName,
                        Up.dx,
                        Up.dy,
                        Up.dz);
    (void) fprintf(stderr,"%s: FocalLength = %f\n", RoutineName, FocalLength);
   }

  if (Preview)
   {
    if (VERBOSE)
     {
      (void) fprintf(stderr,"%s: Previewing\n", RoutineName);
      (void) fprintf(stderr,
		     "%s: InteriorColor = { %f, %f, %f }\n", RoutineName,
                          InteriorColor.r, InteriorColor.g, InteriorColor.b);
      (void) fprintf(stderr,"%s: OutlineColor = { %f, %f, %f }\n", RoutineName,
		     OutlineColor.r, OutlineColor.g, OutlineColor.b);
     }
    RenderMode(Preview, TensorProduct, InteriorColor, OutlineColor);
    sgpColor(InteriorColor); sgpClearScreen();
   }
  else
   {
    Right = Cross3D(OpticalAxis, Up);
    Normalize3D(&Right); Normalize3D(&Up);

    Ambient.r = 0.2; Ambient.g = 0.2; Ambient.b = 0.3;
    SetAmbient(Ambient);
    sgpColor(Ambient); sgpClearScreen();

    LightPosition.x = EyePoint.x + Up.dx + Right.dx;
    LightPosition.y = EyePoint.y + Up.dy + Right.dy;
    LightPosition.z = EyePoint.z + Up.dz + Right.dz;
    Color.r = 0.75; Color.g = 0.75; Color.b = 0.75;
    AddLightSource(LightPosition, Color);

    if (VERBOSE)
     (void) fprintf(stderr,
		    "%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    SetExposure(Exposure);

   if (VERBOSE)
    (void) fprintf(stderr,"%s: Exposure = %f\n", RoutineName, Exposure);
   }

  TheSection = (Section *) malloc(sizeof(Section));
  ReadSection(stdin, TheSection);
  sgpColor(OutlineColor);

  while(NULL != TheSection->TheContours)
   {
    int i;

    for (i=0; i<TheSection->NContours; i++)
     {
      int j, n;
      PointType3D P1, P2;

      n = TheSection->TheContours[i].ContourPoints->n;
      P1.x = TheSection->TheContours[i].ContourPoints->P[0].x;
      P1.y = TheSection->TheContours[i].ContourPoints->P[0].y;
      P1.z = TheSection->TheContours[i].ContourPoints->P[0].z;
      for(j=1; j<=n; j++)
       {
	P2.x = TheSection->TheContours[i].ContourPoints->P[j%n].x;
	P2.y = TheSection->TheContours[i].ContourPoints->P[j%n].y;
	P2.z = TheSection->TheContours[i].ContourPoints->P[j%n].z;
	if(VERBOSE)
	 {
	  (void) fprintf(stderr,
			 "P1.x=%f, P1.y=%f, P1.z=%f\n", P1.x, P1.y, P1.z);
	  (void) fprintf(stderr,
			 "P2.x=%f, P2.y=%f, P2.z=%f\n", P2.x, P2.y, P2.z);
	  (void) fprintf(stderr,
			 "Calling Line3D(P1, P2)\n");
	 }
	Line3D(P1, P2);
	P1.x = P2.x;
	P1.y = P2.y;
	P1.z = P2.z;
       }
     }
    ReadSection(stdin, TheSection);
   }
  sgpQuit();
  if (VERBOSE) (void) fprintf(stderr,"-30-\n");
  exit(0);
 }
