/**********************************************************************
 *
 * setpasswd.cpp -- 
 * Copyright (C) 2000  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

// setpasswd is a windows application that can be used to encrypt a password
// and write it (along with its corresponding username) to a gdbm database.

// it handles writing to the gdbm database itself to avoid having to call
// the txt2db console application (and therefore avoiding the console
// window popping up when called from another windows application).

// note that setpasswd does no checking to make sure that any of it's
// input arguments are valid (or even reasonable) values.

// usage:
// setpasswd -u username -p password -o output_gdbm_file

#include "text_t.h"
#include "crypt.h"
#include "autoconf.h"
#include "systems.h"
#include "gdbmconst.h"
#include "gdbm.h"

#include <windows.h>

text_t username;
text_t password;
text_t output_gdbm_file;

bool parse_cmdline (LPSTR cmdline) {

  bool in_quote = false;
  text_t arg;
  text_tarray args;
  unsigned char *c = (unsigned char *)cmdline;
  while (*c != '\0') {
    if (*c == '"') {
      if (!in_quote) {
	in_quote = true;
      } else {
	in_quote = false;
	if (!arg.empty()) args.push_back (arg);
	arg.clear();
      }
    } else if (*c == ' ' && !in_quote) {
      if (!arg.empty()) args.push_back (arg);
      arg.clear();
    } else {
      arg.push_back (*c);
    }
    c ++;
  }
  if (!arg.empty()) args.push_back (arg);
  
  text_tarray::const_iterator here = args.begin();
  text_tarray::const_iterator end = args.end();
  while (here != end) {
    if (*here == "-u" && (++here != end)) username = *here;
    else if (*here == "-p" && (++here != end)) password = *here;
    else if (*here == "-o" && (++here != end)) output_gdbm_file = *here;
    if (here != end) here ++;
  }
  if (username.empty() || password.empty() || output_gdbm_file.empty()) {
    MessageBox (NULL, "Usage:\n setpasswd -u username -p password -o output_gdbm_file", 
		"setpasswd failed", MB_OK);
    return false;
  }
  return true;
}

text_t crypt_text (const text_t &text) {
  static const char *salt = "Tp";
  text_t crypt_password;

  if (text.empty()) return "";

  // encrypt the password
  char *text_cstr = text.getcstr();
  if (text_cstr == NULL) return "";
  crypt_password = crypt(text_cstr, salt);
  delete text_cstr;

  return crypt_password;
}

bool add_to_db () {

  int block_size = 0;
  GDBM_FILE dbf;
  char *dbname = output_gdbm_file.getcstr();

  // open the database
  int read_write = GDBM_WRCREAT;
  dbf = gdbm_open (dbname, block_size, read_write, 00664, NULL, 1);
  if (dbf == NULL) {
    MessageBox (NULL, "gdbm_open failed\n", "setpasswd", MB_OK);
    return false;
  }

  datum key_data;
  key_data.dptr = username.getcstr();
  if (key_data.dptr == NULL) {
    MessageBox (NULL, "null key_data\n", "setpasswd", MB_OK);
    return false;
  }
  key_data.dsize = strlen(key_data.dptr);

  text_t value = "<comment>\n";
  value += "<enabled>true\n";
  value += "<groups>administrator,colbuilder\n";
  value += "<password>" + password + "\n";
  value += "<username>" + username + "\n";
      
  datum value_data;
  value_data.dptr = value.getcstr();
  if (value_data.dptr == NULL) {
    MessageBox (NULL, "null value_data\n", "setpasswd", MB_OK);
    return false;
  }
  value_data.dsize = strlen(value_data.dptr);
      
  // store the value
  if (gdbm_store (dbf, key_data, value_data, GDBM_REPLACE) < 0) {
    MessageBox (NULL, "gdbm_store failed\n", "setpasswd", MB_OK);
    return false;
  }
  gdbm_close (dbf);

  delete key_data.dptr;
  delete value_data.dptr;
  delete dbname;
  return true;
}

int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                     LPSTR lpCmdLine, int nCmdShow) {

  // parse command line arguments
  if (!parse_cmdline (lpCmdLine)) return 1;

  // encrypt the password
  password = crypt_text (password);

  // append the password and username to database
  add_to_db();
  
  return 0;
}



