/**********************************************************************
 *
 * queryaction.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "queryaction.h"
#include "querytools.h"
#include "formattools.h"
#include "cgiutils.h"
#include "OIDtools.h"
//#include "infodbclass.h"
#include "fileutil.h"
#include "text_t.h"
#include "historydb.h"
#include "htmlutils.h" // for html_safe in do_action
#include "gsdltools.h"

void colinfo_t::clear () {
  formatlistptr = NULL;
  browserptr = NULL;
}

void QueryResult_t::clear() {
  doc.clear();
  collection.clear();
}

queryaction::queryaction () {

  recpt = NULL;
  num_phrases = 0;

  // this action uses cgi variable "a"
  cgiarginfo arg_ainfo;
  arg_ainfo.shortname = "a";
  arg_ainfo.longname = "action";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "q";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

   // "ct" - 0 = mg, 1 = mgpp
  arg_ainfo.shortname = "ct";
  arg_ainfo.longname = "collection type";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);
 
  // "b" - 0 = simple, 1 = advanced
  arg_ainfo.shortname = "b";
  arg_ainfo.longname = "query mode";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "h"
  arg_ainfo.shortname = "h";
  arg_ainfo.longname = "main index";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "h2"
  arg_ainfo.shortname = "h2";
  arg_ainfo.longname = "main index for second query";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "j"
  arg_ainfo.shortname = "j";
  arg_ainfo.longname = "sub collection index";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "j2"
  arg_ainfo.shortname = "j2";
  arg_ainfo.longname = "sub collection index for second query";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "n"
  arg_ainfo.shortname = "n";
  arg_ainfo.longname = "language index";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "n2"
  arg_ainfo.shortname = "n2";
  arg_ainfo.longname = "language index for second query";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);


  // "q"
  arg_ainfo.shortname = "q";
  arg_ainfo.longname = "query string";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "q2"
  arg_ainfo.shortname = "q2";
  arg_ainfo.longname = "query string for second query";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "cq2" ""=don't combine, "and", "or", "not"
  arg_ainfo.shortname = "cq2";
  arg_ainfo.longname = "combine queries";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "t" - 1 = ranked 0 = boolean
  arg_ainfo.shortname = "t";
  arg_ainfo.longname = "search type";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "1";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "k"
  arg_ainfo.shortname = "k";
  arg_ainfo.longname = "casefolding";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "1";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "s"
  arg_ainfo.shortname = "s";
  arg_ainfo.longname = "stemming";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault ="0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "m"
  arg_ainfo.shortname = "m";
  arg_ainfo.longname = "maximum number of documents";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "50";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "o"
  arg_ainfo.shortname = "o";
  arg_ainfo.longname = "hits per page";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "20";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "r"
  arg_ainfo.shortname = "r";
  arg_ainfo.longname = "start results from";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "1";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "ccs"
  arg_ainfo.shortname = "ccs";
  arg_ainfo.longname = "cross collection searching";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "ccp"
  arg_ainfo.shortname = "ccp";
  arg_ainfo.longname = "cross collection page";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "cc"
  arg_ainfo.shortname = "cc";
  arg_ainfo.longname = "collections to search";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "hd" history display  - search history only displayed when
  // this var set to something other than 0
  // this number of records is displayed
  arg_ainfo.shortname = "hd";
  arg_ainfo.longname = "history display";
  arg_ainfo.multiplechar = true;
  arg_ainfo.multiplevalue = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);
	    
  // "hs"  save - set to 1 in query form, so only save when submit
  // query
  // 0 = no save 1 = save
  arg_ainfo.shortname = "hs";
  arg_ainfo.longname = "history save";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // new arg for granularity, for mgpp collections
  arg_ainfo.shortname = "g";
  arg_ainfo.longname = "granularity";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "Document";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "ds" - start date
  arg_ainfo.shortname = "ds";
  arg_ainfo.longname = "start date";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "de" - end date
  arg_ainfo.shortname = "de";
  arg_ainfo.longname = "end date";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "dsbc" - whether or not start date is prechristian
  arg_ainfo.shortname = "dsbc";
  arg_ainfo.longname = "start date bc";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "debc" - whether or not end date is prechristian
  arg_ainfo.shortname = "debc";
  arg_ainfo.longname = "end date bc";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "qt" - 0 = text, 1 = form
  arg_ainfo.shortname = "qt";
  arg_ainfo.longname = "query type";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "qb" - 0 = regular, 1 = large
  arg_ainfo.shortname = "qb";
  arg_ainfo.longname = "query box type";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqn" - number of fields in the query form
  arg_ainfo.shortname = "fqn";
  arg_ainfo.longname = "form query num fields";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "4";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqf" - the list of field names in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqf";
  arg_ainfo.longname = "form query fields";
  arg_ainfo.multiplechar = true;
  //arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
  
  // "fqv" - the list of values in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqv";
  arg_ainfo.longname = "form query values";
  arg_ainfo.multiplechar = true;
  //arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
  

  // "fqs" - the list of stemming options in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqs";
  arg_ainfo.longname = "form query stems";
  arg_ainfo.multiplechar = true;
  //arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
  

  // "fqk" - the list of casefolding options in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqk";
  arg_ainfo.longname = "form query casefolds";
  arg_ainfo.multiplechar = true;
  //arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
  
  // "fqc" - the list of boolean operators in the form query
  // - a comma separated list
  arg_ainfo.shortname = "fqc";
  arg_ainfo.longname = "form query combines";
  arg_ainfo.multiplechar = true;
  //arg_ainfo.multiplevalue = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "fqa" - form query advanced - for "run query"
  arg_ainfo.shortname = "fqa";
  arg_ainfo.longname = "form query advanced query";
  arg_ainfo.multiplechar = false;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "0";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
  
}

void queryaction::configure (const text_t &key, const text_tarray &cfgline) {
  action::configure (key, cfgline);
}

bool queryaction::init (ostream &logout) {
  return action::init (logout);
}

bool queryaction::check_cgiargs (cgiargsinfoclass &argsinfo, cgiargsclass &args, 
				 ostream &logout) {

  // check t argument
  int arg_t = args.getintarg("t");
  if (arg_t != 0 && arg_t != 1) {
    logout << "Warning: \"t\" argument out of range (" << arg_t << ")\n";
    cgiarginfo *tinfo = argsinfo.getarginfo ("t");
    if (tinfo != NULL) args["t"] = tinfo->argdefault;
  }

  // check k argument
  int arg_k = args.getintarg("k");
  if (arg_k != 0 && arg_k != 1) {
    logout << "Warning: \"k\" argument out of range (" << arg_k << ")\n";
    cgiarginfo *kinfo = argsinfo.getarginfo ("k");
    if (kinfo != NULL) args["k"] = kinfo->argdefault;
  }

  // check s argument
  int arg_s = args.getintarg("s");
  if (arg_s != 0 && arg_s != 1) {
    logout << "Warning: \"s\" argument out of range (" << arg_s << ")\n";
    cgiarginfo *sinfo = argsinfo.getarginfo ("s");
    if (sinfo != NULL) args["s"] = sinfo->argdefault;
  }

  // check m argument
  int arg_m = args.getintarg("m");
  if (arg_m < -1) {
    logout << "Warning: \"m\" argument less than -1 (" << arg_m << ")\n";
    cgiarginfo *minfo = argsinfo.getarginfo ("m");
    if (minfo != NULL) args["m"] = minfo->argdefault;
  }

  // check o argument
  int arg_o = args.getintarg("o");
  if (arg_o < -1) {
    logout << "Warning: \"o\" argument less than -1 (" << arg_o << ")\n";
    cgiarginfo *oinfo = argsinfo.getarginfo ("o");
    if (oinfo != NULL) args["o"] = oinfo->argdefault;
  }

  // check r argument
  int arg_r = args.getintarg("r");
  if (arg_r < 1) {
    logout << "Warning: \"r\" argument less than 1 (" << arg_r << ")\n";
    cgiarginfo *rinfo = argsinfo.getarginfo ("r");
    if (rinfo != NULL) args["r"] = rinfo->argdefault;
  }
  //check hd argument
  int arg_hd = args.getintarg("hd");
  if (arg_hd <0 ) {
    logout << "Warning: \"hd\" argument less than 0 (" << arg_hd << ")\n";
    cgiarginfo *hdinfo = argsinfo.getarginfo ("hd");
    if (hdinfo != NULL) args["hd"] = hdinfo->argdefault;
  }  
  
  //check hs argument
  int arg_hs = args.getintarg("hs");
  if (arg_hs !=0 && arg_hs !=1) {
    logout << "Warning: \"hs\" argument out of range (" << arg_hs << ")\n";
    cgiarginfo *hsinfo = argsinfo.getarginfo ("hs");
    if (hsinfo != NULL) args["hs"] = hsinfo->argdefault;
  }

  // check ct argument
  int arg_ct = args.getintarg("ct");
  if (arg_ct !=0 && arg_ct !=1) {
    logout << "Warning: \"ct\" argument out of range (" << arg_ct << ")\n";
    cgiarginfo *ctinfo = argsinfo.getarginfo ("ct");
    if (ctinfo != NULL) args["ct"] = ctinfo->argdefault;
  }

  // check qt argument
  int arg_qt = args.getintarg("qt");
  if (arg_qt !=0 && arg_qt !=1) {
    logout << "Warning: \"qt\" argument out of range (" << arg_qt << ")\n";
    cgiarginfo *qtinfo = argsinfo.getarginfo ("qt");
    if (qtinfo != NULL) args["qt"] = qtinfo->argdefault;
  }

  // check qb argument
  int arg_qb = args.getintarg("qb");
  if (arg_qb !=0 && arg_qb !=1) {
    logout << "Warning: \"qb\" argument out of range (" << arg_qb << ")\n";
    cgiarginfo *qbinfo = argsinfo.getarginfo ("qb");
    if (qbinfo != NULL) args["qb"] = qbinfo->argdefault;
  }

  // check fqa argument
  int arg_fqa = args.getintarg("fqa");
  if (arg_fqa !=0 && arg_fqa !=1) {
    logout << "Warning: \"fqa\" argument out of range (" << arg_fqa << ")\n";
    cgiarginfo *fqainfo = argsinfo.getarginfo ("fqa");
    if (fqainfo != NULL) args["fqa"] = fqainfo->argdefault;
  }

  // check fqn argument
  int arg_fqn = args.getintarg("fqn");
  if (arg_fqn < -1) {
    logout << "Warning: \"fqn\" argument less than -1 (" << arg_fqn << ")\n";
    cgiarginfo *fqninfo = argsinfo.getarginfo ("fqn");
    if (fqninfo != NULL) args["fqn"] = fqninfo->argdefault;
  }

  return true;
}

void queryaction::get_cgihead_info (cgiargsclass &/*args*/, recptprotolistclass * /*protos*/,
				    response_t &response, text_t &response_data, 
				    ostream &/*logout*/) {
  response = content;
  response_data = "text/html";
}

void queryaction::define_internal_macros (displayclass &disp, cgiargsclass &args, 
					  recptprotolistclass * protos, 
					  ostream &logout) {

  // define_internal_macros sets the following macros:

  // _quotedquery_  the part of the query string that was quoted for post-processing



  // The following macros are set later (in define_query_macros) as they can't be set until 
  // the query has been done.

  // _freqmsg_      the term frequency string

  // _resultline_   the "x documents matched the query" string

  // _prevfirst_    these are used when setting up the links to previous/next
  // _prevlast_     pages of results (_thisfirst_ and _thislast_ are used to set
  // _nextfirst_    the 'results x-x for query: xxxx' string in the title bar)
  // _nextlast_
  // _thisfirst_
  // _thislast_


  // get the quoted bits of the query string and set _quotedquery_
  text_tarray phrases;
  get_phrases (args["q"], phrases);
  num_phrases = phrases.size();
  text_tarray::const_iterator phere = phrases.begin();
  text_tarray::const_iterator pend = phrases.end();
  bool first = true;
  text_t quotedquery;
  while (phere != pend) {
    if (!first)
      if ((phere +1) == pend) quotedquery += " and ";
      else quotedquery += ", ";
	
    quotedquery += "\"" + *phere + "\"";
    first = false;
    phere ++;
  }
  if (args.getintarg("s")) quotedquery += "_textstemon_";
  disp.setmacro ("quotedquery", "query", quotedquery);

  define_form_macros(disp, args, protos, logout);
}

// sets the selection box macros _hselection_, _jselection_, and _nselection_.
void queryaction::set_option_macro (const text_t &macroname, text_t current_value,
				    const FilterOption_t &option, displayclass &disp) {
  
  if (option.validValues.empty()) return;
  else if (option.validValues.size() == 1) {
    disp.setmacro (macroname + "selection", "Global", "_" + option.defaultValue + "_");
    return;
  } 
  if (option.validValues.size() < 2) return;
  
  text_t macrovalue = "<select name=\"" + macroname + "\">\n";
  
  if (current_value.empty()) current_value = option.defaultValue;
  
  text_tarray::const_iterator thisvalue = option.validValues.begin();
  text_tarray::const_iterator endvalue = option.validValues.end();

  while (thisvalue != endvalue) {
    macrovalue += "<option value=\"" + *thisvalue + "\"";
    if (*thisvalue == current_value)
      macrovalue += " selected";
    macrovalue += ">_" + *thisvalue + "_\n";
    thisvalue ++;
  }
  macrovalue += "</select>\n";
  disp.setmacro (macroname + "selection", "Global", macrovalue);
}

// sets the selection box macro _fqfselection_.
void queryaction::set_fqfselection_macro (const FilterOption_t &option, 
					 displayclass &disp) {

  if (option.validValues.empty()) return;
  text_t macrovalue = "";

  macrovalue += "<select name=\"fqf\" onChange=\"updatefqf();\">\n";
  
  text_tarray::const_iterator thisvalue = option.validValues.begin();
  text_tarray::const_iterator endvalue = option.validValues.end();

  while (thisvalue != endvalue) {
    macrovalue += "<option value=\"" + *thisvalue + "\">_";    
    macrovalue += *thisvalue + "_\n";
    thisvalue ++;
  }
  macrovalue += "</select>\n";
  disp.setmacro ("fqfselection", "query", macrovalue);

}

void queryaction::define_external_macros (displayclass &disp, cgiargsclass &args, 
					  recptprotolistclass *protos, ostream &logout) {

  // define_external_macros sets the following macros:

  // some or all of these may not be required to be set
  // _hselection_, _h2selection_  the selection box for the main part of the index
  // _jselection_, _j2selection_  the selection box for the subcollection part of the index
  // _nselection_, _n2selection_  the selection box for the language part of the index
  // _cq2selection                the selection box for combining two queries
  

  // can't do anything if collectproto is null (i.e. no collection was specified)
  recptproto *collectproto = protos->getrecptproto (args["c"], logout);
  if (collectproto == NULL) return;

  comerror_t err;
  InfoFilterOptionsResponse_t response;
  InfoFilterOptionsRequest_t request;
  request.filterName = "QueryFilter";
  
  collectproto->get_filteroptions (args["c"], request, response, err, logout);
  if (err == noError) {
    
    FilterOption_tmap::const_iterator it;
    FilterOption_tmap::const_iterator end = response.filterOptions.end();
    
    // _hselection_ and _h2selection_ (Index)
    it = response.filterOptions.find ("Index");
    if (it != end) set_option_macro ("h", args["h"], (*it).second, disp);
    if (it != end) set_option_macro ("h2", args["h2"], (*it).second, disp);
    
    // _jselection_ and _j2selection_ (Subcollection)
    it = response.filterOptions.find ("Subcollection");
    if (it != end) set_option_macro ("j", args["j"], (*it).second, disp);
    if (it != end) set_option_macro ("j2", args["j2"], (*it).second, disp);
    
    // _nselection_ and _n2selection_ (Language)
    it = response.filterOptions.find ("Language");
    if (it != end) set_option_macro ("n", args["n"], (*it).second, disp);
    if (it != end) set_option_macro ("n2", args["n2"], (*it).second, disp);

    // _cq2selection_ (CombineQuery)
    it = response.filterOptions.find ("CombineQuery");
    if (it != end) set_option_macro ("cq2", args["cq2"], (*it).second, disp);

    // _gselection_ (Level) // for mgpp collections
    it = response.filterOptions.find("Level");
    if (it!=end) set_option_macro("g", args["g"], (*it).second, disp);

    // should probably go into define_internal_macros
    // _fqfselection_
    it = response.filterOptions.find("IndexField");
    if (it!=end) {
      set_fqfselection_macro((*it).second, disp);
    }
  }
} // define external macros

void queryaction::define_form_macros (displayclass &disp, cgiargsclass &args, 
			   recptprotolistclass *protos, ostream &logout) {

  // defines the following macros
  // _regformlist_
  // _advformlist_

  if (args["ct"]!="1" || args["qt"]!="1") 
    return; // dont need these macros

    // mgpp & form query only needs the macros defined
  text_t form = "";
  int argfqn = args.getintarg("fqn");
  
  if (args["b"] == "1") { // advanced form
    form += "<tr>_firstadvformelement_</tr>\n";
    for (int i=1; i<argfqn; i++) {      
	form += "<tr>_advformelement_</tr>\n";
    }
    disp.setmacro("advformlist", "query", form);
  }
  else { // simple form
    for (int i=0; i<argfqn; i++) {
      form += "<tr>_regformelement_</tr>\n"; 
    }
    disp.setmacro("regformlist", "query", form);
  }
  
}

void queryaction::define_history_macros (displayclass &disp, cgiargsclass &args, 
					 recptprotolistclass *protos, ostream &logout) {

  // defines the following macros

  // _searchhistorylist_

  text_t historylist;
  int arghd = args.getintarg("hd");
  if (arghd == 0) {
    historylist="";
  }
  else {
    historylist = "<!-- Search History List -->\n";
   
    text_t userid = args["z"];
    text_tarray entries;
    if (get_history_info (userid, entries, gsdlhome, logout)) {
      int count = 1;
      text_tarray::iterator here = entries.begin();
      text_tarray::iterator end = entries.end();
      int numrecords=(int)entries.size();
      if (numrecords>arghd) { // only display some of them 
	numrecords = arghd;
      }
      historylist += "<form name=\"HistoryForm\"><table width=537>\n";

      for (int i=0; i<numrecords;i++) {
	text_t query;
	text_t numdocs;
	text_t cgiargs;
	text_t userinfo;
	text_t escquery;
	split_saved_query(entries[i],numdocs,cgiargs);
	parse_saved_args(cgiargs, "q", query); // get query string out
	decode_cgi_arg(query); // un cgisafe it
	escquery = escape_quotes(query); // escape the quotes
	text_t histvalue = "histvalue";
	histvalue += i;
	disp.setmacro(histvalue, "query", escquery);
	format_user_info(cgiargs, userinfo, args, protos, logout);
	
	historylist += "<tr><td align=right>_imagehistbutton_(";
	historylist += i;
	historylist += ")</td>\n";
	historylist += "<td><nobr><table border=1 cellspacing=0 ";
	historylist += "cellpadding=0><tr><td width=365 align=left>"
	  + query 
	  + "</td></tr></table></td><td width=110 align=center><small>"
	  + numdocs;
	if (numdocs == 1) historylist += " _texthresult_";
	else historylist += " _texthresults_";
	if (!userinfo.empty()) {
	  historylist += "<br>( "+userinfo+" )</small></td>\n";
	}
      }
      historylist+="</table></form>\n\n";
     
    } // if get history info 
    else {
      historylist += "_textnohistory_";
    }
    historylist += "<p><! ---- end of history list ----->\n";
  } // else display list
  disp.setmacro("searchhistorylist", "query", historylist);
  
} // define history macros

void queryaction::output_ccp (cgiargsclass &args, recptprotolistclass *protos,
			      displayclass &disp, outconvertclass &outconvert,
			      ostream &textout, ostream &logout) {

  ColInfoResponse_t *cinfo = NULL;
  comerror_t err;
  InfoFilterOptionsResponse_t fresponse;
  InfoFilterOptionsRequest_t frequest;
  frequest.filterName = "QueryFilter";

  text_t &index = args["h"];
  text_t &subcollection = args["j"];
  text_t &language = args["n"];

  text_tset collections;
  text_t arg_cc = args["cc"];
  decode_cgi_arg (arg_cc);
  splitchar (arg_cc.begin(), arg_cc.end(), ',', collections);

  textout << outconvert << disp << "_query:header_\n"
	  << "<center>_navigationbar_</center><br>\n"
	  << "<form name=QueryForm method=get action=\"_gwcgi_\">\n"
	  << "<input type=hidden name=a value=\"q\">\n"
          << "<input type=hidden name=site value=\"_cgiargsite_\"\n"
	  << "<input type=hidden name=e value=\"_compressedoptions_\">\n"
	  << "<input type=hidden name=ccp value=\"1\">\n"
	  << "<center><table width=_pagewidth_><tr valign=top>\n"
	  << "<td>Select collections to search for \"" << args["q"] 
	  << "\" <i>(index=" << index << " subcollection=" << subcollection
	  << " language=" << language << ")</i></td>\n"
	  << "<td><input type=\"submit\" value=\"_query:textbeginsearch_\"></td>\n"
	  << "</tr></table></center>\n"
	  << "<center><table width=_pagewidth_>\n"
	  << "<tr><td>\n";
  
  recptprotolistclass::iterator rprotolist_here = protos->begin();
  recptprotolistclass::iterator rprotolist_end = protos->end();
  while (rprotolist_here != rprotolist_end) {
    if ((*rprotolist_here).p != NULL) {
      
      text_tarray collist;
      (*rprotolist_here).p->get_collection_list (collist, err, logout);
      if (err == noError) {
	text_tarray::iterator collist_here = collist.begin();
	text_tarray::iterator collist_end = collist.end();
	while (collist_here != collist_end) {
	  
	  cinfo = recpt->get_collectinfo_ptr ((*rprotolist_here).p, *collist_here, logout);
	  //	  if (err == noError && cinfo.isPublic && (cinfo.buildDate > 0)) {
	  if (cinfo != NULL && (cinfo->buildDate > 0)) {
	    
	    (*rprotolist_here).p->get_filteroptions (*collist_here, frequest, fresponse, err, logout);
	    if (err == noError) {
	      
	      FilterOption_tmap::const_iterator it;
	      FilterOption_tmap::const_iterator end = fresponse.filterOptions.end();
	      if (!index.empty()) {
		it = fresponse.filterOptions.find ("Index");
		if (it == end) {collist_here ++; continue;}
		text_tarray::const_iterator there = (*it).second.validValues.begin();
		text_tarray::const_iterator tend = (*it).second.validValues.end();
		while (there != tend) {
		  if (*there == index) break;
		  there ++;
		}
		if (there == tend) {collist_here++; continue;}
	      }
	      if (!subcollection.empty()) {
		it = fresponse.filterOptions.find ("Subcollection");
		if (it == end) {collist_here++; continue;}
		text_tarray::const_iterator there = (*it).second.validValues.begin();
		text_tarray::const_iterator tend = (*it).second.validValues.end();
		while (there != tend) {
		  if (*there == subcollection) break;
		  there ++;
		}
		if (there == tend) {collist_here++; continue;}
	      }
	      if (!language.empty()) {
		it = fresponse.filterOptions.find ("Language");
		if (it == end) {collist_here++; continue;}
		text_tarray::const_iterator there = (*it).second.validValues.begin();
		text_tarray::const_iterator tend = (*it).second.validValues.end();
		while (there != tend) {
		  if (*there == language) break;
		  there ++;
		}
		if (there == tend) {collist_here++; continue;}
	      }
	    
	      // we've got a matching collection
	      textout << "<input type=checkbox";

	      text_tset::const_iterator t = collections.find (*collist_here);
	      if (t != collections.end()) textout << " checked";

	      textout << outconvert 
		      << " name=cc value=\"" << *collist_here << "\">";
	      
	      if (!cinfo->collectionmeta["collectionname"].empty())
		textout << outconvert << disp << cinfo->collectionmeta["collectionname"];
	      else 
		textout << outconvert << *collist_here;
	      
	      textout << "<br>\n";
	      
	    }
	  }
	  collist_here ++;
	}
      }
    }
    rprotolist_here ++;
  }
  textout << outconvert << disp 
	  << "</td></tr></table></center>\n"
	  << "</form>\n"
	  << "_query:footer_\n";
  
}

bool queryaction::do_action (cgiargsclass &args, recptprotolistclass *protos, 
			     browsermapclass *browsers, displayclass &disp, 
			     outconvertclass &outconvert, ostream &textout, 
			     ostream &logout) {
  
  if (recpt == NULL) {
    logout << "ERROR (queryaction::do_action): This action does not contain information\n"
	   << "      about any receptionists. The method set_receptionist was probably\n"
	   << "      not called from the module which instantiated this action.\n";
    return true;
  }


  if (args["ccs"] == "1") {
    if (!args["cc"].empty()) {
      // query the selected collections
      text_t::const_iterator b = args["cc"].begin();
      text_t::const_iterator e = args["cc"].end();
      if (findchar (b, e, ',') != e) {
	if (!search_multiple_collections (args, protos, browsers, disp, outconvert,
					  textout, logout)) return false;
	return true;
      } else {
	if (!search_single_collection (args, args["cc"], protos, browsers, disp, 
				       outconvert, textout, logout)) return false;
  	return true;
      }
    }
  }

  // simply query the current collection
  if (!search_single_collection (args, args["c"], protos, browsers, disp, 
				 outconvert, textout, logout)) return false;
  return true;
}

bool queryaction::search_multiple_collections (cgiargsclass &args, recptprotolistclass *protos,
					       browsermapclass *browsers, displayclass &disp,
					       outconvertclass &outconvert, ostream &textout,
					       ostream &logout) {

  text_tarray collections;
  
  text_t arg_cc = args["cc"];
  decode_cgi_arg (arg_cc);
  splitchar (arg_cc.begin(), arg_cc.end(), ',', collections);

  if (collections.empty()) {
    logout << "queryaction::search_multiple_collections: No collections "
	   << "set for doing multiple query - will search current collection\n";
    textout << outconvert << disp << "_query:textwarningnocollections_\n";
    return search_single_collection (args, args["c"], protos, browsers, disp, 
				     outconvert, textout, logout);
  }

  // queryaction uses "VList" browser to display results,
  // a queries clasification is "Search"
  text_t browsertype = "VList";
  text_t classification = "Search";

  QueryResult_tset results;
  map<text_t, colinfo_t, lttext_t> colinfomap;

  ColInfoResponse_t *cinfo = NULL;
  recptproto *collectproto = NULL;
  comerror_t err;
  FilterRequest_t request;
  FilterResponse_t response;
  request.filterResultOptions = FROID | FRmetadata | FRtermFreq | FRranking;
  text_t formattedstring = args["q"];
  text_t freqmsg = "_textfreqmsg1_";
  int numdocs = 0;
  isapprox isApprox = Exact;

  format_querystring (formattedstring, args.getintarg("b"));
  set_queryfilter_options (request, formattedstring, args);

  // need to retrieve maxdocs matches for each collection
  // (will eventually want to tidy this up, do so caching etc.)
  OptionValue_t option;
  option.name = "StartResults";
  option.value = "1";
  request.filterOptions.push_back (option);
  
  option.name = "EndResults";
  option.value = args["m"];
  request.filterOptions.push_back (option);

  text_tarray::iterator col_here = collections.begin();
  text_tarray::iterator col_end = collections.end();

  map<text_t, int, lttext_t> termfreqs;
  while (col_here != col_end) {

    request.fields.erase (request.fields.begin(), request.fields.end());
    request.getParents = false;

    collectproto = protos->getrecptproto (*col_here, logout);
    if (collectproto == NULL) {
      logout << outconvert << "queryaction::search_multiple_collections: " << *col_here 
	     << " collection has a NULL collectproto, ignoring\n";
      col_here ++;
      continue;
    }
    cinfo = recpt->get_collectinfo_ptr (collectproto, *col_here, logout);
    if (cinfo == NULL) {
      logout << "ERROR (query_action::search_multiple_collections): get_collectinfo_ptr returned NULL\n";
      col_here ++;
      continue;
    }
    
    browserclass *bptr = browsers->getbrowser (browsertype);

    // get the formatstring if there is one 
    text_t formatstring;
    if (!get_formatstring (classification, browsertype, 
			   cinfo->format, formatstring))
      formatstring = bptr->get_default_formatstring();

    bptr->load_metadata_defaults (request.fields);

    format_t *formatlistptr = new format_t();
    parse_formatstring (formatstring, formatlistptr, request.fields, request.getParents);

    colinfo_t thiscolinfo;
    thiscolinfo.formatlistptr = formatlistptr;
    thiscolinfo.browserptr = bptr;
    colinfomap[*col_here] = thiscolinfo;
    
    // do the query
    collectproto->filter (*col_here, request, response, err, logout);
    if (err != noError) {
      outconvertclass text_t2ascii;
      logout << text_t2ascii
	     << "queryaction::search_multiple_collections: call to QueryFilter failed "
	     << "for " << *col_here << " collection (" << get_comerror_string (err) << ")\n";
      return false;
    }

    if (isApprox == Exact)
      isApprox = response.isApprox;
    else if (isApprox == MoreThan)
      if (response.isApprox == Approximate)
	isApprox = response.isApprox;

    TermInfo_tarray::const_iterator this_term = response.termInfo.begin();
    TermInfo_tarray::const_iterator end_term = response.termInfo.end();
    while (this_term != end_term) {
      termfreqs[(*this_term).term] += (*this_term).freq;
      if ((col_here+1) == col_end) {
	freqmsg += (*this_term).term + ": " + termfreqs[(*this_term).term];
	if ((this_term+1) != end_term) freqmsg += ", ";
      }
      this_term ++;
    }
    
    if (response.numDocs > 0) {
      numdocs += response.numDocs;

      QueryResult_t thisresult;
      thisresult.collection = *col_here;
      ResultDocInfo_tarray::iterator doc_here = response.docInfo.begin();
      ResultDocInfo_tarray::iterator doc_end = response.docInfo.end();
      while (doc_here != doc_end) {
	thisresult.doc = *doc_here;
	results.insert (thisresult);
	doc_here ++;
      }
    }
    col_here ++;
  }

  disp.setmacro ("freqmsg", "query", freqmsg);

  int maxdocs = args.getintarg("m");
  int firstdoc = args.getintarg("r");
  int hitsperpage = args.getintarg("o");
  if (num_phrases > 0) isApprox = Exact;
  if (maxdocs == -1) maxdocs = numdocs;
  else if (numdocs > maxdocs) {
    numdocs = maxdocs;
    isApprox = MoreThan;
  }
  if (hitsperpage == -1) hitsperpage = numdocs;

  text_t resline;
  if (isApprox == Approximate) resline = "_textapprox_";
  else if (isApprox == MoreThan) resline = "_textmorethan_";
  
  if (numdocs == 0) resline = "_textnodocs_";
  else if (numdocs == 1) resline += "_text1doc_";
  else resline += text_t(numdocs) + " _textlotsdocs_";
  disp.setmacro("resultline", "query", resline);

  QueryResult_tset::iterator res_here = results.begin();
  QueryResult_tset::iterator res_end = results.end();
  text_tset metadata; // empty !!
  bool getParents = false; // don't care !!
  bool use_table;
  ResultDocInfo_t thisdoc;
  format_t *formatlistptr = NULL;
  browserclass *browserptr = NULL;

  // set up _thisfirst_ and _thislast_ macros
  disp.setmacro ("thisfirst", "query", firstdoc);
  int thislast = firstdoc + (hitsperpage - 1);
  if (thislast > numdocs) thislast = numdocs;
  disp.setmacro ("thislast", "query", thislast);

  // set up _prevfirst_ and _prevlast_ macros
  if (firstdoc > 1) {
    disp.setmacro ("prevlast", "query", firstdoc - 1);
    int prevfirst = firstdoc - hitsperpage;
    if (prevfirst < 1) prevfirst = 1;
    disp.setmacro ("prevfirst", "query", prevfirst);
  }

  // set up _nextfirst_ and _nextlast_ macros
  if (thislast < numdocs) {
    disp.setmacro ("nextfirst", "query", thislast + 1);
    int nextlast = thislast + hitsperpage;
    if (nextlast > numdocs) nextlast = numdocs;
    disp.setmacro ("nextlast", "query", nextlast);
  }

  textout << outconvert << disp << "_query:header_\n"
	  << "_query:content_";

  int count = 1;

  // output results
  while (res_here != res_end) {
    if (count < firstdoc) {count ++; res_here ++; continue;}
    if (count > thislast) break;
    formatlistptr = colinfomap[(*res_here).collection].formatlistptr;
    browserptr = colinfomap[(*res_here).collection].browserptr;
    thisdoc = (*res_here).doc;
    use_table = is_table_content (formatlistptr);

    collectproto = protos->getrecptproto ((*res_here).collection, logout);
    if (collectproto == NULL) {
      logout << outconvert << "queryaction::search_multiple_collections: " << (*res_here).collection
	     << " collection has a NULL collectproto, ignoring results\n";
      res_here ++;
      continue;
    }

    browserptr->output_section_group (thisdoc, args, (*res_here).collection, 0, 
    				      formatlistptr, use_table, metadata, getParents, 
    				      collectproto, disp, outconvert, textout, logout);
    //    textout << outconvert << "(ranking: " << (*res_here).doc.ranking << ")\n";
    res_here ++;
    count ++;
  }

  textout << outconvert << disp << "_query:footer_";
  
  // clean up the format_t pointers
  map<text_t, colinfo_t, lttext_t>::iterator here =  colinfomap.begin();
  map<text_t, colinfo_t, lttext_t>::iterator end =  colinfomap.end();
  while (here != end) {
    delete ((*here).second.formatlistptr);
    here ++;
  }
  return true;
}

bool queryaction::search_single_collection (cgiargsclass &args, const text_t &collection,
					    recptprotolistclass *protos, browsermapclass *browsers, 
					    displayclass &disp, outconvertclass &outconvert, 
					    ostream &textout, ostream &logout) {

  recptproto *collectproto = protos->getrecptproto (collection, logout);
  if (collectproto == NULL) {
    logout << outconvert << "queryaction::search_single_collection: " << collection
	   << " collection has a NULL collectproto\n";
    return false;
  }

  // queryaction uses "VList" browser to display results,
  // a queries clasification is "Search"
  text_t browsertype = "VList";
  text_t classification = "Search";

  comerror_t err;
  ColInfoResponse_t *cinfo = recpt->get_collectinfo_ptr (collectproto, collection, logout);

  if (cinfo == NULL) {
    logout << "ERROR (query_action::search_single_collection): get_collectinfo_ptr returned NULL\n";
    return false;
  }
    
  browserclass *bptr = browsers->getbrowser (browsertype);

  // get the formatstring if there is one 
  text_t formatstring;
  if (!get_formatstring (classification, browsertype, 
			 cinfo->format, formatstring))
    formatstring = bptr->get_default_formatstring();

  FilterRequest_t request;
  FilterResponse_t response;
  bptr->set_filter_options (request, args);
  bptr->load_metadata_defaults (request.fields);

  format_t *formatlistptr = new format_t();
  parse_formatstring (formatstring, formatlistptr, request.fields, request.getParents);
    
  // do the query
  request.filterResultOptions = FROID | FRmetadata | FRtermFreq;
  text_t formattedstring = "";

  if (args["qt"]=="0") { // normal text search
    formattedstring = args["q"];
    format_querystring (formattedstring, args.getintarg("b"));
    add_dates(formattedstring, args.getintarg("ds"), args.getintarg("de"),
	    args.getintarg("dsbc"), args.getintarg("debc"));
  }
  else if (args["qt"]=="1" ){ // form search

    if (args["b"]=="1" && args["fqa"]=="1") { // explicit query
      formattedstring = args["q"];
    }
    else { // form search
      if (args["b"]=="0") { // regular form
	parse_reg_query_form(formattedstring, args);
      }
      else  { // advanced form
	parse_adv_query_form(formattedstring, args);
      }
      args["q"] = formattedstring;
      
      // also reset the _cgiargq_ macro as it has changed now
      disp.setmacro("cgiargq", "Global", html_safe(args["q"]));
      
      // reset the compressed options to include the q arg
      text_t compressedoptions = recpt->get_compressed_arg(args, logout);
      if (!compressedoptions.empty()) {
	disp.setmacro ("compressedoptions", "Global", dm_safe(compressedoptions));
	// need a decoded version of compressedoptions for use within forms
	// as browsers encode values from forms before sending to server
	// (e.g. %25 becomes %2525)
	decode_cgi_arg (compressedoptions);
	disp.setmacro ("decodedcompressedoptions", "Global", dm_safe(compressedoptions));
      }
    } // form search
  } // args["qt"]=1
  else {
    logout << "ERROR (query_action::search_single_collection): querytype not defined\n";
    return false;
  }

  if (!formattedstring.empty()) { // do the query
    set_queryfilter_options (request, formattedstring, args);
    collectproto->filter (collection, request, response, err, logout);
    if (err != noError) {
      outconvertclass text_t2ascii;
      logout << text_t2ascii
	     << "queryaction::search_single_collections: call to QueryFilter failed "
	     << "for " << collection << " collection (" << get_comerror_string (err) << ")\n";
      return false;
    }
 
    define_query_macros (args, disp, response);

    // save the query if appropriate
    if (!save_search_history(args, response))
      logout << "save failed"; 
  }

  define_history_macros (disp, args, protos, logout);
  
  textout << outconvert << disp << "_query:header_\n"
	  << "_query:content_";

  // output the results
  bool use_table = is_table_content (formatlistptr);
  bptr->output_section_group (response, args, collection, 0, formatlistptr, 
			      use_table, request.fields, request.getParents, 
			      collectproto, disp, outconvert, textout, logout);
  

  textout << outconvert << disp << "_query:footer_";
  
  delete (formatlistptr);
  
  return true;
}
  
// define_query_macros sets the macros that couldn't be set until the
// query had been done. Those macros are _freqmsg_, _quotedquery_,
// _resultline_, _nextfirst_, _nextlast_, _prevfirst_, _prevlast_,
// _thisfirst_, and _thislast_
void queryaction::define_query_macros (cgiargsclass &args, displayclass &disp, 
				       const FilterResponse_t &response) { 
  // set up _freqmsg_ and _quotedquery_ macros
  text_t freqmsg = "_textfreqmsg1_";
  TermInfo_tarray::const_iterator this_term = response.termInfo.begin();
  TermInfo_tarray::const_iterator end_term = response.termInfo.end();
  while (this_term != end_term) {
    char *term = (*this_term).term.getcstr();
    //   if(term[0]<'0' || term[0] >'9'){
      freqmsg += (*this_term).term + ": " + (*this_term).freq;
      if ((this_term + 1) != end_term)
	freqmsg += ", ";
      // }
    delete term;
    this_term ++;
  }
  disp.setmacro ("freqmsg", "query", freqmsg);

  // set up _resultline_ macro
  text_t resline;
  int maxdocs = args.getintarg("m");
  int numdocs = response.numDocs;
  isapprox isApprox = response.isApprox;
  if (num_phrases > 0) isApprox = Exact;
  if (maxdocs == -1) maxdocs = numdocs;
  else if (numdocs > maxdocs) {
    numdocs = maxdocs;
    isApprox = MoreThan;
  }

  if (isApprox == Approximate) resline = "_textapprox_";
  else if (isApprox == MoreThan) resline = "_textmorethan_";
  
  if (numdocs == 0) resline = "_textnodocs_";
  else if (numdocs == 1) resline += "_text1doc_";
  else resline += text_t(numdocs) + " _textlotsdocs_";
  
  disp.setmacro("resultline", "query", resline);

  int firstdoc = args.getintarg("r");
  int hitsperpage = args.getintarg("o");
  if (hitsperpage == -1) hitsperpage = numdocs;

  // set up _thisfirst_ and _thislast_ macros
  disp.setmacro ("thisfirst", "query", firstdoc);
  int thislast = firstdoc + (hitsperpage - 1);
  if (thislast > numdocs) thislast = numdocs;
  disp.setmacro ("thislast", "query", thislast);

  // set up _prevfirst_ and _prevlast_ macros
  if (firstdoc > 1) {
    disp.setmacro ("prevlast", "query", firstdoc - 1);
    int prevfirst = firstdoc - hitsperpage;
    if (prevfirst < 1) prevfirst = 1;
    disp.setmacro ("prevfirst", "query", prevfirst);
  }

  // set up _nextfirst_ and _nextlast_ macros
  if (thislast < numdocs) {
    disp.setmacro ("nextfirst", "query", thislast + 1);
    int nextlast = thislast + hitsperpage;
    if (nextlast > numdocs) nextlast = numdocs;
    disp.setmacro ("nextlast", "query", nextlast);
  }
}

bool queryaction::save_search_history (cgiargsclass &args, const FilterResponse_t &response) 
{
  if (args["q"]=="") return true; // null query, dont save
  if (args["hs"]=="0") return true; // only save when submit query
                                    // displayed
  // get userid
  text_t userid = args["z"];

  // the number of docs goes on the front of the query string
  int numdocs= response.numDocs;
  text_t query = text_t(numdocs);
  if (response.isApprox==MoreThan) { // there were more docs found
    query.push_back('+');
  }
  query += "c="+args["c"];
  query += "&h="+args["h"];
  query += "&t="+args["t"];
  query += "&b="+args["b"];
  query += "&j="+args["j"];
  query += "&n="+args["n"];
  query += "&s="+args["s"];
  query += "&k="+args["k"];
  query += "&g="+args["g"]; 
  
  text_t qstring = args["q"];
  text_t formattedquery =cgi_safe(qstring);
  query += "&q="+formattedquery;

  bool display=false;
  int hd = args.getintarg("hd");
  if (hd > 0) display=true;
  if (set_history_info(userid, query, gsdlhome, display)) return true;
  else return false;


}



