###########################################################################
#
# List.pm --
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 1999 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

# simple list classifier plugin 
# options are:
# metadata=Metaname -- (optional) all documents with Metaname metadata
#                      will be included in list. if not included all documents
#                      will be included in list.
# sort=Meta         -- (optional) sort documents in list alphabetically by
#                      Meta. by default it will sort by Metaname, if neither
#                      are set documents will be in build (random) order.
#                      Meta may be Filename to sort by original filename or
#                      nosort to force not to sort
# buttonname=Title  -- (optional) the title field for this classification.
#                      if not included title field will be Metaname. 
#                      if metadata is also not included title will be 'List'

use BasClas;
package List;

use sorttools;

sub BEGIN {
    @ISA = ('BasClas');
}

sub print_usage {
    print STDERR "
  usage: classify List [options]
  options:

   -metadata X    Metadata field used for classification,
                  list will be sorted by this element.

   -buttonname X  (optional) Title field for this classification.
                  if not included title field will be Metaname. 

   -sort X        (optional) Sort documents in list by this metadata field.
                  By default it will sort by Metaname, or (if this is not
                  set) in build (random) order.
";
}

sub new {
    my $class = shift (@_);
    my $self = new BasClas($class, @_);
    
    my ($metaname, $title, $sortname, $list);

    if (!parsargv::parse(\@_, 
			 q^metadata/.*/^, \$metaname,
			 q^buttonname/.*/^, \$title,
			 q^sort/.*/^, \$sortname,
			 "allow_extra_options")) {
	
	print STDERR "\nIncorrect options passed to $class, check your collect.cfg file\n";
	&print_usage();
	die "\n";
    }

    if (!$title) {
	if ($metaname) {
	    $title = $metaname;
	} else {
	    $title = 'List';
	}
    }

    if (!$sortname) {
	if ($metaname) {
	    $sortname = $metaname;
	} else {
	    $sortname = undef;
	}
    }

    if ($metaname) {
	$list = {};
    } else {
	$list = [];
	$metaname = undef;
    }

    $self->{'list'} = $list;
    $self->{'metaname'} = $metaname;
    $self->{'title'} = $title;
    $self->{'sortname'} = $sortname;

    return bless $self, $class;
}

sub init {
    my $self = shift (@_);

    if (defined $self->{'sortname'}) {
	$self->{'list'} = {};
    } else {
	$self->{'list'} = [];
    }
}

sub classify {
    my $self = shift (@_);
    my ($doc_obj) = @_;

    my $doc_OID = $doc_obj->get_OID();

    my $sortmeta = "";
    if (defined $self->{'sortname'}) {
	if ($self->{'sortname'} =~ /^filename$/i) {
	    $sortmeta = $doc_obj->get_source_filename();
	} else {
	    $sortmeta = $doc_obj->get_metadata_element($doc_obj->get_top_section(),
						       $self->{'sortname'});
	    if (defined $sortmeta) {
		if ($self->{'sortname'} eq "Creator") {
		    &sorttools::format_string_name_english (\$sortmeta);
		} else {
		    &sorttools::format_string_english (\$sortmeta);
		}
	    }
	}
	$sortmeta = "" unless defined $sortmeta;

	if (defined $self->{'metaname'}) {
	    my $metavalue = $doc_obj->get_metadata_element ($doc_obj->get_top_section(),
							    $self->{'metaname'});
	    if (defined $metavalue) {
		if (defined $self->{'list'}->{$doc_OID}) {
		    my $outhandle = $self->{'outhandle'};
		    print $outhandle "WARNING: List::classify called multiple times for $doc_OID\n";
		} 
		$self->{'list'}->{$doc_OID} = $sortmeta;
	    }
	} else {
	    if (defined $self->{'list'}->{$doc_OID}) {
		my $outhandle = $self->{'outhandle'};
		print $outhandle "WARNING: List::classify called multiple times for $doc_OID\n";
	    } 
	    $self->{'list'}->{$doc_OID} = $sortmeta;
	}
    } else {
	if (defined $self->{'metaname'}) {
	    my $metavalue = $doc_obj->get_metadata_element ($doc_obj->get_top_section(),
							    $self->{'metaname'});
	    if (defined $metavalue) {
		push (@{$self->{'list'}}, $doc_OID);
	    }
	} else {
	    push (@{$self->{'list'}}, $doc_OID);
	}	    	    
    }
}

sub get_classify_info {
    my $self = shift (@_);
    my ($no_thistype) = @_;
    $no_thistype = 0 unless defined $no_thistype;

    my @list = ();
    if (defined $self->{'sortname'}) {
	if (keys %{$self->{'list'}}) {
	    @list = sort {$self->{'list'}->{$a} 
			  cmp $self->{'list'}->{$b};} keys %{$self->{'list'}};
	}
    } else {
	@list = @{$self->{'list'}};
    }

    # organise into classification structure
    my %classifyinfo = ('childtype'=>'VList',
			'Title'=>$self->{'title'},
			'contains'=>[]);
    $classifyinfo{'thistype'} = 'Invisible' unless $no_thistype;

    foreach $OID (@list) {
	push (@{$classifyinfo{'contains'}}, {'OID'=>$OID});
    }

    return \%classifyinfo;
}


1;
