/**********************************************************************
 *
 * OIDtools.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "OIDtools.h"
#include <assert.h>

// returns (in top) the top level of OID (i.e. everything
// up until the first dot)
void get_top (const text_t &OID, text_t &top) {

  top.clear();
  if (OID.empty()) return;

  text_t::const_iterator begin = OID.begin();
  text_t::const_iterator end = OID.end();

  top.appendrange (begin, findchar(begin, end, '.'));
}	


// checks if OID is top level (i.e. contains no dots)
bool is_top (const text_t &OID) {

  if (OID.empty()) return true;

  text_t::const_iterator here = OID.begin();
  text_t::const_iterator end = OID.end();
  here = findchar (here, end, '.');

  if (here == end) return true;
  return false;
}


// get_parents_array loads the parents array with all the parents of the
// document or classification specified by OID (not including OID itself)
void get_parents_array (const text_t &OID, text_tarray &parents) {

  text_t::const_iterator here = OID.begin ();
  text_t::const_iterator end = OID.end ();
  text_t thisparent;

  while (here != end) {
    if (*here == '.') parents.push_back(thisparent);
    thisparent.push_back(*here);
    here ++;
  }
}


// get_info does a protocol call and returns (in response) the metadata 
// associated with OID. Metadata should be loaded with whatever
// metadata fields are to be requested.

bool get_info (const text_t &OID, const text_t &collection,
	       const text_tset &metadata, bool getParents,
	       recptproto *collectproto, FilterResponse_t &response, 
	       ostream &logout) {

  response.clear();

  comerror_t err;
  FilterRequest_t request;
  request.clear();

  request.filterName = "NullFilter";
  request.filterResultOptions = FRmetadata;
  request.getParents = getParents;
  request.fields = metadata;
  request.docSet.push_back (OID);
  assert (collectproto != NULL);

  collectproto->filter (collection, request, response, err, logout);
  if (err != noError) {
    outconvertclass text_t2ascii;
    logout << text_t2ascii
	   << "Error: call to filter failed for " << OID
	   << " in OIDtools::get_info (" 
	   << get_comerror_string (err) << ")\n";
    return false;
  }
  
  return true;
}

// overloaded, to allow "custom" filter options.
bool get_info (const text_t &OID, const text_t &collection,
	       const text_tset &metadata, const OptionValue_tarray &options,
	       bool getParents,
	       recptproto *collectproto, FilterResponse_t &response, 
	       ostream &logout) {

  response.clear();

  comerror_t err;
  FilterRequest_t request;

  request.filterName = "NullFilter";
  request.filterResultOptions = FRmetadata;
  request.getParents = getParents;
  request.filterOptions = options;
  request.fields = metadata;
  request.docSet.push_back (OID);

  assert (collectproto != NULL);
  collectproto->filter (collection, request, response, err, logout);
  if (err != noError) {
    outconvertclass text_t2ascii;
    logout << text_t2ascii
	   << "Error: call to filter failed for " << OID
	   << " in OIDtools::get_info (" 
	   << get_comerror_string (err) << ")\n";
    return false;
  }
  
  return true;
}

bool get_info (const text_tarray &OIDs, const text_t &collection,
	       const text_tset &metadata, bool getParents,
	       recptproto *collectproto, FilterResponse_t &response, 
	       ostream &logout) {

  response.clear();
  if (OIDs.empty()) return true;

  comerror_t err;
  FilterRequest_t request;

  request.filterName = "NullFilter";
  request.filterResultOptions = FRmetadata;
  request.getParents = getParents;
  request.fields = metadata;

  request.docSet = OIDs;
  
  collectproto->filter (collection, request, response, err, logout);
  if (err != noError) {
    outconvertclass text_t2ascii;
    logout << text_t2ascii
	   << "Error: call to filter failed in OIDtools::get_info (" 
	   << get_comerror_string (err) << ")\n";
    return false;
  }

  return true;
}



// has_children returns true if OID has children
bool has_children (const text_t &OID, const text_t &collection, 
		   recptproto *collectproto, ostream &logout) {

  FilterResponse_t response;
  text_tset metadata;
  metadata.insert ("haschildren");

  if (get_info (OID, collection, metadata, false, collectproto,	response, logout)) {
    if (response.docInfo[0].metadata["haschildren"].values[0] == "1")
      return true;
  }
  return false;
}


// get_children does a protocol call and returns (in response) the OIDs and 
// metadata of all the children of OID. The metadata set should be loaded
// with whatever metadata fields are to be requested.

bool get_children (const text_t &OID, const text_t &collection, 
		   const text_tset &metadata, bool getParents,
		   recptproto *collectproto, FilterResponse_t &response, 
		   ostream &logout) {
  
  response.clear();

  comerror_t err;
  FilterRequest_t request;
  OptionValue_t option;

  option.name = "ParentNode";
  option.value = OID;
  request.filterOptions.push_back (option);
  request.filterName = "BrowseFilter";
  request.filterResultOptions = FROID | FRmetadata;
  request.fields = metadata;
  request.getParents = getParents;

  collectproto->filter (collection, request, response, err, logout);

  if (err != noError) {
    outconvertclass text_t2ascii;
    logout << text_t2ascii
	   << "Error: call to filter failed for " << OID 
	   << " in OIDtools::get_children (" 
	   << get_comerror_string (err) << ")\n";
    return false;
  }
  return true;
}

// get_parent returns the parent of the document or classification
// specified by OID
text_t get_parent (text_t OID) {

  if (OID.empty() || is_top (OID)) return "";

  text_t::const_iterator begin = OID.begin();
  text_t::const_iterator here = (OID.end() - 1);

  while (here >= begin) {
    OID.pop_back();
    if (*here == '.') break;
    here --;
  }
  return OID;
}

// takes an OID like ".2 and replaces the " with parent
void translate_parent (text_t &OID, const text_t &parent) {

  text_t::const_iterator here = OID.begin();
  text_t::const_iterator end = OID.end();
  text_t temp;

  while (here != end) {
    if (*here == '"') temp += parent;
    else temp.push_back (*here);
    here ++;
  }
  OID = temp;
}

// shrink_parent does the opposite to translate_parent
void shrink_parent (text_t &OID) {
  
  text_tarray tmp;
  splitchar (OID.begin(), OID.end(), '.', tmp);
  OID = "\"." + tmp.back();
}

// checks if OID uses ".fc", ".lc", ".pr", ".ns",
// or ".ps" syntax (first child, last child, parent,
// next sibling, previous sibling)
bool needs_translating (const text_t &OID) {

  if (OID.size() < 4) return false;

  text_t tail = substr (OID.end()-3, OID.end());
  if (tail == ".fc" || tail == ".lc" || tail == ".pr" || 
      tail == ".ns" || tail == ".ps") return true;

  return false;
}

// strips the ".fc", ".lc", ".pr", ".ns",
// or ".ps" suffix from the end of OID
void strip_suffix (text_t &OID) {

  text_t tail = substr (OID.end()-3, OID.end());
  while (tail == ".fc" || tail == ".lc" || tail == ".pr" || 
	 tail == ".ns" || tail == ".ps") {
    OID.erase (OID.end()-3, OID.end());
    tail = substr (OID.end()-3, OID.end());
  }
}

static void recurse_contents (ResultDocInfo_t section, const bool &is_classify, 
			      const text_t &collection, const text_tset &metadata, 
			      recptproto *collectproto, FilterResponse_t &response, 
			      ostream &logout) {

  int haschildren = section.metadata["haschildren"].values[0].getint();
  const text_t &doctype = section.metadata["doctype"].values[0];

  if ((haschildren == 1) && ((!is_classify) || (doctype == "classify"))) {
    FilterResponse_t tmp;
    bool getParents = false;
    get_children (section.OID, collection, metadata, getParents, collectproto, tmp, logout);
    ResultDocInfo_tarray::iterator thisdoc = tmp.docInfo.begin();
    ResultDocInfo_tarray::iterator lastdoc = tmp.docInfo.end();
    while (thisdoc != lastdoc) {
      response.docInfo.push_back (*thisdoc);
      recurse_contents (*thisdoc, is_classify, collection, metadata, 
			collectproto, response, logout);
      thisdoc ++;
    }
  }
}

// get_contents returns OIDs and metadata of all contents 
// below (and including) OID.
void get_contents (const text_t &topOID, const bool &is_classify, 
		   text_tset &metadata, const text_t &collection, 
		   recptproto *collectproto, FilterResponse_t &response, 
		   ostream &logout) {

  if (topOID.empty()) return;
  response.clear();

  metadata.insert ("haschildren");
  metadata.insert ("doctype");

  // get topOIDs info
  if (get_info (topOID, collection, metadata, false, collectproto, response, logout))
      recurse_contents (response.docInfo[0], is_classify, collection, 
			metadata, collectproto, response, logout);
}

// is_child_of returns true if OID2 is a child of OID1
bool is_child_of(const text_t &OID1, const text_t &OID2) {

  text_t parent = get_parent(OID2);

  while (!parent.empty()) {
    if (parent == OID1) return true;
    parent = get_parent(parent);
  }
  return false;
}
