/**********************************************************************
 *
 * gsdltools.cpp -- 
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "gsdltools.h"

#if defined(__WIN32__)
#include <windows.h>
#include <process.h>
#endif

bool littleEndian() {
  char s[2] = {'\xFE', '\xEF'};

  if (sizeof(unsigned short) == 2)
    return *(unsigned short*)s == 0xEFFE;
  else if (sizeof(unsigned int) == 2)
    return *(unsigned int*)s == 0xEFFE;
}

text_t dm_safe (const text_t &instring) {

  text_t outstring;
  text_t::const_iterator here = instring.begin();
  text_t::const_iterator end = instring.end();
  while (here != end) {
    if (*here == '_' || *here == '\\') outstring.push_back('\\');
    outstring.push_back(*here);
    here ++;
  }
  return outstring;
}

// gsdl_system creates a new process for the cmd command (which
// may contain arguments).
// cmd should contain the full path of the program to run.
// The child process inherits the environment of the calling
// process.
// If sync is true a synchronous call will be made, otherwise
// an asyncronous call.
// If sync is true the return value will be the exit code of
// the child process or -1 if the child process wasn't started.
// If sync is false the return value will be 0 if the process
// was started ok or -1 if it failed.
int gsdl_system (const text_t &cmd, bool sync, ostream &logout) {
  if (cmd.empty()) return -1;
  char *cmd_c = cmd.getcstr();

#if defined (__WIN32__)
  // the windows version - this is implemented this way
  // to prevent windows popping up all over the place when
  // we call a console application (like perl)
  STARTUPINFO ps = {sizeof(STARTUPINFO), NULL, NULL, NULL,
                    0, 0, 0, 0, 0, 0,
                    0, STARTF_USESHOWWINDOW,
                    SW_HIDE, 0, NULL,
                    NULL, NULL, NULL};
  PROCESS_INFORMATION pi;
  BOOL res = CreateProcess(NULL,
                           cmd_c,
                           NULL,
                           NULL,
                           FALSE,
			   NULL,
                           NULL,
                           NULL,
                           &ps,
                           &pi);
  if (!res) {
    logout << "gsdl_system failed to start " << cmd_c 
	   << " process, error code " << GetLastError();
    delete cmd_c;
    return -1;
  } 

  DWORD ret = 0;
  if (sync) { // synchronous system call
    // wait until child process exits.
    WaitForSingleObject(pi.hProcess, INFINITE);
    // set ret to exit code of child process
    GetExitCodeProcess(pi.hProcess, &ret);
  }

  CloseHandle(pi.hProcess);
  CloseHandle(pi.hThread);

#else
  // the unix version
  int ret = 0;
  if (sync) { // synchronous system call
    // make sure the command interpreter is found
    if (system (NULL) == 0) {
      logout << "gsdl_system failed to start " << cmd_c
	     << " process, command interpreter not found\n";
      delete cmd_c;
      return -1;
    }
    ret = system (cmd_c);

  } else { // asynchronous system call
    int pid = fork();
    if (pid == -1) {
      delete cmd_c;
      return -1;
    }
    if (pid == 0) {
      // child process
      char *argv[4];
      argv[0] = "sh";
      argv[1] = "-c";
      argv[2] = cmd_c;
      argv[3] = 0;
      execv("/bin/sh", argv);
    }
  }
#endif

  delete cmd_c;
  return ret;
}

// attempts to work out if perl is functional
bool perl_ok (ostream &logout) {
#if defined(__WIN32__)
  text_t cmd = "perl -e \"exit 0\"";
#else 
  text_t cmd = "perl -e 'exit 0'";
#endif;
  int i = gsdl_system (cmd, true, logout);
  return (i == 0);
}
