###########################################################################
#
# mgppbuildproc.pm -- 
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 1999 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

# This document processor outputs a document
# for mgpp to process


package mgppbuildproc;

use classify;
use doc;
use docproc;
use util;


BEGIN {
    @ISA = ('docproc');
}


sub new {
    my ($class, $collection, $source_dir, $build_dir, 
	$verbosity, $outhandle) = @_;
    my $self = new docproc ();

    # outhandle is where all the debugging info goes
    # output_handle is where the output of the plugins is piped
    # to (i.e. mg, gdbm etc.)
    $outhandle = STDERR unless defined $outhandle;

    $self->{'collection'} = $collection;
    $self->{'source_dir'} = $source_dir;
    $self->{'build_dir'} = $build_dir;
    $self->{'verbosity'} = $verbosity;
    $self->{'classifiers'} = [];
    $self->{'mode'} = "text";
    $self->{'assocdir'} = $build_dir;
    $self->{'dontgdbm'} = {};
    $self->{'index'} = "text";
    $self->{'indexexparr'} = [];
    $self->{'output_handle'} = "STDOUT";
    $self->{'num_docs'} = 0;
    $self->{'num_sections'} = 0;
    $self->{'num_bytes'} = 0;
    $self->{'num_processed_bytes'} = 0;
    $self->{'outhandle'} = $outhandle;
    $self->{'dontindex'} = {};
    $self->{'indexfieldmap'} = {};

    $self->{'indexing_text'} = 0;
    $self->{'indexfields'} = {};
    $self->{'strip_html'}=1;


    return bless $self, $class;
}

sub reset {
    my $self = shift (@_);
    
    $self->{'num_docs'} = 0;
    $self->{'num_sections'} = 0;
    $self->{'num_processed_bytes'} = 0;
    $self->{'num_bytes'} = 0;
}

sub get_num_docs {
    my $self = shift (@_);

    return $self->{'num_docs'};
}

sub get_num_sections {
    my $self = shift (@_);

    return $self->{'num_sections'};
}

# num_bytes is the actual number of bytes in the collection
# this is normally the same as what's processed during text compression
sub get_num_bytes {
    my $self = shift (@_);

    return $self->{'num_bytes'};
}

# num_processed_bytes is the number of bytes actually passed
# to mgpp for the current index
sub get_num_processed_bytes {
    my $self = shift (@_);

    return $self->{'num_processed_bytes'};
}

sub set_output_handle {
    my $self = shift (@_);
    my ($handle) = @_;

    $self->{'output_handle'} = $handle;
}

sub set_mode {
    my $self = shift (@_);
    my ($mode) = @_;

    $self->{'mode'} = $mode;
}

sub set_assocdir {
    my $self = shift (@_);
    my ($assocdir) = @_;

    $self->{'assocdir'} = $assocdir;
}

sub set_dontgdbm {
    my $self = shift (@_);
    my ($dontgdbm) = @_;

    $self->{'dontgdbm'} = $dontgdbm;
}

sub set_index {
    my $self = shift (@_);
    my ($index, $indexexparr) = @_;

    $self->{'index'} = $index;
    $self->{'indexexparr'} = $indexexparr if defined $indexexparr;
}

sub get_index {
    my $self = shift (@_);

    return $self->{'index'};
}

sub set_classifiers {
    my $self = shift (@_);
    my ($classifiers) = @_;

    $self->{'classifiers'} = $classifiers;
}

sub set_indexing_text {
    my $self = shift (@_);
    my ($indexing_text) = @_;

    $self->{'indexing_text'} = $indexing_text;
}

sub get_indexing_text {
    my $self = shift (@_);

    return $self->{'indexing_text'};
}

sub set_indexfieldmap {
    my $self = shift (@_);
    my ($indexmap) = @_;

    $self->{'indexfieldmap'} = $indexmap;
}

sub get_indexfieldmap {
    my $self = shift (@_);

    return $self->{'indexfieldmap'};
}

sub set_levels {
    my $self = shift (@_);
    my ($levels) = @_;

    $self->{'levels'} = $levels;
}

sub set_strip_html {
    my $self = shift (@_);
    my ($strip) = @_;
    $self->{'strip_html'}=$strip;
}

sub process {
    my $self = shift (@_);
    my $method = $self->{'mode'};

    $self->$method(@_);
}

# use 'Paged' if document has no more than 2 levels
# and each section at second level has a number for
# Title metadata
sub get_document_type {
    my $self = shift (@_);
    my ($doc_obj) = @_;

    my $thistype = "VList";
    my $childtype = "VList";
    my $title;
    my @tmp = ();
    
    my $section = $doc_obj->get_top_section ();
    my $first = 1;
    while (defined $section) {
	@tmp = split /\./, $section;
	if (scalar(@tmp) > 1) {
	    return ($thistype, $childtype);
	}
	if (!$first) {
	    $title = $doc_obj->get_metadata_element ($section, "Title");
	    if (!defined $title || $title !~ /^\d+$/) {
		return ($thistype, $childtype);
	    }
	}
	$first = 0;
	$section = $doc_obj->get_next_section($section);
    }
    if ($doc_obj->get_text_length ($doc_obj->get_top_section())) {
	$thistype = "Paged";
    } else {
	$thistype = "Invisible";
    }
    $childtype = "Paged";
    return ($thistype, $childtype);
}

sub assoc_files {
   my $self = shift (@_);
    my ($doc_obj, $archivedir) = @_;
    my ($afile);
    
    foreach my $assoc_file (@{$doc_obj->get_assoc_files()}) {
	# if assoc file contains directory structure of
	# its own use it, otherwise use HASH... directory
	if ($assoc_file->[1] =~ /[\/\\]/) {
	    $afile = &util::filename_cat($self->{'assocdir'}, $assoc_file->[1]);
	} else {
	    $afile = &util::filename_cat($self->{'assocdir'}, $archivedir, $assoc_file->[1]);
	}
	&util::hard_link ($assoc_file->[0], $afile);
    }
}

sub infodb {
    my $self = shift (@_);
    my ($doc_obj, $filename) = @_;
    my $handle = $self->{'output_handle'};

    my $doctype = $doc_obj->get_doc_type();

    # only output this document if it is one to be indexed
    return if ($doctype ne "indexed_doc");

    #if a Section level index is not built, the gdbm file should be at doc 
    #level not Section
    my $docs_only = 1;
    if ($self->{'levels'}->{'Section'}) {
	$docs_only = 0;
    }

    my ($archivedir) = $filename =~ /^(.*?)(?:\/|\\)[^\/\\]*$/;
    $archivedir = "" unless defined $archivedir;
    $archivedir =~ s/\\/\//g;
    $archivedir =~ s/^\/+//;
    $archivedir =~ s/\/+$//;

    $self->assoc_files ($doc_obj, $archivedir);

    # this is another document
    $self->{'num_docs'} += 1 unless ($doctype eq "classification");

    # is this a paged or a hierarchical document
    my ($thistype, $childtype) = $self->get_document_type ($doc_obj);

    my $section = $doc_obj->get_top_section ();
    my $doc_OID = $doc_obj->get_OID();
    my $first = 1;
    my $url = "";
    while (defined $section) {
	# update a few statistics
	$self->{'num_bytes'} += $doc_obj->get_text_length ($section);
	$self->{'num_sections'} += 1 unless ($doctype eq "classification");

	# output the section name
	if ($section eq "") { print $handle "[$doc_OID]\n"; }
	else { print $handle "[$doc_OID.$section]\n"; }

	# output the fact that this document is a document (unless doctype
	# has been set to something else from within a plugin
	my $dtype = $doc_obj->get_metadata_element ($section, "doctype");
	if (!defined $dtype || $dtype !~ /\w/) {
	    print $handle "<doctype>doc\n";
	}

	# output whether this node contains text
	if ($doc_obj->get_text_length($section) > 0) {
	    print $handle "<hastxt>1\n";
	} else {
	    print $handle "<hastxt>0\n";
	}

	# output all the section metadata
	my $metadata = $doc_obj->get_all_metadata ($section);
	foreach my $pair (@$metadata) {
	    my ($field, $value) = (@$pair);

	    if ($field ne "Identifier" && $field !~ /^gsdl/ && 
		defined $value && $value ne "") {

		# escape problematic stuff
		$value =~ s/\\/\\\\/g;
		$value =~ s/\n/\\n/g;
		$value =~ s/\r/\\r/g;

		# special case for URL metadata
		if ($field =~ /^URL$/i) {
                    $url .= "[$value]\n";
                    if ($section eq "") {$url .= "<section>$doc_OID\n";}
                    else {$url .= "<section>$doc_OID.$section\n";}
                    $url .= '-' x 70 . "\n";
		}

		if (!defined $self->{'dontgdbm'}->{$field}) {
		    print $handle "<$field>$value\n";
		}
	    }
	}

	# output archivedir if at top level
	if ($section eq $doc_obj->get_top_section()) {
	    print $handle "<archivedir>$archivedir\n";
	}

	# output document display type
	if ($first) {
	    print $handle "<thistype>$thistype\n";
	}

	if (!$docs_only) {
	    # output a list of children
	    my $children = $doc_obj->get_children ($section);
	    if (scalar(@$children) > 0) {
		print $handle "<childtype>$childtype\n";
		print $handle "<contains>";
		my $firstchild = 1;
		foreach $child (@$children) {
		    print $handle ";" unless $firstchild;
		    $firstchild = 0;
		    if ($child =~ /^.*?\.(\d+)$/) {
			print $handle "\".$1";
		    } else {
			print $handle "\".$child";
		    }
#		if ($child eq "") { print $handle "$doc_OID"; }
#		elsif ($section eq "") { print $handle "$doc_OID.$child"; }
#		else { print $handle "$doc_OID.$section.$child"; }
		}
		print $handle "\n";
	    }
	    #output the matching doc number
	    print $handle "<docnum>$self->{'num_sections'}\n";
	    
	} # if (!$docs_only)
	else { #docs only, doc num is num_docs not num_sections
	    # output the matching document number
	    print $handle "<docnum>$self->{'num_docs'}\n";
	}
	
	print $handle '-' x 70, "\n";

	
	# output a database entry for the document number
	if ($docs_only) {
	    print $handle "[$self->{'num_docs'}]\n";
	    print $handle "<section>$doc_OID\n";
	}
	else {
	    print $handle "[$self->{'num_sections'}]\n";
	    if ($section eq "") { print $handle "<section>$doc_OID\n"; }
	    else { print $handle "<section>$doc_OID.$section\n"; }
	}
	print $handle '-' x 70, "\n";
	
        # output entry for url
        if ($url ne "") {
            print $handle $url;
        }

	$first = 0;
	$section = $doc_obj->get_next_section($section);
	last if ($docs_only); # if no sections wanted, only gdbm the docs
    }

    # classify this document
    &classify::classify_doc ($self->{'classifiers'}, $doc_obj);

}

sub find_paragraphs {
    $_[1] =~ s/(<p\b)/<Paragraph>$1/gi;
}

#this function strips the html tags from the doc if ($strip_html) and
# if ($para) replaces <p> with <Paragraph> tags.
# if both are false, the original text is returned
#assumes that <pre> and </pre> have no spaces, and removes all < and > inside
#these tags
sub preprocess_text {
    my $self = shift (@_);
    my ($text, $strip_html, $para) = @_;

    my ($outtext) = "";
    if ($strip_html) { 
	while ($text =~ /<([^>]*)>/ && $text ne "") {
	    
	    $tag = $1;
	    $outtext .= $`." "; #add everything before the matched tag
	    $text = $'; #everything after the matched tag
	    if ($para && $tag =~ /^\s*p\s/) {
		$outtext .= "<Paragraph> ";
	    }
	    elsif ($tag =~ /^pre$/) { # a pre tag
		$text =~ /<\/pre>/; # find the closing pre tag
		my $tmp_text = $`; #everything before the closing pre tag
		$text = $'; #everything after the </pre>
		$tmp_text =~ s/[<>]//g; # remove all < and >
		$outtext.= $tmp_text . " ";
	    }
	}
    
	$outtext .= $text; # add any remaining text
	return $outtext;
    } #if strip_html

    if ($para) {
	$text =~ s/(<p\b)/<Paragraph>$1/gi;
	return $text;
    }
    return $text;
}
	
    

sub filter_text {
    # $self->filter_text ($field, $new_text);
    # don't want to do anything for this version, however,
    # in a particular collection you might want to override
    # this method to post-process certain fields depending on
    # the field, or whether we are outputting it for indexing
}

sub text {
    my $self = shift (@_);
    my ($doc_obj) = @_;
    my $handle = $self->{'output_handle'};
    my $indexed_doc = 1;

    # only output this document if it is one to be indexed
    return if ($doc_obj->get_doc_type() ne "indexed_doc");

    # see if this document belongs to this subcollection
    foreach my $indexexp (@{$self->{'indexexparr'}}) {
	$indexed_doc = 0;
	my ($field, $exp, $options) = split /\//, $indexexp;
	if (defined ($field) && defined ($exp)) {
	    my ($bool) = $field =~ /^(.)/;
	    $field =~ s/^.// if $bool eq '!';
	    if ($field =~ /^filename$/i) {
		$field = $doc_obj->get_source_filename();
	    } else {
		$field = $doc_obj->get_metadata_element($doc_obj->get_top_section(), $field);
	    }
	    next unless defined $field;
	    if ($bool eq '!') {
		if ($options =~ /^i$/i) {
		    if ($field !~ /$exp/i) {$indexed_doc = 1; last;}
		} else {
		    if ($field !~ /$exp/) {$indexed_doc = 1; last;}
		}
	    } else {
		if ($options =~ /^i$/i) {
		    if ($field =~ /$exp/i) {$indexed_doc = 1; last;}
		} else {
		    if ($field =~ /$exp/) {$indexed_doc = 1; last;}
		}
	    }
	}
    }

    # this is another document
    $self->{'num_docs'} += 1;

    # get the parameters for the output
    my ($fields) = $self->{'index'};

    my ($sectiontag) = "";
    if ($self->{'levels'}->{'Section'}) {
	$sectiontag = "\n<Section>\n";
    }
    my ($paratag) = "";
    if ($self->{'levels'}->{'Paragraph'}) {
	$paratag = "<Paragraph>";
    }
    my $doc_section = 0; # just for this document
    my $text = "<Document>\n";
   
    # get the text for this document
    my $section = $doc_obj->get_top_section();
    while (defined $section) {
	# update a few statistics
	$doc_section++;
	$self->{'num_sections'} += 1;
	$text .= $sectiontag;

	if ($indexed_doc) {
	    $self->{'num_bytes'} += $doc_obj->get_text_length ($section);
	    foreach my $field (split (/,/, $fields)) {
		# only deal with this field if it doesn't start with top or
		# this is the first section
		my $real_field = $field;
		if (!($real_field =~ s/^top//) || ($doc_section == 1)) {
		    my $new_text = ""; 
		    my $tmp_text = "";
		    if ($real_field eq "text") {
			if ($self->{'indexing_text'}) { #tag the text with <Text>...</Text>, add the <Paragraph> tags and strip out html if needed
			    $new_text .= "<TX>\n";
			    $tmp_text .= $doc_obj->get_text ($section);
			    $tmp_text = $self->preprocess_text($tmp_text, $self->{'strip_html'}, $self->{'levels'}->{'Paragraph'});

			    $new_text .= "$tmp_text</TX>\n";
			    if (!defined $self->{'indexfields'}->{'TextOnly'}) {
				$self->{'indexfields'}->{'TextOnly'} = 1;   
			    }
			}
			else { # leave html stuff in, and dont add Paragraph tags - never retrieve paras at the moment
			    $new_text .= $doc_obj->get_text ($section);
                            #if ($self->{'levels'}->{'Paragraph'}) {
			    #$self->find_paragraphs($new_text);
			    #}                
			}
		    } else { # metadata field
			if ($real_field eq "metadata") { # insert all metadata 
			    #except gsdl stuff
			    my $shortname = "";
			    my $metadata = $doc_obj->get_all_metadata ($section);
			    foreach $pair (@$metadata) {
				my ($mfield, $mvalue) = (@$pair);
				# check fields here, maybe others dont want - change to use dontindex!!
				if ($mfield ne "Identifier" && $mfield ne "classifytype" && 
				    $mfield !~ /^gsdl/ && defined $mvalue && $mvalue ne "") {
				    
				    if (defined $self->{'indexfieldmap'}->{$mfield}) {
					$shortname = $self->{'indexfieldmap'}->{$mfield};
				    }
				    else {
					$shortname = $self->create_shortname($mfield);
					$self->{'indexfieldmap'}->{$mfield} = $shortname;
					$self->{'indexfieldmap'}->{$shortname} = 1;
				    }	   
				    $new_text .= "$paratag<$shortname>$mvalue</$shortname>\n";
				    if (!defined $self->{'indexfields'}->{$mfield}) {
					$self->{'indexfields'}->{$mfield} = 1;
				    }				    
				}
			    }
			    
			}
			else { #individual metadata specified
			    my $shortname="";
			    if (!defined $self->{'indexfields'}->{$real_field}) {
				$self->{'indexfields'}->{$real_field} = 1;
			    }
			    if (defined $self->{'indexfieldmap'}->{$real_field}) {
				$shortname = $self->{'indexfieldmap'}->{$real_field};
			    }
			    else {
				$shortname = $self->create_shortname($real_field);
				$self->{'indexfieldmap'}->{$real_field} = $shortname;
				$self->{'indexfieldmap'}->{$shortname} = 1;
			    }
			    foreach $item (@{$doc_obj->get_metadata ($section, $real_field)}) {
				$new_text .= "$paratag<$shortname>$item</$shortname>\n";
			    }
			}

		    }
		    
		    # filter the text
		    $self->filter_text ($field, $new_text);

		    $self->{'num_processed_bytes'} += length ($new_text);
		    $text .= "$new_text";
		}
	    }
	} # if (indexed_doc)
	
	$section = $doc_obj->get_next_section($section);
    } #while defined section
    print $handle "$text\n</Document>\n"; 
}

sub create_shortname {
    $self = shift(@_);
    
    my ($realname) = @_;
    #take the first two chars
    my ($shortname) = $realname =~ /^(\w\w)/;
    $shortname =~ tr/a-z/A-Z/;

    #if already used, take the first and third letters and so on
    $count = 1;
    while (defined $self->{'indexfieldmap'}->{$shortname}) {
	if ($realname =~ /^(\w).{$count}(\w)/) {
	    $shortname = "$1$2";
	$count++;
	$shortname =~ tr/a-z/A-Z/;
	
	}
	else {
	    $realname =~ s/^.//;
	    $count = 0;
	}
    }

    return $shortname;
}

1;

