#!/usr/bin/perl -w


BEGIN {
    die "GSDLHOME not set\n" unless defined $ENV{'GSDLHOME'};
    unshift (@INC, "$ENV{'GSDLHOME'}/perllib");
}


# create_distributions.pl creates the following distributions from the gsdl
# directory pointed to by $GSDLHOME (although it uses the main CVS
# repository to get the latest source code distribution).

# Windows distribution - gsdl-x.xx-win32.exe
# Creates directory structure for this - use installshield and
# packagefortheweb to create self-extracting executable

# Linux distribution - gsdl-x.xx-linux.tgz

# Source distribution - gsdl-x.xx.tgz

# cd-rom distribution = gsdl-x.xx-cdrom.tgz
# Creates directory structure - use installshield to create media
# for writing to cd-rom

# all collections in $GSDLHOME/collect (except modelcol and those
# explicitly ignored by -ignorecol options):
#  pre-built (collname-prebuilt) .tgz .zip
#  unbuilt (collname) .tgz .zip


#    creates ChangeLog using `cvs2cl.pl -P -F trunk -r -S -l "-d'date<tomorrow'"
#    where date of last distribution is read from the last line of 
#    $GSDLHOME/DistDates or from -ChangeLogDate command line option
#    should use something like cvs2cl.pl ... -l "-r'last_tag' -r'this_tag'" but
#    I can't get cvs to ignore files which contain neither last_tag or this_tag
#    so ChangeLog contains lots of ancient stuff 

#    sets a cvs tag called gsdl-x_xx-distribution (unless -cvs_tag is set
#    in which case it uses that)

#    creates a copy of everything in cvs repository in $tmpdir

#    edits /tmp/gsdl/etc/VERSION, /tmp/gsdl/src/w32server/fnord.cpp
#    and /tmp/gsdl/cgi-bin/gsdlsite.cfg to use new version number
#    and default GSDLHOME

# temporary working directory
my $tmpdir = '/tmp';

# docs directory - up-to-date copy of everything to go in the docs directory 
# (including the README.txt) is expected to live here
my $docdir = '/home/nzdl/gsdl-docs';

# where the windows binaries live (including library.exe and server.exe)
# this currently relies on being a directory ending in "windows"
my $winbin = "$ENV{'GSDLHOME'}/bin/windows";

# ditto for linux binaries (don't forget getpw)
my $linuxbin = "$ENV{'GSDLHOME'}/bin/linux";

my $cvsroot = '/home/nzdl/gsdl-src';

use parsargv;
use util;

sub print_usage {
    print STDERR "\n";
    print STDERR "create_distributions.pl: Packages up Greenstone distributions.\n\n";
    print STDERR "  usage: $0 [options]\n\n";
    print STDERR "  options:\n";
    print STDERR "   -version_num           version number of distribution (x.xx)\n";
    print STDERR "   -cvs_tag               cvs tag from which to create distributions\n";
    print STDERR "                          - if not set latest versions will be used and\n";
    print STDERR "                          a tag will be set\n";
    print STDERR "   -no_cdrom              don't create cd-rom version\n";
    print STDERR "   -no_cols               don't attempt to create any collection distributions\n";
    print STDERR "   -only_cols             create collection distributions only\n";
    print STDERR "   -includecol            collection to include (by default all collections in\n";
    print STDERR "                          $ENV{'GSDLHOME'}/collect will be included\n";
    print STDERR "   -ignorecol             directory in $ENV{'GSDLHOME'}/collect to ignore (i.e.\n";
    print STDERR "                          don't include as collection in distribution)\n";
    print STDERR "   -output_dir            directory to output distributions to\n";
    print STDERR "   -NoChangeLog           don't create ChangeLog\n";
    print STDERR "   -UseExistingLog        use $ENV{'GSDLHOME'}/ChangeLog rather than creating one\n";
    print STDERR "   -ChangeLogDate         date from which to begin ChangeLog - note that ChangeLog\n";
    print STDERR "                          will always run from date to most recent commits, even if\n";
    print STDERR "                          cvs_tag is for a previous revision\n\n";
}

&main ();

sub main {
    if (!parsargv::parse(\@ARGV, 
			 'version_num/\d\.\d\d[a-z]?', \$version_num,
			 'cvs_tag/.*/', \$cvs_tag,
			 'no_cdrom', \$no_cdrom,
			 'no_cols', \$no_cols,
			 'only_cols', \$only_cols,
			 'includecol/.*', \@includecols,
			 'ignorecol/.*', \@ignorecols,
			 'NoChangeLog', \$nochangelog,
			 'UseExistingLog', \$useexistinglog,
			 'ChangeLogDate/.*/', \$changelogdate,
			 'output_dir/.*/', \$output_dir)) {
	&print_usage();
	die "\n";
    }
    
    $output_dir = "." unless $output_dir =~ /\w/;
    mkdir ($output_dir, 0777) unless -d $output_dir;

    my $have_tag = 1;
    if ($cvs_tag !~ /\w/) {
	$cvs_tag = $version_num;
	$cvs_tag =~ s/\./\_/g;
	$cvs_tag = "gsdl-" . $cvs_tag . "-distribution";
	$have_tag = 0;
    }
    
    &create_changelog() unless ($nochangelog || $useexistinglog);
    
    if (!$only_cols) {

	# tag repository
	if (!$have_tag) {
	    print STDERR "\ntagging with $cvs_tag\n";
	    chdir ($ENV{'GSDLHOME'});
	    `cvs tag $cvs_tag`;
	}
	
	# cvs export gsdl to $tmpdir
	print STDERR "\nexporting gsdl ($cvs_tag) to $tmpdir\n\n";
	chdir ($tmpdir);
	`cvs -d $cvsroot export -r $cvs_tag gsdl`;
	
	# copy ChangeLog into $tmpdir/gsdl
	&util::cp (&util::filename_cat($ENV{'GSDLHOME'}, "ChangeLog"), 
		   &util::filename_cat($tmpdir, "gsdl")) unless $nochangelog;

	# edit the VERSION and fnord.cpp files
	&edit_files();
	
	&create_windows_distribution();
	&create_linux_distribution();
	&create_src_distribution();
	&create_cdrom_distribution() unless $no_cdrom;
    }

    &create_collection_distributions() unless $no_cols;
}

sub create_windows_distribution {
    my $windows_dist_dir = &util::filename_cat ($output_dir, "gsdl-" . $version_num . "-win32");
    mkdir ($windows_dist_dir, 0777);

    # docs directory and README.TXT
#    &install_docs ($windows_dist_dir, 1); -- no docs for windows web installation

    # gsdl directory
    &install_gsdl ($windows_dist_dir);

    # src directory
    &install_src ($windows_dist_dir, "windows");

    # Windows directory
    &install_windows_specific ($windows_dist_dir);
}

sub create_linux_distribution {
    my $linux_dir = "gsdl-" . $version_num . "-linux";
    my $linux_dist_dir = &util::filename_cat ($output_dir, $linux_dir);
    mkdir ($linux_dist_dir, 0777);

    # empty "collect" directory
    mkdir (&util::filename_cat($linux_dist_dir, "collect"), 0777);

    # docs directory, README.TXT, COPYING and Support.htm
    &install_docs ($linux_dist_dir, 1);

    # gsdl directory
    &install_gsdl ($linux_dist_dir);

    # src directory
    &install_src ($linux_dist_dir, "unix");

    # Unix directory
    &install_unix_specific ($linux_dist_dir);

    # tar and gzip it
    &zip ($linux_dir, $linux_dir, $output_dir, 2);
    &util::rm_r ($linux_dist_dir);
}

sub create_src_distribution {
    my $src_zip = "gsdl-" . $version_num . "-src";
    # we want the COPYING file in the source distribution too
    &util::cp (&util::filename_cat($docdir, "COPYING"),
	       &util::filename_cat($tmpdir, "gsdl"));
#    &edit_gsdlsite("/home/gsdl");
    &zip ($src_zip, "gsdl", $tmpdir, 2);
    &util::cp (&util::filename_cat($tmpdir, "$src_zip.tgz"), $output_dir);
    # remove the COPYING file again to avoid confusion (it's copied to the right 
    # places by install_docs() for other distributions)
    &util::rm (&util::filename_cat($tmpdir, "COPYING"));
}

sub create_cdrom_distribution {
    my $cdrom_dist_dir = &util::filename_cat ($output_dir, "gsdl-" . $version_num . "-cdrom");
    mkdir ($cdrom_dist_dir, 0777);

    # collect directory (empty for now. we'll add collections manually)
    mkdir (&util::filename_cat ($cdrom_dist_dir, "collect"), 0777);

    # docs directory and README.TXT
    &install_docs ($cdrom_dist_dir, 0);

    # gsdl directory
    &install_gsdl ($cdrom_dist_dir);

    # src directory
    &install_src ($cdrom_dist_dir, "cdrom");

    # Windows directory
    &install_windows_specific ($cdrom_dist_dir);

    # Unix directory
    &install_unix_specific ($cdrom_dist_dir);
}

# isweb is 1 if it's one of the web distributions (i.e. if we don't
# want to install the manual) - this shouldn't be called at all for
# windows web installation as it doesn't use docs at all (not much 
# point for self-extracting exe).
sub install_docs {
    my ($install_dir, $isweb) = @_;

    # docs directory, README.TXT, COPYING and Support.htm
    &util::cp_r (&util::filename_cat($docdir, "docs"), $install_dir);
    &util::cp (&util::filename_cat($docdir, "README.TXT"), $install_dir);
    &util::cp (&util::filename_cat($docdir, "COPYING"), $install_dir);
    &util::cp (&util::filename_cat($docdir, "Support.htm"), $install_dir);
    # don't want manual for web distributions
    if ($isweb) {
	my $manfile = &util::filename_cat ($install_dir, "docs", "Manual.pdf");
	&util::rm ($manfile) if -e $manfile;
    }
}

sub install_gsdl {
    my ($install_dir) = @_;

    my $gsdldir = &util::filename_cat ($install_dir, "gsdl");
    my $gsdlbindir = &util::filename_cat ($gsdldir, "bin");
    mkdir ($gsdldir, 0777);
    mkdir ($gsdlbindir, 0777);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "bin", "script"), $gsdlbindir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "cgi-bin"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "collect"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "etc"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "images"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "macros"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "perllib"), $gsdldir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "mappings"), $gsdldir);

    # make sure that modelcol collection contains all the right 
    # empty directories
    my $modelindex = &util::filename_cat ($tmpdir, "gsdl", "collect", "modelcol", "index");
    &util::mk_all_dir ($modelindex) unless -d $modelindex;
    my $modelbuilding = &util::filename_cat ($tmpdir, "gsdl", "collect", "modelcol", "building");
    &util::mk_all_dir ($modelbuilding) unless -d $modelbuilding;
    my $modelarchives = &util::filename_cat ($tmpdir, "gsdl", "collect", "modelcol", "archives");
    &util::mk_all_dir ($modelarchives) unless -d $modelarchives;
    my $modelimport = &util::filename_cat ($tmpdir, "gsdl", "collect", "modelcol", "import");
    &util::mk_all_dir ($modelimport) unless -d $modelimport;


    # demo collection needs to be pre-built
    my $collectdir = &util::filename_cat ($gsdldir, "collect");
    &util::rm_r (&util::filename_cat($collectdir, "demo"));
    if (!&get_built_collection ("demo", $collectdir)) {
	die "Couldn't get built version of demo collection\n";
    }
}

sub install_src {
    my ($install_dir, $type) = @_;

    my $srcdir = &util::filename_cat ($install_dir, "src");
    my $srcwindir = &util::filename_cat ($srcdir, "Windows");
    my $srcunixdir = &util::filename_cat ($srcdir, "Unix");
    mkdir ($srcdir, 0777);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "lib"), $srcdir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "packages"), $srcdir);
    &util::cp_r (&util::filename_cat ($tmpdir, "gsdl", "src"), $srcdir);

    if ($type ne "unix") {
	mkdir ($srcwindir, 0777);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "setup.bat"), $srcwindir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "win32.mak"), $srcwindir);
    }

    if ($type ne "windows") {
	mkdir ($srcunixdir, 0777);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "Makefile.in"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "acconfig.h"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "aclocal.m4"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "config.h.in"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "configtest.pl"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "configure"), $srcunixdir);
	# make sure configure script is executable
	my $configure_script = &util::filename_cat ($srcunixdir , "configure");
	`chmod a+x $configure_script`;
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "configure.in"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "install-sh"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "setup.bash"), $srcunixdir);
	&util::cp (&util::filename_cat ($tmpdir, "gsdl", "setup.csh"), $srcunixdir);
    }
}

sub install_windows_specific {
    my ($install_dir) = @_;

    my $windir = &util::filename_cat ($install_dir, "Windows");
    my $winbindir = &util::filename_cat ($windir, "bin");
    mkdir ($windir, 0777);
    mkdir ($winbindir, 0777);
    &util::cp_r ($winbin, $winbindir);

    # make sure there aren't any CVS directories laying around
    &remove_cvs_dirs ($windir);
}

sub install_unix_specific {
    my ($install_dir) = @_;

    my $unixdir = &util::filename_cat ($install_dir, "Unix");
    my $unixbindir = &util::filename_cat ($unixdir, "bin");
    mkdir ($unixdir, 0777);
    mkdir ($unixbindir, 0777);
    &util::cp (&util::filename_cat($tmpdir, "gsdl", "Install.sh"), $unixdir);

    # make sure Install.sh is executable
    my $install_sh = &util::filename_cat ($unixdir, "Install.sh");
    `chmod a+x $install_sh`;
    &util::cp_r ($linuxbin, $unixbindir);

    # make sure there aren't any CVS directories laying around
    &remove_cvs_dirs ($unixdir);
}

sub create_collection_distributions {
    
    # work out which collections we want
    my @cols = ();
    if (scalar @includecols) {
	@cols = @includecols;
    } else {
	my $collectdir = &util::filename_cat($ENV{'GSDLHOME'}, "collect");

	opendir (COLLECTDIR, $collectdir) || die;
	my @cdirs = readdir COLLECTDIR;
	closedir COLLECTDIR;
	my %ignore = ();
	map { $ignore{$_} = ""; } @ignorecols;
	foreach $d (@cdirs) {
	    if ((-d &util::filename_cat($collectdir, $d)) && ($d ne ".") && ($d ne "..") && 
		($d ne "modelcol") && ($d ne "CVS") && (!defined $ignore{$d})) {
		push (@cols, $d);
	    }
	}
    }
    
    return unless scalar @cols;

    # create distributions
    foreach $collection (@cols) {
	if (&get_built_collection ($collection, $tmpdir)) {
	    &zip ("$collection-prebuilt", $collection, $tmpdir, 0);
	    &util::cp (&util::filename_cat($tmpdir, "$collection-prebuilt.tgz"), $output_dir);
	    &util::cp (&util::filename_cat($tmpdir, "$collection-prebuilt.zip"), $output_dir);
	} else {
	    print STDERR "ERROR: Couldn't create pre-built $collection collection\n";
	}
	&util::rm_r (&util::filename_cat ($tmpdir, $collection)) 
	    if -d &util::filename_cat ($tmpdir, $collection);

	if (&get_unbuilt_collection ($collection, $tmpdir)) {
	    &zip ($collection, $collection, $tmpdir, 0);
	    &util::cp (&util::filename_cat($tmpdir, "$collection.tgz"), $output_dir);
	    &util::cp (&util::filename_cat($tmpdir, "$collection.zip"), $output_dir);
	} else {
	    print STDERR "ERROR: Couldn't create unbuilt $collection collection\n";
	}
	&util::rm_r (&util::filename_cat ($tmpdir, $collection)) 
	    if -d &util::filename_cat ($tmpdir, $collection);
    }
}

# gets all the right bits of a built collection from 
# $GSDLHOME/collect and copies them to $collect_dir
# returns 1 if successful, 0 if not
sub get_built_collection {
    my ($colname, $collect_dir) = @_;

    my $from_dir = &util::filename_cat ($ENV{'GSDLHOME'}, "collect", $colname);
    if (!-d $from_dir) {
	print STDERR "\nERROR: No collection at $from_dir\n";
	return 0;
    }
    
    my $to_dir = &util::filename_cat ($collect_dir, $colname);
    mkdir ($to_dir, 0777) unless -d $to_dir;
    
    # get the built indexes
    my $index_dir = &util::filename_cat ($from_dir, "index");
    if (-d $index_dir) {
	# if build.cfg exists we'll assume collection is built ok
	if (-e &util::filename_cat ($index_dir, "build.cfg")) {
	    &util::cp_r ($index_dir, $to_dir);
	} else {
	    print STDERR "\nERROR: no build.cfg at $index_dir (collection not built?)\n";
	    rmdir ($to_dir);
	    return 0;
	}
    } else {
	print STDERR "\nERROR: collection at $from_dir appears unbuilt (no index directory)\n";
	return 0;
    }
    &util::cp_r ($index_dir, $to_dir);
    
    # get the collect.cfg file
    mkdir (&util::filename_cat($to_dir, "etc"), 0777);
    &util::cp (&util::filename_cat($from_dir, "etc", "collect.cfg"),
	       &util::filename_cat($to_dir, "etc"));

    # get the images directory
    my $from_images = &util::filename_cat ($from_dir, "images");
    &util::cp_r ($from_images, $to_dir) if -d $from_images;
    
    # make sure there aren't any CVS directories laying around
    &remove_cvs_dirs ($to_dir);

    &edit_collect_cfg (&util::filename_cat($to_dir, "etc", "collect.cfg"), $colname);
    &create_version_file (&util::filename_cat($to_dir, "etc", "VERSION"), 0);

    return 1;
}

# gets all the right bits of an unbuilt collection from 
# $GSDLHOME/collect and copies them to $collect_dir
# returns 1 if successful, 0 if not
sub get_unbuilt_collection {
    my ($colname, $collect_dir) = @_;

    my $from_dir = &util::filename_cat ($ENV{'GSDLHOME'}, "collect", $colname);
    if (!-d $from_dir) {
	print STDERR "\nERROR: No collection at $from_dir\n";
	return 0;
    }
    
    my $to_dir = &util::filename_cat ($collect_dir, $colname);
    mkdir ($to_dir, 0777) unless -d $to_dir;

    # get the unbuilt data (either import or archives)
    my $have_import = 0;
    my $import_dir = &util::filename_cat ($from_dir, "import");
    if (-d $import_dir && opendir (IMPORTDIR, $import_dir)) {
	my @importfiles = readdir IMPORTDIR;
	closedir IMPORTDIR;
	# if the import directory isn't empty we'll assume everything's ok
	if (scalar @importfiles) {
	    $have_import = 1;
	    &util::cp_r ($import_dir, $to_dir);
	    mkdir (&util::filename_cat ($to_dir, "archives"), 0777);
	}
    }
    if (!$have_import) {
	# see if we've got archives then (check for archives.inf)
	if (-e &util::filename_cat ($from_dir, "archives", "archives.inf")) {
	    &util::cp_r (&util::filename_cat($from_dir, "archives"), $to_dir);
	} else {

	    print STDERR "$collection collection appears to have no valid\n";
	    print STDERR "import or archives data\n";
	    &util::rm_r ($to_dir);
	    return 0;
	}
    }
    
    # get the etc directory
    &util::cp_r (&util::filename_cat ($from_dir, "etc"), $to_dir);

    # get the perllib directory
    &util::cp_r (&util::filename_cat ($from_dir, "perllib"), $to_dir);

    # get the images
    &util::cp_r (&util::filename_cat ($from_dir, "images"), $to_dir);
    
    mkdir (&util::filename_cat ($to_dir, "building"), 0777);
    mkdir (&util::filename_cat ($to_dir, "index"), 0777);

    # make sure there aren't any CVS directories laying around
    &remove_cvs_dirs ($to_dir);

    &edit_collect_cfg (&util::filename_cat($to_dir, "etc", "collect.cfg"), $colname);
    my $preimported = 0;
    $preimported = 1 unless $have_import;
    &create_version_file (&util::filename_cat($to_dir, "etc", "VERSION"), $preimported);

    return 1;
}

sub create_changelog {

    my ($tag, $date, $file);

    my $datefile = &util::filename_cat ($ENV{'GSDLHOME'}, "DistDates");
    if ($changelogdate !~ /\w/) {
	# get date from $GSDLHOME/DistDates (and update DistDates)
	open (DATES, $datefile) || die "can't open $datefile\n";
	my $line = "";
	while (defined ($line = <DATES>)) {
	    if ($line =~ /\w/) {
		($tag, $date) = $line =~ /^(\S+)\t(.*)$/;
		$changelogdate = $date unless ($tag eq $cvs_tag);
	    }
	    $file .= $line;
	}
	close DATES;
    }

    if ((!defined $tag) || ($tag ne $cvs_tag)) {
	open (DATES, ">$datefile") || die;
	print DATES $file if defined $file;
	print DATES "$cvs_tag\t" . `date`;
	close DATES;
    }

    print STDERR "Creating ChangeLog from $changelogdate to most recent\n";

    chdir($ENV{'GSDLHOME'});
    my $cmd = "cvs2cl.pl -P -F trunk -r -S -l \"-d'$changelogdate<tomorrow'\"";
    system ($cmd);
}

# edits the gsdlsite.cfg file to set GSDLHOME
# makes a copy of the initial version of gsdlsite.cfg in
# /tmp (if it doesn't exist already).
sub edit_gsdlsite {
    my ($gsdlhome) = @_;

    my $gsdlsite_file = &util::filename_cat ($tmpdir, "gsdl", "cgi-bin", "gsdlsite.cfg");
    my $tmp_gsdlsite_file = &util::filename_cat ($tmpdir, "gsdlsite.cfg");

    if (-e $tmp_gsdlsite_file) {
	&util::cp ($tmp_gsdlsite_file, $gsdlsite_file);
    } else {
	&util::cp ($gsdlsite_file, $tmp_gsdlsite_file);
    }

    open (GSDLSITE, $gsdlsite_file) || die;
    my $line = ""; my $file = ""; my $found = 0;
    while (defined ($line = <GSDLSITE>)) {
	if ($line =~ s/\*\*GSDLHOME\*\*/$gsdlhome/g) {
	    $found = 1;
	}
	$file .= $line;
    }
    close GSDLSITE;

    if (!$found) {
	die "ERROR: $gsdlsite_file contains no **GSDLHOME** string\n";
    }

    open (GSDLSITE, ">$gsdlsite_file") || die;
    print GSDLSITE $file;
    close GSDLSITE;
}

# currently just checks that iconcollection fields are correct and that
# creator and maintainer fields are set to greenstone@cs.waikato.ac.nz
sub edit_collect_cfg {
    my ($collect_cfg_file, $collname) = @_;

    open (FILE, $collect_cfg_file) || die "couldn't open $collect_cfg_file\n";
    my $line = ""; my $file = "";
    while (defined ($line = <FILE>)) {
	$line =~ s/^((?:creator|maintainer)\s+\"?)\w+@\w+(\"?)/$1greenstone\@cs.waikato.ac.nz$2/ix;
	$line =~ s/^(collectionmeta\s+\"?iconcollection(?:small)?\"?\s+\"?).*?($collname\/images\/)/$1_httpprefix_\/collect\/$2/ix;
	$file .= $line;
    }
    close FILE;

    open (FILE, ">$collect_cfg_file") || die;
    print FILE $file;
    close FILE;
}

# the "collection release" is the version of the content - it should only change if the 
# collections content is changed in some way
# the "build version" is the build version of Greenstone when the collection distribution
# was created

# recent build version changes were:
# 2.0 - 2.1: plugins were altered to take input_encoding (and other) options. GB plugins
#            were removed. numwords and numsections statistics were added.
#            All build version 2.0 collections other than those that suddenly required
#            the input_encoding option (i.e. Arabic and Chinese) may still be built and
#            viewed using build verion 2.1 software (numwords and numsections won't be
#            available though).

sub create_version_file {
    my ($version_file, $preimported) = @_;
    
    open (FILE, ">$version_file") || die;
    
    print FILE "collection release: 1.1\n";
    print FILE "build version: 2.1\n";
    print FILE "pre-imported\n" if $preimported;

    close FILE;
}

# simply recurses directory structure beginning at $dir
# and deletes any CVS administrative directories it comes 
# across
sub remove_cvs_dirs {
    my ($dir) = @_;

    if (!-d $dir) {return;}

    opendir (DIR, $dir) || die;
    my @files = readdir DIR;
    closedir DIR;

    foreach $file (@files) {
	next if $file =~ /^\.\.?$/;
	my $fullpath = &util::filename_cat ($dir, $file);
	if (-d $fullpath) {
	    if ($file eq "CVS") {
		&util::rm_r ($fullpath);
	    } else {
		&remove_cvs_dirs ($fullpath);
	    }
	}
    }
}

# mode is 0 to create .zip and .tgz, 1 to create .zip only, and 2 to create
# .tgz only
sub zip {
    my ($zip_to, $zip_from, $dir, $mode) = @_; 

    chdir ($dir);

    if ($mode != 2) {
	my $to = $zip_to . ".zip";
	unlink ($to) if -e $to;
	print STDERR "zipping up $to\n";
	`zip -r $to $zip_from`;
    }
    if ($mode != 1) {
	my $to = $zip_to . ".tgz";
	unlink ($to) if -e $to;
	print STDERR "tarring and gzipping $to\n";
	print STDERR "tar cvzf $to $zip_from\n";
	system ("tar cvzf $to $zip_from");
    }
}

sub edit_files {
    # edit VERSION file
    my $version_file = &util::filename_cat ($tmpdir, "gsdl", "etc" , "VERSION");
    open (VERSION, $version_file) || die;
    my $found = 0; my $line = ""; my $file = "";
    while (defined ($line = <VERSION>)) {
	if ($line =~ s/(gsdl version: )x\.xx/$1$version_num/) {
	    $found ++;
	} elsif ($line =~ s/(cvs tag: )gsdl-x_xx-distribution/$1$cvs_tag/) {
	    $found ++;
	}
	$file .= $line;
    }
    close VERSION;
    if ($found != 2) {
	die "error while editing $version_file\n";
    } 
    
    open (VERSION, ">$version_file") || die;
    print VERSION $file;
    close VERSION;
    
    # edit fnord.cpp
    my $fnord_file = &util::filename_cat ($tmpdir, "gsdl", "src", "w32server", "fnord.cpp");
    open (FNORD, $fnord_file) || die;
    $found = 0; $line = ""; $file = "";
    while (defined ($line = <FNORD>)) {
	if ($line =~ s/(\#define VERSIONSTRING \"version )x\.xx(\")/$1$version_num$2/) {
	    $found ++;
	}
	$file .= $line;
    }
    close FNORD;
    if (!$found) {
	die "error while editing $fnord_file\n";
    } 
    
    open (FNORD, ">$fnord_file") || die;
    print FNORD $file;
    close FNORD;
}

END {
    # remove any temporary files we created
    print STDERR "\ndeleting temporary files\n";
    my $tmp_gsdlsite_file = &util::filename_cat ($tmpdir, "gsdlsite.cfg");
    if (-e $tmp_gsdlsite_file) {
	print STDERR "$tmp_gsdlsite_file\n";
	unlink($tmp_gsdlsite_file);
    }
    my $tmp_gsdlhome = &util::filename_cat ($tmpdir, "gsdl");
    if (-d $tmp_gsdlhome) {
	print STDERR "$tmp_gsdlhome\n";
	&util::rm_r ($tmp_gsdlhome);
    }
}
